#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

__all__=["prettifyXHTMLFile", "formatZoteroRefsFile", "show_msgbox", "writeFile2Epub", "getUserSelectedFileName", "fileSelector", "copyRefs2SelectedSigilFile", "importZoteroFile"]


import os, os.path, sys, shutil, inspect, re
import options
from tempfile import mkdtemp 
import time

from lxml import etree
from io import StringIO, BytesIO

import tkinter as tk
import tkinter.messagebox as mbox
import tkinter.filedialog as fdbox

try:
    from sigil_bs4 import BeautifulSoup, Tag, NavigableString
except:
    from bs4 import BeautifulSoup, Tag, NavigableString
	
	
def prettifyXHTMLFile(wdir, file):
    """ Reformats, cleans up and prettifies the html file
        by adding appropriate vertical spacing and indents. 
    """
    print('\n>>> In prettifyXHTML...')
    output = os.path.join(wdir, 'prettify.xhtml')
    file = os.path.join(wdir, file)
    outfp = open(output, 'wt', encoding=  'utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
        
            line = line.replace('</body>', '')
            line = line.replace('</html>', '')
            if line.strip == '</body></html>':
                line = ''
        
            if 'xhtml11.dtd' in line or \
                line.strip().startswith('<?xml') or \
                line.strip().startswith('<!DOCTYPE') or \
                line.strip().startswith('<html') or \
                line.strip().startswith('</html>') or \
                line.strip().startswith('<title>') or \
                line.strip().startswith('<meta') or \
                line.strip().startswith('<head>') or \
                line.strip().startswith('</head>') or \
                line.strip().startswith('<body') or \
                line.strip().startswith('</body>'):
                outfp.write(line)
                continue      
            
            if line.strip() == '':
                continue
                
            if '.</p>' in line:
                line = line.replace('.</p>', '</p>') 
                
            if line.strip().startswith('</div></body>'):
                line = line.replace('</div></body>', '</body>')   
            
            outfp.write(line + '\n') 
            print('>>> In prettifyXHTML...' + line)
    outfp.write('\n\n</body>\n</html>')     
    outfp.close()
    os.remove(file)
    shutil.copy(output, file)
    return(0)	    
    
def formatZoteroRefsFile(wdir, file):
    """ Reformats the Zotero html file:
        *converts all div tags to p tags,
        *removes all lone span formatting        
        *formats all p tags using inline styles,
        *implements user formatting choices,
        *converts all <i> tags to span style italics,
    """
    print('\n>>> In formatZoteroRefsFile()...') 
    file = os.path.join(wdir, file)
    output = os.path.join(wdir, 'format_refs.xhtml')
    outfp = open(output, 'w', encoding=  'utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
    
        numbers = 1
        for line in infp:
        
        
        # remove html xmlns + meta headers 
            # and tail - leave just the data
            if 'xhtml11.dtd' in line or \
                line.strip().startswith('<?xml') or \
                line.strip().startswith('<!DOCTYPE') or \
                line.strip().startswith('<html') or \
                line.strip().startswith('</html>') or \
                line.strip().startswith('<title>') or \
                line.strip().startswith('<meta') or \
                line.strip().startswith('<head>') or \
                line.strip().startswith('</head>') or \
                line.strip().startswith('<body') or \
                line.strip().startswith('</body>'):
                continue
            
            if '<div>' in line:
                numbers += 1
            
            line = line.replace('<i>', '!~')
            line = line.replace('</i>', '^~') 
            
            # convert div to p tags
            html = BeautifulSoup(line, 'html.parser') 
            
            div_count = 0
            for dtag in html.find_all('div'):
                div_count += 1
                
                
            if div_count > 1 or '<ol>' in line or '<li>' in line or line.strip().startswith('(') :
                msg = "You are only allowed to use CMOS's 'author-date' style with this plugin. Please try again"
                print('\n>>> Error: ' + msg + '\n\nAbort plugin...')
                show_msgbox('Incompatible Zotero Style', msg, msgtype='error')
                options.SYS_EXIT = True
                shutil.rmtree(wdir, ignore_errors=True)
                return(-1)                         
            
            for dtag in html.find_all('div'):        
                if dtag.get_text() == None:
                    dtag.attrs = {}
                    dtag.unwrap()
                    continue                    
                else:
                    print('\n>>> Swap div for p tag...' + str(html.div))
                    dtag.attrs = {}
                    dtag.name = 'p'
                    if options.REMOVE_HANGING_INDENTS == True:
                        dtag['style'] = 'text-align: left;text-indent: 0;margin-left: 1.5em;margin-top: 0.8333em;' 
                    else:                        
                        dtag['style'] = 'text-align: left;text-indent: -2em;margin-left: 2em;margin-bottom: 0.833em;'
                    if options.USE_TEXT_JUSTIFY == True:
                        dtag['style'] = str(dtag['style']).replace('text-align: left;', 'text-align: justify;')           
                    line = str(html)
                       
            
            # remove all lone span formatting        
            if line.strip().startswith('<span'):
                soup = BeautifulSoup(line, 'html.parser') 
                print('\n>>> Removing lone spans...')                
                soup.span.decompose()
                line = str(soup)    
                    
            if '!~' in line and '^~' in line:
                if options.REMOVE_ITALICS == True:       
                    print('\n>>> In remove italics...')
                    line = line.replace('!~','')
                    line = line.replace('^~','')
                    print("After remove italics..." + line)                     

            # remove all hyperlink formatting
            if options.REMOVE_HYPERLINK_FORMAT == True:    
                soup = BeautifulSoup(line, 'html.parser')
                if '<a' in line and 'href' in line:
                    print('\n>>> Removing link formatting...')
                    atag = soup.find('a')
                    atag.attrs = {}
                    atag.unwrap()
                    line = str(soup)                   
            
            # delete all hyperlinks 
            if options.DELETE_HYPERLINKS == True:        
                soup = BeautifulSoup(line, 'html.parser')
                if '<a' in line and 'href' in line:
                    print('\n>>> Delete links...')
                    atag = soup.find('a')
                    atag.clear()
                    atag.decompose()
                    line = str(soup)
                    
                
            print('\n>>> In Add Numbering -- line...' + line)
            soup = BeautifulSoup(line, 'html.parser')
            if soup.find('a') != None:
                # remove italics
                extag = soup.p.a.extract()
                pstring = soup.p.get_text()
                pstring = str(numbers) + '. ' + pstring
                soup.p.string = pstring
                soup.p.append(extag)                    
                line = str(soup)
                numbers = numbers + 1
            
            print('\n>>> In Zotero format refs - Write before...' + line)
            line = line.replace('!~', '<i>')
            line = line.replace('^~', '</i>')                
            print('>>> In Zotero format refs - Write after...' + line)
            outfp.write(line)
             
    outfp.close()
    os.remove(file)
    os.rename(output, file)
    return(0)    
                                
def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
    
def writeFile2Epub(bk, wdir, s_id, file): 
    print('\n>>> In write2Epub id, file...' + s_id + ' ' + file)
    file = os.path.join(wdir, file)
    with open(file, 'rt', encoding='utf-8') as fp:
        data = fp.read()           
        bk.writefile(s_id, data)
        
    return(0)    
 
def getUserSelectedFileName(bk, wdir):
    print('\n>>> In  getUserSelectedFileName...')
    """ Obtains the name of the file selected by the 
        user in Sigil's Book Browser, check it and  
        copy it to the work dir.
    """
    # get user selected file
    files = []
    id = ''
    count= 0
    for (id_type, ident) in bk.selected_iter():
        print('>>> In selected file, id is ' + ident)
        count += 1
        href = bk.id_to_href(ident)
        files.append(bk.href_to_basename(href))
        
    # check if too many files were selected.    
    if len(files) > 1:
       msg = 'Too many files selected!! You must select only one file at a time to add specific endnotes or refs to that particular file. Please try again'
       print('\n>>> Critical Error: ' + msg + '\n\n>>> Abort plugin...')
       show_msgbox('File Selection Error', msg, msgtype='error')
       options.SYS_EXIT = True
       shutil.rmtree(wdir, ignore_errors=True)
       return(-1)               
    
    # check that a file has been selected else error exit
    try:
        file = options.SELECTED_SIGIL_FILE = files[0]
    except:
        msg = "You haven't selected a file from the 'Text' directory of Sigil's Book Browser. Please try again"
        print('>>> Critical Error: ' + msg + "\n\n>>> Abort plugin...\n\n")
        show_msgbox('File Selection Error', msg, msgtype='error')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(-1) 

    # check selected file type
    print('\n>>> Selected file...' + file)
    print(' >>> Selected files[0]...' + file[0])    
    if not file.endswith('.xhtml') and not file.endswith('.html'):
        msg = "Your selected Sigil file is the wrong file type. You should select only files that have the '.xhtml' or '.html' extension from the 'Text' directory in the Book Browser. Please try again"
        print('>>> Critical Error: ' + msg + "\n\n>>> Abort plugin...\n\n")
        show_msgbox('Incorrect File Type', msg, msgtype='error')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(-1)     
    
    # copy selected file to the work dir
    file = os.path.join(wdir, file)
    with open(file, 'wt', encoding='utf-8') as outfp:
        data = bk.readfile(ident)
        outfp.writelines(data)
       
    return(ident, files[0]) 
    
def fileSelector(bk):
    """ Gets the file path of the Zotero html file 
        containg the ref items
    """
    prefs = bk.getPrefs()
    localRoot = tk.Tk()
    localRoot.withdraw()
    
    # borrowed from Kevin's code with thanks
    if sys.platform.startswith('darwin'):
        # localRoot is is an empty topmost root window that is hidden by withdrawing it
        # but localRoot needs to be centred, and lifted and focus_force used
        # so that its child dialog will inherit focus upon launch
        localRoot.overrideredirect(True)
        # center on screen but make size 0 to hide the empty localRoot
        w = localRoot.winfo_screenwidth()
        h = localRoot.winfo_screenheight()
        x = int(w/2)
        y = int(h/2)
        localRoot.geometry('%dx%d+%d+%d' % (0, 0, x, y))
        localRoot.deiconify()
        localRoot.lift()
        localRoot.focus_force()
        
    file_opt = {}
    file_opt['parent'] = localRoot
    file_opt['title']= 'Select Zotero Export File'
    file_opt['defaultextension'] = '.html'
    file_opt['initialdir'] = prefs['user_dir_path']
    file_opt['multiple'] = False
    file_opt['filetypes'] = [('Zotero files', ('*.html'))]
    
    filename = fdbox.askopenfilename(**file_opt)
    localRoot.quit()
    return(filename)
    
def copyRefs2SelectedSigilFile(wdir, zot_file, sig_file):
    """ Copies Zotero note/ref items to the selected 
        Sigil file. Also ensures that the copy will work for 
        a sig_file with or without a div section.
    """
        
    print('\n>>> In copyRefs2SelectedSigilFile...')
    zot_file = os.path.join(wdir, zot_file)
    sig_file = os.path.join(wdir, sig_file)
    
    # saves body and body.div attrs
    output = os.path.join(wdir, 'sig_refs1.html')
    outfp = open(output, 'w', encoding='utf-8')
    html = open(sig_file, 'rt', encoding= 'utf8').read()
    
    has_div_section = False
    body_attrs_save = []
    div_attrs_save = []
    soup = BeautifulSoup(html, 'html.parser')
   
    # test for a div section and save 
    # body and div attrs for later use
    if len(soup.html.body.contents) > 1:
        if soup.html.body.contents[1].name == 'div':
            has_div_section = True
            dtag = soup.html.body.find('div')
            div_attrs_save = soup.html.body.div.attrs  
            body_attrs_save = soup.html.body.attrs
            soup.html.body.div.unwrap()  
     
    outfp.writelines(str(soup))
    outfp.close()
    os.remove(sig_file)
    os.rename(output, sig_file)

    #get the zotero ref/notes data
    output = os.path.join(wdir, 'zot_data.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(zot_file, 'rt', encoding ='utf-8') as infp:
        for line in infp:
            if '<p' in line:
                outfp.write(line)
    outfp.close()
    
    # append the zotero data to the sig data
    output = os.path.join(wdir, 'temp.xhtml')
    zot_html = os.path.join(wdir, 'zot_data.html')    
    outfp = open(output, 'w', encoding='utf-8')
    zot_data = open(zot_html, 'rt', encoding='utf-8').read()
    sig_data = open(sig_file, 'rt', encoding='utf-8').read() 
        
    outfp.writelines(sig_data + zot_data)
    
    outfp.close() 
    os.remove(zot_html)
    os.remove(sig_file)
    os.rename(output, sig_file)
    
    # remove all </body> and </html> tags and re-insert properly at eof
    output = os.path.join(wdir, 'sig_temp.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(sig_file, 'rt', encoding ='utf-8') as infp:
        for line in infp:
            
            if  line.strip().startswith('</body>') or \
                line.strip().startswith('</html>'):
                line = ''   
            outfp.write(line) 

    outfp.write('\n</body>\n</html>')
    outfp.close()
    os.remove(sig_file)
    os.rename(output, sig_file)    
    
    # now add back the div attrs and body 
    # attrs to the selected sigil file
    output = os.path.join(wdir, 'sig_temp2.html')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(sig_file, 'rt', encoding ='utf-8').read()
    
    soup = BeautifulSoup(html, 'html.parser')
    if has_div_section == True:
        soup.html.body.wrap(soup.new_tag('div'))
        soup.html.div.body.unwrap()
        soup.html.div.wrap(soup.new_tag('body'))
        if body_attrs_save != []:
            soup.html.body.attrs = body_attrs_save
        if div_attrs_save != []:
            soup.html.body.div.attrs = div_attrs_save    
    
    outfp.writelines(str(soup))
    outfp.close()
    os.remove(sig_file)
    os.rename(output, sig_file) 
    
    # sort out minor div formatting errors
    output = os.path.join(wdir, 'sig_temp4.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(sig_file, 'rt', encoding ='utf-8') as infp:
        for line in infp:
            if '<body' in line and '<div' in line:
                line = line.replace('<div', '\n<div')              
            outfp.write(line) 
    outfp.close()
    os.remove(sig_file)
    shutil.copy(output, sig_file)     
    return(0)    
    
def importZoteroFile(bk, wdir):
    """ Copies the Zotero export file to the work 
        dir and save the user dir path to prefs
    """
    prefs = bk.getPrefs()
    zot_file_path = fileSelector(bk)
    zot_file = os.path.basename(zot_file_path)
    new_zot_file = ''
    # if zotero file path does not exist - exit
    if zot_file_path == '' or not os.path.exists(zot_file_path):
        print('\n>>> No input file selected!\n\nAbort plugin...')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(0)
    else:#copy zotero file to work dir 
        new_zot_file = os.path.join(wdir, 'Zotero_Refs.html')
        shutil.copy(zot_file_path, new_zot_file)
        folder, _ = os.path.split(zot_file_path)    
        if folder != "":
            prefs['user_dir_path'] = folder
            bk.savePrefs(prefs)
    return(os.path.basename(new_zot_file))            
    
    
    
    