#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

__all__=["prettifyXHTMLFile", "formatZoteroRefsFile", "show_msgbox", "writeFile2Epub", "getUserSelectedFileName", "fileSelector", "copyRefs2SelectedSigilFile", "importZoteroFile", "svgAttributes2CamelCase"]


import os, os.path, sys, shutil, inspect, re, stat
import options
from tempfile import mkdtemp 

from lxml import etree
from io import StringIO, BytesIO

import tkinter as tk
import tkinter.messagebox as mbox
import tkinter.filedialog as fdbox

try:
    from sigil_bs4 import BeautifulSoup, Tag, NavigableString
except:
    from bs4 import BeautifulSoup, Tag, NavigableString
	
	
def prettifyXHTMLFile(wdir, file):
    """ Reformats, cleans up and prettifies the html file
        by adding appropriate vertical spacing and indents. 
    """
    
    # converts vb and par svg attributes to camel case
    svgAttributes2CamelCase(wdir, file)
    
    print('\n>>> In prettifyXHTML...')
    output = os.path.join(wdir, 'prettify.xhtml')
    file = os.path.join(wdir, file)
    outfp = open(output, 'wt', encoding=  'utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
        
            line = line.replace('</body>', '')
            line = line.replace('</html>', '')
            if line.strip == '</body></html>':
                line = ''
        
            if 'xhtml11.dtd' in line or \
                line.strip().startswith('<?xml') or \
                line.strip().startswith('<!DOCTYPE') or \
                line.strip().startswith('<html') or \
                line.strip().startswith('</html>') or \
                line.strip().startswith('<title>') or \
                line.strip().startswith('<meta') or \
                line.strip().startswith('<head>') or \
                line.strip().startswith('</head>') or \
                line.strip().startswith('<body') or \
                line.strip().startswith('</body>'):
                outfp.write(line)
                continue      
            
            if line.strip() == '':
                continue
                
            if '.</p>' in line:
                line = line.replace('.</p>', '</p>')  
            
            outfp.write(line + '\n') 
            print('>>> In prettifyXHTML...' + line)
            
    outfp.write('\n\n</body>\n</html>')     
    outfp.close()
    os.remove(file)
    shutil.copy(output, file)
    return(0)	    
    
def formatZoteroRefsFile(wdir, file):
    """ Reformats the Zotero html file:
        *remove all xmlns + meta headers 
        *implements user formatting choices,
        *converts all <i> tags to span style italics
        *adds appropriate ref spacing
    """
    print('\n>>> In formatZoteroRefsFile()...') 
    file = os.path.join(wdir, file)
    output = os.path.join(wdir, 'format_refs.xhtml')
    outfp = open(output, 'w', encoding=  'utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
    
        numbers = 0
        for line in infp:
            
            # remove html xmlns + meta headers 
            # and tail - leaving just the data
            if 'xhtml11.dtd' in line or \
                line.strip().startswith('<?xml') or \
                line.strip().startswith('<!DOCTYPE') or \
                line.strip().startswith('<html') or \
                line.strip().startswith('</html>') or \
                line.strip().startswith('<title>') or \
                line.strip().startswith('<meta') or \
                line.strip().startswith('<head>') or \
                line.strip().startswith('</head>') or \
                line.strip().startswith('<body') or \
                line.strip().startswith('</body>'):
                continue                 
            
            if '<div' in line:
                msg = "You are only allowed to use Zotero's Chicago Style 'fullnote' and 'note' styles with 'Notes' also selected using this plugin. Please try again"
                print('\n>>> Error: ' + msg + '\n\nAbort plugin...')
                show_msgbox('Incompatible Zotero Style', msg, msgtype='error')
                options.SYS_EXIT = True
                shutil.rmtree(wdir, ignore_errors=True)
                return(-1) 
                
            if '<ol>' in line or '</ol>' in line:
                continue
                    
            if '<li>' in line:
                numbers += 1
                
                # reformat the note list
                soup = BeautifulSoup(line, 'html.parser')  
                
                # get rid of <li> tags and numbering
                if '<li>' in line:
                    soup.li.name = 'p'             
                    print('\n>>> In <p> formatting...') 
                    soup.p['style'] = 'text-align:left;margin-top: 0.8333em;' 
                    line = str(soup)                     
                    
                    if options.USE_SMALL_FONT == True: 
                        soup.p['style'] = soup.p['style'] + 'font-size: 0.875em;'
                        line = str(soup) 
                        
                    if options.USE_TEXT_JUSTIFY == True:                         
                        line = str(soup) 
                        line = line.replace('text-align:left;','text-align:justify') 
                    
                # remove or convert italics
                if '<i' in line:
                    soup = BeautifulSoup(line, 'html.parser')  
                    if options.REMOVE_ITALICS == True:    
                        print('\n>>> Removing <i> tags...')
                        line = line.replace('<i>', '')
                        line = line.replace('</i>', '')    
                        
                # remove all hyperlink formatting
                if options.REMOVE_HYPERLINK_FORMAT == True:
                    soup = BeautifulSoup(line, 'html.parser')                  
                    if '<a' in line and 'href' in line:
                        print('\n>>> Removing link formatting...')
                        atag = soup.find('a')
                        atag.attrs = {}
                        atag.unwrap()                  
                        line = str(soup)
                
                # delete all hyperlinks 
                if options.DELETE_HYPERLINKS == True:        
                    soup = BeautifulSoup(line, 'html.parser')
                    if '<a' in line and 'href' in line:
                        print('\n>>> Delete links...')
                        atag = soup.find('a')
                        atag.clear()
                        atag.decompose()
                        line = str(soup)

                # add consecutive numbering to all ref/notes    
                if options.ADD_NUMBERING == True:
                    print('\n>>> In Add Numbering -- line...' + line)
                    soup = BeautifulSoup(line, 'html.parser')
                    extag = soup.p.a.extract()
                    pstring = soup.p.get_text()
                    pstring = str(numbers) + '. ' + pstring
                    soup.p.string = pstring
                    soup.p.append(extag)                    
                    line = str(soup)
                    
                # add consecutive superscript numbering         
                if options.ADD_SUPER_NUMBERING == True:
                    soup = BeautifulSoup(line, 'html.parser')
                    soup.p.insert(0, soup.new_tag('sup'))
                    soup.p.sup['style'] = 'font-size: 0.75em;'
                    soup.p.sup.string = str(numbers) + ' '                      
                    line = str(soup)
                    
                # add superscript numbering with brackets          
                if options.ADD_SUPER_WITH_BRACKETS == True:
                    print('\n>>> In SUPER WITH BRACKETS -- line...' + line)
                    soup = BeautifulSoup(line, 'html.parser')
                    soup.p.insert(0, soup.new_tag('sup'))
                    soup.p.sup['style'] = 'font-size: 0.75em;'
                    soup.p.sup.string = '[' + str(numbers) + '] '                                 
                    line = str(soup)
                    
                if '</p></body>' in line:
                    line = line.replace('</p></body>', '</p>\n\n</body>')   
                
                print('>>> In Zotero format refs - Write...' + line)
                line = line.replace(', .</p>', '.</p>')          
                outfp.write(line)
                       
            
    outfp.close()
    os.remove(file)
    shutil.copy(output, file)
    return(0)    
                                
def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
    
def writeFile2Epub(bk, wdir, s_id, file): 
    print('\n>>> In write2Epub id, file...' + s_id + ' ' + file)
    file = os.path.join(wdir, file)
    with open(file, 'rt', encoding='utf-8') as fp:
        data = fp.read()           
        bk.writefile(s_id, data)
        
    return(0)    
 
def getUserSelectedFileName(bk, wdir):
    """ Obtains the name of the file selected by the 
        user in Sigil's Book Browser, runs checks and  
        then copies it to the work dir.
    """
    # get user selected file
    files = []
    id = ''
    count= 0
    for (id_type, ident) in bk.selected_iter():
        print('>>> In selected file, id is ' + ident)
        count += 1
        href = bk.id_to_href(ident)
        files.append(bk.href_to_basename(href))
        
    # check if too many files were selected.    
    if len(files) > 1:
       msg = 'Too many files selected!! You must select only one file at a time to add specific endnotes to that particular file. Please try again'
       print('\n>>> Critical Error: ' + msg + '\n\n>>> Abort plugin...')
       show_msgbox('File Selection Error', msg, msgtype='error')
       options.SYS_EXIT = True
       shutil.rmtree(wdir, ignore_errors=True)
       return(-1)               
    
    # check that a file has been selected else error exit
    try:
        file = options.SELECTED_SIGIL_FILE = files[0]
    except:
        msg = "You haven't selected a file from the 'Text' directory of Sigil's Book Browser. Please try again"
        print('>>> Critical Error: ' + msg + "\n\n>>> Abort plugin...\n\n")
        show_msgbox('File Selection Error', msg, msgtype='error')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(-1) 

    # check selected file type    
    if not file.endswith('.xhtml') and not file.endswith('.html'):
        msg = "Your selected Sigil file is the wrong file type. You should select only files that have the '.xhtml' or '.html' extension from the 'Text' directory in Sigil's Book Browser. Please try again"
        print('>>> Critical Error: ' + msg + "\n\n>>> Abort plugin...\n\n")
        show_msgbox('Incorrect File Type', msg, msgtype='error')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(-1)  
    
    # copy selected file to the work dir
    file = os.path.join(wdir, file)
    with open(file, 'wt', encoding='utf-8') as outfp:
        data = bk.readfile(ident)
        outfp.writelines(data)
       
    return(ident, file) 
    
def fileSelector(bk):
    """ Gets the file path of the Zotero html file 
        containg the ref items
    """
    prefs = bk.getPrefs()
    localRoot = tk.Tk()
    localRoot.withdraw()
    
    # gratefully borrowed from Kevin's code
    if sys.platform.startswith('darwin'):
        # localRoot is is an empty topmost root window that is hidden by withdrawing it
        # but localRoot needs to be centred, and lifted and focus_force used
        # so that its child dialog will inherit focus upon launch
        localRoot.overrideredirect(True)
        # center on screen but make size 0 to hide the empty localRoot
        w = localRoot.winfo_screenwidth()
        h = localRoot.winfo_screenheight()
        x = int(w/2)
        y = int(h/2)
        localRoot.geometry('%dx%d+%d+%d' % (0, 0, x, y))
        localRoot.deiconify()
        localRoot.lift()
        localRoot.focus_force()
        
    file_opt = {}
    file_opt['parent'] = localRoot
    file_opt['title']= 'Select Zotero Export File'
    file_opt['defaultextension'] = '.html'
    file_opt['initialdir'] = prefs['user_dir_path']
    file_opt['multiple'] = False
    file_opt['filetypes'] = [('Zotero files', ('*.html'))]
    
    filename = fdbox.askopenfilename(**file_opt)
    localRoot.quit()
    return(filename)
    
def copyRefs2SelectedSigilFile(wdir, zot_file, sig_file):
    """ Copies Zotero note/ref items to the selected 
        Sigil file. Also ensures that the copy will work for 
        a sig_file with or without a div section.
    """
        
    print('\n>>> In copyRefs2SelectedSigilFile...')
    zot_file = os.path.join(wdir, zot_file)
    sig_file = os.path.join(wdir, sig_file)

    # append the zotero data to the sig data file
    output = os.path.join(wdir, 'temp.xhtml')  
    outfp = open(output, 'w', encoding='utf-8')
    zot_data = open(zot_file, 'rt', encoding='utf-8').read()
    sig_data = open(sig_file, 'rt', encoding='utf-8').read() 
        
    outfp.writelines(sig_data + '\n\n' + zot_data)
    
    outfp.close() 
    os.remove(zot_file)
    os.remove(sig_file)
    shutil.copy(output, sig_file)
    
    # remove all </body> and </html> tags and re-insert properly at eof
    output = os.path.join(wdir, 'sig_temp.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(sig_file, 'rt', encoding ='utf-8') as infp:
        for line in infp:
            
            if  line.strip().startswith('</body>') or \
                line.strip().startswith('</html>'):
                line = ''   
            outfp.write(line) 

    outfp.write('\n\n</body>\n</html>')
    outfp.close()
    os.remove(sig_file)
    os.rename(output, sig_file)  
    
def importZoteroFile(bk, wdir):
    """ Copies the Zotero export file to the work 
        dir and save the user dir path to prefs
    """
    prefs = bk.getPrefs()
    zot_file_path = fileSelector(bk)
    zot_file = os.path.basename(zot_file_path)
    new_zot_file = ''
    # if zotero file path does not exist - exit
    if zot_file_path == '' or not os.path.exists(zot_file_path):
        print('\n>>> No input file selected!\n\nAbort plugin...')
        options.SYS_EXIT = True
        shutil.rmtree(wdir, ignore_errors=True)
        return(0)
    else:#copy zotero file to work dir 
        new_zot_file = os.path.join(wdir, 'Zotero_Refs.html')
        shutil.copy(zot_file_path, new_zot_file)
        folder, _ = os.path.split(zot_file_path)    
        if folder != "":
            prefs['user_dir_path'] = folder
            bk.savePrefs(prefs)
    return(os.path.basename(new_zot_file))            
    
def svgAttributes2CamelCase(wdir, file):
       
    output = os.path.join(wdir, 'reformat.html')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()

    soup = BeautifulSoup(html, 'html.parser')
    
    for svg in soup.find_all('svg'):
    
        # reformat the par to camel case
        if svg.has_attr('preserveaspectratio'):
            par = svg['preserveaspectratio']
            del svg['preserveaspectratio']
            svg['preserveAspectRatio'] = par
        
        # reformat the vb to camel case
        if svg.has_attr('viewbox'):
            vb = svg['viewbox']
            del svg['viewbox']
            svg['viewBox'] = vb
               
    outfp.writelines(str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
    outfp.close()
    os.remove(file)
    os.rename(output, file)        
    return(0)            