/************************************************************************
 **
 **  Copyright (C) 2021  Kevin B. Hendricks, Stratford, Ontario, Canada
 **
 **  This file is part of Sigil.
 **
 **  Sigil is free software: you can redistribute it and/or modify
 **  it under the terms of the GNU General Public License as published by
 **  the Free Software Foundation, either version 3 of the License, or
 **  (at your option) any later version.
 **
 **  Sigil is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **
 **  You should have received a copy of the GNU General Public License
 **  along with Sigil.  If not, see <http://www.gnu.org/licenses/>.
 **
 ** Extracted and modified from:
 ** CSSTidy (https://github.com/csstidy-c/csstidy)
 **
 ** CSSTidy Portions Copyright:
 **   Florian Schmitz <floele@gmail.com>
 **   Thierry Charbonnel
 **   Will Mitchell <aethon@gmail.com>
 **   Brett Zamir <brettz9@yahoo.com>
 **   sined_ <sined_@users.sourceforge.net>
 **   Dmitry Leskov <git@dmitryleskov.com>
 **   Kevin Coyner <kcoyner@debian.org>
 **   Tuukka Pasanen <pasanen.tuukka@gmail.com>
 **   Frank W. Bergmann <csstidy-c@tuxad.com>
 **   Frank Dana <ferdnyc@gmail.com>
 **
 ** CSSTidy us Available under the LGPL 2.1
 ** You should have received a copy of the GNU Lesser General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **
 *************************************************************************/

#include <QtCore>
#include <QCoreApplication>
#include <QChar>
#include <QString>
#include <QVector>
#include <QStringList>
#include <QFile>
#include <QTextStream>
#include <QDebug>

#include "qCSSProperties.h"
#include "qCSSUtils.h"
#include "qCSSParser.h"


QString ReadUnicodeTextFile(const QString &fullfilepath)
{
    QFile file(fullfilepath);
    if (!file.open(QFile::ReadOnly)) {
        QString msg = fullfilepath + ": " + file.errorString();
        qDebug() << msg;
    }
    QTextStream in(&file);
    in.setCodec("UTF-8");
    in.setAutoDetectUnicode(true);
    return in.readAll();
}


int main(int argc, char *argv[])
{
    QCoreApplication app(argc, argv);
    QStringList arguments = QCoreApplication::arguments();

    QTextStream out(stdout, QIODevice::WriteOnly);
    CSSParser csst;

    if(arguments.size() > 1)
    {
        QString filein = arguments[1];
        if (!QFile::exists(filein)) 
        {
            out << "The file \"" << filein << "\" does not exist." << endl;
            return EXIT_FAILURE;
        }
        
        QString output_filename = "";
        QString css_file = ReadUnicodeTextFile(filein);

        // The interface

        // valid css levels are "CSS1.0", "CSS2.0", "CSS2.1", "CSS3.0" 
        csst.set_level("CSS3.0");

        // do the actual parsing
        csst.parse_css(css_file);

        // check for any parse errors
        QVector<QString> errors = csst.get_parse_errors();
        out << "Errors: " << errors.size() << endl;
        for(int i = 0; i < errors.size(); i++) {
            out << "  Error: " << errors[i] << endl;
        }

        // check for any parse warnings
        QVector<QString> warnings = csst.get_parse_warnings();
        out << "Warnings: " << warnings.size() << endl;
        for(int i = 0; i < warnings.size(); i++) {
            out << "  Warning: " << warnings[i] << endl;
        }

        // check for any parse information messages
        QVector<QString> infos = csst.get_parse_info();
        out << "Information: " << infos.size() << endl;
        for(int i = 0; i < infos.size(); i++) {
            out << "  Information: " << infos[i] << endl;
        }

        // get any @charset without having to walk the csstokens list
        QString cset = csst.get_charset();

        // get all @import without having to walk the csstokens list
        QVector<QString> imports = csst.get_import();

        // get any @namespace without having to walk the csstokens list
        QString ns = csst.get_namespace();
        
        // The possible token type are an enum:
        // enum token_type:
        //
        //     CHARSET   =  0
        //     IMPORT    =  1
        //     NAMESP    =  2
        //     AT_START  =  3
        //     AT_END    =  4
        //     SEL_START =  5
        //     SEL_END   =  6
        //     PROPERTY  =  7
        //     VALUE     =  8
        //     COMMENT   =  9
        //     CSS_END   = 10

        // now walk the sequence of parsed tokens
        // if you know the location of the token you want in the sequence (starting with 0)
        // simply pass start_ptr in get_next_token set to a valid starting point in the token sequence

        CSSParser::token atoken = csst.get_next_token();
        while(atoken.type != CSSParser::CSS_END) {
            QString ttype = csst.get_type_name(atoken.type);
            out << "Pos: " << atoken.pos << " Line: " << atoken.line << " Type: " << ttype  <<
                         "  Data: " << atoken.data << endl;
            atoken = csst.get_next_token();
        }

        // serialize CSS to stdout by default set tostdout to false to prevent
        QString cssout;
        // cssout = csst.serialize_css(false);
        cssout = csst.serialize_css();
        
        return EXIT_SUCCESS;
    }

    return EXIT_SUCCESS;
}
