#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = '2018, Steven Dick <kg4ydw@gmail.com>'

import time
from datetime import datetime
import re

from PyQt5 import QtGui, QtCore, QtWidgets
from PyQt5.QtWidgets import QAbstractItemView
from PyQt5.Qt import QMenu, QToolButton, QPixmap, Qt, QDialog, QProgressDialog
from calibre.gui2 import info_dialog
from calibre.gui2.actions import InterfaceAction

from calibre.ebooks.metadata.book.base import Metadata

import calibre_plugins.wikidata_gui.config as cfg
from calibre_plugins.wikidata_gui.modeldata import PropIndex as PI, EntityIndex as EI, IdIndex as II
from calibre_plugins.wikidata_gui.query import WikidataQueryBuilder as WQB

from calibre_plugins.wikidata_gui.importbooks import ImportBooks

from calibre_plugins.wikidata_gui.common_utils import (set_plugin_icon_resources, get_icon)

class WikidataAction(InterfaceAction):
    name = 'Wikidata merge'
    action_spec = ('wikidata merge', "images/wikidata.png", "Merge Wikidata into a book", None)
    # action_add_menu = False

    action_type = 'current'
    # accepts_drops = False
    ## books not in library can't have metadata
    dont_add_to = frozenset(['context-menu-device', 'menubar-device'] )
    popup_type = QToolButton.MenuButtonPopup

    def merge_wikidata(self, properties=None):
        # properties must be a valuelist string
        selected_cids, matching_cids, wdids = self.get_selected_books()
        if not wdids: return
        self.merge_actual(selected_cids, matching_cids, wdids, properties)

    def get_selected_books(self):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            info_dialog(self.gui,"Wikidata GUI","Please select books to update",show=True)
            return None, None, None
        view = self.gui.library_view
        db = self.gui.current_db.new_api
        #current_idx = view.currentIndex()
        selected_cids = set(view.get_selected_ids())
        matching_cids = set()
        wdids = {}
        for book_cid in selected_cids:
            mi = db.get_metadata(book_cid)
            ids = mi.get_identifiers()
            if 'wd' in ids:
                matching_cids.add(book_cid)
                wdids[ids['wd']] = book_cid
                # XXF this doesn't handle multiple books on same wdid well
        if not matching_cids:
            info_dialog(self.gui,"Wikidata GUI","No selected books included wikidata IDs.",show=True)
        return selected_cids, matching_cids, wdids

    def merge_actual(self, selected_cids, matching_cids, wdids, properties):
        # properties must be a valuelist string
        view = self.gui.library_view
        db = self.gui.current_db.new_api
        marked_ids = self.gui.current_db.data.marked_ids.copy()
        errors = 0
        error_cids = set()
        updated_cids = set()
        ## in test samples, the first 2-3 steps are really fast, don't bother
        #progress.setValue(2)
        #progress.setLabelText("Querying wikidata")
        wqb = WQB(cfg.prefs)
        if not properties:
            properties =  wqb.getprops() + wqb.getextids()
        starttime = time.time()
        results = wqb.doquery('bookprops', books=wqb.valueList('wd',wdids.keys()), properties=properties)
        endwikidatatime = time.time()
        wikidatatime = endwikidatatime-starttime
        print("Wikidata query took %f seconds"%wikidatatime)
        updated_cids = set()
        newtag_cids = set()
        # books properties and priorities by wdid
        found_cidp = {}
        foundcol = {}
        found_ids = {}
        # pick highest priority (lowest value)
        # filter and collect results
        # these currently come in unsorted but even sorted it's easier to just collect everything first
        if not results:
            if wqb.msg: 
                # XXX opportunity for expanded diagnostics in dialog
                info_dialog(self.gui,"Wikidata GUI",wqb.msg, show=True)
            else:
                info_dialog(self.gui,"Wikidata GUI","No results were returned from Wikidata. (M)",show=True)
            return
        # now we've got a better idea of how much work there is
        #counter=2
        #progress.setLabelText("Scanning wikidata results")
        #sizeguess = 3+len(results)*2
        #progress.setMaximum(sizeguess) 
        for result in results:
            #counter += 1
            #progress.setValue(counter)
            #if progress.wasCanceled(): break;
            if cfg.prefs['options']['debugquery']:
                print("result:"+" ".join(map(lambda i: "%s=%s"%(i[0],i[1]['value']), result.items())))
            book_wdid = wqb.getqid(result,'book')
            if book_wdid not in wdids:
                print("Can't get wdid from book: %s"%result['book']['value'])
                continue
            book_cid = wdids[book_wdid]
            if book_cid not in found_cidp:
                found_cidp[book_cid] = {} # YYY
            prop = wqb.getqid(result,'prop')
            if prop in cfg.prefs['ids']:
                if book_cid not in found_ids:
                    found_ids[book_cid] = { }
                # XX doesn't check for dups, keeps last found
                found_ids[book_cid][cfg.prefs['ids'][prop][II.idtag]] = result['value']['value']
                continue
            elif prop not in cfg.prefs['properties']:
                # this shouldn't happen anyway
                print("Found extra property %s for book %s, skipping"%(prop,book_cid))
                continue
            (v,t) = wqb.convertResultValue(result, 'value')
            if t=='wdid' and cfg.prefs['properties'][prop][PI.efilter]:
                if v not in cfg.prefs['tags']:
                    newtag_cids.add(book_cid) # XXF save property too?
                    # XXF collect new tag and label for future use?
                    continue
                if not cfg.prefs['tags'][v][EI.use]:
                    continue
                else:
                    label = cfg.prefs['tags'][v][EI.ylabel]
            else:
                label = result['valueLabel']['value']
            if prop in found_cidp[book_cid]:
                found_cidp[book_cid][prop].append(label)
            else:
                found_cidp[book_cid][prop] = [label]
            # check what columns need updating; remember highest priority found
            p =  cfg.prefs['properties'][prop]
            if book_cid not in foundcol:
                foundcol[book_cid] = { }
            if p[PI.colname] not in foundcol[book_cid] or p[PI.priority]>foundcol[book_cid][p[PI.colname]]:
                foundcol[book_cid][p[PI.colname]] = cfg.prefs['properties'][prop][PI.priority]
        scantime = time.time()
        print("Scanning found properties took %f seconds"%(scantime-endwikidatatime))
        # now decide which ones to add to each book
        #progress.setLabelText("Updating books")
        #if progress.wasCanceled(): return  # cleanup?
        progress = QProgressDialog("Merging metadata (%d properties for %d books)"%(len(results), len(found_cidp)),"cancel",0, len(found_cidp), self.gui)
        counter=0
        progress.setWindowModality(Qt.WindowModal)
        progress.setValue(counter)
        progress.show()
        sizeguess = counter+len(found_cidp)
        progress.setMaximum(sizeguess)
        for book_cid in found_cidp:
            counter += 1
            progress.setValue(counter)
            if progress.wasCanceled(): break;
            mi = db.get_metadata(book_cid)
            # XXX should make our own instead of copying it?
            changed = False
            for prop in found_cidp[book_cid]:
                p =  cfg.prefs['properties'][prop]
                if foundcol[book_cid][p[PI.colname]] < p[PI.priority]:
                    continue # higher priority found elsewhere
                # This is a bit of a hack to detect tag type fields, but seems to work. XXT
                v =mi.get(p[PI.colname])
                # special cases for overwrite
                if isinstance(v, list) or isinstance(v,set):  # merge or overwrite
                    t = set(found_cidp[book_cid][prop])
                    if not p[PI.overwrite]: # merge in old tags
                        t.update(v)
                    mi.set(p[PI.colname],t)
                    changed = True
                    #print("Updated %s to %s"%(p[PI.colname]," ".join(t)))
                    continue
                #elif not isinstance(v,datetime):
                #    print("merge %s: %s"%(p[PI.colname], type(v)))
                # special detection of "undefined" date
                if isinstance(v,datetime) and (p[PI.overwrite] or v==datetime(101,1,1,0,0,0,0,v.tzinfo)):
                    mi.set(p[PI.colname],found_cidp[book_cid][prop][0])
                    changed = True
                    continue
                # not a list, only save if field is empty or overwrite is true
                if not p[PI.overwrite] and mi.has_key(p[PI.colname]) and not mi.is_null(p[PI.colname]):
                    continue
                try:
                    # XXF XXT set separator character per column??
                    mi.set(p[PI.colname],", ".join(found_cidp[book_cid][prop]))
                    #print("Setting %s with %d"%(p[PI.colname],len(found_cidp[book_cid][prop])))
                    changed=True
                except Exception as ack:
                    error_cids.add(book_cid)
                    errors += 1
                    print("Error while saving property %s: %s"%(prop,ack))
            if book_cid in found_ids:
                idict = mi.get_identifiers()
                for id in found_ids[book_cid]:
                    if id not in idict:
                        try:
                            mi.set_identifier(id,found_ids[book_cid][id])
                            changed=True
                        except Exception as ack:
                            errors += 1
                            error_cids.add(book_cid)
                            print("Error while saving ID $s: %s"%(id,ack))
            if changed:
                #print("changed")
                try:
                    db.set_metadata(book_cid, mi, set_title=False, set_authors=False)
                    updated_cids.add(book_cid)
                except Exception as ack:
                    errors += 1
                    error_cids.add(book_cid)
                    try:
                        print("Error updating metadata:%s\n%s"%(ack,mi))
                    except: # yes really
                        print("Error printing exception (changed) %s, %s"%(ack, type(mi)))
            #else:
            #    print("not changed")
        # make sure progress dialog goes away
        progress.hide()
        inserttime=time.time()
        print("Updating metadata in books took %f seconds"%(inserttime-scantime))
        # update marked books
        opts = cfg.prefs['options']
        smessage = ""
        if opts['enableSeries']:
            smessage, s_updated_cids, s_error_cids = self.actual_merge_series(selected_cids, matching_cids, wdids)
            updated_cids.update(s_updated_cids)
            error_cids.update(s_error_cids)
        if opts['mark_updated']:
            for book_id in updated_cids:
                marked_ids[book_id] = 'updated'
        if opts['mark_newtags']:
            for book_id in newtag_cids:
                marked_ids[book_id] = 'newtag'
        if opts['mark_errors']:
            for book_id in error_cids:
                marked_ids[book_id] = 'error'
        if marked_ids: self.gui.current_db.data.set_marked_ids(marked_ids)

        # update the view so metadata changes actually show up
        # don't know if we need to save current pos since we're reselecting below anyway
        cr = self.gui.library_view.currentIndex().row()
        self.gui.library_view.model().refresh_ids(updated_cids, cr)

        # now select books the way the user wanted
        om = opts['selectMode']

        if om==0:  # updated books
            view.select_rows(updated_cids)
        elif om==1: # books not found, aka, has no wdid
            view.select_rows(selected_cids - matching_cids)
        elif om==2: # new unimported tags
            view.select_rows(newtag_cids)
        elif om==3:
            view.select_rows(error_cids)
        message = """%d books checked
%d books with wikidata IDs
%d properties for %d books"""%(len(selected_cids), len(matching_cids), len(results), len(found_cidp))
        if len(results)>= opts['limit']:
            message += " (query limit reached!)"
        #else:
        #    message += " (%s left)"%(opts['limit']-len(results)) # debug
        message +="""        
%d external IDs found
%d books with new tags not in filter list (marked:newtag)
%d books updated (marked:updated)"""%(len(found_ids), len(newtag_cids), len(updated_cids))
        if errors:
            message += "\n%d errors were caught (check marked:error)"%errors
        print(message)
        message += "\nWikidata query took %f seconds\nMerge took %f seconds."%(wikidatatime, inserttime-endwikidatatime)
        if opts['enableSeries']:
            message += "\n\n"+smessage
        info_dialog(self.gui,"Wikidata import results",message,show=True)

    def merge_series(self):
        selected_cids, matching_cids, wdids = self.get_selected_books()
        if not wdids: return
        message, updated_cids, error_cids = self.actual_merge_series(selected_cids, matching_cids, wdids)
        marked_ids = self.gui.current_db.data.marked_ids.copy()
        opts = cfg.prefs['options']
        if opts['mark_updated']:
            for book_id in updated_cids:
                marked_ids[book_id] = 'updated'
        if opts['mark_errors']:
            for book_id in error_cids:
                marked_ids[book_id] = 'error'
        if marked_ids: self.gui.current_db.data.set_marked_ids(marked_ids)
        info_dialog(self.gui,"Wikidata merge results",message,show=True)
        
    def actual_merge_series(self, selected_cids, matching_cids, wdids):
        wqb = WQB(cfg.prefs)
        mergemode = cfg.prefs['options']['seriesMode']
        series1col = cfg.prefs['options']['series1col']
        series2col = cfg.prefs['options']['series2col']
        errors = 0
        twoseries = 0
        error_cids = set()
        updated_cids = set()
        if not series1col: return "No series column set", updated_cids, error_cids
        # XXX if mode=ifempty, filter out books that are already populated
        # this could have been merged with the main query, with the risk of a lot of dups in the results
        results = wqb.doquery('series', booklist=wqb.valueList('wd',wdids.keys()))
        if not results:
            if wqb.msg:
                msg = wqb.msg
            else:
                msg = "No series data returned from wikidata"
            return msg, updated_cids, error_cids
        found_cidp={}
        for result in results:
            book_wdid = wqb.getqid(result,'book')
            if book_wdid not in wdids:
                print("Can't get wdid from book: %s"%result['book']['value'])
                continue
            book_cid = wdids[book_wdid]
            if book_cid not in found_cidp:
                found_cidp[book_cid] = []
            if 'seriesordinal' in result:
                found_cidp[book_cid].append(
                (result['seriesLabel']['value'], result['seriesordinal']['value']))
            else:
                found_cidp[book_cid].append(
                    (result['seriesLabel']['value'], None))
        # XXP progress?
        db = self.gui.current_db.new_api
        for book_cid in found_cidp:
            mi = db.get_metadata(book_cid)
            # swap values according to seriesOrder
            seriesOrder = cfg.prefs['options']['seriesOrder']
            i = found_cidp[book_cid]
            if len(i)>1 and seriesOrder>0:
                # possible swap
                if i[0][1]!=i[1][1] and (seriesOrder==1)==(i[0][1]<i[1][1]):
                    i = [ i[1], i[0] ]
            # XX assume if series is empty, series2 is also empty
            # reswap if not overwrite and existing data matches
            if len(i)>1:
                twoseries += 1
                seriesv =mi.get(series1col)
                if mergemode!=2 and seriesv and seriesv==i[1][0]:
                    i = [ i[1], i[0] ]
            # all modes save if it's empty
            seriesv =  mi.get(series1col)
            updated = False
            if not seriesv or mergemode==2 or (mergemode==1 and seriesv==i[0][0]):
                mi.set(series1col, i[0][0])
                mi.set(series1col+'_index', i[0][1])
                updated = True
            if series2col: seriesv = mi.get(series2col)
            else: seriesv = None
            if len(i)>1 and series2col and (not seriesv or mergemode==2 or (mergemode==1 and seriesv==i[1][0])):
                mi.set(series2col, i[1][0])
                mi.set(series2col+'_index', i[1][1])
                updated = True
            if updated:
                updated_cids.add(book_cid)
                try:
                    db.set_metadata(book_cid, mi, set_title=False, set_authors=False)
                    updated_cids.add(book_cid)
                except Exception as ack:
                    errors += 1
                    error_cids.add(book_cid)
                    try:
                        print("Error updating metadata:%s\n%s"%(ack,mi))
                    except: # yes really
                        print("Error printing exception (updated) %s, %s"%(ack,type(mi)))
        # infobox goes here
        message = "Series checked %d books with %d results.\n%d books updated."%(len(wdids), len(results), len(updated_cids))
        if twoseries:
            message += "\n%d books found with two series."%twoseries
        if error_cids:
            message += "\n%d errors updating series data (see marked:error)"
        return message, updated_cids, error_cids            
    def doprefs(self,whatbool):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def convert_identifiers(self,whatbool=False, chain=False):
        # XXP mark converted books
        # XXP select converted books
        count_over = 0    # convered from overdrive
        count_url1 = 0    # converted via our urlfixer
        count_url2 = 0    # XXP convert uris from other plugins to ids
        count_updated = 0
        view = self.gui.library_view
        db = self.gui.current_db.new_api
        #current_idx = view.currentIndex()
        selected_cids = set(view.get_selected_ids())
        if not selected_cids:
            if chain: return None
            info_dialog(self.gui,"Wikidata GUI","Please select books to update",show=True)
        progress =  QProgressDialog("Checking %d books for convertible IDs"%len(selected_cids),"cancel",0, len(selected_cids), self.gui)
        progress.setWindowModality(Qt.WindowModal)
        counter = 0
        progress.setValue(0)
        progress.show()
        for cid in selected_cids:
            counter += 1
            progress.setValue(counter)
            if progress.wasCanceled(): break;
            changed = False
            mi = self.gui.current_db.new_api.get_metadata(cid)
            ids = mi.get_identifiers()
            #print("** ids: "+" ".join(ids.keys()))
            # XXP convert overdrive uris 
            if 'gutenberg' not in ids and 'odid' in ids:
                # find first gutenberg ID
                #print("ov: %s"%ids['odid'])
                result = re.match(r"(\d+)@pg/",ids['odid'])
                if result:
                    mi.set_identifier('gutenberg', result.group(1))
                    #print(" found gutenberg:%s"%result.group(1))
                    changed = True
                    count_over+=1
            # XXP convert our managed URIs to ids
            # XXP should this seek out and destroy *all* matching uris or just ones we convert?
            url = None
            if 'url' in ids:
                sid = 'url'
                url = ids[sid]
            if 'uri' in ids:
                sid = 'uri'
                url = ids[sid]
            if url and self.urlfixer:
                id = self.urlfixer.id_from_url(url)
                #print("url: %s"%url)
                if id and id[0] not in ids:
                    mi.set_identifier(id[0], id[1])
                    #print(" found: %s:%s"%(id[0],id[1]))
                    changed = True
                    count_url1 += 1
                    # XXP delete original url if converted?
                    mi.set_identifier(sid,None) # XXX test this
            if changed:
                self.gui.current_db.new_api.set_metadata(cid,mi,set_title=False, set_authors=False, force_changes=True)
                count_updated +=1
        progress.hide()
        # XXG dialog reporting results
        message = """%d books selected
%d new gutenberg ids converted from overdrive
%d new ids converted from URL
%d books updated
"""%(len(selected_cids), count_over, count_url1, count_updated)
        if chain: return message
        info_dialog(self.gui,"Wikidata convert IDs", message, show=True)
        # XXG select updated books?

    def dotest(self, whatbool=False):
        print("dotest: %s"%whatbool)
        # get a selected book
        book_ids = self.gui.library_view.get_selected_ids()
        if not book_ids or len(book_ids)==0: return
        book = book_ids[0]
        ids = self.gui.current_db.new_api.get_metadata(book).get_identifiers()
        print(ids)
        try:
            self.urlfixer
        except:  # try a late import
            from calibre_plugins.wikidata.urlfixer import UrlFixer
            self.urlfixer = UrlFixer()
        if self.urlfixer:
            print(self.urlfixer.get_book_urls(ids))
        else:
            print("Wikidata metadata plugin not found.")

    def search_by_ids(self, whatbool=False):
        message = ""
        if cfg.prefs['options']['preconvert']:
            m = self.convert_identifiers(chain=True)
            if m: message = m+"\n"
        # construct a map of searchable ids to wikidata properties
        idd = cfg.prefs['ids']
        searchids = {}
        for prop in idd:
            if idd[prop][II.search]:
                searchids[idd[prop][II.idtag]] = prop
        # map out which id/val maps to what book
        valuemap = {}
        book_ids = self.gui.library_view.get_selected_ids()
        if not book_ids or len(book_ids)==0:
            info_dialog(self.gui,"Wikidata GUI","Please select books to update.", show=True)
            return
        for cid in book_ids:
            # map out the IDs in each book
            bids = self.gui.current_db.new_api.get_metadata(cid).get_identifiers()
            if 'wd' in bids: continue  # skip books we know about already
            for bid in bids:
                if bid in searchids:  # look for ids we know about
                    # id with this value maps to this book
                    if bid not in valuemap:
                        valuemap[bid] = { bids[bid] : cid }
                    else:
                        valuemap[bid][bids[bid]] = cid
        if not valuemap:
            info_dialog(self.gui, "Wikidata search",message+"No searchable identifiers found in unlinked books.", show=True)
            return
        # construct the search parts
        propkeys = valuemap.keys()
        idlist = " ".join("?%s"%searchids[prop] for prop in propkeys)
        preds = "\n".join("OPTIONAL {{ ?book wdt:{pred} ?{pred}. }}".format(pred=searchids[prop]) for prop in propkeys)
        vals = "\n".join(
            ("VALUES (?%s) { "%searchids[prop]+ " ".
             join(map(lambda val: '("%s")'%val, valuemap[prop].keys()))
             +" }") for prop in propkeys)
        wqb = WQB(cfg.prefs)
        results = wqb.doquery("blankquery",idlist=idlist, where=preds+vals)
        if not results:
            if wqb.msg:
                message = message+wqb.msg+"\n"
            info_dialog(self.gui, "Wikidata search",message+"Wikidata returned no results. (sID)",show=True)
            return
        t = "Wikidata returned %d results."%len(results)
        print(t)
        message += t+"\n"
        count = 0
        bookmap = {}
        # Note: wikidata query is keyd by properties, books are keyed by ID
        # so we have to map them back...
        for result in results:
            wdid = wqb.getqid(result, 'book')
            # we only need one match per book, take the first match and stop
            for k in result:  # these should all be literals
                if not k in idd: continue
                bid = idd[k][II.idtag]
                if result[k]['value'] in valuemap[bid]:
                    bookmap[valuemap[bid][result[k]['value']]] = wdid
                    break;
        t = "%d books with IDs found"%len(bookmap)
        print(t)
        message += t+"\n"
        # XXG progress bar?
        for cid in bookmap:
            # insert wdid into book ids
            mi = self.gui.current_db.new_api.get_metadata(cid)
            mi.set_identifier('wd',bookmap[cid])
            self.gui.current_db.new_api.set_metadata(cid,mi,set_title=False, set_authors=False)
        info_dialog(self.gui, "Wikidata search", message, show=True)
        # XXP select updated books
        self.gui.library_view.select_rows(bookmap.keys())

    # XXP marked:newbooks (not documented)
    # XXX deal with duplicates (merge or let the user choose?)
    def search_by_series(self):
        selected_cids, matching_cids, wdids = self.get_selected_books()
        if not wdids: return
        wqb = WQB(cfg.prefs)
        idlist = wqb.valueList('wd',wdids.keys())
        results = wqb.doquery("othersInSeries",booklist=idlist)
        if not results: # maybe it was a series not a book
            results = wqb.doquery("booksInSeries",booklist=idlist)
        if not results:
            if wqb.msg:
                msg = wqb.msg
            else:
                msg ="No results returned from Wikidata. (ss)"
            info_dialog(self.gui, "Wikidata search", msg, show=True)
            return
        # return from query:
        # ?booko ?bookoLabel ?authorLabel ?seriesLabel ?seriesordinal ?prevbook ?nextbook  ?pubdate ?inception
        # grovel through the results and merge duplicate results
        books = {}
        # build both metadata and table model
        # these could be merged but it would make the model more complex
        for result in results:
            book = wqb.getqid(result,'booko')
            if book not in books: books[book] = {}
            # Copy values from results to book table
            # Translate to final column names while we're at it
            # get the non-special probably singleton fields first
            for i in ('bookoLabel', 'instanceLabel', 'prevbook', 'nextbook', 'pubdate', 'inception'):
                if i=='bookoLabel': ii='title'
                else: ii=i
                if ii not in books[book]:  # ignore unlikely duplicate values
                    (v,t) = wqb.convertResultValue(result,i)
                    if v: books[book][ii] = v
            # now do the fields that need special handling
            # these fields should all be literal values, no conversion needed
            if 'authorset' not in books[book]: books[book]['authorset']=set()
            if 'authorLabel' in result:
                books[book]['authorset'].add(result['authorLabel']['value'])
            # Note: this only gets the first series supplied by wikidata.
            # If you want the second one, import it with a normal merge.
            if 'seriesLabel' in result and 'series' not in books[book]:
                books[book]['series'] = result['seriesLabel']['value']
                if 'seriesordinal' in result:
                    books[book]['seriesordinal'] = result['seriesordinal']['value']
        # XXX do another query if some nextbook/prevbook are missing??
        # make a second pass to create metadata
        for book in books:
            books[book]['authors'] = "&".join(books[book]['authorset'])
            mi = Metadata(books[book]['title'], tuple(books[book]['authorset']))
            if 'series' in books[book]:
                mi.series = books[book]['series']
            if 'seriesordinal' in books[book]:
                try:  # try to convert to number and only then save in metadata
                    v = float(v)
                    # convert to int if we can to get a spinner
                    vv = int(v)
                    if v==vv: v=vv
                    books[book]['seriesordinal'] = v # save so it sorts
                    mi.series_index = v
                except:
                        pass
            books[book]['mi'] = mi
        # XXX sort and link books within series, number
        ImportBooks(self.gui, "Books in series", books,
                    columns=[ 'title', 'authors', 'instanceLabel', 'series', 'seriesordinal', 'prevbook', 'nextbook', 'pubdate','inception'],
                    colmap={'instanceLabel':'type', 'seriesordinal':'ser#'},
                    readonly=set(['instanceLabel','prevbook','nextbook']) )

    def search_by_author_names(self):
        db = self.gui.current_db.new_api
        wqb = WQB(cfg.prefs)
        bookauthors =  db.all_field_for('authors', self.gui.library_view.get_selected_ids())
        print(bookauthors)
        authors = set()
        for book in bookauthors:
            authors.update(bookauthors[book])
        print(authors)
        wqb = WQB(cfg.prefs)
        lang = cfg.prefs['options']['deflang']
        authorlist = "".join("""("{name}"@{lang})""".format(name=author, lang=lang) for author in authors)
        results = wqb.doquery("byauthorname", authornamelist=authorlist)
        if not results:
            if wqb.msg:
                msg = wqb.msg
            else:
                msg = "No results returned from Wikidata for %d authors."%len(authors)
            info_dialog(self.gui, "Wikidata search", msg, show=True)
        self.handle_authors(wqb,results)
        
    # XXX this could search by author label if no wdid is available
    def search_by_authors(self):
        selected_cids, matching_cids, wdids = self.get_selected_books()
        if not wdids: return
        wqb = WQB(cfg.prefs)
        idlist = wqb.valueList('wd',wdids.keys())
        results = wqb.doquery("othersByAuthor",booklist=idlist)
        if not results:
            if wqb.msg:
                msg = wqb.msg
            else:
                msg = "No results returned from Wikidata. (sa)"
            info_dialog(self.gui, "Wikidata search", msg, show=True)
            return
        self.handle_authors(wqb,results)

    def handle_authors(self, wqb, results):
        books = {}
        for result in results:
            book = wqb.getqid(result,'booko')
            if book not in books: books[book] = {}
            # assume both fields 'bookoLabel' 'authorLabel' exist
            if 'title' not in books[book]:  # ignore dups
                books[book]['title'] = result['bookoLabel']['value']
            if 'authorset' not in books[book]: books[book]['authorset']=set()
            # XXX we might have books with no author
            books[book]['authorset'].add(result['authorLabel']['value'])
            if 'instanceLabel' in result:
                books[book]['type'] = result['instanceLabel']['value']
        # second pass to create metadata
        for book in books:
            books[book]['authors'] = "&".join(books[book]['authorset'])
            books[book]['mi'] = Metadata(books[book]['title'], books[book]['authorset'])
        ImportBooks(self.gui,"Books by same author", books)

    def bulk_search(self):
        db = self.gui.current_db.new_api
        wqb = WQB(cfg.prefs)
        selectedbooks = self.gui.library_view.get_selected_ids()
        if not selectedbooks:
            info_dialog(self.gui,"Wikidata search","Please select books to search",show=True)
            return
        # first, filter for only books without wdIDs
        book_cids = set()
        authors = set()
        titles = set()
        count_wd = 0
        count_nowd = 0
        # XXG progress?
        # XXX limit or split queries that are too long
        for book_cid in selectedbooks:
            mi = db.get_metadata(book_cid)
            ids = mi.get_identifiers()
            # can't handle books with no author, don't handle found books
            if 'wd' in ids or mi.is_null('authors'):
                count_wd += 1
            else:
                count_nowd += 1
                book_cids.add(book_cid)
                authors.update(mi.authors)
                titles.add(mi.title)
        if not titles:
            info_dialog(self.gui,"Wikidata search","No searchable books found in %d selected books"%len(selectedbooks),show=True)
            return
        lang = cfg.prefs['options']['deflang']
        bookauthors =  "".join("""("{name}"@{lang})""".format(name=author,lang=lang) for author in authors)
        titlelist =  "".join("""("{name}"@{lang})""".format(name=title,lang=lang) for title in titles)
        results = wqb.doquery("findbooks", authornamelist=bookauthors, titlelist=titlelist)
        if not results:
            if wqb.msg:
                msg = wqb.msg
            else:
                msg = "No results returned from Wikidata. (ha)"
            info_dialog(self.gui, "Wikidata search", msg, show=True)
            # XXX give more details?
            return
        # coalesce books with multiple authors into one entry
        foundbooks = {}
        for result in results:
            wdid = wqb.getqid(result,'book')
            if wdid not in foundbooks: foundbooks[wdid] = [
                    result['titlelist']['value'],
                    [ result['authorLabel']['value'] ] ]
            else: # add another author, ignore title
                foundbooks[wdid][1].append(result['authorLabel']['value'])
        # map cids to multiple wdids
        bookmap_c2w = {}
        for wdid in foundbooks:
            mi = Metadata(foundbooks[wdid][0], tuple(foundbooks[wdid][1]))
            ms = db.find_identical_books(mi, book_ids=book_cids)
            if not ms:
                print("Not matched: %s=%s"%(wdid,foundbooks[wdid]))
            else:
                for m in ms:
                    if m in bookmap_c2w:
                        bookmap_c2w[m].append(wdid)
                    else:
                        bookmap_c2w[m] = [ wdid]
        # print/reject duplicates and marked:dup
        # update metadata for singletons and marked:updated
        dup_cids = set()
        updated_cids = set()
        error_cids = set()
        marked_ids = self.gui.current_db.data.marked_ids.copy()
        for cid in bookmap_c2w:
            if len(bookmap_c2w[cid])==1:
                mi = db.get_metadata(cid)
                mi.set_identifier('wd',bookmap_c2w[cid][0])
                try:
                    db.set_metadata(cid, mi, set_title=False, set_authors=False)
                    updated_cids.add(cid)
                    marked_ids[cid] = 'updated'
                except:
                    error_cids.add(cid)
                    marked_ids[cid] = 'error'
            else:
                dup_cids.add(cid)
                marked_ids[cid] = 'dup'
                # XXX reformat this error
                try:  # XXX
                    print("Duplicates found for %s: %d %s"%(cid, len(bookmap_c2w[cid]), bookmap_c2w[cid]))
                except Exception as ack:
                    print("Duplicate print err",ack)
        self.gui.current_db.data.set_marked_ids(marked_ids)
        info_dialog(self.gui,"Wikidata search","""For {selected} books, searched {searchable} books
Searched for {title} titles, {author} authors
Got {result} results matching {book} books from wikidata
Updated {updated} books (marked:updated)
Didn't update {dup} books (marked:dup)
Got {err} errors while updating (marked:error)
""".format(selected=len(selectedbooks), searchable=count_nowd, title=len(titles), author=len(authors), result=len(results), book=len(foundbooks), updated=len(updated_cids), dup=len(dup_cids), err=len(error_cids)), show=True)
            
    menudata = (
        ("merge wikidata", "merge wikidata into books",merge_wikidata),
        ## these goals are too ambitious for first pass
        ("convert URIs to IDs", "Convert uris in identifiers to their matching (enabled) IDs", convert_identifiers),
        ("Search for books using other IDs", "Find books in wikidata using IDs from other sources", search_by_ids),
        ("Bulk search by author and title", "Find books in wikidata efficently with an exact author/title search, but doesn't handle duplicates or fuzzy matches", bulk_search), 
        ("Other books in series", "Search for other books in series of selected books", search_by_series),
        ("Other books by authors", "Search for other books by same author", search_by_authors),
        ("Other books by same author names", "Search for books by same author name (for books without a wikidata id)", search_by_author_names),
        #("test", "test", dotest),
        ("customize wikidata merge", "Customize plugin", doprefs)
        )
    
    def genesis(self):
        # get icons
        icon_resources = self.load_resources(['images/icon.png','images/okok.png'])
        set_plugin_icon_resources(self.name, icon_resources)
        self.qaction.setIcon(get_icons("images/icon.png"))
        # build menu
        m = self.menu = QMenu(self.gui)
        # XXF create more icons?
        from functools import partial
        for (short, tooltip, action) in self.menudata:
            self.create_menu_action(m, short, short, None, None, tooltip, partial(action,self), None)
        self.qaction.setMenu(m)
        # initialize state vars
        self.is_library_selected = True
        # initialize wikidata metadata interlink
        try:
            self.urlfixer
        except:
            from calibre_plugins.wikidata.urlfixer import UrlFixer
            self.urlfixer = UrlFixer()
        if self.urlfixer:
            pids =  cfg.prefs['ids']
            self.urlfixer.delIDs(filter(lambda x: not pids[x][II.dispenable], pids.keys()))
            self.urlfixer.addIDs(map(
                lambda x: (pids[x][II.idtag], [ pids[x][y] for y in [II.pastepriority, II.template, II.regex] ]) ,
                filter(lambda x: pids[x][II.dispenable], pids.keys())))

    # def library_changed():

    def location_selected(self, loc):
        #  library, main, card and cardb; we can only handle library
        self.is_library_selected = loc == 'library'

    #def shutting_down():
    #def initialization_complete():

    ## possible menu items for selected books:
    ##  find books by external ID (unique only?)
    ##  search book editions for oldest pubdate

    ## Button functions not implemented:
    ## ?? test url template/regex?
    
