#!/usr/bin/env python2
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2018, Steven Dick <kg4ydw@gmail.com>'

from datetime import datetime
from PyQt5.Qt import Qt, QAbstractTableModel, QSortFilterProxyModel, QModelIndex, QPixmap, QIcon, QBrush;
from PyQt5.QtCore import QSize
from PyQt5.QtWidgets import QTableView, QAbstractItemView
from calibre_plugins.wikidata_gui.common_utils import get_icon

class DictDictModel(QAbstractTableModel):
    '''
    Treat a dictionary of dictionaries like a table using row and
    column order arrays.  Note that column[0] is row key and read only.
    (YYY) This may be mangled slightly for our special book views
    '''
    # XXX also take tooltips for headers?
    def mapOrNot(self, colmap, var):
        if var in colmap:
            return colmap[var]
        else:
            return var
    def __init__(self, data, columns=None, colmap=None, exclude=(), readonly=set()):
        QAbstractTableModel.__init__(self)
        self.d = data;
        self.rownames = sorted(self.d.keys())
        self.readonly = readonly
        cn = set()
        for r in self.rownames:
            cn.update(data[r].keys())
        cn -= set(exclude)
        if columns: # impose an order
            # note: key is always first col and never in cn
            self.colnames = ['key']
            for i in columns:
                if i in cn:
                    self.colnames.append(i)
                    cn.discard(i)
            # add the rest anyway
            self.colnames += sorted(cn)
        else:
            self.colnames = ['key'] + sorted(cn)
        if colmap:
            self.colHeads = [ self.mapOrNot(colmap, i) for i in self.colnames]
        else:
            self.colHeads = self.colnames
        #print("rows=%d heads=%d"%(len(self.rownames), len(self.colnames)))
        #try:
        #    print(self.rownames)
        #    print(self.colnames)
        #except Exception as ack:
        #    print(ack)
              
    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.rownames)
    def columnCount(self,parent):
        if parent and parent.isValid():
            return 0
        return len(self.colnames)
    def headerData(self, col, orientation, role):
        if role == Qt.DisplayRole and orientation == Qt.Horizontal and col<len(self.colHeads):
            return self.colHeads[col]
        return None

    def key(self, index):
        if not index.isValid():
            return None;
        row = index.row()
        if row < 0 or row >= len(self.rownames):
            return None
        return self.rownames[row]
    def datarow(self, index):
        return self.d[self.key(index)]
        
    def data(self, index, role):
        if not index.isValid():
            return None;
        row = index.row()
        col = index.column()
        if row < 0 or col<0 or row >= len(self.rownames) or col>=len(self.colnames):
            #print("OOB %d,%d"%(row,col))
            return None
        if role in [Qt.DisplayRole, Qt.UserRole, Qt.EditRole]:
            if col==0:
                return self.rownames[row]
            else:
                try:
                    data = self.d[self.rownames[row]][self.colnames[col]]
                except:
                    return None # not all columns have values
                # Qt doesn't seem to know what to do with dates
                if isinstance(data, datetime):
                    if role==Qt.DisplayRole:
                        data = data.isoformat() # strftime hates dates <1900
                    else:
                        return None # XXX do something else for edit role
                return data
        # XXX other roles?
        #### YYY book view specific roles
        coln = self.colnames[col]
        # display validity of next/prev
        if role == Qt.BackgroundRole and coln in ('prevbook', 'nextbook'):
            rown = self.rownames[row]
            if coln not in self.d[rown] or not self.d[rown][coln]:
                return None
            if self.d[rown][coln] in self.d:
                return QBrush(Qt.green)
            else:
                return QBrush(Qt.red)
        #### YYY end  book specific roles
        return None

    def setData(self, index, value, role):
        if not index.isValid():
            return False
        row, col = index.row(), index.column()
        if row < 0 or col<=0 or row >= len(self.rownames) or col>=len(self.colnames):
            return False
        rown = self.rownames[row]
        coln = self.colnames[col]
        if role == Qt.EditRole:
            if hasattr(value,'strip'):
                value = value.strip()
            self.d[rown][coln] = value
            ##### YYY book model specific stuff
            if coln in ('authors','title'):
                self.d[rown]['edited'] = True # XXX recreate mi later
            elif 'mi' in self.d[rown]: # update mi if we know how
                if coln=='series':
                    self.d[rown]['mi'].series = value
                elif coln=='seriesordinal':
                    try:
                        self.d[rown]['mi'].series_index = float(value)
                    except:
                        return False
                #elif add other fields later?
            ##### YYY end book model
            self.dataChanged.emit(index, index)
            return True
        # XXX other roles?
        return False

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if not index.isValid() or index.column()==0:
            return flags
        rown = self.rownames[index.row()]
        coln = self.colnames[index.column()]
        if coln in self.readonly: return flags
        try:
            data = self.d[rown][coln]
            if isinstance(data, datetime):  # can't edit dates right now
                return flags
        except Exception as ack:
            #print("Bad data: %s"%ack) # empty columns are likely
            pass
        #####YYY This section is specific to book views
        # don't edit author/title of a matched row
        if 'status' in self.d[rown] and self.d[rown]['status'] in (1,2) and coln in ('authors','title'):
            return flags
        ####YY end bookview items
        return flags | Qt.ItemIsEditable

    # implement insertRows removeRows later if we need them

# QSortFilterProxyModel
# QTableView

statusInit=False
statusThings = [
    [ None, "No matching book was found", None ],              # 0
    [ 'ok.png', "A matching book was found", None ],           # 1
    [ 'images/okok.png', "Multiple matching books were found", None], # 2
    [ 'plus.png', "An empty book will be added if you save", None], # 3
]

class StatusModel(DictDictModel):
    '''
    Move a status column to column 0 and display it as an icon.
    '''
    def __init__(self, data, columns=None, colmap=None, exclude=(), readonly=set()):
        global statusInit, statusThings
        ex = set(exclude)
        ex.add('status')
        ex.add('edited')
        DictDictModel.__init__(self, data, columns, colmap, ex, readonly)
        if not statusInit:
            for i in statusThings: # use a number for any icons we don't get
                if i[0]:
                    icon = get_icon(i[0])
                    if icon: i[2] = icon
            statusInit=True
    def columnCount(self, parent):
        return DictDictModel.columnCount(self, parent)+1
    def headerData(self, col, orientation, role):
        if role == Qt.DisplayRole and orientation == Qt.Horizontal and col==0:
            return "status"
        return DictDictModel.headerData(self, col-1, orientation, role)
    def data(self, index, role):
        if not index.isValid() or index.column()<0:
            return None;
        if index.column()>0:
            return DictDictModel.data(self, DictDictModel.index(self, index.row(), index.column()-1), role)
        # column 0/status stuff
        row = self.datarow(index)
        if not row: return None
        st = row['status']
        if role==Qt.DisplayRole:
            if st==0:
                return ""
            if st>len(statusThings) or not statusThings[st][2]:
                return st
            return None
        if st<0 or st>len(statusThings):
            return None
        elif role==Qt.DecorationRole and statusThings[st][2]:
            return statusThings[st][2]
        elif role==Qt.ToolTipRole:
            return statusThings[st][1]
        return None
    def setData(self, index, value, role):
        if not index.isValid():
            return False
        if index.column()==0: # ok, this hack is very model specific
            row = self.datarow(index)
            if 'status' in row and row['status'] in (1,2):
                return False  # can't modify matches
            # save new status, whatever it is
            row['status'] = value
            self.dataChanged.emit(index,index)
            return True
        return DictDictModel.setData(self, DictDictModel.index(self, index.row(), index.column()-1), value, role)
    def flags(self, index):
        if not index.isValid():
            return False
        if index.column()==0:
            return Qt.ItemIsSelectable|Qt.ItemIsEnabled # XXX more later
        return DictDictModel.flags(self, DictDictModel.index(self, index.row(), index.column()-1))

class StatusView(QTableView):
    '''
    View for StatusModel including some display tweaks
    '''
    # columns to tweak:
    # status: icon
    # wdid/key: url link
    def __init__(self, parent):
        QTableView.__init__(self, parent)
        self.setSortingEnabled(True)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.ExtendedSelection)
        # force vert scrollbar on and always reserve space for it
        self.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
    def setModel(self, model):
        QTableView.setModel(self, model)
        # XXX set delegates?
        # XXX set min and max column widths?
        self.resizeColumnsToContents()
        self.adjustSize()

    def sizeHint(self):
        # try to make window big enough to not need a scrollbar
        s = self.viewportSizeHint()
        # is this a bug? it seems to never make room for vertical scroll bar
        v = self.verticalScrollBar()
        s += v.sizeHint()+QSize(2,0)
        return s
