#!/usr/bin/env python2

__license__   = 'GPL v3'
__copyright__ = '2018, Steven Dick <kg4ydw@gmail.com>'

from calibre_plugins.wikidata_gui.modeldata import PropIndex as PI, EntityIndex as EI, IdIndex as II
from urllib import quote
from calibre_plugins.wikidata_gui.SPARQLWrapper import SPARQLWrapper, JSON

QueryId = "# Calibre Wikidata GUI plugin\n"

class WikidataQueryBuilder:
    queries = {
## query a list of books and their properties
'bookprops':
"""SELECT DISTINCT ?book ?prop ?value ?valueLabel WHERE {{
SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
  VALUES (?book) {{ {books} }}
  VALUES (?prop) {{ {properties} }}
  ?book ?prop ?value.
}}""",
'getlabels':
"""SELECT ?item ?itemLabel WHERE {{ VALUES (?item) {{ {items} }}
SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }} }}
""",
# note: all fields in idquery are required; missing field is prob not a good ID
# additional filter:  ?id wikibase:propertyType wikibase:ExternalId
"idquery":
"""SELECT ?id ?idLabel ?fmturl ?fmtregex WHERE {{
  VALUES (?id) {{ {items} }}
  ?id wdt:P1630 ?fmturl.
  ?id wdt:P1793 ?fmtregex.
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }} }} """,
"idsfrombooks":
"""SELECT DISTINCT ?prop ?wdLabel ?fmturl ?fmtregex WHERE {{
  VALUES (?item) {{ {booklist} }}
  ?item ?prop ?something.
  ?wd wikibase:directClaim ?prop.
  ?wd wikibase:propertyType ?wdtype.
  ?wd wikibase:propertyType wikibase:ExternalId.
  ?wd wdt:P1630 ?fmturl.
  ?wd wdt:P1793 ?fmtregex.
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }} }}""",
"propsfrombooks":
"""SELECT DISTINCT ?prop ?wdLabel ?wdtype WHERE {{
  VALUES (?item) {{ {booklist} }}
  ?item ?prop ?something.
  ?wd wikibase:directClaim ?prop.
  ?wd wikibase:propertyType ?wdtype.
  FILTER(?wdtype != wikibase:ExternalId)
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }} }}""",
"tagsfrombookprops":
 """SELECT DISTINCT ?tag ?tagLabel WHERE {{
  VALUES (?item) {{ {booklist} }}
  VALUES (?prop) {{ {proplist} }}
  ?item ?prop ?tag.
  ?wd wikibase:directClaim ?prop.
  ?wd wikibase:propertyType wikibase:WikibaseItem.
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }} }}""",
# for queries that are totally constructed
"blankquery" :
"""SELECT ?book {idlist} WHERE {{
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
  {where}
  }}""",
"series" :
"""SELECT ?book ?seriesLabel ?seriesordinal WHERE {{
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
 VALUES (?book) {{ {booklist} }}
 ?book p:P179 ?seriesstatement.
 ?seriesstatement ps:P179 ?series.
 OPTIONAL {{ ?seriesstatement pq:P1545 ?seriesordinal. }}
  }}""",
# find other books in series of a book
# Note: author will cause dups for coauthors
# use inception or pubdate to order books if ordinal and next/prev missing

# Note: P155=prevbook and P156=nextbook are present both as wdt: and
# pq:/series in different books; hopefully when they are there as pq,
# there is also a series ordinal

"othersInSeries":
"""SELECT ?booko ?bookoLabel ?authorLabel ?seriesLabel ?seriesordinal ?prevbook ?nextbook ?pubdate ?inception WHERE {{
SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
VALUES (?book) {{ {booklist} }}
?book wdt:P179 ?series.  
?booko wdt:P179 ?series.
?booko p:P179 ?seriesstatement.
?seriesstatement ps:P179 ?series.
OPTIONAL {{ ?booko wdt:P155 ?prevbook. }}
OPTIONAL {{ ?booko wdt:P156 ?nextbook. }}
OPTIONAL {{ ?seriesstatement pq:P1545 ?seriesordinal. }}
OPTIONAL {{ ?booko wdt:P50 ?author. }}
OPTIONAL {{ ?booko wdt:P571 ?inception. }}
OPTIONAL {{ ?booko wdt:P577 ?pubdate. }}
}}
""",
# in case a series was saved as a book
# tried to make this a union with above, didn't work
"booksInSeries":
"""SELECT?booko ?bookoLabel ?authorLabel ?instanceLabel ?seriesLabel ?seriesordinal ?prevbook ?nextbook ?pubdate ?inception WHERE {{
SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
VALUES (?series) {{ {booklist} }}
?booko wdt:P179 ?series.
?booko p:P179 ?seriesstatement.
?seriesstatement ps:P179 ?series.
OPTIONAL {{ ?booko wdt:P31 ?instance. }}
OPTIONAL {{ ?booko wdt:P155 ?prevbook. }}
OPTIONAL {{ ?booko wdt:P156 ?nextbook. }}
OPTIONAL {{ ?seriesstatement pq:P1545 ?seriesordinal. }}
OPTIONAL {{ ?booko wdt:P50 ?author. }}
OPTIONAL {{ ?booko wdt:P571 ?inception. }}
OPTIONAL {{ ?booko wdt:P577 ?pubdate. }}
}}
""",

# leave author in query output to collect coauthors
# Should this pick up more metadata? or just let the regular query
# fill it in separately...
# XXX this picks up both books and series (filter out series?)
"othersByAuthor":
"""
SELECT ?booko ?bookoLabel ?authorLabel ?instanceLabel WHERE {{
SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
VALUES (?book) {{ {booklist} }}
?book wdt:P50 ?author.
?booko wdt:P50 ?author.
OPTIONAL {{ ?booko wdt:P31 ?instance. }}
}}
""",
# if book doesn't have a wdid, try searching by author name
'byauthorname':"""
SELECT ?booko ?bookoLabel ?authorLabel ?instanceLabel WHERE {{
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
  VALUES (?authorname) {{
    {authornamelist}
  }}
  ?author rdfs:label ?authorname.
  ?booko wdt:P50 ?author.
  OPTIONAL {{ ?booko wdt:P31 ?instance. }}
}}
""",
# bulk wikidata id search
# issues: may find mismatched title/author
# will find multiple editions -- how to pick one?  UI? or use the other plugin
# P31 filter may be overly restrictive
'findbooks':"""
SELECT distinct ?book ?titlelist ?authorLabel WHERE {{
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
  VALUES ?worktype {{ wd:Q47461344 wd:Q732577 }}
  ?book wdt:P31/wdt:P279* ?worktype.
  VALUES (?authorname) {{ {authornamelist} }}
  VALUES (?titlelist) {{ {titlelist} }}
  {{ ?book rdfs:label ?titlelist. }} UNION {{ ?book skos:altLabel ?titlelist. }}
  ?author rdfs:label ?authorname.
  ?book wdt:P50 ?author.
}}
""",
# XXX not used yet, untested ; alternate series search P527="parts of"
"seriesparts" :  # P527: has part; not reliably filled in
"""SELECT ?book ?bookLabel ?authorLabel ?prevbook ?nextbook WHERE {{
VALUES (?series) {{ (series) }}
?series wdt:P527 ?book.
  SERVICE wikibase:label {{ bd:serviceParam wikibase:language "[AUTO_LANGUAGE],{lang}". }}
 OPTIONAL {{
 ?book p:P179 ?seriesstatement.
 ?seriesstatement ps:P179 ?series.
 ?seriesstatement pq:P1545 ?seriesordinal. }}
}}
"""
}

    def __init__(self, prefs):
        self.prefs = prefs
        self.opts = prefs['options']
    def valueList(self, type, list): # XXX does this need the join space?
        return " ".join(map(lambda(p): "(%s:%s)"%(type,p), list))
    def getprops(self):
        proplist = sorted(filter(lambda(p): self.prefs['properties'][p][PI.use],
                          self.prefs['properties'].keys()))
        return self.valueList('wdt',proplist)
    def getextids(self):
        proplist = sorted(filter(lambda(p): self.prefs['ids'][p][II.importids],
                          self.prefs['ids'].keys()))
        return self.valueList('wdt',proplist)

    # XXF put a query rate limiter on this? 60/min?
    # XXF this should really examine errors from the query and distinguish between error and no results
    def doquery(self, queryname, **kwargs):
        query = QueryId + self.queries[queryname].format(
            lang=self.opts['deflang'], **kwargs) + (" LIMIT %d"%self.opts['limit'])
        if self.opts['debugquery']:
            print('query: %s'%query)
        sparql = SPARQLWrapper("https://query.wikidata.org/sparql")
        try:
            sparql.setQuery(query)
            
            sparql.setReturnFormat(JSON)
            results = sparql.query().convert()
            if self.opts['debugquery']:
                print("results:\n%s\n"%results)
            if results and results['results'] and results['results']['bindings']:
                return results['results']['bindings']
        except Exception as ack:
            print('query exception:\n%s\n\n%s'%(query,ack))
            #log.info('query exception:\n%s\n\n%s'%(query,ack))
        return None
    def getid(self, uri):
        import re
        (wdid, ok) = re.subn(r"http://www.wikidata.org/.*/", "", uri)
        if ok: return wdid
        else: return None

    def getqid(self, match, col):
        if col in match and 'value' in match[col]:
            return self.getid(match[col]['value'])
        else:
            return None
    # do something inteligent with the value and return the type too
    def convertResultValue(self, match, col):
        if not (col in match and 'value' in match[col]): return (None,None)
        t = match[col]['type']
        v = match[col]['value']
        # also lang, datatype
        if t=='uri':
            import re
            (wdid, ok) = re.subn(r"http://www.wikidata.org/.*/", "", v)
            if ok:
                v=wdid
                t="wdid"
        if t=='literal' and 'datatype' in match[col] and match[col]['datatype']=="http://www.w3.org/2001/XMLSchema#dateTime":
            from datetime import datetime
            try:
                v = datetime.strptime(v,"%Y-%m-%dT%H:%M:%SZ")
                t = 'pdate'
            except:
                print("Date conversion failed for %s"%v)
        return (v, t)

        
# functions to query wikidata for insertion into various models

# merge metadata:
#   selected books
#   wikidata query
#     VALUES sets -> booklist, property list, entity list, identifiers
#  VALUES ?book { wd:%s ...
#  VALUES ?prop { wdt:%s ... incl identifiers
#  ?book ?prop ?value
#  series info (from wikidata metadata plugin)

# selected books
# all (properties, labels),

# selected books
# selected properties
# (values, labels)
#   --> pick property colors, color entities to match (use priority list)
#   --> select property --> select entities

# toggle colors?

# query single property for label

# query single entity for label

# query books in a series
#   book / booklabel
#   authors?
#   series name
#   index in series
#   next / previous book in series
# series from multiple columns?
# publication date?

# functions:
# 
# blank -> match ^[PQ]\d* --> change combobox
# 
# add : add item in blank
# delete: delete selected items
# import
# combobox: property, tag/entity filter, ID
# 
