#!/usr/bin/env python2
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2018, Steven Dick <kg4ydw@gmail.com>'

import sys
import re

from PyQt5.Qt import QWidget, QHBoxLayout, QLabel, QLineEdit, QTableWidgetItem

# MVC
from PyQt5 import QtCore, QtGui
from PyQt5.QtCore import Qt, QAbstractTableModel, QModelIndex, QVariant, QSize
from PyQt5.QtWidgets import QMessageBox

from calibre.utils.config import JSONConfig
from calibre_plugins.wikidata_gui.config_ui import Ui_Form

from calibre_plugins.wikidata_gui.modeldata import DefaultModel as DM
from calibre_plugins.wikidata_gui.query import WikidataQueryBuilder as WQB

from calibre_plugins.wikidata_gui.modeldata import PropIndex as PI, EntityIndex as EI, IdIndex as II

from calibre.ebooks.metadata.book.base import STANDARD_METADATA_FIELDS

prefs = JSONConfig('plugins/wikidata-gui')
# initialize or update preferences ###
dm = DM(prefs)
# XXD should add debug consistency check to check length of arrays between defaults, headers, (tooltips?), rowxlate and things in modeldata.py

## headers reflect rowxlate order rather than model data order

# ID search priority(7) removed as not used:
#   header: worder
#   type: I
#   tooltip: "Ranked search order for ID in wikidata (stop on first match)",
# probably actual order is hash order, should this be fixed? XXX

# hidden (properties[6] and tags[3]) not listed here!

headers = {
    'properties' : ['prop', 'label', 'column', 'use', 'filter' , 'ov', 'pri'],
    'tags' : [ 'wdid', 'wikidata label', 'your label', 'use' ],
    'ids' : ['wdid', 'description', 'tag', 'URL template', 'regex match', 'en', 'imp', 'search',  'porder']
    }
tooltips = {
    'properties': ['wikidata property', 'wikidata label', 'destination column', 'Use or ignore this property', 'For wikidata tags, enable tag filter or use label directly',
                   'overwrite existing data for non-tag columns',
                   'lowest value priority is used if multiple properties for the same non-tags column' ],
    'tags': None,
    'ids': [ None, None, "Identifer used in IDs", "insert value into this URL when linking",
             "URIs that match this are translated into this ID on URI paste",
             "Enable identifer display (disable if you get duplicates)",
             "Import this ID from Wikidata", "Search wikidata by this tag",
             "Ranked search order when matching URI for paste" ]
        }

# Reorder and translate fields; coulda fixed this in the data, but I'm
# sure I'll have to reorder them again later, so may as well bake it in.
# Same field could be used multiple times with different formatters
# note that column 0 is the key, so 1 is the first array element in real data
# element zero here is not used
# XXM change rowxlate['ids'][1][6]='c' if wikidata plugin isn't loaded
rowxlate = {
    'properties' : [ [ 0, 2, 3, 0, 1, 5, 4], "ksSCCCI" ],
    'tags' : [ [ 0, 1, 2, 0 ], "kfSC" ],
    'ids' : [ [ 0, 1, 2, 3, 4, 0, 5, 7, 8], "ksSFFCCCI" ],
    }
flagmap = {
    'k': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsSelectable,  # key
    's': Qt.ItemIsSelectable | Qt.ItemIsEnabled, # read only string
    'f': Qt.ItemIsSelectable | Qt.ItemIsEnabled, # read only folded string
    'S': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsEditable,
    'F': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsEditable, # foldable string
    'I': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsEditable, # integer
    'i': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsEditable, # small integer
    'C': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsUserCheckable,
    'D': Qt.ItemIsSelectable | Qt.ItemIsEnabled | Qt.ItemIsEditable | Qt.ItemIsUserCheckable,
    'c': Qt.ItemIsSelectable | Qt.ItemIsUserCheckable,
    }

### array formats
# properties:  P123 : [ mode, valuemode, label, column, priority ]
#  * mode: 0=disable/ignore 1=ifempty 2=append 3=replace 4=ifgreater 5=ifless
#  * valuemode: 0=don't import new tag after saving; >0=value copy to tag state
#  * Note: for tags type column, only actions are ignore(0,1), append(rest)
# tags: Q123 : [ state, wikidata label, your label ]
#  * state: 0=ignore 1=new/ignore 2=uselabel 3=use wdid
## Note: there are >3500 ID's in wikidata, getting them all is insane
# ids:  wdid(P123) : [ modemask, description, calibreIDtype, URLtemplate, matchRE, idorder, wdorder ]
# * modemask: 1=enableview 2=enable paste 4=enable import 8=search wd by
# * modes 1,2 require wikidata metadata plugin

## note: for fast translation, also make reverse map
# idbyhost: extract host part from URLtemplate as key, map to idorder sorted array of ids keys


# browse a dictionary of arrays as a table
# find actual model data in prefs[dictname]
# find model headers in headers, map in rowxlate, view types in typemap

# map columns to view order and lead to correct decorations
class myDictModel(QAbstractTableModel):
    def __init__(self, dictname):
        QAbstractTableModel.__init__(self)
        self.sortcol = 0
        self.sortrev = False
        self.needsort = True
        self.modelName = dictname;
        self.showhidden = False
        self.hidecol = dm.modelHideCol[dictname]
        self.mykeys = None
        self.resort()
    def rowCount(self, parent=QModelIndex()):
        #return len(prefs[self.modelName])
        return len(self.mykeys)
    def columnCount(self, index=QModelIndex()):
        return len(headers[self.modelName])
    def data(self, index, role):
        if not index.isValid() or not (0<=index.row()<len(self.mykeys)):
            return QVariant()
        rowname = self.mykeys[index.row()]
        # XX didn't check if column is valid
        rcol = rowxlate[self.modelName][0][index.column()]
        try:
            value = prefs[self.modelName][rowname][rcol]
        except:
            value = 'ERR' # XXD this shouldn't happen: model drift...
            print("%s tried to get (%d,%d=%d) in role %d"%(self.modelName, index.row(),index.column(), rcol, role))
        ftype = rowxlate[self.modelName][1][index.column()]
        if role==Qt.TextAlignmentRole:
            if ftype=='I' or ftype=='i': return Qt.AlignRight|Qt.AlignVCenter
            if ftype=='C' or ftype=='c': return Qt.AlignCenter # didn't help
            return QVariant() # no other types need special alignment
        if role==Qt.SizeHintRole:
            if ftype=='F' or ftype=='f':
                return QSize(150,20)  # arbitrarily small
            if ftype=='i':  # small integer, should be 2-3 em
                return QSize(15,32)
            if ftype=='C' or ftype=='c':
                return QSize(-32,32)  # quit adding extra space!
            return QVariant() # no other types need special size
        if ftype =='C' or ftype=='c' or ftype=='D':
            if role==Qt.CheckStateRole:
                if not value:
                    return Qt.Unchecked
                else:
                    return Qt.Checked
            return QVariant() # no other roles are valid for checkboxes
        if not (role==Qt.DisplayRole or role==Qt.EditRole):
            return QVariant() # no remaining roles are supported
        if index.column()==0: return rowname
        return value

    def setData(self, index, value, role=Qt.EditRole):
        if index.column()==0: return False
        # XX verify column index?
        rcol = rowxlate[self.modelName][0][index.column()]
        ftype = rowxlate[self.modelName][1][index.column()]
        rowname=self.mykeys[index.row()]
        if ftype == 'C' or ftype=='c':
            if role == Qt.CheckStateRole:
                prefs[self.modelName][rowname][rcol] = value==Qt.Checked
                self.dataChanged.emit(index,index) # XX? ,Qt.CheckStateRole)
                return True
            else:
                return False
        # strip spaces from strings, but don't have a cow if this fails
	try:
          if ftype in ( 'S' , 'F'): value = value.strip()
	except:
	  pass
        if role == Qt.EditRole and 0<=index.row()<len(self.mykeys):
            prefs[self.modelName][rowname][rcol] = value
            self.dataChanged.emit(index,index)
            return True
        else:
            return False
    def flags(self, index):
        r = rowxlate[self.modelName][1]
        if not index.isValid() or not (0<=index.row()<len(self.mykeys)) or not (0<=index.column()<len(r)) or r[index.column()] not in flagmap:
            return Qt.NoItemFlags;
        if index.column()<2 and self.showhidden and prefs[self.modelName][self.mykeys[index.row()]][self.hidecol]:
            return Qt.ItemIsEnabled ^ flagmap[r[index.column()]]
        return flagmap[r[index.column()]]
    def headerData(self, section, orientation, role=Qt.DisplayRole):
        if orientation == Qt.Horizontal:
            if role == Qt.ToolTipRole and tooltips[self.modelName] and tooltips[self.modelName][section]:
                return tooltips[self.modelName][section]
            if role == Qt.DisplayRole:
                return headers[self.modelName][section]
        return QAbstractTableModel.headerData(self, section, orientation, role)

    def setHideMode(self, checkbox):
        newmode = checkbox.isChecked()
        #print("Change state: old=%s new=%s"%(self.showhidden, newmode))
        if self.showhidden == newmode: return
        self.showhidden = newmode
        # XXG this could preserve selection when unhiding rows
        if not self.showhidden:
            # first, unhide anything that is enabled
            # XXX this doesn't work for ids
            for item in prefs[self.modelName].keys():
                if prefs[self.modelName][item][0]:
                    #if prefs[self.modelName][item][self.hidecol]:
                    #   print("Unhiding %s"%item)
                    prefs[self.modelName][item][self.hidecol] = False
        self.beginResetModel()
        self.resort() # XXX or maybe we should just insert them at the end
        self.endResetModel()

    def hideItems(self, itemlist):
        if not itemlist or len(itemlist)<1: return
        # just like removeRows except the data isn't deleted
        # and we have to disable the item
        # XXX this won't work for the id table!!
        # XXG handle sequential indexes to optimize display
        # delete in reverse order so we don't have to think too hard
        # XXP warn user if enabled items are being hidden (instead of just disabling them)
        rowlist = set(i.row() for i in itemlist)
        if self.showhidden: # unhide
            for i in rowlist:
                prefs[self.modelName][self.mykeys[i]][self.hidecol] = False
                self.dataChanged.emit( self.createIndex(i,0,0), self.createIndex(i,0,1))
        else:
            rows = reversed(sorted(rowlist))
            for i in rows:
                self.beginRemoveRows(QModelIndex(), i,i)
                prefs[self.modelName][self.mykeys[i]][self.hidecol] = True
                if prefs[self.modelName][self.mykeys[i]][0]:
                    prefs[self.modelName][self.mykeys[i]][0] = False
                    #print("Disabled %s"%self.mykeys[i])
                self.mykeys = self.mykeys[:i]+self.mykeys[i+1:]
                self.endRemoveRows()

    def sort(self, col, order):
        if self.needsort or col != self.sortcol or ((order==Qt.DescendingOrder) != self.sortrev):
            self.layoutAboutToBeChanged.emit([], QtCore.QAbstractItemModel.VerticalSortHint)
            self.sortcol = col
            self.sortrev = order==Qt.DescendingOrder
            self.resort()
            self.layoutChanged.emit()
            self.dataChanged.emit(QtCore.QModelIndex(), QtCore.QModelIndex())
    def resort(self):
        oldkeys = self.mykeys
        col = self.sortcol
        self.needsort = False
        if col==None: self.sortcol=col=0
        if self.showhidden:
            newkeys = prefs[self.modelName].keys()
        else:
            newkeys = filter(lambda(x):not prefs[self.modelName][x][self.hidecol], prefs[self.modelName].keys())
        if col==0:
            self.mykeys = sorted(newkeys)
        else:
            xcol = rowxlate[self.modelName][0][col]
            self.mykeys = sorted(newkeys, key=lambda(k):prefs[self.modelName][k][xcol])
        if self.sortrev:
            self.mykeys.reverse()
        newmap = dict(map(lambda(a,b):(b,a), enumerate(self.mykeys)))
        # filter old for invalid indexes?
        if oldkeys:
            old = filter(lambda(i): i.row()<len(oldkeys), self.persistentIndexList())
            old = filter(lambda(i): oldkeys[i.row()] in newmap, old)
            new = [self.createIndex(newmap[oldkeys[i.row()]], i.column(), 0)
               for i in old]
            self.changePersistentIndexList(old,new)
    def removeRows(self,rownumber, count, parent):
        end=rownumber+count-1
        self.beginRemoveRows(parent, rownumber, end)
        # delete items from array and from model
        for i in range(rownumber, end):
            print("Deleting %s"%mykeys[i])
            del prefs[modelName][mykeys[i]]
        self.mykeys = self.mykeys[:rownumber-1]+self.mykeys[end:]
        self.endRemoveRows()
        return True
    # local convenience function
    # note: this uses the base model row data order, not the view order
    def addItem(self, itemName, rowdata=None):
        """Add an item, possibly including the whole row data"""
        totalItems = self.rowCount()
        if itemName in prefs[self.modelName]: return
        self.beginInsertRows(QModelIndex(), totalItems+1,totalItems+1 )
        self.mykeys.append(itemName)
        # add missing columns from defaults if necessary
        if len(rowdata) < len(dm.defOptTemplate[self.modelName]):
            rowdata +=  dm.defOptTemplate[self.modelName][len(rowdata):]
        if rowdata:
            prefs[self.modelName][itemName] = rowdata
        self.needsort = True  # no longer sorted
        self.endInsertRows()
    def shortItemDescr(self, row, descrColumn):
        """Get a short displayable string for the row"""
        key = self.mykeys[row]
        return "%s: %s"%(key, prefs[self.modelName][key][descrColumn])
    def deleteRowList(self,sortedrows):
        # just remove all the listed rows and reset
        self.beginResetModel()
        for i in sortedrows:
            del prefs[self.modelName][self.mykeys[i]]
        self.mykeys = filter(lambda(x): x in prefs[self.modelName], self.mykeys)
        self.endResetModel()

class ConfigWidget(QWidget):
    def __init__(self,actual_plugin):
        QWidget.__init__(self)
        self.actual = actual_plugin
        self.ui = Ui_Form()
        self.ui.setupUi(self)
        ### copy prefs to UI ###
        # first handle individual options
        opts = prefs['options']
        if 'deflang' in opts:
            self.ui.deflang.setText(opts['deflang'])
        if 'limit' in opts:
            self.ui.querylimit.setValue(opts['limit'])
        if 'selectMode' in opts:
            self.ui.selectMode.setCurrentIndex(opts['selectMode'])
        if 'pubdateMode' in opts:
            self.ui.pubdateMode.setCurrentIndex(opts['pubdateMode'])
        self.setCheckState('debugquery', self.ui.debugquery)
        self.setCheckState('mark_updated', self.ui.mark_updated)
        self.setCheckState('mark_newtags', self.ui.mark_newtags)
        self.setCheckState('mark_errors', self.ui.mark_errors)
        self.setCheckState('ptconfirmdel', self.ui.ptConfirmDelOne)
        self.setCheckState('preconvert', self.ui.preconvert)
        self.setCheckState('enableSeries', self.ui.enableSeries)
        # series options
        if 'seriesMode' in opts:
            self.ui.seriesMode.setCurrentIndex(opts['seriesMode'])
        self.ui.series1col.setText(opts['series1col'])
        if opts['series2col']: self.ui.series2col.setText(opts['series2col'])
        self.ui.seriesOrder.setCurrentIndex(opts['seriesOrder'])
        ### connect table views to data models ###
        self.property_model = self.setupMVC('properties', self.ui.property_view)
        self.tag_model = self.setupMVC('tags', self.ui.tag_view)
        self.ID_model = self.setupMVC('ids', self.ui.IDView)
        ### set up for wikidata queries ###
        self.query = WQB(prefs)
        ### connect buttons ###
        self.ui.UpdateLabels.clicked.connect(self.updateLabels)
        self.ui.ptadd.clicked.connect(self.ptAdd)
        self.ui.ptdel.clicked.connect(self.ptDelete)
        self.ui.ptimport.clicked.connect(self.ptImport)
        self.ui.idAdd.clicked.connect(self.idAdd)
        self.ui.idDel.clicked.connect(self.idDel)
        self.ui.idImport.clicked.connect(self.idImport)
        self.ui.idTest.clicked.connect(self.idTest)
        self.ui.merge1.clicked.connect(self.merge_wikidata_props)
        self.ui.merge2.clicked.connect(self.merge_wikidata)
        self.ui.ptShowHide.stateChanged.connect(self.ptShowHide)
        self.ui.ptHide.clicked.connect(self.ptHide)
        self.ui.seriesMerge.clicked.connect(self.merge_series)
        ### initialize wikidata interlink ###
        self.updateURLs()
        self.ptCheckValid()

    def ptShowHide(self):
        if self.ui.ptShowHide.isChecked():
            self.ui.ptHide.setText("Unhide")
        else:
            self.ui.ptHide.setText("Hide")
        self.property_model.setHideMode(self.ui.ptShowHide)
        self.tag_model.setHideMode(self.ui.ptShowHide)
    def ptHide(self):
        self.property_model.hideItems(self.ui.property_view.selectedIndexes())
        self.tag_model.hideItems(self.ui.tag_view.selectedIndexes())

    def setCheckState(self, optname, checkbox):
        if not optname in prefs['options']: return
        if prefs['options'][optname]:
            checkbox.setCheckState(Qt.Checked)
        else:
            checkbox.setCheckState(Qt.Unchecked)

    def ptCheckValid(self):
        # data integrity check warnings
        # XXX this check should probably be done in multiple places?
        p = prefs['properties']
        if filter(lambda(x):p[x][PI.efilter] and not p[x][PI.colname], p.keys()):
            self.ui.ptmessage.setText("WARNING: some enabled properties don't have a destination column set.")
            return
        else: #  check if all requested columns exist
            try:
                s = set(filter(lambda(x): x,[p[i][PI.colname] for i in p.keys()]))
                print("set",s)
                s -= STANDARD_METADATA_FIELDS
                s -= set(self.actual.gui.current_db.new_api.field_metadata.custom_field_metadata().keys())
                if s:
                    self.ui.ptmessage.setText("Non-existant columns requested: "+" ".join(sorted(s)))
                    return
            except Exception as ack:
                print("ptcheck: %s"%ack)
        # else if no messages, clear ptMessage
        self.ui.ptmessage.clear()

    def merge_wikidata_props(self):
        # save data just in case
        self.save_settings()
        # if any properties are selected, use those
        props = None
        proprows = self.ui.property_view.selectedIndexes()
        if proprows and len(proprows):
            props = self.query.valueList('wdt',
             [self.property_model.mykeys[i.row()] for i in proprows])
        self.actual.merge_wikidata(props)

    def merge_series(self):
        # save data just in case
        self.save_settings()
        self.actual.merge_series()

    def merge_wikidata(self):
        # save data just in case
        self.save_settings()
        # if any properties are selected, use those
        self.actual.merge_wikidata()

    def updateLabelGroup(self, group, lindex):
        results = self.query.doquery("getlabels",items=self.query.valueList('wd', prefs[group].keys()))
        if not results: return
        for result in results:
            item = self.query.getqid(result,'item')
            label = result['itemLabel']['value']
            prefs[group][item][lindex] = label

    def updateLabels(self):
        # this could be done with one query, but it would be more messy
        # first copy the new language to preferences so it will be used
        prefs['options']['deflang'] = unicode(self.ui.deflang.text())
        # update entities
        self.updateLabelGroup('tags', EI.wlabel)
        # update properties
        self.updateLabelGroup('properties', PI.wlabel)
        # identifiers too
        self.updateLabelGroup('ids', II.description)

    def ptAdd(self):
        # add what is in ptline as a property or tag
        wdid = self.ui.ptline.text()
        if len(wdid)==0:
            self.ptCheckValid()
            return
        if wdid in DM.rejectprops:
            self.ui.ptline.clear()  # clear because it can't be used
            self.ui.ptmessage.setText("%s can not be directly used as an importable property."%wdid)
            return
        if not re.match(r"[PQ]\d+$",wdid): # wdids are case sensitive apparently
            self.ui.ptmessage.setText("Incorrect format for a wikidata ID, must start with P or Q followed by digits")
            return
        if wdid in prefs['properties'] or wdid in prefs['tags']:
            self.ui.ptmessage.setText("%s is already here."%wdid)
            return
        # query first
        results = self.query.doquery('getlabels',items="(wd:%s)"%wdid)
        # this needs to detect non-existant entries somehow.
        # Maybe collect datatype too? wrong/missing datatype -> bad id
        if not results or u'xml:lang' not in results[0]['itemLabel']:
            # don't clear, it might be a fixable typo
            self.ui.ptmessage.setText("%s was not found."%wdid)
            return
        #print(results)
        wdid = self.query.getqid(results[0],'item')
        if wdid in prefs['properties'] or wdid in prefs['tags']:
            self.ui.ptmessage.setText("%s is already here."%wdid)
            return
        label = results[0]['itemLabel']['value']
        # Only enough columns are included here to cover new data
        # remaining colums automatically pulled from defOptTemplate
        if wdid[0]=='P':
            self.property_model.addItem(wdid, [False, False, label ])
        elif wdid[0]=='Q':
            self.tag_model.addItem(wdid, [False, label, label])
        else:
            self.ui.ptmessage.setText("ERR: %s!=%s"%(self.ui.ptline.text(), wdid))
            return
        self.ui.ptmessage.setText("Added.")


    def confirmDelete(self, msg, details):
        box = QMessageBox()
        box.setIcon(QMessageBox.Question)
        box.setText(msg)
        box.setDetailedText(details)
        box.setWindowTitle("Confirm delete")
        box.setStandardButtons(QMessageBox.Yes|QMessageBox.Cancel)
        return box.exec_()

    # XXP make a preference to bypass this (partially done)
    def ptDelete(self):
        propIndexes = self.ui.property_view.selectedIndexes()
        tagIndexes =  self.ui.tag_view.selectedIndexes()
        if not (propIndexes or tagIndexes):
            ptmessage.setText("No IDs are selected for deletion.")
            return
        proprows = self.rowsFromMessyIndexes(propIndexes)
        tagrows = self.rowsFromMessyIndexes(tagIndexes)
        question = "Delete "
        details=""
        if proprows:
            question += "%d properties"%len(proprows)
            details = "Delete these properties:\n"
            details += "\n".join(map(lambda(index): self.property_model.shortItemDescr(index, PI.wlabel), proprows))
        if tagrows and proprows:
            question += " and "
            details += "\n\n"
        if tagrows:
            question += "%d tags"%len(tagrows)
            details += "Delete these tags:\n"
            details += "\n".join(map(lambda(index): self.tag_model.shortItemDescr(index, EI.wlabel), tagrows))
        question += "?"
        if not prefs['options']['ptconfirmdel'] and (not proprows or not tagrows):
            answer = QMessageBox.Yes
        else:
            answer = self.confirmDelete(question, details)
        if answer==QMessageBox.Yes:
            if proprows: self.property_model.deleteRowList(proprows)
            if tagrows: self.tag_model.deleteRowList(tagrows)

    def ptImport(self):
        # import associated properties or tags based on selected books/properties
        # XXF or use saved results from last merge for visual display
        # XXXG this can be slow on large insert/query, add indicator!!
        gui = self.actual.gui
        db = gui.current_db.new_api
        book_cids = gui.library_view.get_selected_ids()
        wqb = self.query
        if not book_cids:
            self.ui.ptmessage.setText("Please select books to import properties and tags from first.")
            return
        # if this is a big query, at least indicate the button was pushed
        if len(book_cids)>20: # XXG progress?
            self.ui.ptmessage.setText("Checking %d books"%len(book_cids))
        # TTT extract list of book wdid's
        # XXF keep a set of books for selection adjustment later?
        book_wdids = set()
        for book_cid in book_cids:
            mi = db.get_metadata(book_cid)
            ids = mi.get_identifiers()
            if 'wd' in ids: book_wdids.add(ids['wd'])
        if not book_wdids:
            self.ui.ptmessage.setText("No selected books have wikidata IDs.")
            return
        # see if we are importing properties or tags
        proprows = self.ui.property_view.selectedIndexes()
        ### property import mode ###
        if not proprows:
            # import properties from books
            results = wqb.doquery('propsfrombooks', booklist=wqb.valueList('wd',book_wdids))
            if not results:
                self.ui.ptmessage.setText("No properties found in selected books.")
                return
            lastrow = len(self.property_model.mykeys)
            count=0
            for result in results:
                wdid = wqb.getqid(result, 'prop')
                if wdid in prefs['properties'] or wdid in DM.rejectprops:
                    continue
                # XXT default columns?  use wdType for something?
                self.property_model.addItem(wdid, [ False, False, result['wdLabel']['value'], '', 1, False])
                count += 1
            self.ui.ptmessage.setText("Imported %d properties from %d books"%(count, len(book_cids)))
            # guess where the rows got inserted and select them
            m = self.property_model
            self.ui.property_view.selectionModel().select(QtCore.QItemSelection(m.createIndex(lastrow,0), m.createIndex(lastrow+count,0)),QtCore.QItemSelectionModel.Select)
            return
        ### book/property tag import mode ###
        # this could be filtered by PI.use or PI.filter but maybe the user knows what they are doing
        proplist = [ self.property_model.mykeys[i.row()] for i in proprows]
        results = wqb.doquery('tagsfrombookprops', booklist=wqb.valueList('wd',book_wdids), proplist=wqb.valueList('wdt', proplist))
        if not results:
            self.ui.ptmessage.setText("No tags found in selected books.")
            return
        # XXF could save a map of properties/tags (if the query included that)
        count=0
        lastrow = len(self.tag_model.mykeys)
        # XXG progress
        for result in results:
            tag = wqb.getqid(result,'tag')
            if tag and tag not in prefs['tags']:
                label =  result['tagLabel']['value']
                self.tag_model.addItem(tag, [ False, label, label] )
                count += 1
        m = self.tag_model
        self.ui.tag_view.selectionModel().select(QtCore.QItemSelection(m.createIndex(lastrow,0), m.createIndex(lastrow+count,0)),QtCore.QItemSelectionModel.Select)
        self.ui.ptmessage.setText("Imported %d new tags from %d books"%(count, len(book_cids)))

    # shared code with import
    def addNewID(self, wdid, label, result):
        fmturl = re.sub(r'\$1', '%s', result['fmturl']['value'])
        # this may not be exactly right, but it's a start
        try:
            fmtregex = fmturl%("("+result['fmtregex']['value']+")")
        except:
            # must be a mess, just add it as is
            fmtregex = '('+result['fmtregex']['value']+")"
        # Don't care if the above worked as long as we got something.
        # The user can edit it and fix it.
        # not much more checking to do?  if we have something, use it.
        # pick a default ID label from the base hostname in the url
        idtag = ""
        m = re.match(r"http.?://([^/]*\.)?([^.]+)\.[^/.]+/", fmturl)
        if m: idtag = m.group(2)
        # XX should probably make sure idtag is not already there and blank it if it is
        # XXX check if generated ID tag is too short and pick a different one?
        if len(idtag)<2: idtag=""  # at least blank one letter tags
        self.ID_model.addItem(wdid, [False, label, idtag, fmturl, fmtregex])

    def idAdd(self):
        wdid = self.ui.idLine.text()
        if not re.match(r"P\d+$",wdid):
            self.ui.idMessage.setText("Incorrect format for a wikidata property, must start with P followed by digits")
            return
        if wdid in prefs['ids']:
            self.ui.idMessage.setText("%s is already here."%wdid)
            return
        results = self.query.doquery('idquery',items="(wd:%s)"%wdid)
        if not results or  u'xml:lang' not in results[0]['idLabel'] or not results[0]['fmturl']['value'] or not results[0]['fmtregex']['value']:
            self.ui.idMessage.setText("Identifier property %s not found."%wdid)
            return
        self.addNewID(wdid, results[0]['idLabel']['value'], results[0])
        self.ui.idMessage.setText("Added.")

    def rowsFromMessyIndexes(self, rowIndexes):
        # unordered with duplicates --> sorted unique list
        rows = set()
        for index in rowIndexes:
            rows.add(index.row())
        return sorted(rows)

    # XXP make a preference to bypass this?
    def idDel(self):
        rowIndexes = self.ui.IDView.selectedIndexes()
        if not rowIndexes:
            self.idMessage.setText("No IDs are selected for deletion.")
            return
        sortedrows = self.rowsFromMessyIndexes(rowIndexes)
        details = "Delete these IDs, books containing them are not affected but the ID won't be displayed anymore.\n"
        details += "\n".join(map(lambda(index): self.ID_model.shortItemDescr(index, II.description), sortedrows))
        answer = self.confirmDelete("Delete %d IDs?"%len(sortedrows),details)
        if answer==QMessageBox.Yes:
            self.ID_model.deleteRowList(sortedrows)

    def idImport(self):
        gui = self.actual.gui
        db = gui.current_db.new_api
        book_cids = gui.library_view.get_selected_ids()
        wqb = self.query
        if not book_cids:
            self.ui.idMessage.setText("Please select books to import IDs from first.")
            return
        book_wdids = set()
        for book_cid in book_cids:
            mi = db.get_metadata(book_cid)
            ids = mi.get_identifiers()
            if 'wd' in ids: book_wdids.add(ids['wd'])
        if not book_wdids:
            self.ui.idMessage.setText("No selected books have wikidata IDs.")
            return
        results = wqb.doquery('idsfrombooks', booklist=wqb.valueList('wd',book_wdids))
        if not results:
            self.ui.idMessage.setText("No IDs found in selected books.")
            return
        lastrow = len(self.ID_model.mykeys)
        count = 0
        for result in results:
            wdid = wqb.getqid(result, 'prop')
            if wdid in prefs['ids']:
                continue
            label = result['wdLabel']['value']
            self.addNewID(wdid, label, result)
            count += 1
        self.ui.idMessage.setText("Imported %d properties from %d books"%(count, len(book_cids)))
        # guess where the rows got inserted and select them
        m = self.ID_model
        self.ui.IDView.selectionModel().select(QtCore.QItemSelection(m.createIndex(lastrow,0), m.createIndex(lastrow+count,0)),QtCore.QItemSelectionModel.Select)
    def idTest(self):
        urlfixer = self.geturlfixer()
        if not urlfixer:
            self.ui.idMessage.setText("Wikidata metadata plugin not found.")
            return
        url = self.ui.idLine.text()
        id = urlfixer.id_from_url(url)
        print(id)
        #print("urlfixer: %s"%id)
        if id:
            self.ui.idMessage.setText(id[0]+':'+id[1])
            self.ui.idLine.setText(id[0]+':'+id[1])
        else:
            self.ui.idMessage.setText("No id returned")

    def geturlfixer(self):
        try:
            self.urlfixer
        except:  # try a late import
            from calibre_plugins.wikidata.urlfixer import UrlFixer
            self.urlfixer = UrlFixer()
        return self.urlfixer


    def setupMVC(self, modelName, view):
        model = myDictModel(modelName)
        view.setModel(model)
        # this should probably be done in column delegates?
        view.resizeColumnsToContents() # XXG does this need to be called often?
        # XXG maybe resize contents to column too?
        return model

    # generate IDs in the format UrlFixer.addIDs wants them
    def iterIDs(self):
        for id in prefs['ids']:
            v = prefs['ids'][id]
            if not v[II.dispenable]: continue
            yield (v[II.idtag], [ v[II.pastepriority], v[II.template], v[II.regex] ])

    def updateURLs(self):
        urlfixer = self.geturlfixer()
        if urlfixer:
            ick = [prefs['ids'][x][II.idtag] for x in filter( lambda(x): not prefs['ids'][x][II.dispenable], prefs['ids'].keys())]
            urlfixer.delIDs(ick)
            urlfixer.addIDs(self.iterIDs())
        else:
            print("No wikidata plugin?")

    def save_settings(self):
        prefs['options']['deflang'] = unicode(self.ui.deflang.text())
        try:
            prefs['options']['limit'] = int(self.ui.querylimit.value())
        except:
            pass # keep previous value
        opts = prefs['options']
        opts['selectMode'] = self.ui.selectMode.currentIndex()
        opts['debugquery'] = self.ui.debugquery.isChecked()
        opts['mark_updated'] = self.ui.mark_updated.isChecked()
        opts['mark_newtags'] = self.ui.mark_newtags.isChecked()
        opts['mark_errors'] = self.ui.mark_errors.isChecked()
        opts['pubdateMode'] = self.ui.pubdateMode.currentIndex()
        opts['ptconfirmdel'] = self.ui.ptConfirmDelOne.isChecked()
        opts['preconvert'] = self.ui.preconvert.isChecked()
        opts['seriesMode'] = self.ui.seriesMode.currentIndex()
        opts['series1col'] = self.ui.series1col.text()
        opts['series2col'] = self.ui.series2col.text()
        opts['seriesOrder'] = self.ui.seriesOrder.currentIndex()
        opts['enableSeries'] = self.ui.enableSeries.isChecked()
        # make sure our settings get saved
        if 'serial' in prefs:
            prefs['serial'] += 1
        else:
            prefs['serial']=1
        ### table values are updated by MVC
        # copy ids settings to wikidata plugin
        self.updateURLs()
