# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.16"  # Miscellany

from PyQt5.Qt import (Qt, QDialog, QLabel,  QFont, QWidget, QApplication,
                                       QIcon, QGridLayout, QGroupBox, QMargins, QScrollArea,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout)

from calibre.constants import DEBUG
from calibre.gui2 import gprefs
from calibre.gui2.ui import get_gui
from calibre.utils.config import JSONConfig

from polyglot.builtins import as_unicode, unicode_type, range

from calibre_plugins.audit_log.config import prefs

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(90, 90)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):

        #~ if DEBUG: self.geom = None

        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class AuditLogDialog(SizePersistedDialog):

    def __init__(self, parent, audit_log_detail,custom_column_dict):
        unique_pref_name = 'Audit_Log:audit_log_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        mytitle = 'Audit Log Activity Detail: [NNNN] Rows'
        mytitle = mytitle.replace("[NNNN]",as_unicode(len(audit_log_detail)))
        self.setWindowTitle(_(mytitle))

        self.custom_column_dict = custom_column_dict

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        self.n_log_rows = len(audit_log_detail)

        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("Time +0000 UTC")
        column_label_list.append("Book")
        column_label_list.append("Table Name")
        column_label_list.append("Table Column")
        column_label_list.append("Table Action")
        column_label_list.append("Value Old")
        column_label_list.append("Value New")
        column_label_list.append("Other")

        self.n_total_cols = 8

        self.logtable = QTableWidget(self.n_log_rows,self.n_total_cols)

        self.logtable.setSortingEnabled(False)

        self.logtable.setHorizontalHeaderLabels(column_label_list)

        self.logtable.setColumnWidth(0, 125)
        self.logtable.setColumnWidth(1,   50)
        self.logtable.setColumnWidth(2, 200)
        self.logtable.setColumnWidth(3, 100)
        self.logtable.setColumnWidth(4, 125)
        self.logtable.setColumnWidth(5, 125)
        self.logtable.setColumnWidth(6, 125)
        self.logtable.setColumnWidth(7, 100)

        self.logtable.clearContents()
        #--------------------------------------------------

        r = 0
        for row in audit_log_detail:
            try:
                #~ if DEBUG: print(as_unicode(row))
                #---------------------------
                #---------------------------
                id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other = row

                #~ id = as_unicode(id)
                book = as_unicode(book)
                time_stamp = as_unicode(time_stamp)

                if table_name == "comments":  #Calibre does INSERT OR REPLACE, not DELETE then INSERT, and not UPDATE...
                    value_old = value_old[0:25].strip()
                    value_new = value_new[0:25].strip()
                    if len(value_old) > 0:
                        value_old = value_old + "....."
                    if len(value_new) > 0:
                        value_new = value_new + "....."

                if not value_old:
                    value_old = ""

                if not value_new:
                    value_new = ""

                if not other:
                    other = ""

                if "custom_column_" in table_name:
                    other = self.get_custom_column_name(other,table_name)

                #---------------------------
                #---------------------------
                time_stamp_ = QTableWidgetItem(time_stamp)
                book_ = QTableWidgetItem(book)
                table_name_ = QTableWidgetItem(table_name)
                table_column_ = QTableWidgetItem(table_column)
                table_action_ = QTableWidgetItem(table_action)
                value_old_ = QTableWidgetItem(value_old)
                value_new_ = QTableWidgetItem(value_new)
                other_ = QTableWidgetItem(other)
                #---------------------------
                #---------------------------
                self.logtable.setItem(r,0,time_stamp_)
                self.logtable.setItem(r,1,book_)
                self.logtable.setItem(r,2,table_name_)
                self.logtable.setItem(r,3,table_column_)
                self.logtable.setItem(r,4,table_action_)
                self.logtable.setItem(r,5,value_old_)
                self.logtable.setItem(r,6,value_new_)
                self.logtable.setItem(r,7,other_)

                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("class AuditLogDialog(SizePersistedDialog):", as_unicode(e))
                return
        #END FOR

        self.n_total_rows = r

        layout.addWidget(self.logtable)

        self.logtable.setSortingEnabled(True)

        self.resize_all_columns()

        #-----------------------------------------------------

        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        layout.addWidget(self.bottom_buttonbox)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,0)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,0)

        self.push_button_save_column_widths = QPushButton(" ", self)
        self.push_button_save_column_widths.setText("Save Optimization")
        self.push_button_save_column_widths.clicked.connect(self.save_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_save_column_widths,0)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Export to Clipboard")
        self.push_button_copy_to_clipboard.setToolTip("<p style='white-space:wrap'>The Audit Log in its entirety will be copied to the Clipboard in a tab-delimited format.  Paste Special into a Spreadsheet, or Paste into a Text Document.")
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_log_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,0)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()

        self.maingui = get_gui()

        #~ msg = "AL: Number of Rows Selected: " + as_unicode(self.n_total_rows)
        #~ self.maingui.status_bar.show_message(_(msg), 10000)

    #-----------------------------------------------------
    def resize_all_columns(self):

        n = prefs['COLUMN__0_WIDTH']
        n = int(n)
        if n == 1:       #config.py default value
            self.logtable.resizeColumnsToContents()
        else:               #previously saved by the user
            column__0_width = int(prefs['COLUMN__0_WIDTH'])
            column__1_width = int(prefs['COLUMN__1_WIDTH'])
            column__2_width = int(prefs['COLUMN__2_WIDTH'])
            column__3_width = int(prefs['COLUMN__3_WIDTH'])
            column__4_width = int(prefs['COLUMN__4_WIDTH'])
            column__5_width = int(prefs['COLUMN__5_WIDTH'])
            column__6_width = int(prefs['COLUMN__6_WIDTH'])
            column__7_width = int(prefs['COLUMN__7_WIDTH'])

            if column__0_width < 50: column__0_width = 50
            if column__1_width < 50: column__1_width = 50
            if column__2_width < 50: column__2_width = 50
            if column__3_width < 50: column__3_width = 50
            if column__4_width < 50: column__4_width = 50
            if column__5_width < 50: column__5_width = 50
            if column__6_width < 50: column__6_width = 50
            if column__7_width < 50: column__7_width = 50

            self.logtable.setColumnWidth(0, column__0_width)
            self.logtable.setColumnWidth(1, column__1_width)
            self.logtable.setColumnWidth(2, column__2_width)
            self.logtable.setColumnWidth(3, column__3_width)
            self.logtable.setColumnWidth(4, column__4_width)
            self.logtable.setColumnWidth(5, column__5_width)
            self.logtable.setColumnWidth(6, column__6_width)
            self.logtable.setColumnWidth(7, column__7_width)

    #-----------------------------------------------------
    def save_column_widths(self):

        column__0_width = self.logtable.columnWidth(0)
        column__1_width = self.logtable.columnWidth(1)
        column__2_width = self.logtable.columnWidth(2)
        column__3_width = self.logtable.columnWidth(3)
        column__4_width = self.logtable.columnWidth(4)
        column__5_width = self.logtable.columnWidth(5)
        column__6_width = self.logtable.columnWidth(6)
        column__7_width = self.logtable.columnWidth(7)

        prefs['COLUMN__0_WIDTH'] = unicode_type(column__0_width)
        prefs['COLUMN__1_WIDTH'] = unicode_type(column__1_width)
        prefs['COLUMN__2_WIDTH'] = unicode_type(column__2_width)
        prefs['COLUMN__3_WIDTH'] = unicode_type(column__3_width)
        prefs['COLUMN__4_WIDTH'] = unicode_type(column__4_width)
        prefs['COLUMN__5_WIDTH'] = unicode_type(column__5_width)
        prefs['COLUMN__6_WIDTH'] = unicode_type(column__6_width)
        prefs['COLUMN__7_WIDTH'] = unicode_type(column__7_width)

        prefs

        self.save_audit_log_dialog_geometry()


    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.logtable.resizeColumnsToContents()

        self.save_audit_log_dialog_geometry()

    #-----------------------------------------------------
    def deoptimize_column_widths(self):
        self.logtable.setColumnWidth(0, 130)
        self.logtable.setColumnWidth(1, 50)
        self.logtable.setColumnWidth(2, 200)
        self.logtable.setColumnWidth(3, 100)
        self.logtable.setColumnWidth(4, 125)
        self.logtable.setColumnWidth(5, 130)
        self.logtable.setColumnWidth(6, 125)
        self.logtable.setColumnWidth(7, 100)
        self.save_audit_log_dialog_geometry()
    #-----------------------------------------------------
    def copy_log_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.logtable.selectAll()
        self.update()
        QApplication.instance().processEvents()

        if DEBUG: print("self.n_total_rows to copy to clipboard:   ", as_unicode(self.n_total_rows))

        s = ''
        s = s + "\t".join([self.logtable.horizontalHeaderItem(i).text() for i in range(0, self.n_total_cols)])
        s = s +  '\n'

        for r in range(0, self.n_total_rows):
            for c in range(0, self.n_total_cols):
                try:
                    s = s + self.logtable.item(r,c).text() + "\t"
                except:
                    s = s + "\t"
            #END FOR
            s = s[:-1] + "\n"        #eliminate last '\t'
        #END FOR

        self.clip.setText(s)

        self.save_audit_log_dialog_geometry()
    #-----------------------------------------------------
    def save_audit_log_dialog_geometry(self):
        self.dialog_closing(None)
    #-----------------------------------------------------
    def get_custom_column_name(self,other,table_name):

        table_name = table_name.replace("books","")
        table_name = table_name.replace("custom_column","")
        table_name = table_name.replace("link","")
        table_name = table_name.replace("_","")
        id = as_unicode(table_name.strip())
        if id in self.custom_column_dict:
            id,label,name = self.custom_column_dict[id]
            other = other + " " + name
        else:
            pass

        return other

    #-----------------------------------------------------


#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------

#END OF audit_log_dialog.py