import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Date;

/**
 * This class providers a formatter that reads an IETF RFC and produces an iLiad HTML repository
 * complete with manifest.xml.
 * Usage: java RFCToiLiad rfc2246.txt RFC2246
 * 
 * @author Scott Turner (scotty1024@mac.com)
 * @version $Revision: 1.0 $
 * @Date October 16, 2006
 */
public class RFCToiLiad {

    /**
     * Method is invoked by the command line.
     * Argument 0 is the name of the IETF RFC file to be formatted into HTML for the iLiad.
     * Argument 1 is the name of a directory to create and places the HTML files in.
     * Application returns a positive non-zero result value if there is an error.
     *
     * @param args[] a <code>String</code> see above.
     * @exception Exception if an error occurs
     */
    public static void main(String args[])
	throws Exception
    {
	// Get RFC file from argument 0
	String rfcName = args[0];
	File rfcFile = new File(rfcName);
	if (!rfcFile.exists()) {
	    System.err.println("RFC File " + rfcName + " not found.");
	    System.exit(1);
	}

	// Get destination directory from argument 1
	File destDir = new File(args[1]);
	if (destDir.exists()) {
	    // Zap its contents if it exists
	    if (!deleteFile(destDir)) {
		System.err.println("Couldn't clear " + args[1]);
		System.exit(2);
	    }
	}
	destDir.mkdirs();

	// Setup to parse RFC file
	ArrayList pageList = new ArrayList(512);
	ArrayList curPage = new ArrayList(70);
	String line = null;

	BufferedReader br = new BufferedReader(new FileReader(rfcFile));

	while ((line = br.readLine()) != null) {
	    //	    System.out.println(line);
	    if ("\014".equals(line)) {
		// Found page break
		pageList.add(curPage);
		curPage = new ArrayList(70);
	    } else {
		curPage.add(line);
	    }
	}

	System.out.println("RFC contains " + pageList.size() + " pages.");

	// Setup to create manifest file
	StringBuffer manifest = new StringBuffer(2048 + (pageList.size() * 64));
	String header = (String)((ArrayList)pageList.get(1)).get(0);
	int spaceIndex = header.indexOf((int)' ', 4);
	Date now = new Date();
	SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");

	manifest.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
	manifest.append("<package>\n");
	manifest.append("  <metadata>\n");
	manifest.append("    <dc-metadata>\n");
	manifest.append("      <Title>").append(header.substring(0, spaceIndex)).append("</Title>\n");
	manifest.append("      <Description>").append(header.substring(spaceIndex).trim()).append("</Description>\n");
	manifest.append("      <Date>").append(format.format(now)).append("</Date>\n");
	manifest.append("      <Format/>\n");
	manifest.append("      <Identifier/>\n");
	manifest.append("      <Language>en-us</Language>\n");
	manifest.append("      <Type>overview</Type>\n");
	manifest.append("    </dc-metadata>\n");
	manifest.append("    <y-metadata>\n");
	manifest.append("      <startpage>page0001.html</startpage>\n");
	manifest.append("      <version>000</version>\n");
	manifest.append("    </y-metadata>\n");
	manifest.append("  </metadata>\n");
	manifest.append("  <index>\n");
	manifest.append("    <pagelist type=\"overview\" count=\"").append(pageList.size()).append("\">\n");

	// Iterator through pages and create an HTML file for each page
	Iterator pages = pageList.iterator();
	int page = 1;
	StringBuffer index = new StringBuffer(8192);
	while (pages.hasNext()) {
	    curPage = (ArrayList)pages.next();
	    index.setLength(0);

	    // Add line to manifest for this HTML page
	    manifest.append("      <page number=\"").append(makePageNumber(page)).append("\" url=\"").append(makePageName(page)).append("\"/>\n");

	    // HTML file pre-amble
	    index.append("<html>\n");
	    index.append("<head>\n");

	    // iLiad links and metadata
	    if (page != 1) {
		index.append("<link rel=\"prev\" href=\"").append(makePageName(page - 1)).append("\"/>\n");
	    }
	    if (page != pageList.size()) {
		index.append("<link rel=\"next\" href=\"").append(makePageName(page + 1)).append("\"/>\n");
	    }
	    index.append("<meta name=\"keywords\" type=\"overview\">\n");

	    // HTML header info
	    index.append("<title>").append(rfcName).append("</title>\n");
	    index.append("</head>\n");
	    index.append("<body>\n");
	    index.append("<font size=\"4\">");
	    index.append("<pre>\n");

	    // Copy lines to HTML, we're <pre> so no formatting needed.
	    Iterator lines = curPage.iterator();
	    while (lines.hasNext()) {
		line = (String)lines.next();
		index.append(line);
		index.append("\n");
	    }

	    // Finish HTML for page
	    index.append("</pre>\n");
	    index.append("</font>");
	    index.append("</body>\n");
	    index.append("</html>\n");


	    // Write HTML file for page
	    File pageFile = new File(destDir, makePageName(page++));
	    PrintWriter out = new PrintWriter( new FileOutputStream( pageFile, false));
	    out.print(index);
	    out.close();
	}

	// Finish manifest
	manifest.append("    </pagelist>\n");
	manifest.append("  </index>\n");
	manifest.append("</package>\n");

	// Write manifest file
	File manifestFile = new File(destDir, "manifest.xml");
	PrintWriter out = new PrintWriter( new FileOutputStream( manifestFile, false));
	out.print(manifest);
	out.close();
    }

    /**
     * Given aPageNumber return a zero padded String value.
     *
     * @param aPageNumber an <code>int</code> value contianing page number, first page is numbered as 1.
     * @return a <code>String</code> zero padded value
     */
    public static String makePageNumber(int aPageNumber) {
	String pageName = null;
	if (aPageNumber < 10) {
	    pageName = "000";
	} else 	if (aPageNumber < 100) {
	    pageName = "00";
	} else 	if (aPageNumber < 1000) {
	    pageName = "0";
	} else 	if (aPageNumber < 10000) {
	    pageName = "";
	} else {
	    System.err.println("RFC is more than 10000 pages and is too large.");
	    System.exit(3);
	}

	return pageName + aPageNumber;
    }

    /**
     * Given a page of aPageNumber returns a string of the name to store that page's HTML in.
     *
     * @param aPageNumber an <code>int</code> value contianing page number, first page is numbered as 1.
     * @return a <code>String</code> value containing name for page's HTML to be stored as.
     */
    public static String makePageName(int aPageNumber) {
	return "page" + makePageNumber( aPageNumber) + ".html";
    }

    /**
     * This method will delete the given File. If the file is a directory then all files in
     * the directory will be deleted and then the directory itself will be deleted.
     * If if File is a simple file, it is deleted.
     *
     * @param aFile a <code>File</code> containing the path to the file/directory to be deleted
     * @return a <code>boolean</code> of true if file was deleted, false if it couldn't be removed.
     */
    public static boolean deleteFile(File aFile) {
        if (aFile.isDirectory()) {
            String[] children = aFile.list();
            for (int i=0; i<children.length; i++) {
                boolean success = deleteFile(new File(aFile, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
    
        // The directory is now empty (or a simple file) so delete it
        return aFile.delete();
    }
}
