from __future__ import (unicode_literals, division, absolute_import, print_function)

import os, shutil, re
import win32com.client #, weakref
from polyglot.builtins import unicode_type

from PyQt5.Qt import (QDialog, Qt, QVBoxLayout, QHBoxLayout, QGridLayout, 
    QGroupBox, QFrame, QSplitter, QDialogButtonBox, QIcon,
    QLabel, QTextBrowser, QPushButton, QTextEdit,
    QComboBox, QSpinBox, QRadioButton, QTableWidget, QTableWidgetItem, QVariant)

from calibre.devices.usbms.driver import debug_print

#import from this plugin
from calibre_plugins.tts_to_mp3_plugin import PLUGIN_NAME    
from calibre_plugins.tts_to_mp3_plugin.tts_typelib import constants
from calibre_plugins.tts_to_mp3_plugin.common_utils import find_icon

class SelNamesDlg(QDialog):
    ''' select ebook spine files '''
    
    def __init__(self, select_spines, spine_dict, name_text_map, voice, parent=None):
        QDialog.__init__(self, parent=parent)
        
        self.select_spines = select_spines
        self.all_spines = [d['name'] for d in spine_dict]
        self.name_text_map = name_text_map
        
        self.vname = None
        self.vrate = 0
        self.isPlaying = False
        self.testVoice = SapiVoice()
        self._advise = None
        
        self.buttonBox = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        
        gptable = QGroupBox('Select &Files:')
        laytable = QHBoxLayout()
        gptable.setLayout(laytable)
        
        self.selnameTable = SelNamesTableWidget(self)
        laytable.addWidget(self.selnameTable)
        
        gpvoice = QGroupBox('&Voice Tester:')
        layvoice = QGridLayout()
        gpvoice.setLayout(layvoice)
        
        self.testvoiceCombo = QComboBox()
        
        self.testrateSpin = QSpinBox()
        self.testrateSpin.setToolTip('Range (-10, 10)')
        self.testrateSpin.setRange(-10, 10)
        self.testrateSpin.setSingleStep(1)
        
        rateLabel = QLabel('Speech &Rate:')
        rateLabel.setAlignment(Qt.AlignRight | Qt.AlignVCenter)
        rateLabel.setBuddy(self.testrateSpin)
        
        layvoice.addWidget(self.testvoiceCombo, 0, 0, 1, 3)
        layvoice.addWidget(rateLabel, 1, 1)
        layvoice.addWidget(self.testrateSpin, 1, 2)
        
        self.playstopButton = QPushButton('x')
        self.playstopButton.setMaximumWidth(100)
        
        layvoice.addWidget(self.playstopButton, 2, 0)
        
        gpbrowser = QGroupBox('Book text:')
        gpbrowser.setToolTip('From first selected file')
        laybrowser = QVBoxLayout()
        gpbrowser.setLayout(laybrowser)
        
        self.browser = QTextBrowser()
        self.browser.setMinimumWidth(300)
        self.browser.setReadOnly(True)
        self.browser.setText('')
        laybrowser.addWidget(self.browser)
        
        gpbrowser2 = QGroupBox('... or enter your own sample text:')
        gpbrowser2.setToolTip('When using the Voice Tester, this text takes priority over the actual book text above')
        laybrowser2 = QGridLayout()
        gpbrowser2.setLayout(laybrowser2)
        
        self.usersampleTextedit = QTextEdit()
        self.usersampleTextedit.setReadOnly(False)
        clearusersampleButton = QPushButton('&Clear')
        clearusersampleButton.setMaximumWidth(100)
        
        laybrowser2.addWidget(clearusersampleButton, 0, 2)
        laybrowser2.addWidget(self.usersampleTextedit, 1, 0, 1, 3)
        
        column2 = QFrame()
        laycolumn2 = QVBoxLayout()
        column2.setLayout(laycolumn2)
        
        laycolumn2.addWidget(gpvoice)
        laycolumn2.addWidget(gpbrowser)
        laycolumn2.addWidget(gpbrowser2)
        
        splitter = QSplitter(Qt.Horizontal)
        splitter.addWidget(gptable)
        splitter.addWidget(column2)
        
        layout = QVBoxLayout()
        layout.addWidget(splitter)
        layout.addWidget(self.buttonBox)
        self.setLayout(layout)
        
        # create connect signals/slots
        self.playstopButton.clicked.connect(self.play_or_stop)
        clearusersampleButton.clicked.connect(self.clearusersampleButton_clicked)
        
        self.testvoiceCombo.currentTextChanged.connect(self.testvoiceCombo_textChanged)
        self.testrateSpin.valueChanged.connect(self.testrateSpin_valueChanged)
        
        self.selnameTable.itemSelectionChanged.connect(self.selnameTable_itemSelectionChanged)
        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)
        
        self.setWindowTitle('{0}: Select HTML files to be recorded to MP3'.format(PLUGIN_NAME))
        
        # initialise data
        self.toggle_player_settings(True)
        
        self.vname = vname = voice.get_voice_longdesc()
        self.vrate = vrate = voice.get_rate()
        
        # search Windows for available Voices
        self.voice_list = voice.get_voices_all()
        vlist = [(d['language'], d['gender'], d['fullname']) for d in self.voice_list]
        
        # populate widgets for voice selection
        vfullnames = [full for (lang, gen, full) in sorted(vlist)]
        self.testvoiceCombo.addItems(vfullnames)
        
        self.testvoiceCombo.setCurrentText(vname)
        self.testrateSpin.setValue(vrate)
        
        self.selnameTable.populate_table([d for d in spine_dict])
        # restore previous selection
        for name in self.select_spines:
            self.selnameTable.set_as_selected(name)

    def testvoiceCombo_textChanged(self, text):
        self.testVoice.set_voice(text)
        self.vname = text
            
    def testrateSpin_valueChanged(self, int):
        self.testVoice.set_rate(int)
        self.vrate = int
        
    def clearusersampleButton_clicked(self):
        self.usersampleTextedit.setText('')
        
    def selnameTable_itemSelectionChanged(self):
        self.stop_speech()
        sel_names = self.selnameTable.get_selected_spine()
        if sel_names:
            # populate the text box to allow voice testing
            name0 = [name for name in self.all_spines if name in sel_names][0]
            booktext = self.name_text_map[name0].get('booktext', '')
            self.browser.setPlainText(booktext)
            
    def accept(self):
        self.select_spines = self.selnameTable.get_selected_spine()
        self.stop()
        QDialog.accept(self)
            
    def reject(self):
        self.stop()
        QDialog.reject(self)
            
    def toggle_player_settings(self, bool):
        self.testvoiceCombo.setEnabled(bool)
        self.isPlaying = not bool
        
        self.playstopButton.setText('&Play' if bool else 'S&top')
        
        icon = find_icon('images/play.png' if bool else 'images/stop.png')
        self.playstopButton.setIcon(icon)
            
    def play_or_stop(self):
        if self.isPlaying:
            self.stop_speech()
        else:
            self.start_speech()
        
    def start_speech(self):
        book_text = self.browser.toPlainText()
        user_text = self.usersampleTextedit.toPlainText().strip()
        saytext = user_text if user_text else book_text
        if saytext:
            self.play(saytext)
        
    def play(self, text):
        self.initialize_speech()
        self.testVoice.say(text, flag=win32com.client.constants.SVSFlagsAsync)
            
    def stop_speech(self):
        self.stop()
        self.toggle_player_settings(True)
            
    def stop(self):
        self.testVoice.pause()

    def initialize_speech(self):
        self.testVoice = SapiVoice()
        self.testVoice.set_voice(self.vname)
        self.testVoice.set_rate(self.vrate)
        self._advise = self.testVoice.handler(SAPI5DriverEventSink)
        self._advise.setDriver(self)
            
    def OnEndStream(self, stream, pos):
        self.stop_speech()
            
    def OnStartStream(self, stream, pos):
        self.toggle_player_settings(False)

        
class SelNamesTableWidget(QTableWidget):
    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.setSelectionBehavior(QTableWidget.SelectRows)
        self.setSelectionMode(QTableWidget.ExtendedSelection)
        self.setEditTriggers(QTableWidget.NoEditTriggers)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table(self, lines):
        # lines = {rownum: 'toc'=?, 'name'=?, 'sample'=?, 'wordcount'=?} 
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(lines))
        header_labels = ['HTML filename', 'Word count', 'ToC entry']
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().hide()
        
        self.lines={}
        for row, line in enumerate(lines):
            self.populate_table_row(row, line)
            self.lines[row] = line
            
        self.resizeColumnsToContents()
        self.horizontalHeader().setStretchLastSection(True)
        self.setMinimumColumnWidth(0, 200)
        self.setMinimumColumnWidth(1, 100)
        self.setMinimumColumnWidth(2, 200)
        self.setMinimumSize(600, 700)
        self.setSortingEnabled(False)

    def populate_table_row(self, row, line):
        # lines = {rownum: 'toc'=?, 'name'=?, 'wordcount'=?}
        toc = line['toc']
        name = line['name']
        words = line['wordcount']

        name_cell = ReadOnlyTableWidgetItem(name)
        name_cell.setToolTip(line['sample'])
        name_cell.setData(Qt.UserRole, QVariant(name_cell))
        self.setItem(row, 0, name_cell)
        
        words_str = str(words)
        words_cell = ReadOnlyTableWidgetItem(words_str)
        words_cell.setData(Qt.UserRole, QVariant(words_str))
        words_cell.setTextAlignment(Qt.AlignCenter|Qt.AlignVCenter)
        self.setItem(row, 1, words_cell)
        
        toc_cell = ReadOnlyTableWidgetItem(toc)
        toc_cell.setData(Qt.UserRole, QVariant(toc_cell))
        self.setItem(row, 2, toc_cell)
        
    def set_as_selected(self, name):
        for row in [r for r in self.lines if self.lines[r]['name'] == name]:
            for col in range(self.columnCount()):
                self.item(row, col).setSelected(True)
            
    def get_selected_spine(self):
        return [self.item(row.row(), 0).text() for row in self.selectionModel().selectedRows()]

        
class ReadOnlyTableWidgetItem(QTableWidgetItem):
    def __init__(self, text): 
        if text is None:
            text = ''
        QTableWidgetItem.__init__(self, text, QTableWidgetItem.UserType)
        self.setFlags(Qt.ItemIsSelectable|Qt.ItemIsEnabled)

        
class EbookSelectFormat(QDialog):
    # select a single format if >1 suitable to be container-ised

    def __init__(self, gui, formats, msg, parent=None):
        QDialog.__init__(self, parent=parent)
        
        self.gui = gui
        self.formats = formats
        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok)
        
        label = QLabel(msg)
        
        self.dradio = {}
        for k in self.formats:
            self.dradio[k] = QRadioButton(k)

        gpbox1 = QGroupBox('Formats available:')
        lay1 = QHBoxLayout()
        gpbox1.setLayout(lay1)
        
        for fmt in self.formats:
            lay1.addWidget(self.dradio[fmt])
            
        if 'EPUB' in self.formats:
            self.dradio['EPUB'].setChecked(True)
        else:
            self.dradio[self.formats[0]].setChecked(True)
        
        lay = QVBoxLayout()
        lay.addWidget(label)
        lay.addWidget(gpbox1)
        lay.addStretch()
        lay.addWidget(buttonBox)
        self.setLayout(lay)
        
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        
        self.setWindowTitle('{0}: Select format'.format(PLUGIN_NAME))
        
        icon = find_icon('images/plugin_icon.png')
        self.setWindowIcon(icon)
        
    @property
    def result(self):
        return [k for k in self.formats if self.dradio[k].isChecked()]

        
class SAPI5DriverEventSink(object):
    def __init__(self):
        self._driver = None

    def setDriver(self, driver):
        self._driver = driver

    def OnEndStream(self, stream, pos):
        self._driver.OnEndStream(stream, pos)
        #print("SAPI5DriverEventSink: OnEndStream, stream, pos")
        
    def OnStartStream(self, stream, pos):
        self._driver.OnStartStream(stream, pos)
        #print('SAPI5DriverEventSink OnStartStream', stream, pos)

        
class SapiVoice(object):
    #A speech API using the Microsoft SAPI through COM
    def __init__(self):
        #self.spv = win32com.client.DispatchWithEvents("SAPI.SpVoice", EventHandler)
        self.spv = win32com.client.Dispatch("SAPI.SpVoice")
        
    def handler(self, event_handler):
        return win32com.client.WithEvents(self.spv, event_handler)
        
    def get_voices_all(self):
        # create a list of dictionaries, 1 per Voice
        # each dict contains key/value for useful voice attributes
        voice_list = []
        try:
            voices = self.spv.GetVoices("", "")
        except:
            voices = [self.spv.Voice]
        idx = 0
        for v in voices:
            temp = {}
            temp['idx'] = idx
            temp['fullname'] = v.GetDescription()
            temp['gender'] = v.GetAttribute('Gender')
            temp['language'] = v.GetAttribute('Language')
            temp['name'] = v.GetAttribute('Name')
            temp['age'] = v.GetAttribute('Age')
            temp['vendor'] = v.GetAttribute('Vendor')
            voice_list.append(temp)
            idx += 1
        return voice_list

    def get_voices(self, name=''):
        #Get a list of voices, search by name optional
        voice_list = []
        try:
            voices = self.spv.GetVoices("", "")
        except:
            voices = [self.spv.Voice]

        if name:
            for v in voices:
                if name in v.GetDescription():
                    voice_list.append(v)
                    break
        else:
            for v in voices:
                voice_list.append(v)

        return voice_list

    def get_voice_names(self):
        #Get the names of all the voices
        return [voice.GetDescription() for voice in self.get_voices()]

    def set_voice(self, voice):
        #Set the voice to the given voice
        if isinstance(voice, unicode_type):
            temps = self.get_voices(voice)
            if temps:
                self.spv.Voice = temps[0]
        else:
            self.spv.Voice = voice
        return

    def get_audio_outputs(self, name=''):
        #Get the audio outputs, search for the one with the name if given
        output_list = []
        outputs = self.spv.GetAudioOutputs()

        if name:
            for ao in outputs:
                if name in ao.GetDescription():
                    output_list.append(ao)
                    break
        else:
            for ao in outputs:
                output_list.append(ao)

        return output_list

    def get_audio_output_names(self):
        #Get the names of all the audio outputs
        return [ao.GetDescription() for ao in self.get_audio_outputs()]

    def set_audio_output(self, output):
        #Set the voice audio output to the given output
        if isinstance(output, unicode_type):
            temps = self.get_audio_outputs(output)
            if temps:
                self.spv.AudioOutput = temps[0]
        else:
            self.spv.AudioOutput = output
        return
    
    def set_rate(self, rate):
        #Set the voice speed. -10 is slowest, 10 is fastest
        if isinstance(rate, (int, float)):
            if -10 <= rate <= 10:
                self.spv.Rate = int(rate)

    def get_voice_longdesc(self):
        #Get current voice long name
        try:
            vname = self.spv.Voice.GetDescription()
        except:
            vname = None
        return vname

    def get_voice(self):
        #Get current voice object
        return self.spv.Voice

    def get_voice_attrib(self, attrib):
        # Get selected attrib of current voice
        # 'Gender', Language, 'Name', 'Age', 'Vendor',
        # 'DataVersion', 'SharedPronunciation', 'Version'
        return self.spv.Voice.GetAttribute(attrib)

    def get_audio_output(self):
        #Get current audio output name
        return self.spv.AudioOutput.GetDescription()

    def get_rate(self):
        #Get current voice rate
        try:
            vrate = self.spv.Rate
        except:
            vrate = 0
        return vrate

    def get_volume(self):
        #Get current voice volume
        return self.spv.Volume

    def say(self, message, flag=win32com.client.constants.SVSFDefault):
        self.spv.Speak(message, flag)
        return

    def pause(self):
        #Get pause current speaking stream
        return self.spv.Pause()

    def resume(self):
        #Get restarte current speaking stream
        return self.spv.Resume()

    def create_recording(self, filename, message, doevents=False):
        ''' Make a recording of the given message to the file
            The file should be a .wav as the output is
            PCM 22050 Hz 16 bit, Little endianness, Signed '''
            
        # save voice current AudioOutputStream
        temp_stream = self.spv.AudioOutputStream
        
        # create stream for wav file
        stream = win32com.client.Dispatch('Sapi.SpFileStream')
        stream.Open(filename, win32com.client.constants.SSFMCreateForWrite, doevents)
        self.spv.AudioOutputStream = stream
        
        self.say(message)
        stream.Close()
        
        # restore voice to original AudioOutputStream
        self.spv.AudioOutputStream = temp_stream

        
if __name__ == "__main__":
    # called from Op sys
    import sys
    from PyQt5.Qt import QApplication

    # create user output dialog    
    app = QApplication(sys.argv)
    
    voice = SapiVoice()
    vname = voice.get_voice_longdesc()
    vrate = voice.get_rate()

    form = SelNamesDlg([], [], {}, voice, parent=None)

    form.show()
    app.exec_()
