# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.170"   #Technical changes after Python 3.8 testing with Calibre 4.99.3

import os,sys,apsw,ast,json
from PyQt5.Qt import Qt, QDialog, QVBoxLayout, QFont, QWidget, QSize, QPushButton, QComboBox
from calibre import isbytestring
from calibre.constants import filesystem_encoding
from calibre.gui2 import gprefs, question_dialog, info_dialog
from calibre.utils.config import to_json
from calibre.utils.serialize import json_dumps, json_loads   #cannot use simple json.dumps, json.loads...

from polyglot.builtins import as_bytes, as_unicode, is_py3, iteritems

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(100, 100)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SynchronizeSavedSearchesDialog(SizePersistedDialog):

    def __init__(self,gui,guidb,icon):

        parent = gui
        unique_pref_name = 'job_spy:synchronize_saved_searches_dialog_'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.gui = gui
        self.guidb = guidb

        self.gprefs = gprefs

        self.setWindowTitle('JS+ GUI Tool: Copy Saved Searches')
        self.setWindowIcon(icon)

        self.layout = QVBoxLayout()
        self.setLayout(self.layout)

        self.setToolTip("<p style='white-space:wrap'>Copy Library-specific Saved Searches to other Libraries.  Existing Saved Searches will be retained. ")

        font = QFont()
        font.setBold(False)
        font.setPointSize(11)

        self.target_library_combobox = QComboBox()
        self.target_library_combobox.setEditable(False)
        self.target_library_combobox.setFont(font)
        self.target_library_combobox.setToolTip("<p style='white-space:wrap'>This is the 'target' Library to which to copy the current Library's 'Saved Searches'. ")
        self.layout.addWidget(self.target_library_combobox)

        self.target_library_combobox.setObjectName("saved_search_target_library_combobox")

        self.get_target_library_list()

        self.target_library_combobox.addItem('Choose a Target Library')
        self.target_library_combobox.setFont(font)
        for row in self.target_library_list:
            self.target_library_combobox.addItem(row)
        #END FOR

        self.target_library_combobox.setMaxVisibleItems(len(self.target_library_list))

        self.layout.addStretch(1)

        self.push_button_copy_saved_searches = QPushButton("Copy Saved Searches to the Selected Library", self)
        self.push_button_copy_saved_searches.clicked.connect(self.copy_saved_searches)
        self.push_button_copy_saved_searches.setFont(font)
        self.push_button_copy_saved_searches.setDefault(False)
        self.push_button_copy_saved_searches.setToolTip("<p style='white-space:wrap'>Copy Saved Searches from the current Library to the selected Library.  Any existing Saved Searches in the selected Library will be retained.")
        self.layout.addWidget(self.push_button_copy_saved_searches)

        self.layout.addStretch(2)
        #-----------------------------------------------------
        self.push_button_cancel = QPushButton("Exit")
        self.push_button_cancel.clicked.connect(self.reject)
        self.push_button_cancel.setDefault(True)
        self.push_button_cancel.setToolTip("<p style='white-space:wrap'>Exit.")
        self.layout.addWidget(self.push_button_cancel)
        #-----------------------------------------------------

        self.target_library_combobox.showPopup()

        self.resize_dialog()
    #----------------------------------------------------
    def copy_saved_searches(self):
        self.save_dialog_geometry()
        selected_target_library = self.target_library_combobox.currentText()
        if selected_target_library == 'Choose a Target Library':
            return
        if question_dialog(self.gui, "JS GUI Tool - Copy Saved Searches", "Copy Saved Searches?  Are you sure?"):
            self.copy_saved_search_to_target(selected_target_library)
        else:
            info_dialog(self.gui, "JS GUI Tool - Copy Saved Searches", "Nothing done.").show()
            return
    #----------------------------------------------------
    def copy_saved_search_to_target(self,selected_target_library):

        self.get_current_library_saved_searches()
        if len(self.saved_searches) == 0 or self.saved_searches == "{}":
            msg = "Sorry, but this current Library has no saved searches to copy to another Library.<br><br>Nothing done."
            return info_dialog(self.gui, "JS GUI Tool - Copy Saved Searches", msg).show()

        my_db,my_cursor,is_valid = self.apsw_connect_to_target_library(selected_target_library)
        if not is_valid:
             return error_dialog(self.gui, _('Job Spy'),_('Database Connection Error.  Cannot Connect to the Selected Target Library.'), show=True)

        mysql = "SELECT key,val FROM preferences WHERE key = 'saved_searches' "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
            tmp_rows.append(u'saved_searches', u'{}')
        for row in tmp_rows:
            key,val = row
            if is_py3:
                val = as_unicode(val)
            else:
                val = as_bytes(val)
            tmp_dict = ast.literal_eval(val)
        #~ for k,v in self.saved_searches.iteritems():
        for k,v in iteritems(self.saved_searches):
            tmp_dict[k] = v
        if is_py3:
            val = as_unicode(tmp_dict)
        else:
            val = as_bytes(tmp_dict)
        val = ast.literal_eval(val)
        val =  json_dumps(val, indent=2, default=to_json)
        #~ print("saved searches dict: ", val)

        try:
            my_cursor.execute("begin")
            mysql = "INSERT OR REPLACE INTO preferences (id,key,val) VALUES (null,'saved_searches',?) "
            my_cursor.execute(mysql,([val]))
            my_cursor.execute("commit")
            msg = "Saved Searches for Library " + selected_target_library + "<br>were successfully updated."
        except Exception as e:
            msg = "Error: " + as_unicode(e)

        my_db.close()

        return info_dialog(self.gui, "JS GUI Tool - Copy Saved Searches", msg).show()

    #-----------------------------------------------------------------------------------------------
    def get_current_library_saved_searches(self):
        self.saved_searches = self.guidb.prefs.get('saved_searches')
    #-----------------------------------------------------------------------------------------------
    def get_target_library_list(self):
        # has the complete list of all currently known Calibre libraries

        self.current_library_path = self.gui.library_view.model().db.library_path
        if isbytestring(self.current_library_path):
            self.current_library_path = self.current_library_path.decode(filesystem_encoding)
        self.current_library_path = self.current_library_path.replace(os.sep, '/')

        self.target_library_list = []

        stats = self.gprefs.get('library_usage_stats', {})

        #~ for k,v in stats.iteritems():
        for k,v in iteritems(stats):
            k = k.strip()
            if k != self.current_library_path:
                self.target_library_list.append(k)
        #END FOR

        del stats

        self.target_library_list.sort()

    #----------------------------------------------------
    def apsw_connect_to_target_library(self,selected_target_library):

        if selected_target_library == self.current_library_path:
            return

        path = selected_target_library
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            print("path to metadata.db is: ", path)
            print("error: ", as_unicode(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#END OF synchronize_saved_searches_dialog.py