# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.170"   #Technical changes after Python 3.8 testing with Calibre 4.99.3


from PyQt5.Qt import (Qt, QDialog, QLabel,  QFont, QWidget, QApplication,
                                       QIcon, QGridLayout, QGroupBox, QMargins, QScrollArea,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout)

from calibre.constants import DEBUG
from calibre.gui2 import gprefs
from calibre.gui2.ui import get_gui

from polyglot.builtins import as_unicode, range

from calibre_plugins.job_spy.config import prefs

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(90, 90)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class ShortcutsListingDialog(SizePersistedDialog):

    def __init__(self, parent, tmp_list):
        unique_pref_name = 'Job_Spy:shortcut_listing_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        mytitle = 'JS+ GUI Tool:  Active Keyboard Shortcut Assignments'
        self.setWindowTitle(_(mytitle))

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        shortcut_assignments_list = []

        for row in tmp_list:       #row = source, group, name, shortcut
            source,group,name,shortcut = row
            if not isinstance(shortcut,tuple):
                continue
            shortcut = list(shortcut)  # shortcut is a 1-many tuple...  ('Shift+N', 'Shift+F3')
            l = len(shortcut)
            if l == 0:
                continue
            elif l == 1:
                shortcut = shortcut[0]
            else:
                s = ""
                for sc in shortcut:
                    s = s + sc + ", "
                #END FOR
                s = s.strip()
                s = s[0:-1]
                shortcut = s
            shortcut = as_unicode(shortcut)
            newrow = name,group,shortcut,source               # sort by name,group,shortcut,source
            shortcut_assignments_list.append(newrow)
        #END FOR
        del tmp_list

        shortcut_assignments_list.sort()

        self.n_log_rows = len(shortcut_assignments_list)

        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("Name")
        column_label_list.append("Group")
        column_label_list.append("Shortcut")
        column_label_list.append("Source")

        self.n_total_cols = 4

        self.logtable = QTableWidget(self.n_log_rows,self.n_total_cols)

        self.logtable.setSortingEnabled(False)

        self.logtable.setHorizontalHeaderLabels(column_label_list)

        self.logtable.setColumnWidth(0, 200)
        self.logtable.setColumnWidth(1, 200)
        self.logtable.setColumnWidth(2, 200)
        self.logtable.setColumnWidth(3, 200)

        self.logtable.clearContents()
        #--------------------------------------------------

        r = 0
        for row in shortcut_assignments_list:
            try:
                #~ if DEBUG: print(as_unicode(row))
                #---------------------------
                #---------------------------
                name,group,shortcut,source = row

                source = as_unicode(source)
                group = as_unicode(group)
                name = as_unicode(name)
                shortcut = as_unicode(shortcut)

                #---------------------------
                #---------------------------
                source_ = QTableWidgetItem(source)
                group_ = QTableWidgetItem(group)
                name_ = QTableWidgetItem(name)
                shortcut_ = QTableWidgetItem(shortcut)
                #---------------------------
                #---------------------------
                self.logtable.setItem(r,0,name_)
                self.logtable.setItem(r,1,group_)
                self.logtable.setItem(r,2,shortcut_)
                self.logtable.setItem(r,3,source_)
                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("class ShortcutsListingDialog(SizePersistedDialog):", as_unicode(e))
                return
        #END FOR

        self.n_total_rows = r

        layout.addWidget(self.logtable)

        self.logtable.sortItems(0,Qt.AscendingOrder )

        self.logtable.setSortingEnabled(True)

        self.resize_all_columns()

        #-----------------------------------------------------

        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        layout.addWidget(self.bottom_buttonbox)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.setToolTip("<p style='white-space:wrap'>The listing columns will be resized based on their longest row contents for each column.")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,0)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.setToolTip("<p style='white-space:wrap'>The listing columns will be resized to a fixed width regardless of their contents.")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,0)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Export to Clipboard")
        self.push_button_copy_to_clipboard.setToolTip("<p style='white-space:wrap'>The Shortcuts Listing in its entirety will be copied to the Clipboard in a tab-delimited format.  Paste Special into a Spreadsheet, or Paste into a Text Document.")
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_log_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,0)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()

        self.maingui = get_gui()

        self.deoptimize_column_widths()

    #-----------------------------------------------------
    def resize_all_columns(self):
        self.logtable.resizeColumnsToContents()
        self.save_shortcuts_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.logtable.resizeColumnsToContents()
        self.save_shortcuts_listing_dialog_geometry()
    #-----------------------------------------------------
    def deoptimize_column_widths(self):
        self.logtable.setColumnWidth(0, 200)
        self.logtable.setColumnWidth(1, 200)
        self.logtable.setColumnWidth(2, 200)
        self.logtable.setColumnWidth(3, 100)
    #-----------------------------------------------------
    def copy_log_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.logtable.selectAll()

        s = ''
        s = s + "\t".join([as_unicode(self.logtable.horizontalHeaderItem(i).text()) for i in range(0, self.n_total_cols)])
        s = s +  '\n'
        for r in range(0, self.n_total_rows):
            for c in range(0, self.n_total_cols):
                try:
                    s = s + as_unicode(self.logtable.item(r,c).text()) + "\t"
                except AttributeError:
                    s = s + "\t"
            s = s[:-1] + "\n"        #eliminate last '\t'
        self.clip.setText(s)

        self.save_shortcuts_listing_dialog_geometry()
    #-----------------------------------------------------
    def save_shortcuts_listing_dialog_geometry(self):
        self.dialog_closing(None)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------

#END OF shortcuts_listing_dialog.py