# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.170"   #Technical changes after Python 3.8 testing with Calibre 4.99.3

import os,sys,apsw
from PyQt5.Qt import Qt, QDialog, QVBoxLayout, QFont, QWidget, QSize, QPushButton, QComboBox
from calibre import isbytestring
from calibre.constants import filesystem_encoding
from calibre.gui2 import gprefs, question_dialog, info_dialog

from polyglot.builtins import as_unicode

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(100, 100)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class PurgePrefsDialog(SizePersistedDialog):

    def __init__(self,gui,icon,preferences_list):

        parent = gui
        unique_pref_name = 'job_spy:purge_prefs_dialog_'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.gui = gui

        self.setWindowTitle('JS+ GUI Tool: Purge Preferences')
        self.setWindowIcon(icon)

        self.layout = QVBoxLayout()
        self.setLayout(self.layout)

        self.setToolTip("<p style='white-space:wrap'>Purge the (orphaned) Library-specific preferences for the selected Plug-in.")

        font = QFont()
        font.setBold(False)
        font.setPointSize(11)

        self.preferences_combobox = QComboBox()
        self.preferences_combobox.setEditable(False)
        self.preferences_combobox.setFont(font)
        self.preferences_combobox.setToolTip("<p style='white-space:wrap'>This is the (orphaned) Plug-in for which you wish to remove the (orphaned) Library-specific preferences. ")
        self.layout.addWidget(self.preferences_combobox)

        self.preferences_combobox.setObjectName("preferences_combobox")   # required for .findChildren()

        self.preferences_combobox.addItem('Choose a Plug-in')
        preferences_list.sort()
        for plugin in preferences_list:
            self.preferences_combobox.addItem(plugin)
        #END FOR

        self.preferences_combobox.setMaxVisibleItems(len(preferences_list))

        self.layout.addStretch(1)

        self.push_button_purge_prefs = QPushButton("Purge Library-specific Preferences for Selected Plug-in", self)
        self.push_button_purge_prefs.clicked.connect(self.purge_selected_plugin)
        self.push_button_purge_prefs.setFont(font)
        self.push_button_purge_prefs.setDefault(False)
        self.push_button_purge_prefs.setToolTip("<p style='white-space:wrap'>Purge the Library-specific preferences for the selected Plug-in.")
        self.layout.addWidget(self.push_button_purge_prefs)

        self.layout.addStretch(2)
        #-----------------------------------------------------
        self.push_button_cancel = QPushButton("Exit")
        self.push_button_cancel.clicked.connect(self.reject)
        self.push_button_cancel.setDefault(True)
        self.push_button_cancel.setToolTip("<p style='white-space:wrap'>Exit.")
        self.layout.addWidget(self.push_button_cancel)
        #-----------------------------------------------------

        self.preferences_combobox.showPopup()

        self.resize_dialog()
    #----------------------------------------------------
    def purge_selected_plugin(self):
        self.save_dialog_geometry()
        selected_plugin = self.preferences_combobox.currentText()
        if selected_plugin == 'Choose a Plug-in':
            return
        if question_dialog(self.gui, "Job Spy: Bonus GUI Tool - Purge Preferences", "Purge the preferences for the selected Plug-in?  Are you sure?"):
            self.delete_table_preferences(selected_plugin)
            self.preferences_combobox.removeItem(self.preferences_combobox.currentIndex())
            self.preferences_combobox.setCurrentText('Choose a Plug-in')
            msg = "Purged from the Preferences table in metadata.db: " + selected_plugin + "<br><br>You must manually delete the .json file (if any) for the Plug-in from the Calibre configuration directory."
            info_dialog(self.gui, "Job Spy: Bonus GUI Tool - Purge Preferences", msg).show()
            return
        else:
            info_dialog(self.gui, "Job Spy: Bonus GUI Tool - Purge Preferences", "Nothing done.").show()
            return
    #----------------------------------------------------
    def delete_table_preferences(self,selected_plugin):

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(self.gui, _('Job Spy'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)

        try:
            my_cursor.execute("begin")
            mysql = "DELETE FROM preferences WHERE key LIKE 'namespaced:%' AND key LIKE '%[PI]%'  "
            mysql = mysql.replace("[PI]",selected_plugin)
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
        except Exception as e:
            print(as_unicode(e))

        my_db.close()

    #----------------------------------------------------
    def apsw_connect_to_library(self):

        my_db = self.gui.library_view.model().db

        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            print("path to metadata.db is: ", path)
            print("error: ", as_unicode(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#END OF purge_prefs_dialog.py