# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import absolute_import
from six.moves import range
from polyglot.builtins import is_py3
from functools import cmp_to_key

__license__   = 'GPL v3'
__copyright__ = '2011-2018, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Sort names Kindle style
#####################################################################

import re

import calibre_plugins.kindle_collections.config as cfg
from calibre_plugins.kindle_collections.utilities import debug_print

# Articles ignored at start of title, unless two identical names have articles
SORT_ARTICLES = [ u'a', u'an', u'the' ]

# Kindle ignores these characters and sorts them randomly (unless name is only invisible characters:!
SORT_INVISIBLE = u'-?/]#.\'\\*})&:@%;"!,'  # means the visible are " _`^~'([{$+<=>|" + 0-9, a-z
# And on the Touch...  Didn't care to test the 1 char invisible stuff, so the order might not be completely accurate...
TOUCH_SORT_INVISIBLE = u' #[:?)\\"{.!/%…;&]},*\'(@'
# And on the PaperWhite... (Yeah, completely different than on the Touch... Oh, joy...)
PW_SORT_INVISIBLE = u' _·™…].¢#*`)"¬£-;@&{^/,©%~=|®}€>[($+:?!¥\\<§'

# Sort order for all chars (invisible needed in case the collection is 1 char long)
SORT_CHARACTERS = u'- _,;:!?/.`^~\'"()[]{}@$*&#%+<=>|0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'
# And on the Touch... (Note that you shouldn't be able to put a leading/trailing whitespace in a collection name,
# be it via the default UI or Collections Manager, that's why I put it in the invisible list on the Touch).
TOUCH_SORT_CHARACTERS = u'_-`^§©®+<=>¬|~¢$£¥€0123456789 abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ#[:·?)\\"{.!/%…;&]},*\'(@'
# FIXME: There's unicode stuff in there that might make Python implode...
# These are the unicode chars that might have been removed to avoid issues, and their place in the sort order:
# >, ¬, |   :, ·, ?   %, …, ;   @ possibly sorted like A   ™ possibly sorted like tm or TM
# PaperWhite... Don't have a lot of choices anymore to put stuff in front... A single quote, and that's it :/.
# The case sorting interacts with the invisible chars in a weird way, some of them sit between lowercase and uppercase if they precede the exact same letter...
PW_SORT_CHARACTERS = u'\'012345789 abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_·™…].¢#*`)"¬£-;@&{^/,©%~=|®}€>[($+:?!¥\\<§'

#####################################################################

class KindleSort():

    # Sort collections as the Kindle 3 sorts them in sort 'By Title'
    # No leading/trailing spaces, lowercase/uppercase ignored and sorted randomly
    # Invisible characters are ignored unless there are only invisible characters in which case sorted randomly
    # If you put a space after a symbol you can't type first letter of title and jump to it
    # When sorting by Collections/Most Recent Kindle uses last access time

    def __init__(self):
        # Change strings to lists for easier lookup
        # Kindle PaperWhite handling...
        if cfg.config_settings['kindle_model_version'] >= 5020:
            self.sort_list = dict([(PW_SORT_CHARACTERS[x],x) for x in range(len(PW_SORT_CHARACTERS))])
            self.invisible_list = set([ PW_SORT_INVISIBLE[x] for x in range(len(PW_SORT_INVISIBLE)) ])
        # Kindle Touch handling...
        elif cfg.config_settings['kindle_model_version'] >= 5000:
            self.sort_list = dict([(TOUCH_SORT_CHARACTERS[x],x) for x in range(len(TOUCH_SORT_CHARACTERS))])
            self.invisible_list = set([ TOUCH_SORT_INVISIBLE[x] for x in range(len(TOUCH_SORT_INVISIBLE)) ])
        else:
            self.sort_list = dict([(SORT_CHARACTERS[x],x) for x in range(len(SORT_CHARACTERS))])
            self.invisible_list = set([ SORT_INVISIBLE[x] for x in range(len(SORT_INVISIBLE)) ])
        self.initialize()

    def initialize(self):
        debug_print('BEGIN Initializing KindleSort')
        self.invisible_version = {}
        self.no_article_version = {}
        debug_print('END Initializing KindleSort')

    # Sort collection Kindle style
    # Either just sort names, or also pass in a dictionary of names with values to sort on
    def sort_names(self, names, reverse_sort=False, sort_text=None):
        debug_print('BEGIN Kindle sorting')
        if is_py3:
            # c.f., http://python3porting.com/preparing.html#keycmp-section
            if sort_text is not None:
                names.sort(key=lambda path: sort_text[path])
                # FIXME: cmp_to_key will slow this down. Convert it manually?
                names.sort(key=cmp_to_key(self.sort_compare_kindle_names), reverse=reverse_sort)
            else:
                names.sort(key=cmp_to_key(self.sort_compare_kindle_names), reverse=reverse_sort)
        else:
            if sort_text is not None:
                names.sort(cmp=self.sort_compare_kindle_names, key=lambda path: sort_text[path], reverse=reverse_sort)
            else:
                names.sort(cmp=self.sort_compare_kindle_names, reverse=reverse_sort)

        debug_print('END Kindle sorting')
        return names

    # Sort order of invisible characters is random
    def strip_invisible(self, text):
        # Save previous value if there is one to speed up lookups
        if text not in self.invisible_version:
            temp = ''
            for i in range(len(text)):
                if text[i] not in self.invisible_list:
                    temp += text[i]
            if text != temp and len(temp) > 0:
                self.invisible_version[text] = temp
            else:
                self.invisible_version[text] = text

        return self.invisible_version[text]

    # Remove articles at start of text
    def strip_articles(self, text):
        # Save previous value if there is one to speed up lookups
        if text not in self.no_article_version:
            self.no_article_version[text] = (text, '')
            for article in SORT_ARTICLES:
                temp = re.sub('^' + article + '[^_0-9a-z]', '', text)
                if text != temp:
                    self.no_article_version[text] = (temp, article)
                    break;
        return (self.no_article_version[text])

    # Compare two names Kindle style
    def sort_compare_kindle_names(self, a, b):
        # Ignore case - Kindle doesn't sort Aa and aA consistently!
        a = a.lower()
        b = b.lower()

        # Strip invisible characters from the entire name, unless just invisible
        a = self.strip_invisible(a)
        b = self.strip_invisible(b)

        # Remove articles at start of text
        (a, articlea) = self.strip_articles(a)
        (b, articleb) = self.strip_articles(b)

        # Compare the two strings, character by character using the defined Kindle order
        alen = len(a)
        blen = len(b)
        aval = 0
        bval = 0
        invalid_char = False
        try:
            for i in range(min(alen,blen)):
                if a[i] in self.sort_list and b[i] in self.sort_list:
                    aval = self.sort_list[a[i]]
                    bval = self.sort_list[b[i]]
                    if aval != bval:
                        break
                else:
                    invalid_char = True
                    break
        except:
            compare = True
            invalid_char = True

        if invalid_char:
                try:
                    compare = cmp(a, b)
                except:
                    compare = True
                    # Force repr to avoid blowing up on UnicodeDecode implicit conversion errors...
                    debug_print('Skipping comparison - invalid character found when comparing "{0!r}" and "{1!r}"'.format(a, b))
        else:
            if aval == bval:
                if alen == blen and articlea and articleb:
                    compare = cmp(articlea, articleb)
                else:
                    compare = (alen > blen) - (alen < blen)
            else:
                compare = (aval > bval) - (aval < bval)

        return compare
