# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import absolute_import

__license__   = 'GPL v3'
__copyright__ = '2011-2018, meme'
__docformat__ = 'restructuredtext en'

############################################################
# Kindle Edit window
############################################################

import re
from functools import partial

try:
    from PyQt5 import Qt as QtGui
    from PyQt5.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QLineEdit, QTableWidget, QTableWidgetItem, \
                         QAbstractItemView, QToolButton, QDialog, QDialogButtonBox, \
                         QCheckBox, QGridLayout, QSpinBox, QGroupBox
except:
    from PyQt4 import QtGui
    from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QLineEdit, QTableWidget, QTableWidgetItem, \
                         QAbstractItemView, QToolButton, QDialog, QDialogButtonBox, \
                         QCheckBox, QGridLayout, QSpinBox, QGroupBox


from calibre.gui2 import error_dialog, question_dialog

import calibre_plugins.kindle_collections.config as cfg
import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.kindle_device as kindle_device
import calibre_plugins.kindle_collections.calibre_info as calibre_info
import calibre_plugins.kindle_collections.save as save
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME
from calibre_plugins.kindle_collections.utilities import debug_print, SizePersistedDialog, get_icon, \
        ComboTableWidgetItem, CheckableTableWidgetItem, CheckableBoxWidgetItem, ReadOnlyTableWidgetItem

READER_PREFS_DIALOG_GEOMETRY = 'kindle_collections plugin:edit settings dialog: geometry'

DEFAULTS = {    'HORIZONTAL_MARGIN': '23',
                'FONT_SIZE': '21',
                'LINE_SPACING': '2',
                'JUSTIFICATION': 'full',
                'FONT_FAMILY': 'serif',
                'DICTIONARY': '',
                'LAST_BOOK_READ': '',
                'ALLOW_JUSTIFICATION_CHANGE': 'false',
                'ALLOW_TWO_COLUMN_VIEW': 'false',
                'ALLOW_ARTICLE_THUMBNAIL': 'false',
                'ALLOW_READING_INDICATOR': 'true',
                'ALLOW_USER_FONT': 'false',
                'ALLOW_USER_LINE_SPACING': 'false'
}

FONT_FAMILY_DEFAULTS = [ 'serif', 'condensed', 'sanserif' ]
JUSTIFICATION_OPTIONS = [ 'full', 'left' ]

############################################################


def run(parent):
        debug_print('BEGIN Modifying Settings')
        gui = parent.gui

        if not cfg.init(parent):
            msg.message.display()
            debug_print('END Modifying Settings - Initialization failed')
            return

        # Make sure the settings are up to date, we make extensive use of model checking here...
        if not cfg.store.is_current_store_format():
            msg.message.info('Please run "Customize collections" and select OK to save your customizations before running Modify Kindle Settings.')
            msg.message.display()
            return

        # Not supported on Kindle Touch/PaperWhite
        if cfg.config_settings['kindle_model_version'] < 5000:
            kindle_device.init(calibre_info.ci.device_path)
            reader_prefs = kindle_device.kdevice.read_reader_prefs()
            old_fontname = reader_prefs['FONT_FAMILY'] if 'FONT_FAMILY' in reader_prefs else None
        else:
            reader_prefs = {}
            old_fontname = None

        # Set defaults for all values
        for key in DEFAULTS.keys():
            if key not in reader_prefs:
                reader_prefs[key] = DEFAULTS[key]

        # Get user fonts
        warnings = ""
        (font_family_options, warnings) = get_font_names()
        continue_modify = True
        if warnings != "":
            if not question_dialog(gui, _('Modify Settings - ' + PLUGIN_NAME), '<P>'+
                    warnings + '<P>Do you want to continue?', show_copy_button=False):
                continue_modify = False
        if continue_modify:
            d = KindleEditDialog(gui, reader_prefs, font_family_options)
            d.exec_()
            if d.result() == d.Accepted:
                # Get the preferences set in the dialog
                debug_print('Saving preferences')
                # Not supported on Kindle Touch/PaperWhite
                if cfg.config_settings['kindle_model_version'] < 5000:
                    new_prefs = d.get_prefs()
                    update_font_files(old_fontname, new_prefs['FONT_FAMILY'])
                    if kindle_device.kdevice.write_reader_prefs(new_prefs):
                        save.save_fast_reboot()
                    else:
                        msg.message.error('Unable to write new preferences to the Kindle.')
                else:
                    msg.message.error('Unavailable on the Kindle Touch/PaperWhite.')
                msg.message.display()

        debug_print('END Modifying Settings')

def update_font_files(old_fontname, new_fontname):
    debug_print('Checking if font files need updating from "%s" to "%s"' % (old_fontname, new_fontname))
    # Not supported on Kindle Touch/PaperWhite
    if cfg.config_settings['kindle_model_version'] < 5000:
        if new_fontname not in FONT_FAMILY_DEFAULTS and new_fontname != 'alt' and new_fontname != old_fontname:
            if not kindle_device.kdevice.update_font_files(new_fontname):
                msg.message.error('Unable to update font files.')

def get_font_names():
    warnings = ""
    custom_font_names = []
    # Not supported on Kindle Touch/PaperWhite
    if cfg.config_settings['kindle_model_version'] < 5000:
        fontfiles = set(kindle_device.kdevice.get_font_filenames())
    else:
        fontfiles = None

    if fontfiles:
        fontnames = set([])
        for filename in fontfiles:
            fontname = re.sub('^(.*)-.*\.ttf$', r'\1', filename)
            if fontname:
                fontnames.add(fontname)
        missing_any_variants = False
        for fontname in fontnames:
            if fontname != 'alt':
                all_variants= True
                for variant in [ 'Regular', 'Bold', 'Italic', 'BoldItalic' ]:
                    if fontname + '-' + variant + '.ttf' not in fontfiles:
                        all_variants = False
                        missing_any_variants = True
                        warnings += 'Font "%s" missing file "%s-%s.ttf"<BR>' % (fontname, fontname, variant)
                if all_variants:
                    custom_font_names.append(fontname)
                    debug_print('Found valid font %s' % fontname)
        if missing_any_variants:
            warnings += '<BR>At least one custom font is missing a file in the Kindle fonts directory and will not be selectable.'
    return (FONT_FAMILY_DEFAULTS + custom_font_names, warnings)

class KindleEditDialog(SizePersistedDialog):

    def __init__(self, parent, prefs, font_options):
        SizePersistedDialog.__init__(self, parent, READER_PREFS_DIALOG_GEOMETRY)

        self.prefs = prefs
        self.font_options = font_options

        self.setWindowTitle(_('Modify Settings - ' + PLUGIN_NAME))

        # Not supported on Kindle Touch/PaperWhite
        if cfg.config_settings['kindle_model_version'] >= 5000:
            button_box = QDialogButtonBox(QDialogButtonBox.Cancel)
        else:
            button_box = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        v = QVBoxLayout(self)

        # Not supported on Kindle Touch/PaperWhite
        if cfg.config_settings['kindle_model_version'] < 5000:
            button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)

        # Add the window for editing
        v.addWidget(self.edit_widget())
        v.addWidget(button_box)

        self.resize_dialog()

    def edit_widget(self):
        self.edit_w = QWidget()
        layout = QVBoxLayout(self.edit_w)
        self.edit_w.setLayout(layout)

        # Not supported on Kindle Touch/PaperWhite
        if cfg.config_settings['kindle_model_version'] >= 5000:
            warning_label = QLabel('Modify your Kindle settings.<P>Sorry, this is unavailable on your Kindle model.', self)
            warning_label.setToolTip('Unavailable on your Kindle model!')

            layout.addWidget(warning_label)
        else:
            warning_label = QLabel('Modify your Kindle settings.<P>Make sure you are on the Kindle\'s HOME screen before you connect the Kindle to your PC.', self)
            warning_label.setToolTip('Click on the Kindle\'s Home button before connecting your Kindle to your PC.<BR>Otherwise the settings may be ignored.')

            # Settings

            # Horizontal Margin
            horizontal_margin_label = QLabel('Horizontal Margin:', self)
            horizontal_margin_label.setToolTip('Kindle defaults:<P>fewest = 40<BR>fewer = 80<BR>default = 40')
            self.horizontal_margin_spinbox = QSpinBox(self)
            self.horizontal_margin_spinbox.setRange(0, 200)
            self.horizontal_margin_spinbox.setValue(int(self.prefs['HORIZONTAL_MARGIN']))
            horizontal_margin_label.setBuddy(self.horizontal_margin_spinbox)

            # Justification
            justification_label = QLabel('Justification:', self)
            justification_label.setToolTip('Kindle default:<P>full<P>Set whether right margin is "jagged" (left) or "straight" (full).  Does not work for books that override justification.')
            self.justification_combobox = ComboTableWidgetItem(self, self.prefs['JUSTIFICATION'], JUSTIFICATION_OPTIONS)
            justification_label.setBuddy(self.justification_combobox)

            # Line Spacing
            line_spacing_label= QLabel('Line Spacing:', self)
            line_spacing_label.setToolTip('Kindle defaults:<P>1, 2, 3<P>4 also appears to work. ')
            self.line_spacing_spinbox = QSpinBox(self)
            self.line_spacing_spinbox.setRange(1, 10)
            self.line_spacing_spinbox.setValue(int(self.prefs['LINE_SPACING']))
            line_spacing_label.setBuddy(self.line_spacing_spinbox)

            # Font Size
            font_size_label = QLabel('Font Size:', self)
            font_size_label.setToolTip('Kindle defaults (from smallest "Aa" to largest):<P>17, 18, 21, 25, 31, 36, 60, 88')
            self.font_size_spinbox = QSpinBox(self)
            self.font_size_spinbox.setRange(1, 200)
            self.font_size_spinbox.setValue(int(self.prefs['FONT_SIZE']))
            font_size_label.setBuddy(self.font_size_spinbox)

            # Font Family
            font_family_label = QLabel('Font Family:', self)
            font_family_label.setToolTip('Kindle choices:<P>serif (regular), condensed, sans serif<P>Choose the font to use on your Kindle.<P>See Help for this plugin to add additional fonts.')
            self.font_family_combobox = ComboTableWidgetItem(self, self.prefs['FONT_FAMILY'], self.font_options)
            font_family_label.setBuddy(self.font_family_combobox)

            # Allows
            self.allow_justification_change_checkbox = CheckableBoxWidgetItem(self.prefs['ALLOW_JUSTIFICATION_CHANGE'].lower() == 'true', 'Allow changing justification','Enable or disable showing the Justification choices (left/full) on the Kindle')
            self.allow_user_line_spacing_checkbox = CheckableBoxWidgetItem(self.prefs['ALLOW_USER_LINE_SPACING'].lower() == 'true', 'Allow additional user line spacing options','If checked the line spacing options available on the Kindle change to read<BR>"112 125-small 136-med 150-lg 162" giving up to 5 line spacing options instead of 3.')
            self.allow_user_font_checkbox = CheckableBoxWidgetItem(self.prefs['ALLOW_USER_FONT'].lower() == 'true', 'Allow using user font','Must be checked to use a custom font or the font will be ignored even if selected.')
            self.allow_reading_indicator_checkbox = CheckableBoxWidgetItem(self.prefs['ALLOW_READING_INDICATOR'].lower() == 'true', 'Allow reading indicator','Does not appear to actually change anything on the Kindle.  Unfortunately this does not affect the reading indicator for books.')
            self.allow_article_thumbnail_checkbox = CheckableBoxWidgetItem(self.prefs['ALLOW_ARTICLE_THUMBNAIL'].lower() == 'true', 'Allow article thumbnail','Does not appear to actually change anything on the Kindle.')
            self.allow_two_column_view_checkbox = CheckableBoxWidgetItem(self.prefs['ALLOW_TWO_COLUMN_VIEW'].lower() == 'true', 'Allow two column view', 'Does not appear to actually change anything on the Kindle.')

            # Settings Grid
            settings_groupbox = QGroupBox('Kindle Settings')
            settings_layout = QGridLayout()
            settings_groupbox.setLayout(settings_layout)

            settings_layout.addWidget(horizontal_margin_label, 0, 0, 1, 1)
            settings_layout.addWidget(self.horizontal_margin_spinbox, 0, 1, 1, 1)

            settings_layout.addWidget(justification_label, 1, 0, 1, 1)
            settings_layout.addWidget(self.justification_combobox, 1, 1, 1, 1)

            settings_layout.addWidget(line_spacing_label, 2, 0, 1, 1)
            settings_layout.addWidget(self.line_spacing_spinbox, 2, 1, 1, 1)

            settings_layout.addWidget(font_family_label, 3, 0, 1, 1)
            settings_layout.addWidget(self.font_family_combobox, 3, 1, 1, 1)

            settings_layout.addWidget(font_size_label, 4, 0, 1, 1)
            settings_layout.addWidget(self.font_size_spinbox, 4, 1, 1, 1)

            settings_layout.setColumnMinimumWidth(2,10)

            settings_layout.addWidget(self.allow_justification_change_checkbox, 1, 3, 1, 1)
            settings_layout.addWidget(self.allow_user_line_spacing_checkbox, 2, 3, 1, 1)
            settings_layout.addWidget(self.allow_user_font_checkbox, 3, 3, 1, 1)

            settings_layout.setColumnStretch(4,1)

            unknown_settings_groupbox = QGroupBox('Kindle Settings with no known effect')
            unknown_settings_layout = QGridLayout()
            unknown_settings_groupbox.setLayout(unknown_settings_layout)

            unknown_settings_layout.addWidget(self.allow_reading_indicator_checkbox, 0, 0, 1, 1)
            unknown_settings_layout.addWidget(self.allow_article_thumbnail_checkbox, 1, 0, 1, 1)
            unknown_settings_layout.addWidget(self.allow_two_column_view_checkbox, 2, 0, 1, 1)

            settings_layout.setColumnStretch(1,1)

            # Information
            last_book_label = QLabel('Last Book Read: ', self.edit_w)
            last_book_label.setToolTip('Read Only')
            last_book2_label = QLabel(self.prefs['LAST_BOOK_READ'], self.edit_w)

            dictionary_label = QLabel('Dictionary:', self.edit_w)
            dictionary_label.setToolTip('Read Only')
            dictionary2_label = QLabel(self.prefs['DICTIONARY'], self.edit_w)

            # Information Grid
            general_groupbox = QGroupBox('Kindle Information')
            general_layout = QGridLayout()
            general_groupbox.setLayout(general_layout)

            general_layout.addWidget(last_book_label, 0, 0, 1, 1)
            general_layout.addWidget(last_book2_label, 0, 1, 1, 2)
            general_layout.addWidget(dictionary_label, 1, 0, 1, 1)
            general_layout.addWidget(dictionary2_label, 1, 1, 1, 2)

            general_layout.setColumnStretch(2,1)

            # Show each section
            layout.addWidget(warning_label)
            layout.addSpacing(20)
            layout.addWidget(settings_groupbox)
            layout.addSpacing(10)
            layout.addWidget(unknown_settings_groupbox)
            layout.addSpacing(10)
            layout.addWidget(general_groupbox)

            layout.addStretch(1)

        return self.edit_w

    def get_prefs(self):
        self.prefs['LINE_SPACING'] = self.line_spacing_spinbox.value()
        self.prefs['JUSTIFICATION'] = JUSTIFICATION_OPTIONS[self.justification_combobox.currentIndex()]
        self.prefs['FONT_SIZE'] = self.font_size_spinbox.value()
        font_index = self.font_family_combobox.currentIndex()
        if font_index >= 0:
            self.prefs['FONT_FAMILY']  = self.font_options[font_index]
        self.prefs['HORIZONTAL_MARGIN'] = self.horizontal_margin_spinbox.value()
        self.prefs['ALLOW_JUSTIFICATION_CHANGE'] = 'true' if self.allow_justification_change_checkbox.checkState() else 'false'
        self.prefs['ALLOW_USER_LINE_SPACING'] =  'true' if self.allow_user_line_spacing_checkbox.checkState() else 'false'
        self.prefs['ALLOW_USER_FONT'] =  'true' if self.allow_user_font_checkbox.checkState() else 'false'
        self.prefs['ALLOW_READING_INDICATOR'] =  'true' if self.allow_reading_indicator_checkbox.checkState() else 'false'
        self.prefs['ALLOW_ARTICLE_THUMBNAIL'] =  'true' if self.allow_article_thumbnail_checkbox.checkState() else 'false'
        self.prefs['ALLOW_TWO_COLUMN_VIEW'] =  'true' if self.allow_two_column_view_checkbox.checkState() else 'false'
        return self.prefs

