# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import absolute_import
import six
from six.moves import range

__license__   = 'GPL v3'
__copyright__ = '2011-2018, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Customization window
#####################################################################

import os, re
from functools import partial

# PyQt5/PyQt4 QVariant conversion handling... (cf. http://www.mobileread.com/forums/showthread.php?t=242223)
try:
    from calibre.gui2 import QVariant
    del QVariant
except ImportError:
    is_qt4 = False
    convert_qvariant = lambda x: x
else:
    is_qt4 = True

    def convert_qvariant(x):
        vt = x.type()
        if vt == x.String:
            return six.text_type(x.toString())
        if vt == x.List:
            return [convert_qvariant(i) for i in x.toList()]
        return x.toPyObject()

try:
    from PyQt5 import Qt as QtGui
    from PyQt5.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QLineEdit, QFormLayout, QTableWidget, QTableWidgetItem, QFrame, \
                         QAbstractItemView, QComboBox, QMenu, QToolButton, QCheckBox, QGridLayout, \
                         QVariant
except:
    from PyQt4 import QtGui
    from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                         QLineEdit, QFormLayout, QTableWidget, QTableWidgetItem, QFrame, \
                         QAbstractItemView, QComboBox, QMenu, QToolButton, QCheckBox, QGridLayout, \
                         QVariant

from calibre.gui2 import question_dialog, info_dialog

import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.config as cfg
import calibre_plugins.kindle_collections.calibre_info as calibre_info
from calibre_plugins.kindle_collections.utilities import debug_print, array_to_csv, csv_to_array, \
         ComboTableWidgetItem, CheckableTableWidgetItem, CheckableBoxWidgetItem, get_icon, get_pref, set_pref
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME

#####################################################################

class ConfigWidget(QWidget):

    # Starting point when Customize preferences is selected for the plugin
    def __init__(self, actual_plugin):
        debug_print('BEGIN ConfigWidget')

        QWidget.__init__(self)
        self.actual_plugin = actual_plugin

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # Sort the rows in the table based on the real name not the label
        self.table_sorted_indexes = sorted(list(calibre_info.ci.column_labels.keys()), key=lambda x: calibre_info.ci.column_labels[x].lower())

        # Create and add table containing rows/columns for configuration
        self._table = ConfigTableWidget(self.table_sorted_indexes, cfg.config_table, self)

        # Add the 'menu' at the top right

        words_layout = QHBoxLayout()
        layout.addLayout(words_layout)
        words_layout.addStretch(1)

        # Add Preview link by itself
        preview_label = QLabel('<a href="http://www.foo.com/">Preview without saving collections</a>', self)
        preview_label.setToolTip('Show what would happen if you created collections.<P>Does not modify any files on the Kindle.')
        preview_label.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.LinksAccessibleByKeyboard)
        preview_label.linkActivated.connect(self.preview_activated)
        words_layout.addWidget(preview_label)

        # Add Help link - same line as heading to improve look
        help_label = QLabel(' | <a href="http://www.foo.com/">Help</a>', self)
        help_label.setToolTip('Go to the Mobileread forum posting for this plugin')
        help_label.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.LinksAccessibleByKeyboard)
        help_label.linkActivated.connect(self.help_link_activated)
        words_layout.addWidget(help_label)

        layout.addSpacing(10)

        # Add the intro and reset button above the table

        header_layout = QGridLayout()
        layout.addLayout(header_layout)

        # Add a comment describing the window
        heading_label = QLabel('Configure which Kindle collections to update:', self)
        heading_label.setToolTip('Use this form to customize exactly which collections to create on your Kindle from Calibre.<P>For example, try setting the Action for the "Series" entry to "Create" and then running Preview.<P>When entering words/patterns be aware that<BR>there are some special pattern characters that need to be preceded by "\\"<BR>to use them as normal characters (e.g. "\\[" - without quotes): | [ ] . + ? * ^ $')

        # Add a reset button
        reset_button = QtGui.QToolButton(self)
        reset_button.setToolTip('Reset to defaults')
        reset_button.setIcon(get_icon('clear_left.png'))
        reset_button.clicked.connect(partial(self.reset_to_defaults, self))

        header_layout.addWidget(heading_label, 0, 0, 1, 1)
        header_layout.addWidget(reset_button, 0, 1, 1, 1)

        #Add the table

        # Add a horizontal layout containing the table and the buttons next to it
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        # Add the table created by the class
        table_layout.addWidget(self._table)

        layout.addSpacing(10)

        # Add the lower section of settings under the table

        # Define a grid for the settings under the table
        lower_layout = QGridLayout()

        # Ignore all collections edit box
        ignore_all_label = QLabel('Never delete or modify Kindle collections that match these patterns:', self)
        ignore_all_label.setToolTip('These Kindle collections will never be modified or deleted, whether or not they are in Calibre or Preserve Kindle-only is checked.<BR>    e.g.: __Currently Reading, .*Kindle, ~.*')

        ignore_all_layout = QHBoxLayout()
        self._ignore_all_ledit = QLineEdit(array_to_csv(cfg.config_settings['ignore_all']), self)
        ignore_all_layout.addSpacing(20)
        ignore_all_layout.addWidget(self._ignore_all_ledit)

        # Menu style choice
        menu_label = QLabel('Toolbar icon action:', self)
        menu_label.setToolTip('Choose what the plugin icon on the toolbar does when clicked<P>You can always access the menu by clicking on the small drop-down arrown on the right of the icon.')

        menu_layout = QHBoxLayout()
        menu_default = get_pref(cfg.MENU_CLICK_STYLE)
        if menu_default not in cfg.MENU_OPTIONS:
            menu_default = cfg.MENU_DEFAULT
        self.original_menu_style = menu_default
        self._menu_choices = ComboTableWidgetItem(self, menu_default, cfg.MENU_OPTIONS)
        menu_layout.addSpacing(20)
        menu_layout.addWidget(self._menu_choices)
        menu_layout.addStretch(1)

        # Keep Kindle-only collections checkbox
        self._keep_kindle_only = CheckableBoxWidgetItem(cfg.config_settings['keep_kindle_only'], 'Preserve existing Kindle-only collections','Keep any collections on the Kindle that aren\'t managed by Calibre. (A collection is managed by Calibre if it is selected using the settings you have above.)<P>If you uncheck this, your Kindle-only collections will be deleted (unless their name is in the Never delete box above).')

        # Ignore prefix/suffix when comparing
        self._ignore_prefix_suffix = CheckableBoxWidgetItem(cfg.config_settings['ignore_prefix_suffix'], 'Ignore prefix and suffix when comparing collection names', 'Replace Kindle collections generated from Calibre even if prefix/suffix changes (prefix/suffix must be non-alphanumeric).<BR>Does not affect the never delete/modify list.')

        # Case sensitive checkbox
        self._ignore_case = CheckableBoxWidgetItem(cfg.config_settings['ignore_case'], 'Ignore uppercase / lowercase when matching patterns','If unchecked, all pattern matches will be case-sensitive.')

        # Fast reboot - show only if hacks are installed on Kindle (but define it so it can be saved)
        self._fast_reboot = CheckableBoxWidgetItem(cfg.config_settings['fast_reboot'], 'Trigger fast reboot on USB unplug if hacks are installed','Requires the Fonts or Screensavers hack to be installed on the Kindle. Disabled on Touch devices, where it serves no purpose.')
        # Disable this on Kindle Touch/PaperWhite, it makes no sense there.
        if cfg.config_settings['kindle_model_version'] >= 5000:
            self._fast_reboot.setCheckState(Qt.Unchecked)
            self._fast_reboot.setDisabled(True)

        # Reset collection times checkbox
        self._reset_collection_times = CheckableBoxWidgetItem(cfg.config_settings['reset_collection_times'], 'Reset collection times to force alphabetical sorting in Sort by Collections','If unchecked, collections will retain the timestamps of when you last opened them on the Kindle.')

        # Don't use potentially stale exported json db (Kindle Touch/PaperWhite only, grey it out on other models)
        self._ignore_json_db = CheckableBoxWidgetItem(cfg.config_settings['ignore_json_db'], 'Ignore exported collections database','If unchecked, you\'ll probably want to have a recent export done from Collections Manager or LibrarianSync, because the plugin will use it.<P>If it\'s not in sync with the actual Touch/PaperWhite database, you may lose items in Kindle only collections (especially if \'Preserve existing Kindle-only collections\' is enabled).')
        # Disable it on non-Touch/PaperWhite Kindles...
        if cfg.config_settings['kindle_model_version'] < 5000:
            self._ignore_json_db.setCheckState(Qt.Unchecked)
            self._ignore_json_db.setDisabled(True)

        # Only generate a json db of collections created/updated since the last time (Kindle Touch/PaperWhite only, grey it out on other models)
        self._diff_db_only = CheckableBoxWidgetItem(cfg.config_settings['diff_db_only'], 'Only generate data for created/updated collections','When this is checked, we will only populate the json db with collections that have been updated or created since the last plugin run.<P>Since Collections Manager never deletes a collection, and always rebuild a modified collection from scratch, this will save time during the import.<P>Collections Manager won\'t have to iterate over collections that didn\'t change, thus avoiding a potentially huge waste of time for large libraries.<P>Note that while this works with LibrarianSync too, it\'s completely redundant there, since LS does its own optimisation to avoid doing unnecessary work ;).')
        # Disable it on non-Touch/PaperWhite Kindles...
        if cfg.config_settings['kindle_model_version'] < 5000:
            self._diff_db_only.setCheckState(Qt.Unchecked)
            self._diff_db_only.setDisabled(True)

        # Kindle model selection
        # Put a simple label describing our combobox first
        self._kindle_model_label = QLabel('Kindle model:')
        self._kindle_model_label.setToolTip('Choose your Kindle model. By default, an auto-detection should have taken place. If you want to run it again, simply select \'Auto-detect\'.<P>If, for some reason, it failed, you\'ll have to select your Kindle model manually.')
        # Then the ComboBox ;).
        self._kindle_model = QComboBox()
        # Populate
        kindle_list = [("Auto-detect", -1), ("Unknown", 0), ("Kindle 1", 1000), ("Kindle 2/DX/DXG", 2000), ("Kindle 3", 3000), ("Kindle 4", 4000), ("Kindle Touch", 5000), ("Kindle PaperWhite", 5020), ("Kindle PaperWhite 2", 5040), ("Kindle Basic", 5060), ("Kindle Voyage", 5050), ("Kindle PaperWhite 3", 5061), ("Kindle Oasis", 5071), ("Kindle Basic 2", 5081), ("Kindle Oasis 2", 5090), ("Kindle PaperWhite 4", 5100), ("Kindle Basic 3", 5110), ("Kindle Oasis 3", 5120)]
        for i in kindle_list:
            self._kindle_model.addItem(i[0], QVariant(i[1]))
        # Catch the state changes to launch the auto-detect logic...
        self._kindle_model.currentIndexChanged.connect(self.kindle_model_changed)
        # Restore current settings.
        self._kindle_model.setCurrentIndex(self._kindle_model.findData(cfg.config_settings['kindle_model_version']))
        # Hackish. If the setting doesn't exists yet, or is invalid, fire the signal automatically to trigger auto-detect (can't seem to get one from the setCurrentIndex on the previous line...)
        if not 'kindle_model_version' in cfg.config_settings or cfg.config_settings['kindle_model_version'] not in [1000, 2000, 3000, 4000, 5000, 5020, 5040, 5060, 5050, 5061, 5071, 5081, 5090, 5100, 5110, 5120]:
            self._kindle_model.currentIndexChanged.emit(0)

        # Play with the layouts to match the layout of the indented combobox on the right column...
        kindle_model_layout = QHBoxLayout()
        kindle_model_layout.addSpacing(20)
        kindle_model_layout.addWidget(self._kindle_model)
        kindle_model_layout.addStretch(1)

        # VSpace
        vspace_layout = QVBoxLayout()
        vspace_layout.addSpacing(15)

        # Space
        space_layout = QHBoxLayout()
        space_layout.addSpacing(50)

        # Populate the grid
        lower_layout.addWidget(self._keep_kindle_only, 0, 0, 1, 1)
        lower_layout.addWidget(self._ignore_prefix_suffix, 1, 0, 1, 1)
        lower_layout.addWidget(self._ignore_case, 2, 0, 1, 1)
        lower_layout.addWidget(self._fast_reboot, 3, 0, 1, 1)
        lower_layout.addWidget(self._reset_collection_times, 4, 0, 1, 1)
        lower_layout.addWidget(self._ignore_json_db, 5, 0, 1, 1)
        lower_layout.addWidget(self._diff_db_only, 6, 0, 1, 1)
        lower_layout.addLayout(vspace_layout, 7, 0, 1, 1)
        lower_layout.addWidget(self._kindle_model_label, 8, 0, 1, 1)
        lower_layout.addLayout(kindle_model_layout, 9, 0, 1, 1)

        lower_layout.addLayout(space_layout, 0, 1, 1, 1)

        lower_layout.addWidget(ignore_all_label, 0, 2, 1, 1)
        lower_layout.addLayout(ignore_all_layout, 1, 2, 1, 1)
        lower_layout.addWidget(menu_label, 2, 2, 1, 1)
        lower_layout.addLayout(menu_layout, 3, 2, 1, 1)

        layout.addLayout(lower_layout)

        debug_print('END ConfigWidget')

    # Run the auto-detection if needed when kindle_model changes state...
    def kindle_model_changed(self, idx):
        # Huh, on Py3, this fires early while _kindle_model.itemData(idx) is not yet available...
        if convert_qvariant(self._kindle_model.itemData(idx)) is None:
            return

        if convert_qvariant(self._kindle_model.itemData(idx)) == -1:
            debug_print('BEGIN Get Device Model')

            kindle_model = 0
            try:
                kindle_model = TEST_DEVICE
                # Assume unknown
                kindle_model = 0
                debug_print('RUNNING IN TEST MODE')
            except:
                device_connected = self.actual_plugin.gui.library_view.model().device_connected
                try:
                    device_connected = self.actual_plugin.gui.library_view.model().device_connected
                except:
                    debug_print('No device connected')
                    device_connected = None

                # We got a device, check if it's a Kindle, and if we can get its S/N
                if device_connected is not None:
                    try:
                        device = self.actual_plugin.gui.device_manager.device
                        device_name = device.get_gui_name()
                        debug_print('Device Name: %s' %(device_name))
                        # Once more, with feeling... We need a platform-specific quirk on Windows, because the scanner doesn't export the serial (AFAICT)
                        # This basically reimplements what was removed in r15214, but with the new backend ;).
                        from calibre.constants import iswindows
                        if iswindows:
                            device_serial = None
                            from calibre.devices.winusb import get_usb_info, scan_usb_devices
                            # c.f., calibre.devices.winusb
                            vendor_id = device.detected_device.idVendor if \
                                hasattr(device.detected_device, 'idVendor') else None
                            product_id = device.detected_device.idProduct if \
                                hasattr(device.detected_device, 'idProduct') else None
                            usb_devices = scan_usb_devices()
                            for usbdev in usb_devices:
                                if usbdev.vendor_id == vendor_id and (product_id is None or usbdev.product_id == product_id):
                                    usb_info = get_usb_info(usbdev, debug=True)
                                    debug_print('USB info: %s' %(usb_info))
                                    device_serial = usb_info["serial_number"]
                        else:
                            device_serial = device.detected_device.serial if \
                                hasattr(device.detected_device, 'serial') else None
                        if device_serial:
                            debug_print('Device Serial Prefix: %s' %(device_serial[:4]))
                        else:
                            debug_print('Unable to figure out the device serial number!')

                        # Check if we have a Kindle
                        if device_name and 'kindle' in device_name.lower():
                            # Do the serial dance...
                            if device_serial:
                                if device_serial[2:4] in ['01']:
                                    # K1
                                    kindle_model = 1000
                                elif device_serial[2:4] in ['02', '03', '04', '05', '09']:
                                    # K2/DX
                                    kindle_model = 2000
                                elif device_serial[2:4] in ['08', '06', '0A']:
                                    # K3
                                    kindle_model = 3000
                                elif device_serial[2:4] in ['0E', '23']:
                                    # K4
                                    kindle_model = 4000
                                elif device_serial[2:4] in ['0F', '11', '10', '12']:
                                    # KT
                                    kindle_model = 5000
                                elif device_serial[2:4] in ['24', '1B', '1C', '1D', '1F', '20']:
                                    # PW1
                                    kindle_model = 5020
                                elif device_serial[2:4] in ['D4', '5A', 'D5', 'D6', 'D7', 'D8', 'F2', '17', '60', 'F4', 'F9', '62', '61', '5F']:
                                    # PW2
                                    kindle_model = 5040
                                    # Keep the PWs separate, we need to differentiate it from the Touch because of a different sorting algo....
                                    # (And if it's tied to FW 5.2.0, and it gets released for the Touch, it'll be even more fun...
                                    # We'll have to check system/version.txt [it's a slightly processed copy of prettyversion.txt, it appeared in 5.2.0]...)
                                    # Store this as an int more or less matching the full firmware version without dots, because I've encountered weird rounding issues with floats...
                                elif device_serial[2:4] in ['C6', 'DD']:
                                    # KT2
                                    kindle_model = 5060
                                elif device_serial[2:4] in ['13', '54', '2A', '4F', '52', '53']:
                                    # KV
                                    kindle_model = 5050
                                elif device_serial[3:6] in ['0G1', '0G2', '0G4', '0G5', '0G6', '0G7', '0KB', '0KC', '0KD', '0KE', '0KF', '0KG', '0LK', '0LL']:
                                    # PW3, new device ID scheme...
                                    kindle_model = 5061
                                elif device_serial[3:6] in ['0GC', '0GD', '0GR', '0GS', '0GT', '0GU']:
                                    # Oasis
                                    kindle_model = 5071
                                elif device_serial[3:6] in ['0DU', '0K9', '0KA']:
                                    # KT3
                                    kindle_model = 5081
                                elif device_serial[3:6] in ['0LM', '0LN', '0LP', '0LQ', '0P1', '0P2', '0P6', '0P7', '0P8', '0S1', '0S2', '0S3', '0S4', '0S7', '0SA']:
                                    # KOA2
                                    kindle_model = 5090
                                elif device_serial[3:6] in ['0PP', '0T1', '0T2', '0T3', '0T4', '0T5', '0T6', '0T7', '0TJ', '0TK', '0TL', '0TM', '0TN', '102', '103', '16Q', '16R', '16S', '16T', '16U', '16V']:
                                    # PW4
                                    kindle_model = 5100
                                elif device_serial[3:6] in ['10L', '0WF', '0WG', '0WH', '0WJ', '0VB']:
                                    # KT4
                                    kindle_model = 5110
                                elif device_serial[3:6] in ['11L', '0WQ', '0WP', '0WN', '0WM', '0WL']:
                                    # KOA3
                                    kindle_model = 5120
                                else:
                                    # Unknown Kindle...
                                    kindle_model = 0
                            else:
                                # Couldn't get the serial from the Kindle...
                                kindle_model = 0
                        else:
                            # Not a Kindle
                            kindle_model = 0
                    except:
                        debug_print('A device appears to be connected, but we couldn\'t get its serial number')
                        # This might still happen on Windows...
                        kindle_model = 0
                else:
                    debug_print('No device appears to be connected')

            debug_print('Kindle Model Number: %d' %(kindle_model))
            debug_print('END Get Device Model')

            # Set the combobox to the detected model...
            self._kindle_model.setCurrentIndex(self._kindle_model.findData(kindle_model))
        elif convert_qvariant(self._kindle_model.itemData(idx)) >= 5000:
            # We have a Touch or a PaperWhite, enable the Collections Manager/LibrarianSync stuff, and disable the Kindle Hacks autoreboot stuff
            self._fast_reboot.setDisabled(True)
            self._fast_reboot.setCheckState(Qt.Unchecked)
            self._ignore_json_db.setDisabled(False)
            self._diff_db_only.setDisabled(False)
        else:
            # We have a legacy device, disable the Collections Manager/LibrarianSync stuff, and enable the Kindle Hacks autoreboot stuff
            self._fast_reboot.setDisabled(False)
            self._ignore_json_db.setDisabled(True)
            self._ignore_json_db.setCheckState(Qt.Unchecked)
            self._diff_db_only.setDisabled(True)
            self._diff_db_only.setCheckState(Qt.Unchecked)

    # Called by Calibre before save_settings
    def validate(self):
        debug_print('BEGIN Validate')
        valid = True
        # Only save if we were able to get data to avoid corrupting stored data
        try:
            if convert_qvariant(self._kindle_model.itemData(self._kindle_model.currentIndex())) == -1 or convert_qvariant(self._kindle_model.itemData(self._kindle_model.currentIndex())) == 0:
                msg.message.error('Not saving customizations due to an unknown kindle model.')
                valid = False
            if not cfg.validate_configuration(self.get_custom_window_table_data(), self.get_custom_window_ignore_all()):
                msg.message.error('Not saving customizations due to invalid patterns.')
                valid = False
        except:
                msg.message.error('Not saving customizations due to unexpected errors.')
                valid = False

        if not valid:
            msg.message.display()

        debug_print('END Validate, status = %s' % valid)
        return valid

    # Invoked when user clicks OK in preferences dialog.  Saves customizations to file store for later use.
    # Already validated by validate() - if the customization pre-checks did not generate an error
    def save_settings(self):
        debug_print('Customization trying to save data')
        collections_data = {}
        try:
            # _table will not be defined if the customization script did not run due to pre-check errors
            collections_data[cfg.STORE_KEY_ROWS] = self._table.get_data()
            collections_data[cfg.STORE_KEY_SETTINGS] =  {
                                                 'ignore_all': csv_to_array(self.get_custom_window_ignore_all()),
                                                 'keep_kindle_only': self._keep_kindle_only.isChecked(),
                                                 'ignore_prefix_suffix': self._ignore_prefix_suffix.isChecked(),
                                                 'ignore_case': self._ignore_case.isChecked(),
                                                 'reset_collection_times': self._reset_collection_times.isChecked(),
                                                 'fast_reboot': self._fast_reboot.isChecked(),
                                                 'ignore_json_db': self._ignore_json_db.isChecked(),
                                                 'diff_db_only': self._diff_db_only.isChecked(),
                                                 'kindle_model_version' : convert_qvariant(self._kindle_model.itemData(self._kindle_model.currentIndex())),
                                                 'store_version': cfg.STORE_VERSION
                                                }
            cfg.store.write_store_data(collections_data)
        except:
            debug_print('Customizations not saved - unexpected errors in saving customization.')

        # Save the menu style
        new_menu_style = cfg.MENU_OPTIONS[self._menu_choices.currentIndex()]
        set_pref(cfg.MENU_CLICK_STYLE, new_menu_style)
        if new_menu_style != self.original_menu_style:
            info_dialog(self.actual_plugin.gui, _(PLUGIN_NAME), _('You need to restart Calibre to see the new toolbar icon style.'), '', show=True)

        debug_print('Customization saved data')

    # Define the help message shown when customize-calibre -l is run
    # Would also be displayed if config_widget is not defined, but it is for this plugin
    def customization_help(self, gui=False):
        return 'This plugin can only be customized using the GUI.'

    # Show the help URL
    def help_link_activated(self, url):
        self.actual_plugin.help()

    # Run preview while currently in customize window
    def preview_activated(self):
        debug_print('Preview Activated within customization window')
        # Run in preview mode
        if self.validate():
            debug_print('BEGIN saving configuration temporarily for preview')
            # Save current settings so we can run with temporary settings - so Cancel in Customization works ok
            current_data = cfg.store.read_store_data()
            # Temporarily update version so preview will run
            old = 'No store version'
            if current_data and cfg.STORE_KEY_SETTINGS in current_data:
                if 'store_version' in current_data[cfg.STORE_KEY_SETTINGS]:
                    old = current_data[cfg.STORE_KEY_SETTINGS]['store_version']
            self.save_settings()
            debug_print('END saving configuration temporarily for preview')
            self.actual_plugin.preview_kindle_collections()
            current_data[cfg.STORE_KEY_SETTINGS]['store_version'] = old
            debug_print('Restoring pre-preview configuration')
            cfg.store.write_store_data(current_data)
        else:
            msg.message.error('Cannot run preview until customization errors are corrected.')
        msg.message.display()

    def reset_to_defaults(self, w):
        if question_dialog(w, _('Question'), '<P>'+
                'Are you sure you want to reset to default values?<P>Any modified settings will be discarded.', show_copy_button=False):
            cfg.load_config_table()
            w._table.populate_table(self.table_sorted_indexes, cfg.config_table)

            cfg.load_config_settings()
            w._ignore_all_ledit.setText(array_to_csv(cfg.config_settings['ignore_all']))
            w._keep_kindle_only.setChecked(cfg.config_settings['keep_kindle_only'])
            w._ignore_prefix_suffix.setChecked(cfg.config_settings['ignore_prefix_suffix'])
            w._ignore_case.setChecked(cfg.config_settings['ignore_case'])
            w._reset_collection_times.setChecked(cfg.config_settings['reset_collection_times'])
            w._fast_reboot.setChecked(cfg.config_settings['fast_reboot'])
            w._ignore_json_db.setChecked(cfg.config_settings['ignore_json_db'])
            w._diff_db_only.setChecked(cfg.config_settings['diff_db_only'])
            w._kindle_model.setCurrentIndex(w._kindle_model.findData(cfg.config_settings['kindle_model_version']))

            w._menu_choices.setCurrentIndex(0)

    def get_custom_window_table_data(self):
        return self._table.get_data()

    def get_custom_window_ignore_all(self):
        return six.text_type(self._ignore_all_ledit.text())


#####################################################################

# Create and populate rows of collection configuration data
class ConfigTableWidget(QTableWidget):

    def __init__(self, row_sorted_indexes, row_data, *args):
        QTableWidget.__init__(self, *args)
        self.revrow = {}
        self.header_labels = self.get_header_labels()
        self.populate_table(row_sorted_indexes, row_data)

    # Extract the column header labels in the right order from the main dictionary
    def get_header_labels(self):
        header_labels = [ '' ] * len(cfg.CUSTOMIZE_COLUMNS)
        for k in cfg.CUSTOMIZE_COLUMNS.keys():
            header_labels[cfg.CUSTOMIZE_COLUMNS[k]['pos']] = cfg.CUSTOMIZE_COLUMNS[k]['title']
        return header_labels

    # Fill the table with the given data
    def populate_table(self, row_sorted_indexes, row_data):
        self.clear()

        # Define the row/columns before putting any data in table
        h = self.horizontalHeader()
        h.setMinimumSectionSize(50)
        self.setRowCount(len(row_sorted_indexes))
        self.setColumnCount(len(self.header_labels))

        # Fill the table, save column label for each row number to  make it easier to lookup when saving
        #debug_print('Populate Config table, rows = %s' % self.rowCount())
        for row, index in enumerate(row_sorted_indexes):
            self.revrow[str(row)] = index
            self.populate_table_row(row, row_data[index])

        # Set style of table, and set selection to first row
        self.setAlternatingRowColors(True)
        self.setSortingEnabled(False)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.selectRow(0)
        self.setMinimumSize(700, 200)

        # Set the column headers - after the data so they are resized correctly
        self.verticalHeader().setDefaultSectionSize(24)
        self.setHorizontalHeaderLabels(self.header_labels)
        self.resizeColumnsToContents()

    # Return the column in the row for the given label
    def pos(self, column):
        return cfg.CUSTOMIZE_COLUMNS[column]['pos']

    def populate_table_row(self, row, data):
        self.blockSignals(True)
        item = ComboTableWidgetItem(self, data['action'], cfg.ACTION_OPTIONS)
        item.setToolTip('<dt>Create</dt><dd>Generate collections on the Kindle using entries from Calibre columns/categories.</dd><dt>Delete</dt> <dd>Remove collections from the Kindle that match entries from the Calibre column.</dd>')
        self.setCellWidget(row, self.pos('action'), item)

        item = QTableWidgetItem(data['column'])
        item.setToolTip('The source column to use from Calibre to get the list of collection names.')
        self.setItem(row, self.pos('column'), item)

        item = QTableWidgetItem(data['prefix'])
        item.setToolTip('Add text to start of collection names for extra sorting control when using Sort by Title.<BR>You should only use symbols to allow changing the prefix later.<BR>The Kindle will sort in this order:  _`^~\'([{$+<=>|<BR>Most other symbols are "invisible" to the Kindle.')
        self.setItem(row, self.pos('prefix'), item)

        item = QTableWidgetItem(data['suffix'])
        item.setToolTip('Add text to the end of collection name.<BR>You should only use symbols to allow changing the suffix later.')
        self.setItem(row, self.pos('suffix'), item)

        item = QTableWidgetItem(data['minimum'])
        item.setToolTip('Set the minimum number of books a collection must contain before <BR>creating, updating, or deleting the collection on the Kindle.<BR>Blank is equivalent to 1.')
        self.setItem(row, self.pos('minimum'), item)

        item = QTableWidgetItem(array_to_csv(data['ignore']))
        item.setToolTip('Identify the entries to ignore when creating collections on the Kindle,<BR>using a comma separate list of words or patterns.<BR>e.g.: IgnoreMe, .*ignorecontaining.*, ignorestarting.*, .*ignoreending')
        self.setItem(row, self.pos('ignore'), item)

        item = QTableWidgetItem(array_to_csv(data['include']))
        item.setToolTip('Identify the entries to include when creating collections on the Kindle.<BR>Leave blank to include all entries except ones marked ignore.')
        self.setItem(row, self.pos('include'), item)

        item = QTableWidgetItem(data['rename_from'])
        item.setToolTip('Advanced feature to allow renaming collection names and add/remove/substitute characters.<BR>Uses the python command re.sub(from_pattern, to_pattern, collection_name).<BR>Rename is applied after ignore/include. e.g.:<BR>from: "^+(.*)$"  to: "\\1" will remove the "+" at the start of any collection name.')
        self.setItem(row, self.pos('rename_from'), item)

        item = QTableWidgetItem(data['rename_to'])
        item.setToolTip('See Rename these patterns...')
        self.setItem(row, self.pos('rename_to'), item)

        item = QTableWidgetItem(data['split_char'])
        item.setToolTip('Split the text in the Calibre field into separate collections using this character.<BR>Split is applied before ignore/include or rename.')
        self.setItem(row, self.pos('split_char'), item)

        self.blockSignals(False)

    # Get all the data from the updated customization window to save for later
    def get_data(self):
        row_data = {}
        for row in range(self.rowCount()):
            row_data[self.revrow[str(row)]] = self.convert_row_to_data(row)
        return row_data

    # Get data from one row in the Customization window to save for later
    def convert_row_to_data(self, row):
        # Blank data since we'll populate it with specific data, leave spaces in prefix
        data = {}
        data['action'] = six.text_type(self.cellWidget(row, self.pos('action')).currentText()).strip()
        data['prefix'] = six.text_type(self.item(row, self.pos('prefix')).text())
        data['suffix'] = six.text_type(self.item(row, self.pos('suffix')).text())
        data['minimum'] = six.text_type(self.item(row, self.pos('minimum')).text()).strip()
        data['ignore'] = csv_to_array(six.text_type(self.item(row, self.pos('ignore')).text()))
        data['include'] = csv_to_array(six.text_type(self.item(row, self.pos('include')).text()))
        data['rename_from'] = six.text_type(self.item(row, self.pos('rename_from')).text())
        data['rename_to'] = six.text_type(self.item(row, self.pos('rename_to')).text())
        data['split_char'] = six.text_type(self.item(row, self.pos('split_char')).text())
        return data
