#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

# Copyright 2020 Kevin B. Hendricks, Stratford Ontario Canada

# This plugin's source code is available under the GNU LGPL Version 2.1 or GNU LGPL Version 3 License.
# See https://www.gnu.org/licenses/old-licenses/lgpl-2.1.en.html or
# https://www.gnu.org/licenses/lgpl.html for the complete text of the license.

import sys

_lvls = ["h1","h2","h3","h4","h5","h6"]


# the plugin entry point
def run(bk):

    try:
        if bk.launcher_version() < 20190927:
            print("Requires Sigil 1.0 or later")
            return -1
    except:
        print("Requires Sigil 1.0 or later")
        return -1

    epubversion = bk.epub_version()

    # first parse the ncx to get the toc
    ncxid = bk.gettocid()
    if not ncxid or ncxid == "":
        if epubversion.startswith("3"):
            print("NCX file not present, use Sigil's ePub3 Tools to create one before launch.")
        else:
            print("An epub2 must contain a valid NCX, none found")
        return -1
            
    ncxid = bk.gettocid()
    ncxbookhref = bk.id_to_bookpath(ncxid)
    print("Parsing: ", ncxbookhref)
    toclist = parse_ncx(bk, ncxbookhref)


    # now walk the toclist updating each link target
    qp = bk.qp
    for bookhref in toclist:
        alist = toclist[bookhref]
        id_targets = {}
        top_targets = []
        # convert the targets in this file to a more useable form
        for lvl, fragment, navlabel in alist:
            # FIXME - the ncx could have two different navlabels  pointing to the same fragment
            if fragment:
                id_targets[fragment] = (_lvls[lvl], navlabel)
            else:
                top_targets.append((_lvls[lvl], navlabel))

        print("Processing: ", bookhref)
        mid = bk.bookpath_to_id(bookhref)
        qp.setContent(bk.readfile(mid))
        res = []
        first_child_tag_of_body = False

        # properly inject the correct hidden tags and title attributes
        for text, tprefix, tname, ttype, tattr in qp.parse_iter():
            if text is not None:
                res.append(text)
                last_indent = ""
                if text.startswith("\n") and text[1:].isspace():
                    last_indent = " " * (len(text) - 1)
                continue

            if first_child_tag_of_body:
                if ttype in ["begin", "single"]:
                    if tname not in _lvls:
                        if len(top_targets) > 0:
                            # need to inject hidden heading tag here for all top_targets in order
                            for htag, navlabel in top_targets:
                                new_tag = '<%s hidden="hidden" style="display:none">%s</%s>\n' % (htag, navlabel, htag)
                                print("    ...injecting: ", new_tag)
                                res.append(new_tag + last_indent)
                        top_targets = []
                        first_child_tag_of_body = False
                    else:
                        # we have a heading, so add in title if matching and one exists
                        for htag, navlabel in top_targets:
                            if htag == tname:
                                top_targets.remove((htag, navlabel))
                                if not tattr:
                                    tattr = {}
                                title = tattr.get("title", "")
                                if title != "":
                                    print("    ...heading ", tname, "already has a title present so appending")
                                    title = title + " "
                                title = title + navlabel
                                tattr["title"] = title
                                print("    ...adding title attribute to heading: ", title)
                        first_child_tag_of_body = len(top_targets) > 0
                                
            if tname == "body" and ttype == "begin":
                first_child_tag_of_body = True
                # output body tag then check for any fragments on it
                res.append(qp.tag_info_to_xml(tname, ttype, tattr))
                if tattr:
                    id = tattr.get("id",None)
                    if id and id in id_targets:
                        htag, navlabel = id_targets[id]
                        # need to inject hidden heading tag here
                        new_tag = '<%s hidden="hidden" style="display:none">%s</%s>\n' % (htag, navlabel, htag)
                        print("    ...injecting: ", new_tag)
                        res.append(new_tag + last_indent)
                continue

            # handle the normal case for id fragments
            if ttype in ["begin", "single"]:
                if tattr:
                    id = tattr.get("id",None)
                    if id and id in id_targets:
                        htag, navlabel = id_targets[id]
                        if tname not in _lvls:
                            # need to inject hidden heading tag here
                            new_tag = '<%s hidden="hidden" style="display:none">%s</%s>\n' % (htag, navlabel, htag)
                            print("    ...injecting: ", new_tag)
                            res.append(new_tag + last_indent)
                        else:
                            # we have a heading tag so we need to add title attribute here
                            title = tattr.get("title", "")
                            if title != "":
                                print("    ...heading ", tname, "already has a title present so appending")
                                title = title + " "
                            title = title + navlabel
                            tattr["title"] = title
                            print("    ...adding title attribute to heading: ", title)
                res.append(qp.tag_info_to_xml(tname, ttype, tattr))
                continue

            # convert tag to xml
            res.append(qp.tag_info_to_xml(tname, ttype, tattr))

        # done with edits/injections
        if len(top_targets) > 0:
            print("Error: Lost top targets: ", top_targets) 
        newdata = "".join(res)
        bk.writefile(mid, newdata)

    print("TocSaver complete")
    # Setting the proper Return value is important.
    # 0 - means success
    # anything else means failure
    return 0


# parse the current toc.ncx to extract toc info, and pagelist info
# note all hrefs returned in toclist and pagelist are converted to be book hrefs
def parse_ncx(bk, ncxbookhref):
    ncx_id = bk.gettocid()
    ncxdata = bk.readfile(ncx_id)
    bk.qp.setContent(ncxdata)
    toclist = {}
    navlabel = None
    lvl = -1
    for txt, tp, tname, ttype, tattr in bk.qp.parse_iter():
        if txt is not None:
            if tp.endswith('.navpoint.navlabel.text'):
                navlabel = txt
        else:
            if tname == "navpoint" and ttype == "begin":
                lvl += 1
            elif tname == "navpoint" and ttype == "end":
                lvl -= 1
            elif tname == "content" and tattr is not None and "src" in tattr and tp.endswith("navpoint"):
                href =  tattr["src"]
                ahref, asep, afrag = href.partition('#')
                base = bk.get_startingdir(ncxbookhref)
                bookhref = bk.build_bookpath(ahref, base)
                fragment = afrag
                if afrag == "":
                    fragment = None
                alist = toclist.get(bookhref, [])
                alist.append((lvl, fragment, navlabel))
                toclist[bookhref] = alist
                print("ToC: ", bookhref, lvl, fragment, navlabel)
                navlabel = None
    return toclist

def main():
    print("I reached main when I should not have\n")
    return -1
    
if __name__ == "__main__":
    sys.exit(main())

