# ebook.py 0.1 2006-10-13
# Copyright (C) 2006 Igor Skochinsky <skochinsky@mail.ru>
#
# This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with this program; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

import struct, time, sys, os
from ctypes import *

reqFileOpen     = 0x10
reqFileClose    = 0x11
reqGetFileSize  = 0x12
reqSetFileSize  = 0x13
reqFileRead     = 0x16
reqGetFileInfo  = 0x18
reqDirEnumStart = 0x33
reqDirEnumStop  = 0x34
reqDirEnumNext  = 0x35

def debug(msg):
    #print "[DEBUG]", msg
    pass

def _complain_ifclosed(closed):
    if closed:
        raise ValueError, "I/O operation on closed file"

class EbookFile:
    def __init__(self, ebook):
        self.ebook = ebook
        self.handle = None
        self.closed = True

    def open(self, filename, mode="r"):
        info = self.ebook.GetFileInfo(filename)
        #dbg="Filename: %s\n"%path+"\tsize: %d\n"%info[0]+"\ttype: %d\n"%info[1]+"\tcreate time: %s\n"%time.ctime(info[2])+"\twrite time: %s\n"%time.ctime(info[3])+"\tflags: %d\n"% info[4]
        if info:
            if info[1]!=1:
                raise os.error, "'%s' is not a file"%filename
            self.size = info[0]
            self.ctime = time.gmtime(info[2])
            self.wtime = time.gmtime(info[3])
            self.flags = time.gmtime(info[4])
            self.pos = 0
            self.handle = self.ebook.FileOpen(filename, "w" in mode)
            if not self.handle:
                raise os.error, "Could not open file '%s'"%filename
            self.closed = False
        else:
            raise os.error, "File '%s' does not exist"%filename

    def seek(self, pos, mode = 0):
        """Set the file's current position.

        The mode argument is optional and defaults to 0 (absolute file
        positioning); other values are 1 (seek relative to the current
        position) and 2 (seek relative to the file's end).

        There is no return value.
        """
        _complain_ifclosed(self.closed)
        if mode == 1:
            pos += self.pos
        elif mode == 2:
            pos += self.size
        self.pos = max(0, pos)

    def tell(self):
        """Return the file's current position."""
        _complain_ifclosed(self.closed)
        return self.pos

    def read(self, n = -1):
        """Read at most size bytes from the file
        (less if the read hits EOF before obtaining size bytes).

        If the size argument is negative or omitted, read all data until EOF
        is reached. The bytes are returned as a string object. An empty
        string is returned when EOF is encountered immediately.
        """
        _complain_ifclosed(self.closed)
        if n < 0:
            newpos = sys.maxint
        else:
            newpos = min(self.pos+n, self.size)
        left = newpos-self.pos
        r = ""
        while left>0:
            toread = left
            if toread>0x8000:
                toread=0x8000
            debug("Reading %d at %08X"%(toread, self.pos))
            chunk = self.ebook.FileRead(self.handle, self.pos, toread)
            if chunk is None or len(chunk)==0: break
            self.pos += len(chunk)
            left -= len(chunk)
            r += chunk
        #self.pos = newpos
        debug("Total read: %d (of %d)"%(len(r), n))
        return r

    def close(self):
        if not self.closed and self.handle:
            self.ebook.FileClose(self.handle)
            self.closed = True

class Ebook:
    def __init__(self):
        self.dll = windll.ebookUsb
        self.sendProc = getattr(self.dll,"_UsbSendProc@16")
        self.receiveProc = getattr(self.dll,"_UsbReceiveProc@12")
        self.bufFreeProc = getattr(self.dll,"_UsbBuffFree@4")
        res = getattr(self.dll,"_UsbInitCheck@0")()
        self.connected = False
        self.present = False
        if res!=0:
            debug("checkConnect err=%d"%res)
            return
        else:
            debug("Device is present")
            self.present = True
        proto = create_string_buffer(8)
        res = getattr(self.dll,"_UsbGetProtcolVer@8")(1,proto)
        if res!=0:
            debug("UsbGetProtcolVer err=%d"%res)
            self.present = False
            return
        else:
            debug("Protocol: %s"%proto.value)

    def connect(self):
        if not self.present:
            debug("Device is not connected!")
            return

        res = getattr(self.dll,"_UsbUnlockDevice@4")("-1")
        if res!=0:
            debug("UsbUnlockDevice err=%d"%res)
            return
        else:
            debug("Unlocked OK")

        res = getattr(self.dll,"_UsbConnect@0")()
        if res!=0:
            debug("UsbConnect err=%d"%res)
            self.connected = False
            return
        else:
            debug("Connected OK")
            self.connected = True


    def disconnect(self):
        if not self.connected: return
        res = getattr(self.dll,"_UsbDisConnect@0")()
        if res!=0:
            debug("UsbDisConnect err=%d"%res)
        else:
            debug("Disconnected OK")
        self.connected = False

    def requestReceive(self, reqNo, extradata, readsize):
        request = struct.pack("<IIII",reqNo, 0, 0, len(extradata))+extradata
        answer = c_int(0)
        debug("Request: %02X, extra len: %d" % (reqNo, len(extradata)))
        res = self.receiveProc(request, readsize, byref(answer))
        if res!=0:
            debug("Error from UsbReceiveProc: %d!"%res)
            return
        pa = answer.value
        if pa:
            ans_unp = struct.unpack("<IIII",string_at(pa, 16))
            debug("Answer: "+str(ans_unp))
            ret_len = ans_unp[3]
            #print "pa:",pa
            res = string_at(pa+16, ret_len)
            self.bufFreeProc(pa)
            return res

    def FileOpen(self, path, write=False):
        mode = 0
        if write: mode=1
        extra = struct.pack("<II", mode, len(path))+path
        res = self.requestReceive(reqFileOpen, extra, 4)
        if res:
            handle = struct.unpack("<I",res)[0]
            debug("Opened '%s', handle: %02X"%(path, handle))
            return handle

    def FileClose(self, handle):
        extra = struct.pack("<I", handle)
        self.requestReceive(reqFileClose, extra, 0)

    def FileRead(self, handle, offset, size):
        extra = struct.pack("<IQI", handle, offset, size)
        res = self.requestReceive(reqFileRead, extra, size)
        return res

    def GetFileInfo(self, path):
        extra = struct.pack("<I", len(path))+path
        res = self.requestReceive(reqGetFileInfo, extra, 0x18)
        if res:
            info = struct.unpack("<QIIII",res)
            dbg="Filename: %s\n"%path+"\tsize: %d\n"%info[0]+"\ttype: %d\n"%info[1]+"\tcreate time: %s\n"%time.ctime(info[2])+"\twrite time: %s\n"%time.ctime(info[3])+"\tflags: %d\n"% info[4]
            debug(dbg)
            return info


    def open(self, path, mode="r"):
        f = EbookFile(self)
        f.open(path,mode)
        return f

    def isdir(self,path):
        if not path.endswith('/'): path+='/'
        info = self.GetFileInfo(path)
        return (info and info[1]==2)

    #ripped almost verbatim from os.path
    def walk(self, top, topdown=True, onerror=None):
        from posixpath import join
        try:
            names = self.listdir(top)
        except os.error, err:
            if onerror is not None:
                onerror(err)
            return

        dirs, nondirs = [], []
        for name in names:
            if self.isdir(join(top, name)):
                dirs.append(name)
            else:
                nondirs.append(name)

        if topdown:
            yield top, dirs, nondirs
        for name in dirs:
            path = join(top, name)
            if not (path.startswith('/dev/') or path.startswith('/proc/')): #skip dirs inside /dev and /proc, these can lead to recursion
                for x in walk(path, topdown, onerror):
                    yield x
        if not topdown:
            yield top, dirs, nondirs

    def listdir(self, path):
        #if path.startswith('/dev/') or path.startswith('/proc/'):
        #    return
        if not path.endswith('/'): path+='/'
        if not self.isdir(path):
            return []
        extra = struct.pack("<I", len(path))+path
        enumHandle = self.requestReceive(reqDirEnumStart, extra, 4)
        names = []
        if enumHandle:
            while True:
                nextFile = self.requestReceive(reqDirEnumNext, enumHandle, 0)
                if nextFile:
                    #dword type, dword name_len, char[] name
                    typ = struct.unpack("<I", nextFile[:4])[0]
                    name = nextFile[8:]
                    names.append(name)
                else:
                    self.requestReceive(reqDirEnumStop, enumHandle, 0)
                    return names
        else:
            #print "Could not list directory!"
            raise os.error, "Could not list directory '%s'"%path

def Usage():
    print "Usage: ebook.py cmd [params]"
    print "\tls <dir> [-R]: list device directory <dir> [recursively]"
    print "\tget <path>: download <path> from the device to current directory"

def forcedir(path):
    if os.path.isdir(path):
        return
    if os.path.exists(path):
        os.remove(path)
    os.mkdir(path)

def download(ebook, path, targetdir):
    from posixpath import basename, join
    print "%s -> %s"%(path, targetdir),
    if path.endswith("/"): path=path[:-1]
    base = basename(path)
    if ebook.isdir(path):
        print
        base = basename(path)
        newdir = os.path.join(targetdir, base)
        forcedir(newdir)
        files = ebook.listdir(path)
        for f in files:
            download(ebook, join(path,f), newdir)
    else:
        f = b.open(path)
        file(os.path.join(targetdir, base),"wb").write(f.read())
        print "%d bytes"%f.tell()
        f.close()

def print_dir(ebook, path, recurse):
    files = ebook.listdir(path)
    files.sort()
    from posixpath import join
    dirs = []
    if not path.endswith("/"): path+="/"
    print path+":"
    adddirs = recurse and not (path.startswith('/dev') or path.startswith('/proc'))
    for f in files:
        fpath = join(path,f)
        info = ebook.GetFileInfo(fpath)
        if info[1]==2:
            if adddirs: dirs.append(fpath)
            typ='drwxrwxr-x '
        else:
            typ='-rw-rw-r-- '
        print typ+"%13d %s %s"%(info[0], time.strftime("%b %d %Y %H:%M",time.gmtime(info[3])), f)
        #print "Filename: %s\n"%fpath+"\tsize: %d\n"%info[0]+"\ttype: %d\n"%info[1]+"\tcreate time: %s\n"%time.ctime(info[2])+"\twrite time: %s\n"%time.ctime(info[3])+"\tflags: %d\n"% info[4]
    if recurse: print
    if len(dirs):
        for d in dirs: print_dir(ebook, d, recurse)

print "Sony Reader utility 0.1 (c) 2006 Igor Skochinsky"

if len(sys.argv)>2:
    b = Ebook()
    try:
        b.connect()
        if b.connected:
            cmd = sys.argv[1]
            if cmd=="ls":
                d = sys.argv[2]
                recurse = len(sys.argv)>3 and sys.argv[3]=="-R"
                print_dir(b, d, recurse)
            elif cmd=="get":
                filename = sys.argv[2]
                download(b, filename, ".")
            else:
                Usage()
        else:
            print "Device is not connected to PC"
    finally:
        b.disconnect()
else:
    Usage()