#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

# Get required support modules for all plugin actions
import os, shutil, glob, re

from PyQt5.Qt import QMenu, QToolButton
    
# The class that all interface action plugins must inherit from
from calibre.gui2.actions import InterfaceAction

from calibre.gui2 import error_dialog, info_dialog, question_dialog

from calibre_plugins.copy_cover_to_device.config import prefs
from calibre_plugins.copy_cover_to_device.dialogs import CopyCoverDlg, CoverProgressDialog

MAX_COPY = 100
DEVICE_DESTPATH = {
                'sony': 'database/media/images/calibre_cover.jpg',
                'prs600': 'database/system/PRSPlus/wallpaper/calibre_cover.jpg',
                'sony_prsplus': 'database/system/PRSPlus/wallpaper/calibre_cover.jpg',
                #'pocketbook_8': 'system/logo/calibre_cover.bmp',
                #'pocketbook_16': 'system/logo/calibre_cover.bmp',
                'nook_str': 'screensavers/calibre/calibre_cover.jpg',
                'nook_1': 'my screensavers/calibre/calibre_cover.jpg',
                'kindle_customss': 'linkss/screensavers/calibre_cover.jpg',
                'kindle_customss_touch': 'linkss/screensavers/bg_medium_ss.png',
                'kindle_customss_pw': 'linkss/screensavers/bg_medium_ss.png',
                'kindle_customss_voyage': 'linkss/screensavers/bg_ss.png',
                'kindle_simpless': 'screensaver/bg_xsmall_ss.png'
                }
KINDLE_REBOOT_HACKS = ['linkfonts', 'linkss']
KINDLE_CUSTOMSS_IS_TOUCH = 'linkss/etc/is_a_touch'
KINDLE_CUSTOMSS_IS_PW = 'linkss/etc/is_a_pw'
KINDLE_CUSTOMSS_IS_VOYAGE = 'linkss/etc/is_a_kv'
SONY_INTMEM_X50 = 1500000000
TESTFILE = 'copycover_jackiew'

class CopyCoverUiAction(InterfaceAction):
    name = 'Copy Cover To Device'

    # Declare the main action associated with this plugin
    # The keyboard shortcut can be None if you dont want to use a keyboard
    # shortcut. Remember that currently calibre has no central management for
    # keyboard shortcuts, so try to use an unusual/unused shortcut.
    
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Copy cover to device', 'book.png',
            'Copy cover from selected books to connected device', _('Ctrl+Shift+C'))
    dont_add_to = frozenset(['toolbar', 'context-menu-device'])
    action_type = 'current'
    popup_type = QToolButton.MenuButtonPopup

    def genesis(self):
        # This method is called once per plugin, do initial setup here
        
        self.menu = QMenu(self.gui)
        
        # Set the icon for this interface action
        # The get_icons function is a builtin function defined for all your
        # plugin code. It loads icons from the plugin zip file. It returns
        # QIcon objects, if you want the actual data, use the analogous
        # get_resources builtin function.
        #
        # Note that if you are loading more than one icon, for performance, you
        # should pass a list of names to get_icons. In this case, get_icons
        # will return a dictionary mapping names to QIcons. Names that
        # are not found in the zip file will result in null QIcons.
        
        # icon = get_icons('images/cover.png')

        # The qaction is automatically created from the action_spec defined above
        
        self.rebuild_menu()
        self.qaction.setMenu(self.menu)
        # self.qaction.setIcon(icon)
        self.qaction.triggered.connect(self.copycover_selected)
        
    def apply_settings(self):
        # run at the end of customization
        from calibre_plugins.copy_cover_to_device.config import prefs
        pass

    def rebuild_menu(self):
        m = self.menu
        m.clear()
        ac1 = self.create_action(spec=('Copy Cover', 'book.png', None, None),
                                  attr='Copy Cover')
        ac1.triggered.connect(self.copycover_selected)
        m.addAction(ac1)
        ac2 = self.create_action(spec=('Clear existing cover images', 'trash.png', None, None),
                                  attr='Clear existing cover images')
        ac2.triggered.connect(self.clearcover_selected)
        m.addAction(ac2)
        m.addSeparator()
        ac3 = self.create_action(spec=('Customize plugin...', 'config.png', None, None),
                                  attr='Customize plugin...')
        ac3.triggered.connect(self.show_configuration)
        m.addAction(ac3)
    
    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def clearcover_selected(self):
        # delete all existing cover images
        dev_name, dest_path = self.check_connection()
        if not dev_name:
            return
        
        dest_dir, dest_file = os.path.split(dest_path)
        basename, dest_ext = os.path.splitext(dest_file)
        filelist = glob.glob(os.path.join(dest_dir, basename + '*' + dest_ext))
        logtext = '%s - %d cover(s) for deletion\n%s\nContinue?' % (
                    dev_name.upper(), len(filelist), dest_dir)
        if question_dialog(self.gui, 'Copy Cover - clear existing images', logtext):
            for f in filelist:
                os.remove(f)
        
    def copycover_selected(self):
        class SelectedBookError(Exception): pass
        
        maxcopy = prefs['multi_copy'] and MAX_COPY or 1
        overwrite = prefs['overwrite_prev']
        
        dev_name, dest_path = self.check_connection()
        if not dev_name:
            return
        
        try:        
            # Selected book checks. If error raise SelectedBookError
            book_rows = self.gui.library_view.selectionModel().selectedRows()
            
            if not book_rows:
                errmsg = 'No book selected'
                raise SelectedBookError(errmsg)
            
            total_books = len(book_rows)
            if total_books > maxcopy:
                errmsg = '%d books selected, max. is %d' % (total_books, maxcopy)
                if maxcopy == 1:
                    errmsg += '\nRe-customize to allow copying of multiple covers'
                raise SelectedBookError(errmsg)
                             
            # Set image destination path. Normalize for current OS standards
            dest_dir, dest_file = os.path.split(dest_path)
            basename, dest_ext = os.path.splitext(dest_file)
            
            # Create a list of all available filenames on device
            if dev_name in ['kindle_simpless',
                            'kindle_customss_touch',
                            'kindle_customss_pw']:
                fnames = [os.path.join(dest_dir, '%s%02d%s' % (basename, i, dest_ext)) \
                                    for i in range(0, maxcopy)]
            else:
                fnames = [dest_path]
                fnames.extend([os.path.join(dest_dir, '%s%02d%s' % (basename, i, dest_ext)) \
                                    for i in range(1, maxcopy)])
            if not overwrite:
                fnames = [f for f in fnames if not os.path.exists(f)]
                if total_books > len(fnames):
                    errmsg = 'You have selected %d books.' \
                            '\nYou already have %d covers on this device.' \
                            '\nThis exceeds the max. limit of %d.' \
                            '\n\nReduce selection or delete some covers ' \
                            '\nbefore trying again.' \
                            % (total_books, maxcopy - len(fnames), maxcopy)
                    raise SelectedBookError(errmsg)
                    
        except SelectedBookError as e:
            return error_dialog(self.gui, 'Copy Cover - book error', str(e), show=True)
                
        # all OK, proceed with copy
        
        info_log = ['%d cover(s) selected for: %s' % (total_books, dev_name.upper())]
        info_log.append('%s' % dest_dir)
        
        if overwrite:
            # remove all existing cover images
            filelist = glob.glob(os.path.join(dest_dir, basename + '*' + dest_ext))
            for f in filelist:
                os.remove(f)
            info_log.append('- all previous covers removed\n')
        else:
            info_log.append('- adding to existing covers\n')
        
        if prefs['keep_aspect']:
            info_log.append('Aspect ratio kept\n')
        if prefs['random'] and total_books > 1:
            info_log.append('Covers sent in random order\n')
        
        # get a handle to the database object.
        db = self.gui.library_view.model().db

        # copy cover loop with progress dialog bar
        dlg1 = CoverProgressDialog(self.gui, total_books, db, book_rows, fnames, dev_name)
               
        # check whether any covers copied
        if dlg1.good_log:
            # If Kindle pref checked, make sure Kindle auto reboots when disconnected
            if dev_name.startswith('kindle_customss') and prefs['kindle_restart']:
                # create empty file with correct Kindle path, patch from NiLuJe
                # If both Kindle hacks are installed, Fonts hack should take precedence,
                # but check status of  watchdog itself, to detect correct one
                kroot = self.gui.device_manager.connected_device._main_prefix
                for khack in KINDLE_REBOOT_HACKS:
                    # if watchdog is running for this hack, can drop reboot trigger file
                    if os.path.isfile(os.path.join(kroot, khack, 'run', 'usb-watchdog.pid')):
                        open(os.path.join(kroot, khack, 'reboot'), 'wb').close()
                        info_log.append("Prepared Kindle for autoreboot after disconnect using NiLuJe's %s hack\n" % khack)
                
            # show summary, custom widget as info_dialog unsuited to large text
            info_log.extend(dlg1.bad_log)
            info_log.extend(dlg1.good_log)

            if dlg1.wasCanceled():
                info_log.append('\n*** Copying was cancelled before completion ***')
            
            logtext = '\n'.join(info_log)
            dlg2 = CopyCoverDlg(self.gui, logtext)
            dlg2.exec_()
        else:
            if dlg1.wasCanceled():
                errmsg = 'Copying was cancelled'
            else:    
                errmsg = 'Selected book(s) have no cover'
            error_dialog(self.gui, 'Copy Cover', str(errmsg), show=True)

    def check_connection(self):
        # Device connection checks. If error raise ConnectedDeviceError
        class ConnectedDeviceError(Exception): pass
        try:
            # check whether a device is currently connected
            if self.gui.device_manager.connected_device is None:
                errmsg = 'Cannot proceed, no device connected...'
                raise ConnectedDeviceError(errmsg)
                             
            # determine which device is connected
            root_path = self.gui.device_manager.connected_device._main_prefix
            if not root_path:
                errmsg = 'Cannot proceed, no device connected...'
                raise ConnectedDeviceError(errmsg)
            
            dev_name = self.gui.device_manager.connected_device.name.split()
            dev_name = dev_name[0].lower()
            
            # test purposes only: simulate various devices
            if os.path.isfile(os.path.join(root_path, TESTFILE)):
                with open(os.path.join(root_path, TESTFILE), 'r') as ftest:
                    devlines = ftest.readlines()
                    dev_name = devlines[0].strip().lower()
           
            if hasattr(self, 'which_' + dev_name):
                dev_name = getattr(self, 'which_' + dev_name)(dev_name, root_path)
            
            # Set image destination path depending on device
            img_path = DEVICE_DESTPATH.get(dev_name, '')
            if not img_path:
                if dev_name == 'kindle_bothss':
                    errmsg = 'Sorry, unsure which Kindle Screensaver directory to use as both exist.'
                    path1 = os.path.normpath(root_path + DEVICE_DESTPATH['kindle_customss'])
                    path2 = os.path.normpath(root_path + DEVICE_DESTPATH['kindle_simpless'])
                    errmsg += '\n   Custom Screensaver - %s' % os.path.dirname(path1)
                    errmsg += '\n   Simple Screensaver - %s' % os.path.dirname(path2)
                    errmsg += '\n\nPlease rename or delete the directory no longer required, then try again.'
                else:
                    errmsg = 'Sorry, this %s device is not currently supported' % dev_name.upper()
                raise ConnectedDeviceError(errmsg)

            # Set image destination path. Normalize for current OS standards
            dest_path = os.path.normpath(root_path + img_path)
            dest_dir = os.path.dirname(dest_path)
            # The folder in the path must exist
            if not os.path.exists(dest_dir):
                errmsg = '%s\n\nDestination folder does not exist on %s' % (dest_dir, dev_name.upper())
                errmsg += '\n\nPlease create it manually then try again.'
                raise ConnectedDeviceError(errmsg)
        except ConnectedDeviceError as e:
            error_dialog(self.gui, 'Copy Cover - device error', str(e), show=True)
            dev_name, dest_path = None, None
        return dev_name, dest_path

    def which_sony(self, device, root):
        prsplus_wall = os.path.dirname(os.path.normpath(root + DEVICE_DESTPATH['prs600']))
        # get int memory size from connected device
        intmem = self.gui.device_manager.connected_device.total_space()[0]
        if intmem < SONY_INTMEM_X50:
            if os.path.exists(prsplus_wall):
                return 'prs600'
            else:
                return device
        elif os.path.exists(prsplus_wall):
            return 'sony_prsplus'
        return device

    def which_nook(self, device, root):
        for model in [k for k in sorted(DEVICE_DESTPATH, reverse=True) if k.startswith(device)]:
            ssdir = DEVICE_DESTPATH[model].split('/')[0]
            if os.path.exists(os.path.normpath(root + ssdir)):
                return model
        return device

    def which_kindle(self, device, root):
        model = device
        for mod in ['kindle_customss', 'kindle_simpless']:
            ssdir = os.path.split(DEVICE_DESTPATH[mod])[0]
            if os.path.exists(os.path.normpath(root + ssdir)):
                if model == device:
                    model = mod
                else:
                    model = 'kindle_bothss'
        if model == 'kindle_customss':
            # check to see whether attached device is a kindle touch, pw, voyage (or other)
            if os.path.isfile(os.path.join(root, KINDLE_CUSTOMSS_IS_TOUCH)):
                model += '_touch'
            elif os.path.isfile(os.path.join(root, KINDLE_CUSTOMSS_IS_PW)):
                model += '_pw'
            elif os.path.isfile(os.path.join(root, KINDLE_CUSTOMSS_IS_VOYAGE)):
                model += '_voyage'
        return model
        
    '''def which_pocketbook(self, device, root):
        # get driver name of connected PB
        driver_name = self.gui.device_manager.connected_device.__class__.__name__
        if driver_name in ['POCKETBOOK360', 'POCKETBOOK301']:
            model = 'pocketbook_8'
        else:
            model = 'pocketbook_16'
        return model'''
