#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

import random, re, os
from PyQt5.Qt import (QDialog, QVBoxLayout, QTextBrowser, QDialogButtonBox,
                    QProgressDialog, QTimer)

from calibre_plugins.copy_cover_to_device.config import prefs
from calibre.utils.magick.draw import save_cover_data_to
from calibre.utils.magick import Image, create_canvas

class CopyCoverDlg(QDialog):

    def __init__(self, gui, text):
        QDialog.__init__(self, gui)
        self.gui = gui

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.browser = QTextBrowser()
        self.browser.setMinimumWidth(600)
        self.browser.setMinimumHeight(400)
        self.browser.setText(text)
        self.l.addWidget(self.browser)

        self.closeButton = QDialogButtonBox(QDialogButtonBox.Close)
        self.l.addWidget(self.closeButton)
        self.closeButton.rejected.connect(self.reject)
        
        self.setWindowTitle('Copy Cover - summary')

class CoverProgressDialog(QProgressDialog):

    def __init__(self, gui, total_books, db, book_rows, fnames, dev_name):
        self.total_books = total_books
        QProgressDialog.__init__(self, '', 'Cancel', 0, self.total_books, gui)
        self.gui = gui
        self.db = db
        self.book_rows = book_rows
        self.fnames = fnames
        self.dev_name = dev_name
        
        self.i = 0
        self.count = 0
        self.good_log = []
        self.bad_log = []
        self.bookseq = list(range(0, self.total_books))
        if prefs['random'] and self.total_books > 1:
            random.shuffle(self.bookseq)
        
        swidth, sheight = prefs['screen_size'].split(',')
        self.screen_size = (int(swidth), int(sheight))
        self.keep_aspect = prefs['keep_aspect']
        
        self.setWindowTitle('Copying covers to %s ...' % self.dev_name.upper())
        self.setMinimumWidth(500)
        
        QTimer.singleShot(0, self.do_book_action)
        self.exec_()

    def do_book_action(self):
        ''' Process single book '''
        if self.wasCanceled():
            return self.do_close()
        if self.i >= self.total_books:
            return self.do_close()
            
        # translate the row object to actual row index. This index can
        # be used with the database functions if index_is_id=False
        book_id = self.book_rows[self.bookseq[self.i]].row()

        # use database object to get data for current book
        cover_data = self.db.cover(book_id, index_is_id=False, as_path=False)
        btitle = self.db.title(book_id, index_is_id=False)
        bauthors = self.db.authors(book_id, index_is_id=False)
        if cover_data is not None:
            dest_path = self.fnames[self.count]
            self.count += 1
            rescale = getattr(self, 'copy_to_' + self.dev_name) \
                    (self.screen_size, self.keep_aspect, cover_data, dest_path)
            info_msg = rescale and '(%d, %d)' % rescale or ''
            self.good_log.append('%d. %s - %s %s' % (self.count, bauthors, btitle, info_msg))
        else:    
            self.bad_log.append('*** No cover %s - %s' % (bauthors, btitle))
            
        self.i += 1
        self.setLabelText('%d of %d' % (self.i, self.total_books))
        self.setValue(self.i)
        QTimer.singleShot(0, self.do_book_action)

    def do_close(self):
        self.hide()
        self.gui = None
        
    def copy_to_sony(self, screen, aspect, cdata, dest):
        # if not keep_aspect, resize to screen size,
        # otherwise copy as-is, Sony will resize correctly
        rescale = not aspect and screen or None
        save_cover_data_to(cdata, dest, resize_to=rescale)
        return rescale
        
    def copy_to_prs600(self, screen, aspect, cdata, dest):
        # Same as Sony, diff dest path
        return self.copy_to_sony(screen, aspect, cdata, dest)
        
    def copy_to_sony_prsplus(self, screen, aspect, cdata, dest):
        # Same as Sony, possible diff dest path
        return self.copy_to_sony(screen, aspect, cdata, dest)

    def copy_to_pocketbook_8(self, screen, aspect, cdata, dest):
        img, rescale = self.cover_resize(screen, aspect, cdata)
        # Resize image then convert to greyscale
        if rescale:
            img.size = rescale
        img.quantize(number_colors=8, colorspace='GRAYColorspace')
        # save_cover_data_to(img, dest)
        img.type = "PaletteType"
        img.save(dest, 'BMP3')
        return rescale

    def copy_to_pocketbook_16(self, screen, aspect, cdata, dest):
        img, rescale = self.cover_resize(screen, aspect, cdata)
        # Resize image then convert to greyscale
        if rescale:
            img.size = rescale
        img.quantize(number_colors=16, colorspace='GRAYColorspace')
        # save_cover_data_to(img, dest)
        img.type = "PaletteType"
        img.save(dest, 'BMP3')
        return rescale
        
    def copy_to_kindle_customss(self, screen, aspect, cdata, dest):
        # resize image if necessary
        if aspect:
            swidth, sheight = screen
            img, rescale = self.cover_resize(screen, aspect, cdata)
            iwidth, iheight = rescale
            img.size = rescale
            # convert to greyscale
            img.quantize(number_colors=16, colorspace='GRAYColorspace')
            img.type = "PaletteType"
            # create white canvas at full screensize, then centre img
            left = 0
            top = 0
            canvas = create_canvas(swidth, sheight, bgcolor='#ffffff')
            if iwidth < swidth:
                left = int((swidth - iwidth) / 2)
            if iheight < sheight:
                top = int((sheight - iheight) / 2)
            canvas.compose(img, left, top)
            if os.path.splitext(dest)[1] == '.jpg':
                save_cover_data_to(canvas, dest, resize_to=None)
            else:
                newcdata = canvas.export('jpg')
                save_cover_data_to(newcdata, dest, resize_to=None)
        else:
            # dummy, rescale = self.cover_resize(screen, aspect, cdata)
            rescale = screen
            save_cover_data_to(cdata, dest, resize_to=rescale)
        return rescale
        
    def copy_to_kindle_customss_touch(self, screen, aspect, cdata, dest):
        # Same as Kindle custom screensaver, diff dest path
        # ignore user prefs - force screensize to 600x800
        screen = (600, 800)
        return self.copy_to_kindle_customss(screen, aspect, cdata, dest)
        
    def copy_to_kindle_customss_pw(self, screen, aspect, cdata, dest):
        # Same as Kindle custom screensaver, diff dest path
        # ignore user prefs - force screensize to 758x1024
        screen = (758, 1024)
        return self.copy_to_kindle_customss(screen, aspect, cdata, dest)
        
    def copy_to_kindle_customss_voyage(self, screen, aspect, cdata, dest):
        # Same as Kindle custom screensaver, diff dest path
        screen = (1072, 1448)
        return self.copy_to_kindle_customss(screen, aspect, cdata, dest)
        
    def copy_to_kindle_simpless(self, screen, aspect, cdata, dest):
        # Same as Kindle custom screensaver, diff dest path
        return self.copy_to_kindle_customss(screen, aspect, cdata, dest)
        
    def copy_to_nook_1(self, screen, aspect, cdata, dest):
        dummy, rescale = self.cover_resize(screen, aspect, cdata)
        save_cover_data_to(cdata, dest, resize_to=rescale)
        return rescale
        
    def copy_to_nook_str(self, screen, aspect, cdata, dest):
        # Same as Nook original custom screensaver, diff dest path
        return self.copy_to_nook_1(screen, aspect, cdata, dest)
        
    def cover_resize(self, screen, aspect, cdata):
        img = Image()
        img.load(cdata)
        rescale = None
        if aspect:
            swidth, sheight = screen
            cwidth, cheight = img.size[:2]
            # calc new image size
            # Set rescale factor to match screen height
            rescale = (int(sheight * cwidth / cheight), sheight)
            if rescale[0] > swidth:
                # if too wide for screen, set to match screen width
                rescale = (swidth, int(swidth * cheight / cwidth))
        else:
            rescale = screen
        return img, rescale

