#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
try:
    from PyQt5.Qt import QMenu, QToolButton
except ImportError:
    from PyQt4.Qt import QMenu, QToolButton
from calibre.gui2 import error_dialog, gprefs
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import prefs

import calibre_plugins.quick_preferences.config as cfg
from calibre_plugins.quick_preferences.common_utils import (set_plugin_icon_resources, get_icon,
                                                     create_menu_item, create_menu_action_unique)

PLUGIN_ICONS = ['images/quick_preferences.png']

class QuickPreferencesAction(InterfaceAction):

    name = 'Quick Preferences'
    action_spec = ('Quick Preferences', None, None, None)
    action_type = 'current'
    popup_type = QToolButton.InstantPopup

    automerge_choices = [
                (_('Ignore duplicate incoming formats'), 'ignore'),
                (_('Overwrite existing duplicate formats'), 'overwrite'),
                (_('Create new record for each duplicate format'), 'new record')]

    def genesis(self):
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        self.rebuild_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        # Setup hooks so that we rebuild the dropdown menus each time to represent latest history
        self.menu.aboutToShow.connect(self.about_to_show_menu)

    def rebuild_menus(self):
        m = self.menu
        m.clear()
        cfp = cfg.plugin_prefs[cfg.STORE_FILE_PATTERN_NAME]
        cos = cfg.plugin_prefs[cfg.STORE_OTHER_SHORTCUTS_NAME]

        # Add the file pattern regular expression menu items
        self.pattern_menus = []
        for menu_item in cfp[cfg.KEY_MENUS]:
            if menu_item[cfg.KEY_ACTIVE]:
                text = menu_item[cfg.KEY_TITLE]
                file_pattern = menu_item[cfg.KEY_REGEX]
                swap_names = menu_item[cfg.KEY_SWAP_NAMES]
                ac = create_menu_action_unique(self, m, text, tooltip=file_pattern,
                         triggered=partial(self.switch_file_pattern, file_pattern, swap_names),
                         is_checked=False, shortcut_name='Apply Add Filename Pattern: ' + text)
                self.pattern_menus.append((ac, file_pattern, swap_names))

        # Add the static checkbox driven menu items
        m.addSeparator()
        self.swap_author_names = create_menu_action_unique(self, m, 'Swap author names',
                        is_checked=False, shortcut_name='Toggle Add Option: Swap author names',
                        triggered=partial(self.switch_checkbox_preference, 'swap_author_names'))
        self.swap_author_names.setVisible(cos[cfg.OPT_SWAP_AUTHOR_NAMES][0])
        self.read_file_metadata = create_menu_action_unique(self, m, 'Read metadata from file contents',
                        is_checked=False, shortcut_name='Toggle Add Option: Read metadata from file contents',
                        triggered=partial(self.switch_checkbox_preference, 'read_file_metadata'))
        self.read_file_metadata.setVisible(cos[cfg.OPT_READ_FILE_METADATA][0])
        self.add_formats_to_existing = create_menu_action_unique(self, m, 'Automerge added books if exist',
                        is_checked=False, shortcut_name='Toggle Add Option: Automerge added books if exist',
                        triggered=partial(self.switch_checkbox_preference, 'add_formats_to_existing'))
        self.add_formats_to_existing.setVisible(cos[cfg.OPT_ADD_FORMAT_EXISTING][0])
        self.automerge_sub_menu = QMenu('Automerge type', self.gui)
        if cos[cfg.OPT_ADD_FORMAT_EXISTING][0]:
            m.addMenu(self.automerge_sub_menu)
        self.automerge_menus = []
        for row, value in enumerate(self.automerge_choices):
            text, key = value
            ac = create_menu_action_unique(self, self.automerge_sub_menu, text, is_checked=False, shortcut=False,
                                           triggered=partial(self.switch_automerge_setting, key))
            self.automerge_menus.append(ac)

        # Add a menu item to invoke the standard preferences dialog
        m.addSeparator()
        create_menu_action_unique(self, m, _('Preferences')+'...', 'config.png',
                                  shortcut=False, triggered=self.open_preferences_dialog)
        create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
                                  shortcut=False, triggered=self.show_configuration)
        self.gui.keyboard.finalize()

    def about_to_show_menu(self):
        # Look to set the checked state of the file pattern menu items
        cfp = cfg.plugin_prefs[cfg.STORE_FILE_PATTERN_NAME]
        current_file_pattern = prefs['filename_pattern']

        is_swap_names_checked = prefs['swap_author_names']
        for action, file_pattern, swap_names in self.pattern_menus:
            checked = False
            if file_pattern == current_file_pattern:
                # We have a match on the regex, but it might not be our row
                if swap_names is None:
                    # This file pattern doesn't alter the swap names value
                    checked = True
                elif swap_names == is_swap_names_checked:
                    checked = True
            action.setChecked(checked)

        # Now set the checked state of other static menu items
        self.swap_author_names.setChecked(prefs['swap_author_names'])
        self.read_file_metadata.setChecked(prefs['read_file_metadata'])
        self.add_formats_to_existing.setChecked(prefs['add_formats_to_existing'])

        # Set the checked state of the automerge submenu
        automerge_type = gprefs['automerge']
        for row, action in enumerate(self.automerge_menus):
            text, key = self.automerge_choices[row]
            action.setChecked(automerge_type == key)

    def switch_file_pattern(self, file_pattern, swap_names):
        if len(file_pattern) == 0:
            return error_dialog(self.gui, 'Cannot switch regular expression',
                        'No regular expression specified in this menu item.',
                        show=True)
        prefs['filename_pattern'] = file_pattern
        if swap_names is not None:
            prefs['swap_author_names'] = swap_names

    def switch_checkbox_preference(self, prefs_name):
        prefs[prefs_name] = not prefs[prefs_name]

    def switch_automerge_setting(self, automerge_type):
        gprefs['automerge'] = automerge_type

    def open_preferences_dialog(self):
        self.gui.iactions['Preferences'].do_config()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
