#!/usr/bin/perl

use strict;

#--- Declare variabels
my ($PRG, $VERSION, $AUTHOR, $output, $verbose);
my (@input, @words, @output);
my ($lf, $ff, $dash, $par_beg, $par_end, $out, $rcfile,
    %head_beg, %head_end, %head_keys, $defLvl,
    $quote, $line, $last, $len, $logging, $toc, $level,
    $toc_str, @toc, @toc_lvl, $tocNo, $indent, $ignore,
    $pagenos, $number, $parmode, $title, $author, $fmt,
    $style,  $package, $charset, $align, $cmd, 
    $center_beg, $center_end, $it_beg, $it_end, $doc_foot,
    $newline, $words, $par_finished, $newpar);

#--- Intitialise variables
$PRG="txt2html";
$AUTHOR="T.Berndt";
$VERSION='3.4';

$verbose=0;
$toc_str='Contents';
$tocNo=0;
$fmt="html";
$align="JUSTIFY";
$par_beg='<p ALIGN=$align>';
$par_end='</p>'."\n";
$center_beg="<br>\n<center>";
$center_end="</center>\n<br>";
$it_beg="<i>";
$it_end="</i>";
$newline="<br>";
$quote='&quot;';
$doc_foot='</body>\n</html>';
$lf='
';
$ff='';
$dash="—";
$defLvl="2";
$indent="style='text-indent:0.75cm'";
$parmode=1;
$logging=0;
$pagenos=0;
$toc=0;
$ignore=0;

$rcfile="$ENV{HOME}/.$PRG"."rc";
#--- an alternative rc-file given on the cmdline?
foreach $_ (@ARGV){
    if (/-R\b/){
	shift();
	$rcfile=shift();
	$rcfile=~s/^~/$ENV{HOME}/;
	last;
    }
}

#--- get the config file, if present
if (open(RCFILE, "<$rcfile")){
    my @args=@ARGV;                # get a copy of the cmdline, for later use 
    undef(@ARGV);
    while(<RCFILE>){
	chomp();
	next if (/^\s*$/);         # ignore empty lines
	next if (/^\#/);	   # ignore lines starting with '#'
	s/\#+.*$//;                # ignore stuff trailing a '#'
	@words=split();
	if ($words[0]=~/^-H\b/ || $words[0]=~/^-C\b/){
	    $line=shift(@words);
	    $_=join(" ", @words);
	    @words=($line, $_);
	}
	push(@ARGV, @words);
    }
    push(@ARGV, @args);            # let the cmdline to override the defaults
    close(RCFILE);
    undef($line);
    undef(@words);
}
#--- Parse the command line
while ($_=shift(@ARGV)){
    if (/^-o\b/){
	$output=shift();
    } elsif (/^-h\b/){
	helpMsg();
    } elsif (/^-t\b/){
	$title.=shift()." ";
    } elsif (/^-a\b/){
	$author.=shift()." ";
    } elsif (/^-A\b/){
	$align=shift();
    } elsif (/^-S\b/){
	$style=shift();
    } elsif (/^-P\b/){
	$package=shift();
    } elsif (/^-E\b/){
	$charset=shift();
    } elsif (/^-T\b/){
	unless ($fmt eq "tex"){
	    $toc=($toc?0:1);
	} else {
	    # some special logic for my pageless toc in latex-mode:
	    # $toc=0 -> no toc
	    # $toc=1 -> standard latex \tableofcontents
	    # $toc=2 -> Tommy's manually made toc
	    $toc++;
	    $toc=0 if ($toc==3);
	}
    } elsif (/^-tocStr\b/){
	$toc_str=shift();
	$toc=1;
    } elsif (/^-f\b/){
	push(@input, shift())
    } elsif (/^-i\b/){
	$ignore=($ignore?0:1);
    } elsif (/^-v\b/){
	$verbose++;
    } elsif (/^-N\b/){
	$pagenos=($pagenos?0:1);
    } elsif (/^-H\b/){
	$_=shift();
	my @keys=split();
	if ($keys[0]>0){
	    my $l=shift(@keys);
	    $head_keys{$l}=\@keys if ($l);
	} elsif ($keys[0]=~/^D\b/){
	    $defLvl=eval($keys[1]);
	    $defLvl=2 unless ($defLvl);
	}
    } elsif (/^-C\b/){
	$cmd=shift();
    } elsif (/^-F\b/){
	$fmt=lc(shift());
	unless ($fmt=~/htm/ || $fmt=~/tex/){
	    print(STDERR "WARNING: invalid format ($fmt) given. ".
		  "Resetting to default: html\n");
	    $fmt="html";
	}
	$fmt="html" if ($fmt=~/htm/);
    } elsif (/^-p\b/){
	$parmode=shift();
#	if (($parmode<0 || $parmode>2)&& $parmode!~/auto/){
	if (($parmode<0 || $parmode>2)&& !$parmode eq "auto"){
	    print(STDERR "WARNING: invalid mode ($parmode)".
		  "given for paragraph-detetction\n".
		  "\tResetting to default: 1\n");
	    $parmode=1;
	}
    } elsif (/^-l\b/){
	$logging=($logging?0:1);
    } else {
	push(@input, $_);
    }
}
if ($fmt eq "html"){
    $par_beg=($align=~/off\b/i?"<p>":"<p ALIGN=$align>");
    $head_beg{1}="<h1>";
    $head_end{1}="</h1>";
    $head_beg{2}="<h2>";
    $head_end{2}="</h2>";
    $head_beg{3}="<h3>";
    $head_end{3}="</h3>";
    $head_beg{4}="<h4>";
    $head_end{4}="</h4>";
} elsif ($fmt eq "tex"){
    $newline="\n";#$newline='\\\\'."\n";
    $par_beg="";
    $par_end="\n";#"\n\n";
    $center_beg='\mbox{}\\\\\centerline{';
    $center_end='}'."\n";
    $it_beg='\emph{';
    $it_end='}';
    $quote='"';
    $defLvl=2;
    $doc_foot='\end{document}';

    $head_beg{1}=($toc==1?'\part{':'\part*{');
    $head_beg{2}=($toc==1?'\section{':'\section*{');
    $head_beg{3}=($toc==1?'\subsection{':'\subsection*{');
    $head_beg{4}=($toc==1?'\subsubsection{':'\subsubsection*{');
    $head_end{1}=$head_end{2}=$head_end{3}=$head_end{4}='}';
}

if ($logging){
    $verbose=1 unless ($verbose);
    msgLog("\nDebugging info are written to '$PRG.log'\n\n");
    open(STDERR, ">$PRG.log") || errorMsg(2);
}

#--- replace ~ by the proper home-directory
$output=~s/^~/$ENV{HOME}/;
$style=~s/^~/$ENV{HOME}/;
$package=~s/^~/$ENV{HOME}/;
for (my $i=0; $i<$#input; $i++){
    $input[$i]=~s/^~/$ENV{HOME}/;
}

msgLog("Settings:\n".
       "\tinput     \t= @input\n".
       "\toutput    \t= $output\n".
       "\ttitle     \t= $title\n".    
       "\tauthor    \t= $author\n".    
       "\tfmt       \t= $fmt\n".
       "\tstyle     \t= $style\n".
       "\tpackage   \t= $package\n".
       "\tdefLvl    \t= $defLvl\n".
       "\tpagenos   \t= $pagenos\n".
       "\tparmode   \t= $parmode\n".
       "\ttoc       \t= $toc\n".
       "\tignore    \t= $ignore\n".
       "\tcommand   \t= $cmd\n".
       "\tverbose   \t= $verbose\n"
       );
if ($verbose>1){
    msgLog("Keys for chapter detection:\n");
    foreach my $i (keys %head_keys){
	msgLog("\t$i        \t= @{$head_keys{$i}}\n");
    }
}
msgLog("-" x 79 ."\n");

#--- copy file(s) given on the cmd-line to the input 
@ARGV=@input unless ($#input==-1);
@input=<>; 

#--- Print the usage stuff if no files are given
#errorMsg(1) unless @input;

#--- set parmode according to autoanalysis?
analyseText() if ($#input>0 && $parmode eq "auto");

#--- Open the outfile, if requested
if ($output){
    open(OUT, ">$output") or errorMsg(2);
    $out=\*OUT;
} else {
    $out=\*STDOUT;
}

#--- write a docuement-head
if ($fmt eq "html"){
    print($out "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n");
    print($out "<html>\n");
    print($out "\n<head>\n");
    print($out "<title>$title</title>\n") if $title;
    print($out "  <meta http-equiv=\"content-type\" content=\"text/html".($charset?"; charset=$charset":"")."\">\n");
    print($out "  <meta name=\"generator\" content=\"$PRG V$VERSION\">\n");
    if ($style){
	open(STYLE, "<$style");
	print($out <STYLE>);
	print($out "\n");
    }
    print($out "</head>\n");
    print($out "<body>\n");
    print($out "<h1>\n<br><i><big><big>$title</big></big></i></h1>\n") if ($title);
    print($out "<p align=\"right\"><big><big><strong><i>$author</i></strong></big></big></p>\n") if ($author);
} elsif ($fmt eq "tex"){
    print($out 
	  "%" x 79 ."\n".
	  "%%%\n".
	  "%%% generator: $PRG V$VERSION\n".
	  "%%%\n".
	  "%" x 79 ."\n".
	  '\documentclass{article}'."\n");
    if (!$style && !$package){#($style=~/\.tex\b|\.sty\b/){
	print($out 
	      '\usepackage{hyperref}'."\n".
	      '\evensidemargin0cm'."\n".
	      '\oddsidemargin0cm'."\n".
	      '\voffset-0.5cm'."\n".
	      '\textwidth15cm'."\n".
	      '\textheight24cm'."\n");
    } elsif ($style){
	open(STYLE, "<$style");
	print($out <STYLE>);
	print($out "\n");
	close(STYLE);
    } elsif ($package){
	$package=~s/\.sty//;
	print($out "\\usepackage{$package}\n");
    }
    print($out '\begin{document}'."\n");
    if ($title || $author){
	print($out '\noindent\rule{\textwidth}{0.7pt}\\\ '."\n");
	print($out '\mbox{ }\\\[3ex]'."\n");
	print($out '{\Huge\bf '.$title.'}\\\[3ex]'."\n") if ($title);
	print($out '\begin{flushright}'."\n".'\emph{'.$author.'}\end{flushright} '."\n") if ($author); 
	print($out '\noindent\rule{\textwidth}{0.7pt}\\\ '."\n");
	print($out '\mbox{ }\\\[8ex]'."\n");
    }
}

#--- collect lines up to ... depending on $parmode
msgLog("collect lines according to \$parmode=$parmode\n");

#--- the main loop over the text to be processed
for (my $i=0; $i<=$#input; $i++){
    #--- this is to skip the acroread's newapge (lf + ff)
    next if (length($input[$i])==0 && length($input[$i+1])==0);
    $_=$input[$i];
    my $next=$input[$i+1];

    chomp();
    next if /^$/;
    s/$lf//g;
    next if (/$ff/);

    #--- make some replacements
    s/ /""/ig;
    s/“/$quote/g;
    s/”/$quote/g;   
    s/»/$quote/g;
    s/«/$quote/g;

    if ($fmt eq "tex"){
	s|\$|\\\$|g;             # dollar
	s/#/\#/g;		 # hash
	s/\^/\\^/g;              # circumflex
	s/%/\\%/g;               # per cent
	s/&/\\&/g;               # ampersand
  	s//\$^\circ\$/g;        # degree
	s/^(\t|\s+)/\n/;         # leading (tab|space) + newline -> new par
	s/^\s*\"/\n`\`/g;        # leading " + newline  -> ``
	s/\s\"/ `\`/g;           # leading " -> ``
	s/\"$/\'\'/g;            # trailing " at the end of the line
	s/\b\"/\'\'/g;           # trailing " after a word
	s/(\.|\,|\?|!|\:|;|-)\"/$1\'\'/g; # punctuation + " -> ''
	s/^(\s|\t)*_+$/\\mbox{}\\\\ \\centerline{\\hfill\\hrulefill\\hfill} \\\\ \\noindent /;
	s/\cl/\\newpage/;
	s/\(c\)/\\copyright/;
	s/‘/\`/ig;       # â == &lsquo;
	s/’/\'/ig;       # â == &rsquo;
	s/—/---/ig;      # â == &mdash;
    } else {
	s/--/—/g;
    }
    #s/(^|\s)_/ $it_beg/g;        # start italics
    #s/(\w)_\s/\1$it_end /g;      # end italics
    s/(`|^|\s)_/ $it_beg/g;        # start italics
    s/(\w*)_(\s|,|\.|\?|\!|\$|\n)/\1$it_end /g;      # end italics
    s/_(\$|\b|\n)/\1$it_end /g;      # end italics
       
    @words=split();
    $words=eval($#words+1);
    $len=length();
    msgLog("\tinput[$len; $words]\t>". 
	  sprintf("%.50s", "$_").($len>50?"...":"")."<\n", 1);
       
    if ($pagenos>0){
	if ($len && ($words==0 || $words==2)){
	    s/-|<|>//g;
	    $number=eval($_);
	    msgLog("\t\t#words: $number\tval=$number\n", 1);
	    next if ($number!=0);
	}
    }
    next if chapter();
    if (/\*\s*\*\s*\*/){
	if ($parmode!=1){
	    push(@output, "$center_beg$_$center_end\n");
	    $par_finished=1 if ($parmode==0);
	    next;
	} else {
	    $line="$center_beg$_$center_end";
	    $par_finished=1;
	    undef($_);
	}
    }

    #--- collect the chunks of text into paragraphs
    if ($parmode==0) {			# seperator: leading spaces
	if (/^\s{2}|^\t|^$/ || $len==0){
# 	    unless ($len){
# 		if ($fmt eq "html"){
# 		    $line="\n&nbsp;<br>\n";
# 		} elsif ($fmt eq "tex"){
# 		    $line="\n$_\n";
# 		}
# 		$par_finished=1;
# 	    }
	    $par_finished=1;
	    push(@output, "$par_beg$line$par_end");
	    if ($par_finished){
		my $i=$#output;
		my $j=$i-1;
		$line=$output[$i];
		$output[$i]=$output[$j];
		$output[$j]=$line;
		undef($par_finished);
	    }
	    $line=$_;
	} else {
	    $line.=$_." \n"; 
	}
    } elsif ($parmode==1){		# seperator: empty line
	s/^\s+//g;
	if (length()==0){
	    # ignore newlines if the next line starts lowecase
	    next if ($next=~/^[a-zäöü]/); 
	    push(@output, ($line!~/^<br>\n<center>/?
			   "$par_beg$line$par_end":
			   "$line"));
	    undef($line);
	} else {
	    $line.=$_." "; # " ".$_; produces leading spaces
	}
    } elsif ($parmode==2) {		# seperator: new-line; 1 line = 1 par
	if (/^\s*$/ && !$ignore){ # this line is empty and not to be ignored
	    if ($fmt eq "html"){
		$_="&nbsp;<br>";
	    } else {
		$_='\mbox{}\\\\';
	    }
	}
	#$_.="&nbsp;<br>" if (!$ignore && /^\s*$/ && $fmt eq "html");
	push(@output, "$par_beg$_$par_end");
	$line=$_;	  # get a copy of the last line
    }
}
#--- for parmode=1 the last paragraph needs to be added afterwards
push(@output, "$par_beg$line$par_end\n");# if ($parmode==1);

#--- some additional info
msgLog("-" x 79 ."\n".
       "\tinput      : $#input\n".
       "\toutput     : $#output\n".
       "-" x 79 ."\n");

#--- insert the table of contents, if desired
if ($toc){
    if ($fmt eq "html"){
	my $last=0;
	print($out "\n<br><hr>\n<h2>$toc_str</h2>\n");

	for ($tocNo=0; $tocNo<=$#toc; $tocNo++){
	    if ($last!=$toc_lvl[$tocNo]){
		print($out "<ul>"  x eval($toc_lvl[$tocNo]-$last)) if ($last<$toc_lvl[$tocNo]);
		print($out "</ul>" x eval($last-$toc_lvl[$tocNo])) if ($last>$toc_lvl[$tocNo]);
		print($out "\n");
	    }
	    $last=$toc_lvl[$tocNo];
	    chomp($toc[$tocNo]);
	    print($out "<li><a href=\"#CHAP$tocNo\"> $toc[$tocNo]</a>\n");
	}
	print($out "</ul>" x eval($last-$toc_lvl[$tocNo]-1)) if ($last>$toc_lvl[$tocNo]);
	print($out "</ul><hr><br>\n");
    } elsif ($fmt eq "tex") {
	$last=0;
	if ($toc==1){
	    print($out '\tableofcontents \mbox{ }\\\[3ex]'."\n");
	} else {
	    print($out "\n".$head_beg{$defLvl}.$toc_str.$head_end{$defLvl}."\n");
	    print($out '\noindent'."\n");
	    for ($tocNo=0; $tocNo<=$#toc; $tocNo++){
		chomp($toc[$tocNo]);
		print($out '\\\ '."\n") if ($last>$toc_lvl[$tocNo]);
		print($out '\mbox{}');
		print($out '$\quad$' x eval($toc_lvl[$tocNo]-1));
		print($out "$toc[$tocNo]\\dotfill ");
		print($out "\\hyperlink{CHAP$tocNo}{\$\\to\$}\\\\ \n");
		$last=$toc_lvl[$tocNo];
	    }
	    print($out '\\\[3ex]'."\n");
	}
    }
}

#--- print the actual output
foreach $_ (@output){
    print($out "$_");
    msgLog($_, 2);
}

#--- write the document foot
print($out "$doc_foot\n");
close($out) if $output;

#--- run $cmd
exec("$cmd $output") if ($cmd && $output);

#--- END OF MAIN --------------------------------------------------------------
sub chapter{
    my(@levels, @keys);

    #--- this returns the chapter if given as number and a dash
    return addChapter($defLvl) if (($words==1 && $_>0) || (/^\s*\d+\d*\d*\s*-/));

    #--- this deals with keywords
    @levels=sort(keys(%head_keys));
    foreach my $level (@levels){
	@keys=@{$head_keys{$level}};

	foreach my $key (@keys){
	    return addChapter($level) if (/^\s*$key\b/i);
	}
    }
}
#----------------------------------------------------------------------
sub addChapter{
    my ($level, $beg, $end, $anchor);
    $level=$_[0];
    $beg=$head_beg{$level};
    $end=$head_end{$level};
    chomp();
    if ($toc){
	push(@toc, $_);
	if ($fmt eq "html"){
	    $anchor="<a name=\"CHAP$tocNo\"></a>";
	} elsif ($fmt eq "tex"){
	    if ($toc==1){
		$anchor="\\label{CHAP$tocNo}";
	    } elsif ($toc==2){
		$anchor="\\hypertarget{CHAP$tocNo}{}";
	    }
	}
	push(@toc_lvl, $level);
	$tocNo++;
    }
    if ($fmt eq "html"){
	push(@output, "\n$anchor$beg$_ $end\n");
    } elsif ($fmt eq "tex"){
	push(@output, "\n$beg$_ $end\n$anchor\n");
    }
    $par_finished=1 if ($parmode==0);
    return 1;
}
#----------------------------------------------------------------------
sub analyseText{
    my ($lines, $len, $count, $avglen, $empties, $nonempties, $indents,
	@ratio);
    undef($lines);
    foreach $_ (@input){
	chomp();
	s/$lf//g;
	s/$ff//g;
	$lines++;
	$len=length();
	
	$indents++ if (/^\s+/&&$len>1);
	if ($len<2){
	    $empties++;
	} else {
	    $count+=$len;
	}
    }
    $nonempties=$lines-$empties;
    if ($nonempties!=0){
	$avglen=eval($count/$nonempties);
	$ratio[0]=eval($indents/$nonempties);
	$ratio[1]=eval($empties/$nonempties);
    } else {
	$avglen="---";
	$ratio[0]="---";
	$ratio[1]="---"
    }
    if ($indents!=0){
	$ratio[2]=eval($empties/$indents);
    } else {
	$ratio[2]="999";
    }
    msgLog("\tlines in total  : ".eval($lines)."\n".
	   "\tindented lines  : ".eval($indents)."\n".
	   "\tempties         : ".eval($empties)."\n".
	   "\tnon-empty lines : ".eval($nonempties)."\n".
	   "\ttotal length    : ".eval($count)."\n".
	   "\tavg length      : ".eval($avglen)."\n".
	   "\tindents/lines   : ".eval($ratio[0])."\n".
	   "\tempties/lines   : ".eval($ratio[1])."\n".
	   "\tempties/indents : ".eval($ratio[2])."\n".
	   "\n"
	  );
    
#--- CAVEAT: These rules are based only on some heuristics, 
#            i.e. no more than an educated guess, based on a few examples only
    if ($avglen<80){
	$parmode = 1 if ($ratio[2]>1);
	$parmode = 0 if ($ratio[0]>0.1);
    } else {
	$parmode = 2 if ($avglen>80);
    }
    if ($parmode eq "auto"){
	msgLog("\tanalyseText failed to determine a mode!\n");
	$parmode = 1;
    }
    msgLog("\tparmode   ".
	   "\t= $parmode (determined by auto-text analysis)\n".
	   "-" x 79 ."\n");
}
#------------------------------------------------------------------------------
sub errorMsg{
    my ($errorNo);
    $errorNo=$_[0];
    if ($errorNo==1){
	print(STDERR "ERROR($errorNo): No input file(s) given.\n\n" );
	helpMsg();
    } elsif ($errorNo==2){
	print(STDERR "ERROR($errorNo): cannot open output file.\n");
    } elsif ($errorNo==3){
	print(STDERR "ERROR($errorNo): cannot open style file.\n");
    } else {
	print(STDERR "ERROR: $errorNo.\n");
    }
    exit($errorNo);
}
#------------------------------------------------------------------------------
sub msgLog{
    my $txt=$_[0];
    my $vl_min=$_[1];
    print(STDERR $_[0]) if ($verbose>$vl_min);
}
#------------------------------------------------------------------------------
sub helpMsg{
    print("$PRG V$VERSION (c) by $AUTHOR\n".
	  "This program comes with ABSOLUTELY NO WARRANTY.\n\n".
	  "usage: $PRG [...] [-o <outfile>] [[-f] <file1>] [<file2> ...] \n".
	  "  -o <outfile>            : prints output into <outfile>\n".
	  "  -f <file> [<file2> ...] : reads from the given files\n".
	  "                            If no files are given STDIN is read\n".
	  "  -t <title>              : sets the document\'s title\n".
	  "  -a <author>             : inserts the author\'s name\n".
	  "  -p <mode>               : sets the way how paragraphs are detected\n".
	  "                            mode=0: line starting with spaces\n".
	  "                            mode=1: empty line is seperator\n".
	  "                            mode=2: a single line is a prargraph\n".
	  "                            mode=auto: tries to autodetect the mode\n".
	  "                            Default: mode=1\n".
	  "  -A <align>              : aligns paragraphs accordingly\n".
	  "                            Default: justified alignment\n".
	  "                            <align>=off turns off alignment\n".
	  "  -F <fmt>                : determines the output format: html, tex.\n".
	  "                            Default: html\n".
	  "  -i                      : ignore empty lines(only with -p 2)\n".
	  "                            Default: keep empty lines\n".
	  "  -v                      : verbose output\n".
	  "  -l                      : write debug-info to logfile\n". 
#	  "  -I                      : prompts the user for input\n".
	  "  -h                      : prints this help and terminates\n".
	  "\n");
	  
    
    print("Run $PRG -v -h to get more information on the switches.\n".
	  "Stacked -v accumulate, i.e. $PRG -v -v produces more output.\n"
	  ) if ($verbose==0);

    print("Advanced options/switches:\n".
	  "  -C <cmd>                : sets the command to be executed\n".
	  "  -H <n> <key1 key2 ...>  : sets the key words for chapter headings\n".
	  "                            for level <n>, used to make\n".
	  "                            <hn>...</hn> headings. \n".
	  "  -H D <n>                : sets the default header level to <n>\n".
	  "                            Default: 2\n".
	  "  -S <stylefile>          : insert <stylefile> as STYLE.\n".
	  "  -P <package>            : Adds a \\usepackage{<package>} into the LaTeX-file\n".
	  "  -E <charset>            : use <charset> for encoding.\n".
	  "                            Default: utf-16\n".
	  "  -T                      : creates a table of contents.\n".
	  "                            Default: off\n".
	  "  -tocStr <string>        : sets the header of the TOC to <string>.\n".
	  "                            Default: Content\n".
	  "  -N                      : keep page numbers\n".
	  "                            Default: keep them if present at all\n". 
	  "  -R <rcfile>             : Read <rcfile> instead of the defaul one.\n".
	  "\n"
	  ) if ($verbose);

    print("- On the config-file: The config file's name is hard-wired to\n".
	  "  '.txt2htmlrc'. If this file exists in the user's home directory\n".
	  "  (assuming perl's \$ENV{HOME} works under OSs other than (LI|U)N*X\n".
	  "  as well) its content is interpreted as parameters in the above\n".
	  "  format.\n".
	  "  The settings given on the command line overwrite those found in\n".
	  "  the rc-file.\n".
	  "\n".
	  "- On paragraphs: The default behaviour assumes that\n".
	  "  paragraphs are seperated by empty lines (mode=1).\n".
	  "  If mode=auto is chosen $PRG tries to evaluate the paragraphing\n".
	  "  automatically based on some heuristical rules. This rules are not\n".
	  "  carved out of stone, but hardly more than educated guesses.\n".
	  "  If these rules fail to find an answer the default value mode=1\n".
	  "  will be used.\n".
	  "  There are cases where a single line makes up a paragraph (mode=2) AND\n".
	  "  there is an additional empty line between all the paragraphs.\n".
	  "  In this case the -i switch comes handy, as it will remove those\n".
	  "  empty lines.\n".
	  "\n".
	  "- On chapters: $PRG is capable to detect chapters(and the like) by the\n".
	  "  keys given with the -H <n> parameter list. \n".
	  "  The list of words given, will be used to identify chapter headers.\n".
	  "  A chapter key is assumed to be a the beginning of a line. \n".
	  "  Aside from these key words a chapter header is assumed to be present\n".
	  "  if the line contains only and begins with a number, or\n".
	  "  if the line starts with a number followed by (at least) a single dash (-)\n".
	  "  In these two cases a chapter header is inserted according to the\n".
	  "  value specified by the -H D switch.\n".
	  "  If the -T switch is set, a table of contents made up of hyperlinks\n".
	  "  will be inserted. \n".
	  "  Unless -tocStr <string> is given, the table of contents is introduced\n".
	  "  by 'Content', otherwise <string> will be used instead.\n".
 	  "  If a LaTeX file is generated and the creation of a table of \n".
	  "  content is turned off, the usual unnumberd LaTeX command are employed,\n".
	  "  e.g. \\section*{...} instead of \\section{...}\n".
	  "  On the other hand, if -toc is specified once, a the standard LaTeX\n".
	  "  \\tableofcontents is used.\n".
	  "  However, if -toc is specified twice, again the unnumbered commands\n".
	  "  are used, but some sort of table of contents is generated, that is\n".
	  "  hyperlinked to the respective locations in the text, but skips the\n".
	  "  auto-numbering, that might be annoying, if the chapters are numbered\n".
	  "  already in the source text.\n".
	  "\n".
	  "- On alignment: By default the paragraphs are aligned as justified.\n".
	  "  This can be changed by the -align switch, then the formatting will\n".
	  "  be made according to <p ALIGN=...>.\n".
	  "  If no alignment is desired pass \'off\' as value for this switch.\n".
	  "\n".
	  "- On StyleSheets: If -style stylefile is given, the content of this\n".
	  "  file is inserted without modification into the output-file.\n".
	  "  If the stylefile does not exist, no further warnings/errors are\n".
	  "  given.\n".
	  "  Instead of a style file to be included, for LaTex files a package file\n".
	  "  can be specified to be used for LaTeX's \\usepackage{<package>} specifying\n".
	  "  -P <package>\n".
	  "\n".
	  "- On page numbers: If the input text contains pagenumbers\n".
	  "  (which I consider rather annoying in html) one can try the\n".
	  "  -pn switch to get rid of them. In order to work the pagenumber\n".
	  "  must be the only content of the line (and evaluate to a number)\n".
	  "  This might produce some annoying but completely harmless messages\n".
	  "  So, if you are sure there aren't any pagenumbers present in the\n".
	  "  text and the messages *do* annoy you just tell $PRG to keep the\n".
	  "  pagenumbers and don't eval the lines.\n".
	  "\n".
	  "- On the -t and -a parameter: These two switches are additive,\n".
	  "  meaning that passing e.g. -a foo -a bar will result in 'foo bar'\n".
	  "  in the output html. This was necesarry to correctly parse the \n".
	  "  arguments when started from the jave-GUI.\n"
	  )  if ($verbose>1);

    exit(0);
}
