# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import, print_function)
__license__   = 'GPL v3'
__copyright__ = '2019 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.155"   #Miscellany

import os, apsw, ast, re
from functools import partial

from PyQt5.Qt import (Qt, QDialog, QLabel,  QFont, QWidget, QApplication, QFileDialog,
                                       QIcon, QComboBox, QSize, QPushButton, QVBoxLayout, QHBoxLayout,
                                       QFrame, QCheckBox, QDialogButtonBox, QButtonGroup, QLineEdit)

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import gprefs, error_dialog, info_dialog
from calibre.utils.html2text import html2text

from calibre_plugins.job_spy.config import prefs

MODE_PREVIEW = "Preview"
MODE_UPDATE = "Update"

COMMENTS = "Comments"
DATEADDED = "Date Added"
IDENTIFIERS = "Identifiers"
PUBLISHER = "Publisher"
PUBDATE = "Published Date"
SERIES = "Series"
TAGS = "Tags"
TITLE = "Title"

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(10, 10)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        #~ if DEBUG: self.geom = None
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class ImportCSVFileToUpdateMetadataDialog(SizePersistedDialog):

    def __init__(self,maingui,parent,icon,csv_path):
        unique_pref_name = 'Job_Spy:import_csv_file_to_update_metadata_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.hide()

        self.maingui = maingui
        self.guidb = self.maingui.library_view.model().db

        self.font = QFont()
        self.font.setBold(False)
        self.font.setPointSize(10)

        mytitle = 'JS+ GUI Tool:  Import CSV File to Update Metadata'
        self.setWindowTitle(mytitle)
        self.setWindowIcon(icon)

        t = "<p style='white-space:wrap'>This tool is used to update metadata using values from a CSV File.  View the ToolTips of the selected CSV File name for CSV instructions."

        self.setToolTip(t)

        self.layout_top = QVBoxLayout(self)
        self.layout_top.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout_top)

        self.imported_csv_list,is_valid,msg = self.import_csv_file(csv_path)
        if not is_valid:
            error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)
            self.show()
            self.hide()
            self.reject()
            self.close()
            return

        if not len(self.imported_csv_list) > 1:
            msg = "The number of CSV Rows is less than 2; nothing can be done..."
            error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)
            self.show()
            self.hide()
            self.reject()
            self.close()
            return

        self.csv_is_bad = False
        self.csv_is_bad_msg = None

        self.regex = re
        self.regex.escape("\\")
        self.regexpr = r'(?:^|,)(?=[^"]|(")?)"?((?(1)[^"]*|[^,"]*))"?(?=,|$)'    # https://stackoverflow.com/questions/18144431/regex-to-split-a-csv
        self.regex_compiled_pattern = self.regex.compile(self.regexpr,  re.IGNORECASE|re.DOTALL|re.MULTILINE)

        header_row = unicode(self.imported_csv_list[0])

        first_char = header_row[0:1]
        if ord(first_char) == 65279:          # 65279   'ZERO WIDTH NO-BREAK SPACE' (U+FEFF)   BOM Byte Order Mark
            header_row = header_row[1: ]
            #~ if DEBUG: print("BOM in Header Row:  first_char:>>>", str(first_char),"<<< unicode ", str(ord(first_char)))

        s_split = self.split_row_using_regex(header_row)

        if self.csv_is_bad:
            if DEBUG: print("CSV header is malformed: ", str(s_split))
            msg = "CSV Header is too malformed to use.  Open it in a simple text editor and examine its structure.  Refer to the ToolTips for CSV requirements.  Nothing done." + self.csv_is_bad_msg
            error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)
            self.show()
            self.hide()
            self.reject()
            self.close()
            return

        header_list = []
        self.header_colnum_dict = {}

        colnum = -1
        for header_name in s_split:
            colnum = colnum + 1
            if header_name is None:
                if DEBUG: print(" header_name is None.")
                continue
            if not header_name > " ":
                if DEBUG: print(" header_name is blank")
                continue

            if colnum == 0:
                first_char = header_name[0:1]
                if ord(first_char) == 65279:          # 65279   'ZERO WIDTH NO-BREAK SPACE' (U+FEFF)   BOM Byte Order Mark
                    header_name = header_name[1: ]
                    #~ if DEBUG: print("BOM Byte Order Mark first_char:>>>", str(first_char),"<<< unicode ", str(ord(first_char)), "  in ", header_name)

            header_name = header_name.replace('"','').strip()
            header_name = str(header_name)
            header_list.append(header_name)
            self.header_colnum_dict[header_name] = colnum
        #END FOR

        self.n_header_columns = len(header_list)

        if self.n_header_columns < 2:
            if DEBUG: print("self.n_header_columns < 2")
            msg = "The number of CSV Header Columns is less than 2; nothing can be done..."
            error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)
            self.show()
            self.hide()
            self.reject()
            self.close()
            return

        self.custom_columns_metadata_dict = self.maingui.current_db.field_metadata.custom_field_metadata()

        self.current_custom_columns_list = []
        tmp_list = self.guidb.custom_field_keys()
        for label in tmp_list:
            if not label.startswith("#"):
                label = "#" + label
            if not label in self.custom_columns_metadata_dict:
                continue
            custcol = self.custom_columns_metadata_dict[label]   # should be a '#label'
            cc_datatype = custcol['datatype']
            if (cc_datatype == "float") or (cc_datatype == "int") or (cc_datatype == "rating") or (cc_datatype == "composite"):
                continue
            self.current_custom_columns_list.append(label)
        #END FOR
        del tmp_list
        self.current_custom_columns_list.sort()
        #-----------------------------------------------------
        #-----------------------------------------------------
        t = "<p style='white-space:wrap'>"
        t = t + "A comma-separated values (CSV) file is a delimited text file that uses a comma to separate values. A CSV file stores tabular data in plain text. \
        Each line of the file is a data record. Each record consists of one or more fields, separated by commas. \
        The use of the comma as a field separator is the source of the name for this file format. \
        A record separator must consist of either a line feed or a carriage return and line feed pair (spreadsheet applications handle this automatically, \
        but simple text editors used to create a CSV file may not). "
        t = t + "<br><br>These rules <b>must</b> be followed for this tool to work for you:"
        t = t + "<br><br>[1] The entire file must be 'saved' in the Unicode (UTF-8) character set (not ASCII)."
        t = t + "<br><br>[2] The field delimiter (column separator) must be a single comma (not tab-separated or fixed width)."
        t = t + "<br><br>[3] All text cells (<i>which means all cells</i>) must be double-quoted."
        t = t + "<br><br>[4] All cells must be text (not integer or float numerics). Spreasheets default to numeric, so a value like 24 is treated as a 'number', not text, unless you enter it as '24 or format its column as 'text'.  This tool does not update integer, float, or 'built from others' Custom Columns.  "
        t = t + "<br><br>[5] The first row must be a 'header' row which contains the unique double-quoted textual name of each column."
        t = t + "<br><br>[6] All rows after the first row must contain either textual values or 'empty text' within each and every column.  Leading and trailing spaces will be removed from each value automatically. \
                                        To indicate that the existing Custom Column value should be 'removed' (changed to 'nothing'), use the special keyword 'NULL'.  <u>For 'nothing', use a blank space so it will become double-quoted text.</u>"
        t = t + "<br><br>[7] All rows must have the same number of double-quoted textual columns as the 'header' row.  "
        t = t + "<br><br>[8] There must be at least one (1) row after the first 'header' row."
        t = t + "<br><br>[9] There is no limit to the numer of 'header' columns.  Example:  Zotero export .csv files have at least 87 columns."
        t = t + "<br><br>[10] If you open your CSV file in a very simple text editor, should not find a row that ends with a simple comma (,), or has 2 (,,) or more (,,,,) commas together.  See [6], above. "
        t = t + "<br><br>[11] The <i>minimum</i> number of columns is two (2):  one for the value to be updated into Calibre, and one with the values to match to book metadata to select the appropriate books to be updated."
        t = t + "<br><br>See: <i>'import_csv_to_update_metadata_example.csv'</i> located in your .../calibre/plugins/job_spy/resources  directory.  "

        self.font.setBold(True)
        self.font.setPointSize(10)
        self.csv_selected_input_file_qlabel = QLabel(csv_path)
        self.csv_selected_input_file_qlabel.setAlignment(Qt.AlignCenter)
        self.csv_selected_input_file_qlabel.setFont(self.font)
        self.csv_selected_input_file_qlabel.setToolTip(t)
        self.layout_top.addWidget(self.csv_selected_input_file_qlabel)
        #-----------------------------------------------------
        #-----------------------------------------------------
        s = "Columns: " + str(len(header_list))
        s = s + "  Rows: " + str(len(self.imported_csv_list))
        self.csv_selected_input_rows_qlabel = QLabel(s)
        self.csv_selected_input_rows_qlabel.setAlignment(Qt.AlignCenter)
        self.csv_selected_input_rows_qlabel.setFont(self.font)
        self.csv_selected_input_rows_qlabel.setToolTip("<p style='white-space:wrap'>This is the total number of columns and rows, including the Header row, of the selected CSV File.")
        self.layout_top.addWidget(self.csv_selected_input_rows_qlabel)
        self.font.setPointSize(10)
        self.font.setBold(False)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #~ self.spacer_header_import_1_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_header_import_1_label)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.layout_column_header_import = QHBoxLayout()
        self.layout_column_header_import.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_column_header_import)
        #-----------------------------------------------------
        self.spacer_header_import_2_label = QLabel("     ")
        self.spacer_header_import_2_label.setMinimumWidth(25)
        self.spacer_header_import_2_label.setMaximumWidth(25)
        self.spacer_header_import_2_label.setFont(self.font)
        self.layout_column_header_import.addWidget(self.spacer_header_import_2_label)

        t = "<p style='white-space:wrap'>Select a single Column Header containing the values in the CSV file that you wish to have updated into the metadata column specified just below.\
            All rows after the first row must contain either textual values or 'empty text' within each and every column.  Leading and trailing spaces will be removed from each value automatically.\
            Empty values will be ignored even for 'matched' books. To indicate that an existing Custom Column value should be 'removed' (changed to 'nothing'), use the special keyword 'NULL' \
            instead of 'empty text' or blank spaces.  Standard Columns that support the 'NULL' keyword are Comments, Date Added, Published Date, Publisher, Series, and Tags (but not Identifers or Title)."

        self.csv_column_header_import_qlabel = QLabel("CSV Column Values to Import:   ")
        self.csv_column_header_import_qlabel.setFont(self.font)
        self.csv_column_header_import_qlabel.setToolTip(t)
        self.csv_column_header_import_qlabel.setMinimumWidth(275)
        self.csv_column_header_import_qlabel.setMaximumWidth(275)
        self.layout_column_header_import.addWidget(self.csv_column_header_import_qlabel)

        self.csv_column_header_import_combobox = QComboBox()
        self.csv_column_header_import_combobox.setFont(self.font)
        self.csv_column_header_import_combobox.setEditable(False)
        self.csv_column_header_import_combobox.setFrame(True)
        self.csv_column_header_import_combobox.setMaxVisibleItems(25)
        self.csv_column_header_import_combobox.setMinimumWidth(250)
        self.csv_column_header_import_combobox.setMaximumWidth(250)
        self.csv_column_header_import_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.csv_column_header_import_combobox.setToolTip(t)
        self.layout_column_header_import.addWidget(self.csv_column_header_import_combobox)

        for row in header_list:
            self.csv_column_header_import_combobox.addItem(row)
        #END FOR
        self.csv_column_header_import_combobox.setCurrentIndex(-1)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_target_custom_column = QHBoxLayout()
        self.layout_target_custom_column.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_target_custom_column)
        #--------------------------------------------------
        self.target_custom_column_checkbox = QCheckBox("")
        self.target_custom_column_checkbox.setMinimumWidth(25)
        self.target_custom_column_checkbox.setMaximumWidth(25)
        self.target_custom_column_checkbox.setFont(self.font)
        self.target_custom_column_checkbox.setToolTip("<p style='white-space:wrap'>Target Column is the selected #Custom Column.")
        self.layout_target_custom_column.addWidget(self.target_custom_column_checkbox)

        t = "<p style='white-space:wrap'>Select the 'text', 'comments', 'enumeration' / 'fixed values', 'series-like', 'yes/no', or 'datetime' datatype Custom Column that you wish to have updated in the current Library for any matching Books. \
        Note that Calibre 'Edit Metadata' function that this tool uses will ignore any attempts to update a 'fixed values' Custom Column with any CSV value that was not a predefined 'fixed value' when that Custom Column was created."
        self.target_custom_column_qlabel = QLabel("Target #Column to Update:   ")
        self.target_custom_column_qlabel.setFont(self.font)
        self.target_custom_column_qlabel.setToolTip(t)
        self.target_custom_column_qlabel.setMinimumWidth(275)
        self.target_custom_column_qlabel.setMaximumWidth(275)
        self.layout_target_custom_column.addWidget(self.target_custom_column_qlabel)

        self.target_custom_column_combobox = QComboBox()
        self.target_custom_column_combobox.setFont(self.font)
        self.target_custom_column_combobox.setEditable(False)
        self.target_custom_column_combobox.setFrame(True)
        self.target_custom_column_combobox.setMaxVisibleItems(25)
        self.target_custom_column_combobox.setMinimumWidth(250)
        self.target_custom_column_combobox.setMaximumWidth(250)
        self.target_custom_column_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.target_custom_column_combobox.setToolTip(t)
        self.layout_target_custom_column.addWidget(self.target_custom_column_combobox)

        for row in self.current_custom_columns_list:
            self.target_custom_column_combobox.addItem(row)
        #END FOR

        self.target_custom_column_combobox.setCurrentIndex(-1)
        self.target_custom_column_combobox.setEnabled(True)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_target_standard_column = QHBoxLayout()
        self.layout_target_standard_column.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_target_standard_column)
        #--------------------------------------------------
        self.target_standard_column_checkbox = QCheckBox("")
        self.target_standard_column_checkbox.setMinimumWidth(25)
        self.target_standard_column_checkbox.setMaximumWidth(25)
        self.target_standard_column_checkbox.setFont(self.font)
        self.target_standard_column_checkbox.setToolTip("<p style='white-space:wrap'>Target Column is the selected Standard Column.")
        self.layout_target_standard_column.addWidget(self.target_standard_column_checkbox)

        t = "<p style='white-space:wrap'>The following 8 Standard Columns are available to be updated using the values provided in the CSV File:\
        <br><br><b>Comments:</b>  Comments are always 'appended' to the bottom of any pre-existing Comments (i.e., they are always 'merged).  Use empty double-quoted text for rows requiring no action to be taken.  The Keyword 'NULL' removes all pre-existing Comments.\
        Otherwise, use 'simple' formatting that a simple CSV File can reliably parse. \
        <br><br><b>Date Added:</b>  Dates must be text in the format of YYYYMMDD or YYYY-MM-DD  (or empty double-quoted text for rows requiring no action to be taken).  Ensure that the 2nd format is not interpreted by your spreadsheet application as an arithmetic subtraction expression.\
        <br><br><b>Identifiers:</b>  Identifiers must use the single-quoted format of:  'type':'value' .  For multiple identifiers, separate each single-quoted pair with a comma:  'type':'value','type':'value','type':'value'  .  Identifiers are always 'merged' with any existing Identifiers.  Use empty double-quoted text for rows requiring no action to be taken. The keyword 'NULL' is ignored.\
        <br><br><b>Publisher:</b>  Publisher is a simple text value (or empty double-quoted text for rows requiring no action to be taken).  The Keyword 'NULL' removes the pre-existing Publisher.\
        <br><br><b>Published Date:</b>  Dates must be in the format of YYYYMMDD or YYYY-MM-DD  (or empty double-quoted text for rows requiring no action to be taken).  Ensure that the 2nd format is not interpreted by your spreadsheet application as an arithmetic subtraction expression.\
        <br><br><b>Series:</b>  Series must be in a format of  'name' or 'name[i]' (or empty double-quoted text for rows requiring no action to be taken).\
        <br><br><b>Tags:</b>  Use a comma (,) with no spaces to separate multiple Tags.  Example:  blue,green,red,yellow.  The Keyword 'NULL' removes all pre-existing Tags. \
        <br><br><b>Title:</b>  Title is a Required Column in Calibre, so NULL is invalid with Title.  Use empty double-quoted text for rows requiring no action to be taken.  \
        <br><br>See: <i>'import_csv_to_update_metadata_example.csv'</i> located in your .../calibre/plugins/job_spy/resources directory. "

        self.target_standard_column_qlabel = QLabel("Target Column to Update:   ")
        self.target_standard_column_qlabel.setFont(self.font)
        self.target_standard_column_qlabel.setToolTip(t)
        self.target_standard_column_qlabel.setMinimumWidth(275)
        self.target_standard_column_qlabel.setMaximumWidth(275)
        self.layout_target_standard_column.addWidget(self.target_standard_column_qlabel)

        self.target_standard_column_combobox = QComboBox()
        self.target_standard_column_combobox.setFont(self.font)
        self.target_standard_column_combobox.setEditable(False)
        self.target_standard_column_combobox.setFrame(True)
        self.target_standard_column_combobox.setMaxVisibleItems(25)
        self.target_standard_column_combobox.setMinimumWidth(250)
        self.target_standard_column_combobox.setMaximumWidth(250)
        self.target_standard_column_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.target_standard_column_combobox.setToolTip(t)
        self.layout_target_standard_column.addWidget(self.target_standard_column_combobox)

        self.target_standard_column_combobox.addItem(COMMENTS)
        self.target_standard_column_combobox.addItem(DATEADDED)
        self.target_standard_column_combobox.addItem(IDENTIFIERS)
        self.target_standard_column_combobox.addItem(PUBLISHER)
        self.target_standard_column_combobox.addItem(PUBDATE)
        self.target_standard_column_combobox.addItem(SERIES)
        self.target_standard_column_combobox.addItem(TAGS)
        self.target_standard_column_combobox.addItem(TITLE)

        self.target_standard_column_combobox.setCurrentIndex(-1)
        self.target_standard_column_combobox.setEnabled(False)
        #--------------------------------------------------
        #--------------------------------------------------
        self.target_column_checkbox_buttongroup = QButtonGroup()
        self.target_column_checkbox_buttongroup.setExclusive(True)
        self.target_column_checkbox_buttongroup.addButton(self.target_custom_column_checkbox)
        self.target_column_checkbox_buttongroup.addButton(self.target_standard_column_checkbox)

        self.target_custom_column_checkbox.setChecked(True)
        self.target_column_is_custom = True

        self.target_custom_column_checkbox.stateChanged.connect(self.event_target_column_checkbox_statechanged)
        self.target_standard_column_checkbox.stateChanged.connect(self.event_target_column_checkbox_statechanged)
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_header_import_2_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_header_import_2_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.frame_1 = QFrame()
        self.frame_1.setFrameShape(QFrame.HLine)
        self.frame_1.setFrameShadow(QFrame.Sunken)
        self.layout_top.addWidget(self.frame_1)
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_book_match_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_book_match_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_column_header_match = QHBoxLayout()
        self.layout_column_header_match.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_column_header_match)
        #--------------------------------------------------
        t = "<p style='white-space:wrap'>Select the Column Header containing the values in the CSV file that you wish to have matched to the Custom Column specified below.\
                    <br><br>For Tag-Like Custom Columns, the CSV match values should be in the format 'green,blue,yellow' and not 'green, blue, yellow' and not 'green & blue & yellow'."
        self.csv_column_header_match_qlabel = QLabel("CSV Column Values to Match to Books:   ")
        self.csv_column_header_match_qlabel.setFont(self.font)
        self.csv_column_header_match_qlabel.setToolTip(t)
        self.csv_column_header_match_qlabel.setMinimumWidth(300)
        self.csv_column_header_match_qlabel.setMaximumWidth(300)
        self.layout_column_header_match.addWidget(self.csv_column_header_match_qlabel)

        self.csv_column_header_match_combobox = QComboBox()
        self.csv_column_header_match_combobox.setFont(self.font)
        self.csv_column_header_match_combobox.setEditable(False)
        self.csv_column_header_match_combobox.setFrame(True)
        self.csv_column_header_match_combobox.setMaxVisibleItems(25)
        self.csv_column_header_match_combobox.setMinimumWidth(250)
        self.csv_column_header_match_combobox.setMaximumWidth(250)
        self.csv_column_header_match_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.csv_column_header_match_combobox.setToolTip(t)
        self.layout_column_header_match.addWidget(self.csv_column_header_match_combobox)

        for row in header_list:
            self.csv_column_header_match_combobox.addItem(row)
        #END FOR
        self.csv_column_header_match_combobox.setCurrentIndex(-1)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_book_match_table = QHBoxLayout()
        self.layout_book_match_table.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_book_match_table)
        #-----------------------------------------------------
        t = "<p style='white-space:wrap'>Select the Custom Column in the current Library that should be matched to the CSV File column specified above."
        self.book_match_table_qlabel = QLabel("Book Match #Column:   ")
        self.book_match_table_qlabel.setFont(self.font)
        self.book_match_table_qlabel.setToolTip(t)
        self.book_match_table_qlabel.setMinimumWidth(300)
        self.book_match_table_qlabel.setMaximumWidth(300)
        self.layout_book_match_table.addWidget(self.book_match_table_qlabel)

        self.book_match_table_combobox = QComboBox()
        self.book_match_table_combobox.setFont(self.font)
        self.book_match_table_combobox.setEditable(False)
        self.book_match_table_combobox.setFrame(True)
        self.book_match_table_combobox.setMaxVisibleItems(25)
        self.book_match_table_combobox.setMinimumWidth(250)
        self.book_match_table_combobox.setMaximumWidth(250)
        self.book_match_table_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.book_match_table_combobox.setToolTip(t)
        self.layout_book_match_table.addWidget(self.book_match_table_combobox)

        for row in self.current_custom_columns_list:
            self.book_match_table_combobox.addItem(row)
        #END FOR

        self.book_match_table_combobox.setCurrentIndex(-1)
        #-----------------------------------------------------
        self.layout_book_match_expression = QHBoxLayout()
        self.layout_book_match_expression.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_book_match_expression)
        #-----------------------------------------------------
        t = "<p style='white-space:wrap'>Select the matching expression that should be used to compare the value from the selected CSV column \
        to the value from the selected Custom Column for all books to find any matching books.  \
        Remember that leading and trailing spaces will always have been removed automatically from both values prior to comparison. \
        Letter 'case' is always ignored (the comparison is case-insensitive).\
        <br><br>'equals':  the two values are identical.  Example: 'green,red,yellow' = 'green,red,yellow'.\
        <br><br>'contains':  the CSV value contains the value from the Custom Column.  Example: 'green,red,yellow' contains 'red,yellow', 'green,red', 'green', 'red', 'yellow' but <b>not</b> 'green,yellow'.\
       <br><br>'is contained in':  the value from the Custom Column contains the CSV value.  This is the reverse of 'contains', explained above."
        self.book_match_expression_qlabel = QLabel("Book Match Expression:   ")
        self.book_match_expression_qlabel.setFont(self.font)
        self.book_match_expression_qlabel.setToolTip(t)
        self.book_match_expression_qlabel.setMinimumWidth(300)
        self.book_match_expression_qlabel.setMaximumWidth(300)
        self.layout_book_match_expression.addWidget(self.book_match_expression_qlabel)

        self.book_match_expression_combobox = QComboBox()
        self.book_match_expression_combobox.setFont(self.font)
        self.book_match_expression_combobox.setEditable(False)
        self.book_match_expression_combobox.setFrame(True)
        self.book_match_expression_combobox.setMaxVisibleItems(25)
        self.book_match_expression_combobox.setMinimumWidth(250)
        self.book_match_expression_combobox.setMaximumWidth(250)
        self.book_match_expression_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.book_match_expression_combobox.setToolTip(t)
        self.layout_book_match_expression.addWidget(self.book_match_expression_combobox)

        self.book_match_expression_combobox.addItem("equals")
        self.book_match_expression_combobox.addItem("contains")
        self.book_match_expression_combobox.addItem("is contained in")

        self.book_match_expression_combobox.setCurrentIndex(0)
        #--------------------------------------------------
        self.layout_use_case_sensitive_matching = QHBoxLayout()
        self.layout_use_case_sensitive_matching.setAlignment(Qt.AlignRight)
        self.layout_top.addLayout(self.layout_use_case_sensitive_matching)
        #--------------------------------------------------
        self.use_case_sensitive_matching_checkbox = QCheckBox("Case-Sensitive Matching?")
        self.use_case_sensitive_matching_checkbox.setFont(self.font)
        self.use_case_sensitive_matching_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to match values using their exact letter cases (uppercase and lowercase)?")
        self.layout_use_case_sensitive_matching.addWidget(self.use_case_sensitive_matching_checkbox)

        self.use_case_sensitive_matching_checkbox.setChecked(False)

        self.is_case_sensitive = False
        #--------------------------------------------------
        #--------------------------------------------------
        self.frame_3 = QFrame()
        self.frame_3.setFrameShape(QFrame.HLine)
        self.frame_3.setFrameShadow(QFrame.Sunken)
        self.layout_top.addWidget(self.frame_3)
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_checkbox_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_checkbox_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.update_only_matched_books_checkbox = QCheckBox("Update only matched Books with proper CSV Column Values.")
        self.update_only_matched_books_checkbox.setFont(self.font)
        self.update_only_matched_books_checkbox.setToolTip("<p style='white-space:wrap'>Update only Books that were matched using the match criteria above (assuming a non-blank CSV File 'update value' exists).")
        self.layout_top.addWidget(self.update_only_matched_books_checkbox)

        self.update_only_matched_books_checkbox.setChecked(True)
        #--------------------------------------------------
        self.layout_update_matched_books_using_default = QHBoxLayout()
        self.layout_update_matched_books_using_default.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_update_matched_books_using_default)
        #--------------------------------------------------
        self.update_matched_books_using_default_checkbox = QCheckBox("Use 'Default' for matched:")
        self.update_matched_books_using_default_checkbox.setMinimumWidth(243)
        self.update_matched_books_using_default_checkbox.setMaximumWidth(243)
        self.update_matched_books_using_default_checkbox.setFont(self.font)
        self.update_matched_books_using_default_checkbox.setToolTip("<p style='white-space:wrap'>Update only Books that were matched using the match criteria above, but if they have a blank CSV File 'update value', \
                                                                                                    use the stated Default Value for those matched books.  ")
        self.layout_update_matched_books_using_default.addWidget(self.update_matched_books_using_default_checkbox)

        self.update_matched_books_using_default_checkbox.setChecked(False)

        t = "<p style='white-space:wrap'>'Default'.  This value will be updated in the specified Column for all 'matched' books if the book's CSV File value is blank.\
        <br><br>Blank spaces and empty text are not allowed here.\
        <br><br><b>Custom Columns:</b> The appropriate Default for a particular Custom Column depends on its specific 'datatype'.  Use the Standard Column formats below for guidance.\
        <br><br><b>Comments:</b>  Comments are always 'appended' to the bottom of any pre-existing Comments (i.e., they are always 'merged).  The Keyword 'NULL' removes all pre-existing Comments.  \
         Otherwise, use 'simple' formatting that a simple CSV File can reliably parse.  Do not use HTML or other text containing double-quotes to avoid possible parsing errors.\
        <br><br><b>Date Added:</b>  Dates must be in the format of YYYYMMDD or YYYY-MM-DD.   Ensure that the 2nd format is not interpreted by your spreadsheet application as an arithmetic subtraction expression.\
        <br><br><b>Identifiers:</b>  Identifiers must use the single-quoted format of:  'type':'value' .  For multiple identifiers, separate each single-quoted pair with a comma:  'type':'value','type':'value','type':'value'  .  Identifiers are always 'merged' with any existing Identifiers.  The keyword 'NULL' is ignored.\
        <br><br><b>Publisher:</b>  Publisher is a simple text value.  The Keyword 'NULL' removes the pre-existing Publisher.\
        <br><br><b>Published Date:</b>  Dates must be in the format of YYYYMMDD or YYYY-MM-DD.   Ensure that the 2nd format is not interpreted by your spreadsheet application as an arithmetic subtraction expression.\
        <br><br><b>Series:</b>  Series must be in a format of  'name' or 'name[index]'.  If the [index] is missing, then only the Series Name will be updated.\
        <br><br><b>Tags:</b>  Use a comma (,) with no spaces to separate multiple Tags.  Example:  blue,green,red,yellow.  The Keyword 'NULL' removes all pre-existing Tags. \
        <br><br><b>Title:</b>  Title is a Required Column in Calibre, so NULL is invalid with Title. \
        <br><br>See: <i>'import_csv_to_update_metadata_example.csv'</i> located in your .../calibre/plugins/job_spy/resources directory. "

        self.update_matched_books_using_default_qlineedit = QLineEdit(self)
        self.update_matched_books_using_default_qlineedit.setText("")
        self.update_matched_books_using_default_qlineedit.setFont(self.font)
        self.update_matched_books_using_default_qlineedit.setToolTip(t)
        self.update_matched_books_using_default_qlineedit.setCursorPosition(0)
        self.update_matched_books_using_default_qlineedit.setMinimumWidth(200)
        self.update_matched_books_using_default_qlineedit.setMaximumWidth(200)
        self.layout_update_matched_books_using_default.addWidget(self.update_matched_books_using_default_qlineedit)

        #--------------------------------------------------
        self.layout_update_non_matched_books = QHBoxLayout()
        self.layout_update_non_matched_books.setAlignment(Qt.AlignLeft)
        self.layout_top.addLayout(self.layout_update_non_matched_books)
        #--------------------------------------------------
        self.update_non_matched_books_checkbox = QCheckBox("Update only unmatched Books with: ")
        self.update_non_matched_books_checkbox.setMaximumWidth(300)
        self.update_non_matched_books_checkbox.setMaximumWidth(300)
        self.update_non_matched_books_checkbox.setFont(self.font)
        self.update_non_matched_books_checkbox.setToolTip("<p style='white-space:wrap'>Update only Books that were <b>not</b> matched using the match criteria above.\
                                                                                <br><br>The CSV File values for unmatched books are ignored, and a single value to be used is defined here instead.")
        self.layout_update_non_matched_books.addWidget(self.update_non_matched_books_checkbox)

        self.update_non_matched_books_qlineedit = QLineEdit(self)
        self.update_non_matched_books_qlineedit.setText("")
        self.update_non_matched_books_qlineedit.setFont(self.font)
        self.update_non_matched_books_qlineedit.setToolTip("<p style='white-space:wrap'>This value will be updated in the specified Column for all 'unmatched' books.\
        <br><br>Blank spaces and empty text are not allowed here.\
        <br><br><b>Custom Columns:</b> The appropriate Default for a particular Custom Column depends on its specific 'datatype'.  Use the Standard Column formats below for guidance.\
        <br><br><b>Comments:</b>  Comments are always 'appended' to the bottom of any pre-existing Comments (i.e., they are always 'merged).  The Keyword 'NULL' removes all pre-existing Comments.\
        <br><br><b>Date Added:</b>  Dates must be in the format of YYYYMMDD or YYYY-MM-DD.\
        <br><br><b>Identifiers:</b>  Identifiers must use the single-quoted format of:  'type':'value' .  For multiple identifiers, separate each single-quoted pair with a comma:  'type':'value','type':'value','type':'value'  .  Identifiers are always 'merged' with any existing Identifiers.  The keyword 'NULL' is ignored.\
        <br><br><b>Publisher:</b>  Publisher is a simple text value.  The Keyword 'NULL' removes the pre-existing Publisher.\
        <br><br><b>Published Date:</b>  Dates must be in the format of YYYYMMDD or YYYY-MM-DD.\
        <br><br><b>Series:</b>  Series must be in a format of  'name' or 'name[index]'.  If the [index] is missing, then only the Series Name will be updated.\
        <br><br><b>Tags:</b>  Use a comma (,) with no spaces to separate multiple Tags.  Example:  blue,green,red,yellow.  The Keyword 'NULL' removes all pre-existing Tags. \
        <br><br><b>Title:</b>  Title is a Required Column in Calibre, so NULL is invalid with Title. \
        <br><br>See: <i>'import_csv_to_update_metadata_example.csv'</i> located in your .../calibre/plugins/job_spy/resources directory. ")
        self.update_non_matched_books_qlineedit.setCursorPosition(0)
        self.update_non_matched_books_qlineedit.setMinimumWidth(200)
        self.update_non_matched_books_qlineedit.setMaximumWidth(200)
        self.layout_update_non_matched_books.addWidget(self.update_non_matched_books_qlineedit)

        self.update_non_matched_books_checkbox.stateChanged.connect(self.event_non_matched_checkbox_statechanged)
        #--------------------------------------------------
        self.update_all_books_checkbox = QCheckBox("Update all Books (both matched and unmatched).")
        self.update_all_books_checkbox.setFont(self.font)
        self.update_all_books_checkbox.setToolTip("<p style='white-space:wrap'>Update all matched books as normal, then update all unmatched books with their special value (which may not be blank).")
        self.layout_top.addWidget(self.update_all_books_checkbox)
        #--------------------------------------------------
        self.update_checkbox_buttongroup = QButtonGroup()
        self.update_checkbox_buttongroup.setExclusive(True)
        self.update_checkbox_buttongroup.addButton(self.update_only_matched_books_checkbox)
        self.update_checkbox_buttongroup.addButton(self.update_non_matched_books_checkbox)
        self.update_checkbox_buttongroup.addButton(self.update_all_books_checkbox)
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_selected_only_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_selected_only_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.update_selected_only_checkbox = QCheckBox("Additionally Restrict Updates to Only Manually Selected Books")
        self.update_selected_only_checkbox.setFont(self.font)
        self.update_selected_only_checkbox.setToolTip("<p style='white-space:wrap'>After the 'matched' and/or 'unmatched' books are Marked, actually update their metadata only if they were originally Selected by the user.")
        self.layout_top.addWidget(self.update_selected_only_checkbox)
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_matching_sql_only_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_matching_sql_only_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.update_matching_sql_only_checkbox = QCheckBox("Additionally Restrict Updates to Only Books Matching SQL Snippet")
        self.update_matching_sql_only_checkbox.setFont(self.font)
        self.update_matching_sql_only_checkbox.setToolTip("<p style='white-space:wrap'>After all other matching options have already been performed, actually update a book's metadata only if that book has also been selected using the raw SQL stated below.")
        self.layout_top.addWidget(self.update_matching_sql_only_checkbox)

        self.update_matching_sql_only_checkbox.stateChanged.connect(self.event_sql_checkbox_statechanged)
        #--------------------------------------------------
        #--------------------------------------------------
        t = "<p style='white-space:wrap'>Raw SQL.  The Rules: \
                <br>[1] Must start with SELECT \
                <br>[2] The very first column selected must be an integer book id (e.g. 'books.id' or 'books_authors_link.book') \
                <br>[3] May not contain SQLite Keywords that add, change, or delete data or objects \
                <br>[4] May not contain semi-colons \
                <br>[5] May not contain question marks \
                <br>[6] May not contain double quotes \
                <br>[7] Must otherwise comply with the syntax in: https://www.sqlite.org/lang_corefunc.html \
                <br>[8] May optionally use the REGEXP (Regular Expression) Function as shown in the examples below.\
                <br><br>SQLite Syntax questions?  Answer them at http://www.w3schools.com/sql/ or http://www.sqlcourse.com/.<br>"
        t = t + "<br>Example:  SELECT id FROM books WHERE title LIKE '%of the%'  \
                     <br>Example:  SELECT id FROM books WHERE timestamp REGEXP '^2019-02-10'   \
                     <br>Example:  SELECT id FROM books WHERE pubdate REGEXP '^2018-'  \
                     <br>Example:  SELECT id FROM books WHERE last_modified REGEXP '^2019-03-13'  \
                     <br>Example:  SELECT id FROM books WHERE series_index > 1.0  \
                     <br>Example:  SELECT id FROM books WHERE author_sort LIKE '%Niven%' \
                     <br>Example:  SELECT book FROM books_tags_link WHERE tag IN (SELECT id FROM tags WHERE name REGEXP 'biography'  ) \
                     <br>Example:  SELECT book FROM books_publishers_link WHERE publisher IN (SELECT id FROM publishers WHERE name NOT LIKE '%TBD%' )\
                     <br>Example:  SELECT book FROM books_series_link WHERE series IN (SELECT id FROM series WHERE name LIKE '%Fire%Ice%' ) \
                     <br>Example:  SELECT book FROM comments WHERE text LIKE '%Pulitzer%' \
                     <br>Example:  SELECT book FROM identifiers WHERE type = 'isbn' OR type = 'lccn' OR type = 'viaf_author_id' \
                     <br>Example:  SELECT book FROM data WHERE format = 'EPUB'  \
                     <br>Example:  SELECT id FROM books WHERE pubdate REGEXP '^2019'  AND id IN (SELECT book FROM books_tags_link WHERE tag IN (SELECT id FROM tags WHERE name LIKE '%biography%'  )) \
                     <br><br>Tip: Date-times are stored using the UTC Time Zone, not a local Time Zone."

        self.font.setPointSize(8)

        sql = prefs['GUI_TOOLS_IMPORT_CSV_FILE_TO_UPDATE_METADATA_RAW_SQL'].strip()
        if sql == "":
            sql = prefs.defaults['GUI_TOOLS_IMPORT_CSV_FILE_TO_UPDATE_METADATA_RAW_SQL'].strip()
        self.update_matching_sql_only_qlineedit = QLineEdit(self)
        self.update_matching_sql_only_qlineedit.setText(sql)
        self.update_matching_sql_only_qlineedit.setAlignment(Qt.AlignLeft)
        self.update_matching_sql_only_qlineedit.setFont(self.font)
        self.update_matching_sql_only_qlineedit.setToolTip(t)
        self.update_matching_sql_only_qlineedit.setCursorPosition(0)
        self.update_matching_sql_only_qlineedit.setMinimumWidth(400)
        self.update_matching_sql_only_qlineedit.setMaximumWidth(600)
        self.layout_top.addWidget(self.update_matching_sql_only_qlineedit)

        self.update_matching_sql_only_qlineedit.hide()

        self.font.setPointSize(10)
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_merge_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_merge_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.merge_existing_tags_checkbox = QCheckBox("If Tags/Tag-Like, merge CSV Tags with existing Tags?")
        self.merge_existing_tags_checkbox.setFont(self.font)
        self.merge_existing_tags_checkbox.setToolTip("<p style='white-space:wrap'>If Tags/Tag-like, merge CSV Tags with existing Tags?  Otherwise, the CSV Tags will replace any existing Tags.")
        self.layout_top.addWidget(self.merge_existing_tags_checkbox)
        self.merge_existing_tags_checkbox.setChecked(True)
        self.merge_all_tags = True
        #--------------------------------------------------
        #--------------------------------------------------
        #~ self.spacer_checkboxes_label = QLabel("")
        #~ self.layout_top.addWidget(self.spacer_checkboxes_label)
        #--------------------------------------------------
        #--------------------------------------------------
        self.frame_4 = QFrame()
        self.frame_4.setFrameShape(QFrame.HLine)
        self.frame_4.setFrameShadow(QFrame.Sunken)
        self.layout_top.addWidget(self.frame_4)
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_pushbuttons = QHBoxLayout()
        self.layout_pushbuttons.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_pushbuttons)
        #--------------------------------------------------
        #--------------------------------------------------
        self.push_button_show_preview = QPushButton(" ", self)
        self.push_button_show_preview.setText("Preview Books to be Updated")
        self.push_button_show_preview.setFont(self.font)
        self.push_button_show_preview.setToolTip("<p style='white-space:wrap'>Show a preview of which Books would be actually updated if the above settings were actually executed.")
        self.push_button_show_preview.clicked.connect(self.show_preview)
        self.layout_pushbuttons.addWidget(self.push_button_show_preview)

        self.push_button_execute_updates = QPushButton(" ", self)
        self.push_button_execute_updates.setText("Actually Update Indicated Books")
        self.push_button_execute_updates.setFont(self.font)
        self.push_button_execute_updates.setToolTip("<p style='white-space:wrap'>Actually update the indicated Books using the above settings.")
        self.push_button_execute_updates.clicked.connect(self.execute_updates)
        self.layout_pushbuttons.addWidget(self.push_button_execute_updates)
        #--------------------------------------------------
        #--------------------------------------------------
        self.frame_5 = QFrame()
        self.frame_5.setFrameShape(QFrame.HLine)
        self.frame_5.setFrameShadow(QFrame.Sunken)
        self.layout_top.addWidget(self.frame_5)
        #--------------------------------------------------
        #--------------------------------------------------
        self.bottom_buttonbox = QDialogButtonBox()
        self.bottom_buttonbox.rejected.connect(self.reject)
        self.layout_top.addWidget(self.bottom_buttonbox)

        self.push_button_save_and_exit = QPushButton(" ", self)
        self.push_button_save_and_exit.setText("Exit")
        self.push_button_save_and_exit.setFont(self.font)
        self.push_button_save_and_exit.setToolTip("<p style='white-space:wrap'>Exit.")
        self.push_button_save_and_exit.clicked.connect(self.save_and_exit)
        self.bottom_buttonbox.addButton(self.push_button_save_and_exit,0)

        self.bottom_buttonbox.setCenterButtons(True)

        self.resize_dialog()

        self.dialog_closing(None)

        self.show()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_sql_checkbox_statechanged(self,e):
        if self.update_matching_sql_only_checkbox.isChecked():
            self.update_matching_sql_only_qlineedit.show()
        else:
            self.update_matching_sql_only_qlineedit.hide()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_target_column_checkbox_statechanged(self,e):
        if self.target_custom_column_checkbox.isChecked():
            self.target_custom_column_combobox.setEnabled(True)
            self.target_standard_column_combobox.setCurrentIndex(-1)
            self.target_standard_column_combobox.setEnabled(False)
            self.target_column_is_custom = True
        elif self.target_standard_column_checkbox.isChecked():
            self.target_standard_column_combobox.setEnabled(True)
            self.target_custom_column_combobox.setCurrentIndex(-1)
            self.target_custom_column_combobox.setEnabled(False)
            self.target_column_is_custom = False
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_non_matched_checkbox_statechanged(self,e):
        if self.update_non_matched_books_checkbox.isChecked():
            self.update_matched_books_using_default_checkbox.setChecked(False)
            self.update_matched_books_using_default_qlineedit.setText("")
    #-----------------------------------------------------
    #-----------------------------------------------------
    def show_preview(self):
        self.mode = MODE_PREVIEW
        msg = "Working in Preview Mode...Wait..."
        self.maingui.status_bar.show_message(_(msg), 5000)
        self.updates_control()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def execute_updates(self):
        self.mode = MODE_UPDATE
        msg = "Working in Update Mode...Wait..."
        self.maingui.status_bar.show_message(_(msg), 5000)
        self.updates_control()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def updates_control(self):

        if self.target_standard_column_checkbox.isChecked():
            self.target_column_is_custom = False
        else:
            self.target_column_is_custom = True

        if self.use_case_sensitive_matching_checkbox.isChecked():
            self.is_case_sensitive = True
        else:
            self.is_case_sensitive = False

        self.update_only_matched = self.update_only_matched_books_checkbox.isChecked()
        self.use_matched_default = self.update_matched_books_using_default_checkbox.isChecked()
        self.update_only_unmatched = self.update_non_matched_books_checkbox.isChecked()
        self.update_all_books = self.update_all_books_checkbox.isChecked()

        if self.update_only_unmatched:
            self.update_matched_books_using_default_checkbox.setChecked(False)

        if self.update_matching_sql_only_checkbox.isChecked():
            is_valid = self.validate_raw_sql_query()
            if not is_valid:
                msg = "Invalid Raw SQL"
                return False,msg

        is_valid,msg = self.create_main_list()
        if not is_valid:
            return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return False,'Database Connection Error.  Cannot Connect to the Current Library.'

        self.get_custom_column_technical_details(my_db,my_cursor)              #self.custom_columns_label_dict

        self.get_all_possible_bookids(my_db,my_cursor)                                   #self.all_possible_bookids_list

        if len(self.all_possible_bookids_list) == 0:
            my_db.close()
            msg = "No books exist in this Library.  Nothing can be done."
            return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        self.get_selected_books()                                                                        #self.selected_books_set

        if len(self.selected_books_set) == 0:
            my_db.close()
            msg = "No books have been selected.  Nothing can be done."
            return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        is_valid,msg = self.create_mysql_match(my_db,my_cursor)
        if not is_valid:
            my_db.close()
            return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        is_valid,msg = self.select_matching_books(my_db,my_cursor)
        my_db.close()
        if not is_valid:
            return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        if self.update_only_unmatched or self.update_all_books:
            is_valid,msg = self.select_unmatched_books()
            if not is_valid:
                return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        if self.update_only_matched:
            _books_to_be_updated_list = self.matching_books_list
        elif self.update_only_unmatched:
            _books_to_be_updated_list = self.unmatched_books_list
        elif self.update_all_books:
            _books_to_be_updated_list = self.matching_books_list + self.unmatched_books_list

        if self.update_matching_sql_only_checkbox.isChecked():
            is_valid,msg = self.filter_using_raw_sql()
            if not is_valid:
                return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        self.maingui.status_bar.show_message(_("Still Working...Wait..."), 10000)

        self.books_to_be_updated_list = []

        for book in _books_to_be_updated_list:
            if book in self.selected_books_set:
                if self.update_matching_sql_only_checkbox.isChecked():
                    if book in self.raw_sql_selected_books_set:
                        self.books_to_be_updated_list.append(book)
                else:
                    self.books_to_be_updated_list.append(book)
        #END FOR
        del _books_to_be_updated_list

        is_valid,msg = self.update_books()
        if not is_valid:
            return error_dialog(self.maingui, _('JS+ GUI Tool'),_(msg), show=True)

        self.unmark_all_books()

        self.mark_updated_books()

        self.show_results_message()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_all_possible_bookids(self,my_db,my_cursor):

        self.all_possible_bookids_list = []

        my_cursor.execute("SELECT id FROM books")
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        for row in tmp_rows:
            for id in row:
                self.all_possible_bookids_list.append(id)
            #END FOR
        #END FOR
        del tmp_rows
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_selected_books(self):

        if not self.update_selected_only_checkbox.isChecked():
            self.selected_books_set = set(self.all_possible_bookids_list)
            return

        self.selected_books_set = set([])

        book_ids_list = map(partial(self.convert_id_to_book), self.maingui.library_view.get_selected_ids() )
        for item in book_ids_list:
            book = item['calibre_id']
            self.selected_books_set.add(book)
        #END FOR
        del book_ids_list
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_id_to_book(self, idval):
        book = {}
        book['calibre_id'] = idval
        return book
    #-----------------------------------------------------
    #-----------------------------------------------------
    def create_main_list(self):

        self.import_column_header = self.csv_column_header_import_combobox.currentText()

        if self.target_column_is_custom:
            self.target_column = self.target_custom_column_combobox.currentText()
        else:
            self.target_column = self.target_standard_column_combobox.currentText()

        self.match_column_header = self.csv_column_header_match_combobox.currentText()
        self.match_table = self.book_match_table_combobox.currentText()
        self.match_expression = self.book_match_expression_combobox.currentText()


        if not self.import_column_header > " ":
            return False,"Criteria Incomplete...[1]"
        elif not self.target_column > " ":
            return False,"Criteria Incomplete...[2]"

        if not self.update_all_books:
            if not self.match_column_header > " ":
                return False,"Criteria Incomplete...[3]"
            elif not self.match_table > " ":
                return False,"Criteria Incomplete...[4]"

        if not self.import_column_header in self.header_colnum_dict:
            return False,("not self.import_column_header in self.header_colnum_dict: " + self.import_column_header)

        if not self.match_column_header in self.header_colnum_dict:
            return False,("not self.match_column_header in self.header_colnum_dict: " + self.match_column_header)

        if self.use_matched_default:
            self.matched_books_default_value = self.update_matched_books_using_default_qlineedit.text()
            self.matched_books_default_value = self.matched_books_default_value.strip()
            if not self.matched_books_default_value > " ":
                return False,"Default Value for Matched Books is missing but its checkmark has been checked 'yes'..."
            if self.target_column.lower() == IDENTIFIERS:
                default_idents = self.matched_books_default_value
                if not ":" in default_idents:
                    msg = "Default Identifiers:  Format Error (:)" + "  Value: " + str(default_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if not default_idents.count("'") > 3:
                    msg = "Default Identifiers:  Format Error (')" + "  Value: " + str(default_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if "{" in default_idents or "}" in default_idents:
                    msg = "Default Identifiers:  Format Error  ({)" + "  Value: " + str(default_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if default_idents.startswith(",") or default_idents.endswith(","):
                    msg = "Default Identifiers:  Format Error (,)" + "  Value: " + str(default_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if not len(default_idents) > 6:    #  '':''  actually passes thru ast.literal_eval, so 'x':'y' is the smallest valid value.
                    msg = "Default Identifiers:  Format Error (?)" + "  Value: " + str(default_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
            else:
                pass
        else:
            self.matched_books_default_value = None

        self.import_column_header_colnum = self.header_colnum_dict[self.import_column_header]
        self.match_column_header_colnum = self.header_colnum_dict[self.match_column_header]

        self.main_list = []

        self.csv_is_bad = False

        for i in xrange(1,len(self.imported_csv_list)):  #skip header row...
            row = self.imported_csv_list[i]
            row = row.replace('""',"''").strip()   # change any "" to '' to avoid regex parsing errors
            s_split = self.split_row_using_regex(row)
            if len(s_split) <> self.n_header_columns:
                n1 = str(len(s_split))
                n2 = str(self.n_header_columns)
                msg = "Number of row columns <> Number of header columns:  \n" + n1 + "<>" + n2 + " for row #" + str(i+1)
                msg = msg + '\n\n[1] If you have odd words, such as My\"\"Name, that have double-quotes in nonsensical places, that will cause this problem.  Simply change those double-quotes to single-quotes. '
                msg = msg + "\n\n[2] If you have complex Coments-like text with multiple paragraphs (via using Enter/Return) and special HTML formatting, that could cause this error.  If so, either simplify your Comments-like text or delete the entire CSV File Column for that Comments-like Column."
                msg = msg + "\n\n[3] If the problem persists, check for cells with no double-quote text format such as found in unsupported numeric columns, or cells with 'nothing' that should at least have empty text or a space. "
                msg = msg + "  Open your CSV File with a simple text editor, and search for two commas together (i.e. ,, ).  "
                msg = msg + "\n\n[4] If you run Calibre in debugging mode, the problem row will be printed in the debug log both 'before' and 'after' it was parsed."
                if DEBUG: print(msg)
                if DEBUG: print("-----------------------------------------------")
                if DEBUG: print("[raw row:] ", str(row))
                if DEBUG: print("-----------------------------------------------")
                if DEBUG: print("[split row:] ", str(s_split))
                if DEBUG: print("-----------------------------------------------")
                return False,msg
            update_value = s_split[self.import_column_header_colnum].replace('"','').strip()
            match_value = s_split[self.match_column_header_colnum].replace('"','').strip()
            if match_value > "":
                r = update_value,match_value   #do not filter by update_value > " " here since matched books default value may be specified...
                self.main_list.append(r)
            else:
                continue
        #END FOR

        if self.csv_is_bad:
            return False,("CSV is too malformed to use.  Open it in a simple text editor and examine its structure.  Refer to the ToolTips for CSV requirements.  Nothing done." + self.csv_is_bad_msg)

        if len(self.main_list) == 0:
            if DEBUG: print("len(self.main_list) == 0:")
            return False,"No valid CSV rows to process; Nothing done."

        return True,None
    #-----------------------------------------------------
    def split_row_using_regex(self,row):
        s_split = []
        tmp = []
        try:
            for m in self.regex.finditer(self.regex_compiled_pattern, row):
                tmp.append(unicode(m.group(0)))
            #END FOR
        except Exception as e:
            self.csv_is_bad = True
            self.csv_is_bad_msg = str(e)
            return s_split

        for r in tmp:
            if r is None:     # caused by ,,,,,
                r = " "
            r = unicode(r)
            r = r.strip()
            if r.startswith(","):
                r = r[1: ].strip()
            if r.startswith('"'):
                r = r[1: ].strip()
            if r.endswith('"'):
                r = r[0:-1].strip()
            if r.endswith(","):
                r = r[0:-1].strip()
            s_split.append(r)
            #~ if DEBUG: print("new column in s_split: ", r)
        #END FOR
        del tmp
        m = None
        del m
        return s_split
    #-----------------------------------------------------
    def create_mysql_match(self,my_db,my_cursor):
        self.mysql_match = None

        custcol = self.custom_columns_metadata_dict[self.match_table]   # should be a '#label'
        cc_table = custcol["table"]
        cc_datatype = custcol['datatype']

        id,label,name,datatype,mark_for_delete,editable,display,is_multiple,normalized = self.custom_columns_label_dict[self.match_table]   # should be a '#label'
        if normalized == 1:
            is_normalized = True
        else:
            is_normalized = False

        if is_multiple == 1:
            is_multiple = True
        else:
            is_multiple = False

        if cc_datatype == "text":
            cc_table_link = "books_" + cc_table + "_link"
        else:
            cc_table_link = "None"

        #~ if DEBUG: print("match table: ", cc_datatype,cc_table,cc_table_link, is_normalized, is_multiple)
        #----------------------------
        if self.match_expression == "equals":  #table value equals csv match value        table = ?
            if is_normalized:
                if is_multiple:
                    table1 = cc_table
                    table2 = cc_table_link
                    self.create_temp_views(my_db,my_cursor,table1,table2)
                    self.mysql_match = "SELECT book,'dummy' FROM __js_tags_by_book_concatenate WHERE  book IS NOT NULL AND tagsconcat = ?  " +  self.collate_keywords
                else:
                    self.mysql_match = "SELECT book,'dummy' FROM " +cc_table_link  + " WHERE  book IS NOT NULL AND " + cc_table_link  + ".value IN " + \
                                 " (SELECT id FROM " + cc_table + " WHERE " + cc_table + ".id = " + cc_table_link  +  ".value " + \
                                 " AND "  + cc_table + ".value = ?  " +  self.collate_keywords + " AND " + cc_table + ".value  IS NOT NULL  AND id IS NOT NULL )"
            else:
                self.mysql_match = "SELECT book,value FROM " + cc_table + " WHERE book IS NOT NULL AND value IS NOT NULL AND value = ?  " +  self.collate_keywords
        #----------------------------
        #----------------------------
        elif self.match_expression == "contains":   # csv match value contains table value      csv LIKE '%table%'       ? LIKE '%table%'
            if is_normalized:
                if is_multiple:
                    table1 = cc_table
                    table2 = cc_table_link
                    self.create_temp_views(my_db,my_cursor,table1,table2)
                    self.mysql_match = "SELECT book,'dummy' FROM __js_tags_by_book_concatenate WHERE book IS NOT NULL AND ? LIKE '%'||tagsconcat||'%' ESCAPE '\\' AND tagsconcat IS NOT NULL"
                else:
                    self.mysql_match = "SELECT book,'dummy' FROM " +cc_table_link  + " WHERE book IS NOT NULL AND " + cc_table_link  + ".value IN " + \
                                 " (SELECT id FROM " + cc_table + " WHERE " + cc_table + ".id = " + cc_table_link  +  ".value " + \
                                 " AND  ? LIKE '%'||"   + cc_table + ".value||'%' ESCAPE '\\' AND " + cc_table + ".value IS NOT NULL AND id IS NOT NULL )"
            else:
                self.mysql_match = "SELECT book,value FROM " + cc_table + " WHERE book IS NOT NULL AND ? LIKE '%'||value||'%' ESCAPE '\\' AND value IS NOT NULL"
        #----------------------------
        #----------------------------
        elif self.match_expression == "is contained in":  #table value contains csv match value      table LIKE '%csv%       table LIKE '%?%
            if is_normalized:
                if is_multiple:
                    table1 = cc_table
                    table2 = cc_table_link
                    self.create_temp_views(my_db,my_cursor,table1,table2)
                    self.mysql_match = "SELECT book,'dummy' FROM __js_tags_by_book_concatenate WHERE tagsconcat LIKE '%'||?||'%' ESCAPE '\\'  "
                else:
                    self.mysql_match = "SELECT book,'dummy' FROM " +cc_table_link  + " WHERE book IS NOT NULL AND " + cc_table_link  + ".value IN " + \
                                 " (SELECT id FROM " + cc_table + " WHERE " + cc_table + ".id = " + cc_table_link  +  ".value " + \
                                 " AND "  + cc_table + ".value LIKE '%'||?||'%' ESCAPE '\\' AND " + cc_table + ".value  IS NOT NULL AND id IS NOT NULL )"
            else:
                self.mysql_match = "SELECT book,value FROM " + cc_table + " WHERE book IS NOT NULL AND value LIKE  '%'||?||'%' ESCAPE '\\' AND value IS NOT NULL "
        #----------------------------
        #----------------------------
        else:
            return False,("Unsupported Match Expression: " + self.match_expression)
        #----------------------------

        if DEBUG: print("match expression", self.match_expression, " ---->> using sql: ", self.mysql_match)

        return True,None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def select_matching_books(self,my_db,my_cursor):
        self.matching_books_list = []
        self.book_value_dict = {}

        n = self.mysql_match.count("?")

        for pair in self.main_list:
            update_value,match_value = pair
            update_value = update_value.strip()
            if self.match_expression <> "equals":
                if "_" in match_value:  # _ is a wildcard in LIKE...      value LIKE  '%'||?||'%' ESCAPE '\\'
                    match_value = match_value.replace("_","\\_")
                if "%" in match_value:  # % is a wildcard in LIKE...   value LIKE  '%'||?||'%' ESCAPE '\\'
                    match_value = match_value.replace("%","\\%")
            if n == 1:
                my_cursor.execute(self.mysql_match,([match_value]))
            elif n == 2:
                my_cursor.execute(self.mysql_match,(match_value,match_value))
            else:
                break
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                if DEBUG: print("NOT FOUND FOR:  update_value>>match_value:", update_value, ">>",match_value)
                tmp_rows = []
            else:
                if DEBUG: print("FOUND FOR:           update_value>>match_value:", update_value, ">>",match_value)
            for row in tmp_rows:
                book,matched_value = row
                if not book in self.book_value_dict:  #a single book can have thousands of matching tags, but it only needs 1 to be updated from the csv file...
                    if update_value > " ":
                        self.book_value_dict[book] = update_value
                        self.matching_books_list.append(book)
                        if DEBUG: print("matched book: ", str(book), "match expression: ", self.match_expression, " first matched value: ", str(matched_value), " update value: ", str(update_value))
                    else:
                        if DEBUG: print("update_value is NOT > " ":", update_value)
                        if self.matched_books_default_value is not None:
                            self.book_value_dict[book] = self.matched_books_default_value
                            self.matching_books_list.append(book)
                            if DEBUG: print("Matched Book Value Defaulted for : ", str(book))
                        else:
                            continue
            #END FOR
            del tmp_rows
        #END FOR

        self.matching_books_list = list(set(self.matching_books_list))
        self.matching_books_list.sort()

        return True,None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def select_unmatched_books(self):
        #~ self.unmatched_books_list = self.all_possible_bookids_list minus self.matching_books_list
        self.unmatched_books_list = self.all_possible_bookids_list
        for book in self.all_possible_bookids_list:
            if book in self.matching_books_list:
                self.unmatched_books_list.remove(book)
        #END FOR
        self.unmatched_books_list.sort()
        update_value = self.update_non_matched_books_qlineedit.text()
        if update_value is None:
            update_value = ""
        update_value = update_value.strip()
        if update_value > " ":
            if update_value == "NULL":
                update_value = ""
            for book in self.unmatched_books_list:
                if not book in self.book_value_dict:
                    self.book_value_dict[book] = update_value
            #END FOR
            return True,None
        else:
            return False,"Update Value for Unmatched Books is blank; Nothing Done."
    #-----------------------------------------------------
    #-----------------------------------------------------
    def filter_using_raw_sql(self):

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            msg = 'Database Connection Error.  Cannot Connect to the Current Library.'
            return False,msg

        raw_sql = self.update_matching_sql_only_qlineedit.text()

        if raw_sql.count("REGEXP") > 0:
            func = "REGEXP"
            is_valid,emsg = self.apsw_create_user_functions(my_db,my_cursor,func)
            if not is_valid:
                my_db.close()
                msg = 'APSW Error:  Cannot Create REGEX User Function. ' + emsg
                return False,msg

        found_list,is_valid,msg = self.execute_raw_sql_query(my_db,my_cursor,raw_sql)
        my_db.close()
        if not is_valid:
            del found_list
            return False,msg

        self.raw_sql_selected_books_set = set(found_list)
        del found_list
        return True,None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def validate_raw_sql_query(self):
        raw_sql = self.update_matching_sql_only_qlineedit.text()

        lc_sql = str(raw_sql.lower())
        if lc_sql.count("select") == 0:
            error_dialog(self, _('JS+'),_(("SQL must be for a SELECT query.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False
        elif lc_sql.count(" create ") > 0 or lc_sql.count(" drop ") > 0 or lc_sql.count(" insert ") > 0 or lc_sql.count(" update ") > 0 \
                                                          or lc_sql.count(" delete ") > 0 or lc_sql.count(" replace ") > 0 or lc_sql.count(" reindex ") > 0 \
                                                          or lc_sql.count(" pragma ") > 0 or lc_sql.count(" alter ") > 0 or lc_sql.count(" vacuum ") > 0 \
                                                          or lc_sql.count(" create ") > 0 or lc_sql.count(" begin ") > 0 or lc_sql.count(" commit ") > 0:
            error_dialog(self, _('JS+'),_(("SQL must be for a SELECT query.  CREATE, DROP, INSERT, UPDATE, REPLACE, DELETE and so forth are strictly forbidden.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False
        elif lc_sql.count("?") > 0:
            error_dialog(self, _('JS+'),_(("SQL may not contain a ? (Question Mark).<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False
        elif lc_sql.count('"') > 0:
            error_dialog(self, _('JS+'),_(("SQL may not contain Double Quotes.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False
        elif lc_sql.count(";") > 0:
            error_dialog(self, _('JS+'),_(("SQL may not contain semi-colons.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False

        sql = raw_sql

        sql = sql.replace("select "," SELECT ")
        sql = sql.replace(" from "," FROM ")
        sql = sql.replace(" where "," WHERE ")
        sql = sql.replace(" all "," ALL ")
        sql = sql.replace(" by "," BY ")
        sql = sql.replace(" in "," IN ")
        sql = sql.replace(" in("," IN(")
        sql = sql.replace(" is "," IS ")
        sql = sql.replace(" like "," LIKE ")
        sql = sql.replace(" as "," AS ")
        sql = sql.replace(" and "," AND ")
        sql = sql.replace(" not "," NOT ")
        sql = sql.replace(" or "," OR ")
        sql = sql.replace(" null "," NULL ")
        sql = sql.replace(" exists "," EXISTS ")
        sql = sql.replace(" union "," UNION ")
        sql = sql.replace(" join "," JOIN ")
        sql = sql.replace(" inner "," INNER ")
        sql = sql.replace(" outer "," OUTER ")
        sql = sql.replace(" cross "," CROSS ")
        sql = sql.replace(" using "," USING ")
        sql = sql.replace(" group "," GROUP ")
        sql = sql.replace(" order "," ORDER ")
        sql = sql.replace(" distinct "," DISTINCT ")
        sql = sql.replace(" having "," HAVING ")
        sql = sql.replace(" between "," BETWEEN ")
        sql = sql.replace(" case "," CASE ")
        sql = sql.replace(" when "," WHEN ")
        sql = sql.replace(" end "," END ")
        sql = sql.replace(" else "," ELSE ")
        sql = sql.replace(" then "," THEN ")
        sql = sql.replace(" cast "," CAST ")
        sql = sql.replace(" regexp "," REGEXP ")
        sql = sql.replace(" similarto "," SIMILARTO ")
        sql = sql.replace(" similarto"," SIMILARTO")

        sql = sql.strip()

        self.update_matching_sql_only_qlineedit.setText(sql)
        self.update_matching_sql_only_qlineedit.update()

        nl = sql.count("(")
        nr = sql.count(")")
        if nl <> nr:
            error_dialog(self.maingui, _('JS+'),_(("SQL must have an equal number of left and right parentheses.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False

        if (not sql.startswith("SELECT id ")) and (not sql.startswith("SELECT book ")) \
            and (not sql.startswith("SELECT ALL id ")) and (not sql.startswith("SELECT ALL book ")) \
            and (not sql.startswith("SELECT DISTINCT id ")) and (not sql.startswith("SELECT DISTINCT book ")):
            error_dialog(self.maingui, _('JS+'),_(("SQL must start with: SELECT and the column to be returned must be either the integer 'id' or the integer 'book'.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return False

        prefs['GUI_TOOLS_IMPORT_CSV_FILE_TO_UPDATE_METADATA_RAW_SQL'] = sql
        prefs

        return True
    #-----------------------------------------------------
    #-----------------------------------------------------
    def execute_raw_sql_query(self,my_db,my_cursor,raw_sql):
        found_list = []
        try:
            my_cursor.execute(raw_sql)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                tmp_rows = []
            for row in tmp_rows:
                for col in row:
                    id = int(col)
                    found_list.append(id)
                    break
                #END FOR
            #END FOR
            del tmp_rows
            return found_list,True,None
        except Exception as e:
            msg = "Raw SQL Query Fatal Error:   " + str(e)
            return found_list,False,msg
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
    def update_books(self):
        if self.mode <> MODE_UPDATE:
            return True,None

        if len(self.books_to_be_updated_list) == 0:
            return False,"No Relevant Books Found"

        if self.target_column_is_custom:
            is_valid,msg = self.update_custom_metadata()
            if not is_valid:
                return False,msg
        else:
            is_valid,msg = self.update_standard_metadata()
            if not is_valid:
                return False,msg

        return True,None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def update_custom_metadata(self):

        if self.merge_existing_tags_checkbox.isChecked():
            self.merge_all_tags = True
        else:
            self.merge_all_tags = False

        mi_field = self.target_column.lower()

        is_datetime = False
        is_series = False
        is_bool = False
        is_text = False
        is_taglike = False

        id,label,name,datatype,mark_for_delete,editable,display,is_multiple,normalized = self.custom_columns_label_dict[mi_field]
        if datatype == "datetime":
            is_datetime = True
            from dateutil.parser import parse as dt_parse
            self.dt_parse = dt_parse
            from calibre.utils.date import format_date
            self.format_date = format_date
        elif datatype == "series":
            is_series = True
        elif datatype == "bool":
            is_bool = True
        elif datatype == "text":
            is_text = True
            if is_multiple == 0:
                is_taglike = False
            else:
                is_taglike = True
        elif datatype == "enumeration":
            is_text = True
            is_taglike = True
        elif datatype == "comments":
            is_text = True
            is_taglike = False

        payload = self.books_to_be_updated_list

        id_map = {}

        for book in self.books_to_be_updated_list:
            book = int(book)
            if not book in self.book_value_dict:
                continue
            value = self.book_value_dict[book]
            if is_text:
                if is_taglike:
                    value = self.convert_taglike(value,book)
                else:
                    pass
            elif is_datetime:
                value = self.convert_datetime(value)
            elif is_bool:
                value = self.convert_boolean(value)
            elif is_series:
                value,index = self.convert_series(value,book)
            else:
                continue

            if value is None:
                continue

            mi = Metadata(_('Unknown'))
            custcol = self.custom_columns_metadata_dict[mi_field]   # should be a '#label'
            custcol['#value#'] = value
            mi.set_user_metadata(mi_field, custcol)
            if is_series:
                if index is not None:
                    custcol['#extra#'] = index
                    mi.set_user_metadata(mi_field, custcol)
            id_map[book] = mi
        #END FOR
        if len(id_map) == 0:
            return False,"No relevant books had anything to update via 'Edit Metadata'.  Nothing done."

        edit_metadata_action = self.maingui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map, callback=None)

        del id_map
        del mi

        return True,None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_taglike(self,value,book):
        mi = self.guidb.new_api.get_metadata(book)
        val = mi.get(self.target_column)
        if isinstance(val,list):
            tag_list = []
            if self.merge_all_tags:
                for v in val:
                    tag_list.append(v)
                #END FOR
            tags = value
            if tags == "":
                tags = "NULL"
            tags = tags + ","
            s_split = tags.split(",")
            for tag in s_split:
                tag = tag.strip()
                if tag > " ":
                    tag_list.append(tag)
            #END FOR
            if "NULL" in tag_list:
                tag_list = [""]
            value = list(set(tag_list))
            del tag_list
        else:
            value = None
        return value
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_datetime(self,value):
        #~ The user has been instructed to always use a datetime format of  YYYY-MM-DD (with no time element) as the 'value', but dt_parse is very flexible.
        dtobject = self.dt_parse(value, default=None, dayfirst=False)  #dt_parse comes from an open-source python date utils package used by Calibre
        date_format = "YYYY-MM-DD"
        value = self.format_date(dtobject,date_format)  #format_date translates the user-friendly format string into a valid python date format string
        return value
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_boolean(self,value):
        if value == "True" or value == "true" or value == "1":
            value = True
        else:
            value = False
        return value
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_series(self,value,book):
        #~ formats allowed:  'name[i]'   'name'
        if not "[" in value:
            value = value + "[ ]"
        value = value.split("[")
        series = value[0].strip()
        index = value[1].strip()
        index = index.replace("]","").strip()
        if not index > " ":
            index = None
        else:
            try:
                index = float(index)
            except:
                index = None
        return series,index
    #-----------------------------------------------------
    #-----------------------------------------------------
    def update_standard_metadata(self):

        if self.merge_existing_tags_checkbox.isChecked():
            self.merge_all_tags = True
        else:
            self.merge_all_tags = False

        payload = self.books_to_be_updated_list

        id_map = {}

        db = self.maingui.library_view.model().db

        for book in self.books_to_be_updated_list:
            book = int(book)
            if not book in self.book_value_dict:
                continue
            #------------------------
            mi = Metadata(_('Unknown'))
            #------------------------
            if self.target_column == COMMENTS:
                new_comments = self.book_value_dict[book]
                if new_comments == "NULL":
                    mi.comments = ""
                else:
                    orig_comments = db.comments(book, index_is_id=True)
                    if not orig_comments:
                        orig_comments = ""
                    else:
                        orig_comments = orig_comments + "\n\n------------------\n\n"
                    mi.comments =  orig_comments + new_comments                               # always merge comments
            #------------------------
            elif self.target_column == IDENTIFIERS:
                new_idents = self.book_value_dict[book]  # MUST be in the format:   'type':'val'  ...including the single quotes...   'daltonst':'01234'
                new_idents = new_idents.strip()
                if not ":" in new_idents:
                    msg = "Identifiers:  Format Error (:)" + "  Value: " + str(new_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if not new_idents.count("'") > 3:
                    msg = "Identifiers:  Format Error (')" + "  Value: " + str(new_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if "{" in new_idents or "}" in new_idents:
                    msg = "Identifiers:  Format Error  ({)" + "  Value: " + str(new_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if new_idents.startswith(",") or new_idents.endswith(","):
                    msg = "Identifiers:  Format Error (,)" + "  Value: " + str(new_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if not len(new_idents) > 6:    #  '':''  actually passes thru ast.literal_eval, so 'x':'y' is the smallest valid value.
                    msg = "Identifiers:  Format Error (?)" + "  Value: " + str(new_idents) + "  --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                new_idents = "{" + new_idents.strip() + "}"
                new_idents = str(new_idents)
                try:
                    new_idents = ast.literal_eval(new_idents)
                except Exception as e:
                    msg = "Identifiers:  Format Error-" + str(e) + "  Value: " + str(new_idents) + " --- identifiers must be in the format  'type':'val'  (including the single quotes)"
                    return False,msg
                if not isinstance(new_idents,dict):
                    continue
                old_idents = db.get_identifiers(book, index_is_id=True)
                if old_idents is None:
                    old_idents = {}
                if not isinstance(old_idents,dict):
                    continue
                idents = {}     # always merge identifiers
                for k,v in old_idents.iteritems():
                    idents[k] = v
                for k,v in new_idents.iteritems():
                    idents[k] = v
                mi.identifiers = idents
                del new_idents
                del old_idents
                del idents
            #------------------------
            elif self.target_column == PUBLISHER:
                value = self.book_value_dict[book].strip()
                if value > " ":
                    if value == "NULL":
                        value = ""
                    mi.publisher = value
                else:
                    continue
            #------------------------
            elif self.target_column == TAGS:
                orig_tags = db.tags(book, index_is_id=True)
                if orig_tags is None:
                    orig_tags = []
                else:
                    if not isinstance(orig_tags,list):
                        orig_tags = orig_tags + ","
                        orig_tags = orig_tags.split(",")
                tag_list = []
                if self.merge_all_tags:      # user option to merge tags or not
                    for v in orig_tags:
                        v = v.strip()
                        if v > " ":
                            tag_list.append(v)
                    #END FOR
                del orig_tags
                tags = self.book_value_dict[book].strip()
                if tags == "":
                    tags = "NULL"
                tags = tags + ","
                s_split = tags.split(",")
                for tag in s_split:
                    tag = tag.strip()
                    if tag > " ":
                        tag_list.append(tag)
                #END FOR
                if "NULL" in tag_list:
                    value = list(" ")
                    mi.tags = value
                else:
                    value = list(set(tag_list))
                    mi.tags = value
                del tag_list
            #------------------------
            elif self.target_column == PUBDATE:
                value = self.book_value_dict[book]
                value = self.convert_datetime(value)
                mi.pubdate = value
            #------------------------
            elif self.target_column == DATEADDED:
                value = self.book_value_dict[book]
                value = self.convert_datetime(value)
                mi.timestamp = value
            #------------------------
            elif self.target_column == SERIES:
                value = self.book_value_dict[book]
                value,index = self.convert_series(value,book)
                mi.series = value
                mi.series_index = index
            #------------------------
            elif self.target_column == TITLE:
                mi.title = self.book_value_dict[book]
            #------------------------
            else:
                if DEBUG: print("Standard Target Column is Invalid: ", self.target_column)
                break
            #------------------------
            id_map[book] = mi
        #END FOR
        if len(id_map) == 0:
            return False,"No relevant books had anything to update via 'Edit Metadata'.  Nothing done."

        edit_metadata_action = self.maingui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map, callback=None,merge_tags=False)  #merge_tags is always False here; handled book-by-book

        del id_map
        del mi

        return True,None
    #-----------------------------------------------------
    #-----------------------------------------------------
    def unmark_all_books(self):
        #~ if DEBUG: print("unmark_all_books")
        found_dict = {}
        s_true = 'true'
        marked_ids = dict.fromkeys(found_dict, s_true)
        self.maingui.current_db.set_marked_ids(marked_ids)
        self.maingui.search.clear()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def mark_updated_books(self):
        #~ if DEBUG: print("mark_updated_books")
        found_dict = {}
        s_true = 'true'
        for book in self.books_to_be_updated_list:
            found_dict[book] = s_true
        #END FOR

        marked_ids = dict.fromkeys(found_dict, s_true)
        self.maingui.current_db.set_marked_ids(marked_ids)
        self.maingui.search.clear()
        self.maingui.search.set_search_string('marked:true')

        del found_dict
    #-----------------------------------------------------
    #-----------------------------------------------------
    def show_results_message(self):
        if self.mode == MODE_UPDATE:
            msg = "Metadata has been actually updated for the indicated book(s)."
        else:
            msg = "Preview Only; nothing has been updated for the indicated book(s)."
        self.maingui.status_bar.show_message(_(msg), 10000)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_and_exit(self):
        sql = self.update_matching_sql_only_qlineedit.text()
        prefs['GUI_TOOLS_IMPORT_CSV_FILE_TO_UPDATE_METADATA_RAW_SQL'] = sql.strip()
        prefs
        self.dialog_closing(None)
        self.close()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def apsw_connect_to_library(self):

        path = self.guidb.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", str(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        if self.is_case_sensitive:
            mysql = "PRAGMA case_sensitive_like=ON;"
            my_cursor.execute(mysql)
            self.collate_keywords = "COLLATE BINARY"    #~ SELECT * FROM tags WHERE name = 'Crime fiction' COLLATE BINARY
        else:
            mysql = "PRAGMA case_sensitive_like=OFF;"
            my_cursor.execute(mysql)
            self.collate_keywords = "COLLATE NOCASE"  #~ SELECT * FROM tags WHERE name = 'Crime fiction' COLLATE NOCASE

        return my_db,my_cursor,is_valid
    #-----------------------------------------------------
    #-----------------------------------------------------
    def apsw_create_user_functions(self,my_db,my_cursor,func):
        try:
            if func == "REGEXP":
                my_db.createscalarfunction("regexp", self.apsw_user_function_regexp)
                #~ if DEBUG: print("Create_SQLite_User_Function REGEXP was successful...")
                return True,None
            else:
                msg = "Create_SQLite_User_Function " + str(func) + " failed...cannot proceed..."
                return False,msg
        except Exception as e:
            if DEBUG: print("Create_SQLite_User_Function REGEXP failed...cannot proceed...")
            if DEBUG: print(str(e))
            msg = str(e)
            return False,msg
    #-----------------------------------------------------
    #-----------------------------------------------------
    def apsw_user_function_regexp(self,regexpr,avalue):
        #http://www.sqlite.org/lang_expr.html:  The "X REGEXP Y" operator will be implemented as a call to "regexp(Y,X)"
        #---------------------------------------------------------------------------------------------------------------------------------------
        #mysql = 'SELECT id FROM custom_column_8 WHERE value REGEXP '^.+$'
        #---------------------------------------------------------------------------------------------------------------------------------------
        if regexpr:
            if avalue:
                try:
                    s_string = unicode(avalue)
                    re_string = unicode(regexpr)
                    re.escape("\\")
                    if self.is_case_sensitive:
                        p = re.compile(re_string, re.DOTALL|re.MULTILINE)
                    else:
                        p = re.compile(re_string, re.IGNORECASE|re.DOTALL|re.MULTILINE)
                    match = p.search(s_string)
                    if match:
                        return True
                    else:
                        return False
                except Exception as e:
                    if DEBUG: print(str(e))
                    return False
    #-----------------------------------------------------
    #-----------------------------------------------------
    def import_csv_file(self,csv_path):
        imported_csv_list = []

        if csv_path == unicode(""):
            if DEBUG: print("csv_path == unicode(""); nothing to do.")
            return imported_csv_list

        CR = str(chr(13))
        LF = str(chr(10))
        FF = str(chr(12))

        UCR = u"\u000D"
        ULF =  u"\u000A"
        UFF = u"\u000C"
        NEL = u"\u0085"
        ADC = u"\u21B5"
        SCR = u"\u240D"

        shortest = 9999999999999999
        longest = 0

        msgr  = ""
        msg0 = ""
        msgc = "<br><br>Caution: the Calibre 'Create Catalog as CSV File' function often creates a malformed CSV File if Comments are included in the selection.  Comments are very problematic because of their line-breaks etc. which corrupt the Calibre output CSV File."
        msgr = "CSV File Seems Corrupt.  Open it in a simple text editor (not a spreadsheet application) to examine it for formatting errors.  Example: Comments line-breaks and other common formatting. \
        <br><br>Run Calibre in Debugging Mode for more information about your problem with this particular CSV File."

        is_valid = True

        try:
            with open (csv_path,'rb') as csvfile:
                while True:
                    line = csvfile.readline()
                    if not line:
                        break
                    line = html2text(line)
                    line = line.replace("\r'\n"," ").strip()
                    line = line.replace('\r\n',' ').strip()
                    line = line.replace('\r',' ').strip()
                    line = line.replace('\n',' ').strip()
                    line = line.replace('\f',' ').strip()
                    line = line.replace('\t','    ').strip()
                    line = line.replace(CR," ").strip()
                    line = line.replace(LF," ").strip()
                    line = line.replace(FF," ").strip()
                    if not isinstance(line,unicode):
                        line = unicode(line)
                    line = line.replace(UCR," ").strip()
                    line = line.replace(ULF," ").strip()
                    line = line.replace(UFF," ").strip()
                    line = line.replace(NEL," ").strip()
                    line = line.replace(ADC," ").strip()
                    line = line.replace(SCR," ").strip()
                    imported_csv_list.append(line)
                    if len(line) < shortest:
                        l = len(line)
                        if l == 0:
                            msg0 = "<br><br>CSV File was created with a zero-length row.  This CSV File may have been originally created malformed."
                            if DEBUG: print("==================zero-length row follows next===============================")
                        else:
                            shortest = l
                    if len(line) > longest:
                        longest = len(line)
                    if DEBUG: print(str(line))
                    if DEBUG: print("================New CSV File Row====================")
                #END WHILE
            #END WITH
            csvfile.close()
            del csvfile
            del csv_path
        except Exception as e:
            if DEBUG: print("Import CSV File Error: " + str(e))
            msge = "<br><br>" + str(e)
            is_valid = False
            imported_csv_list = []
            return imported_csv_list,is_valid,msge

        if DEBUG: print("finished importing lines: ", str(len(imported_csv_list)))
        if DEBUG: print("shortest line: ", str(shortest), "  longest line: ", str(longest))

        if (not shortest == 0) and (not longest == 0):
            ratio = shortest/longest
            r = str("<br><br>(Shortest row length / Longest row length) = " + "{:.9f}".format(ratio))
            if DEBUG: print("ratio = shortest row length/longest row length: ",  "{:.9f}".format(ratio))
            if ratio < .001 :
                msgr = msgr + msg0 + msgc + r
                is_valid = False
        elif shortest == 0:
            msgr = msgr + msg0
            is_valid = False
        else:
            msgr = ""

        return imported_csv_list,is_valid,msgr
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_custom_column_technical_details(self,my_db,my_cursor):
        self.custom_columns_label_dict = {}
        mysql = "SELECT id,label,name,datatype,mark_for_delete,editable,display,is_multiple,normalized FROM custom_columns"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        for row in tmp_rows:
            id,label,name,datatype,mark_for_delete,editable,display,is_multiple,normalized = row
            label = "#" + label
            self.custom_columns_label_dict[label] = id,label,name,datatype,mark_for_delete,editable,display,is_multiple,normalized
        #END FOR
        del tmp_rows
    #-----------------------------------------------------
    #-----------------------------------------------------
    def create_temp_views(self,my_db,my_cursor,table1,table2):
        mysql = \
        """
        CREATE TEMP VIEW IF NOT EXISTS __js_tags_by_book AS SELECT [BOOK,TAG],
        (SELECT [NAME] FROM [TAGS] WHERE id = [BOOKS_TAGS_LINK].[TAG]) AS tagname
        FROM [BOOKS_TAGS_LINK]
        ORDER BY [BOOK,TAG]
        """
        mysql = mysql.replace("[BOOK,TAG]","book,value")
        mysql = mysql.replace("[NAME]","value")
        mysql = mysql.replace("[TAGS]",table1)
        mysql = mysql.replace("[BOOKS_TAGS_LINK]",table2)
        mysql = mysql.replace("[TAG]","value")
        my_cursor.execute("begin")
        my_cursor.execute(mysql)
        my_cursor.execute("commit")

        my_cursor.execute("begin")
        mysql = \
        """
        CREATE TEMP VIEW IF NOT EXISTS __js_tags_by_book_concatenate
        AS  SELECT book, group_concat(myconcat,',') AS tagsconcat
        FROM (SELECT book,
        CASE
        WHEN tagname IS NULL THEN 'unknown'
        WHEN tagname = ',' THEN 'unknown'
        WHEN tagname = ' ' THEN 'unknown'
        WHEN tagname = ''  THEN 'unknown'
        ELSE tagname
        END AS myconcat
        FROM __js_tags_by_book ORDER BY tagname  )
        GROUP BY book;
        """
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
#END OF import_csv_file_to_update_metadata_dialog.py