# -*- coding: utf-8 -*-
from __future__ import (unicode_literals, division, absolute_import, print_function)
__license__   = 'GPL v3'
__copyright__ = '2017 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.5"


from PyQt5.Qt import (Qt, QDialog, QLabel,  QFont, QWidget,QApplication,
                                       QIcon, QGroupBox, QMargins,QScrollArea,
                                       QDialogButtonBox,QComboBox,
                                       QTableWidget, QTableWidgetItem,QAbstractItemView,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout)

import os, sys
import apsw
from datetime import datetime
from functools import partial


from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG, iswindows
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import gprefs,error_dialog, info_dialog, question_dialog
from calibre.gui2.ui import get_gui
from calibre.utils.config import JSONConfig

from calibre_plugins.audit_log.config import prefs

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(90, 90)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):

        #~ if DEBUG: self.geom = None

        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class AuditUndoDialog(SizePersistedDialog):

    def __init__(self,parent,guidb,selected_books_list,audit_log_detail_list,custom_column_name_dict,custom_column_list):
        unique_pref_name = 'Audit_Log:audit_undo_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.guidb = guidb

        mytitle = 'Audit Undo Criteria: [NNNN] Rows'
        mytitle = mytitle.replace("[NNNN]",str(len(audit_log_detail_list)))
        self.setWindowTitle(_(mytitle))

        self.selected_books_list = selected_books_list
        self.audit_log_detail_list = audit_log_detail_list
        self.custom_column_name_dict = custom_column_name_dict
        self.custom_column_list = custom_column_list

        self.maingui = get_gui()


        self.timestamp_list = []
        self.book_list = []
        self.table_name_list = []
        self.table_column_list = []
        self.table_action_list = []
        self.value_old_list = []
        self.value_new_list = []
        self.other_list = []

        self.selected_books_set = set(self.selected_books_list)

        match_found = False

        for row in self.audit_log_detail_list:
            id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other = row
            book = str(book)
            if not book in self.selected_books_set:  # should never happen since ui.py filters first...
                continue
            else:
                match_found = True

            id = str(id)
            time_stamp = str(time_stamp)

            self.timestamp_list.append(time_stamp)
            self.book_list.append(book)
            self.table_name_list.append(table_name)
            self.table_column_list.append(table_column)
            self.table_action_list.append(table_action)
            self.value_old_list.append(value_old)
            self.value_new_list.append(value_new)
            self.other_list.append(other)

        #END FOR

        if not match_found:   # i.e., the selected books are not found in the current audit log...
            return

        self.timestamp_list = list(set(self.timestamp_list))
        self.book_list = list(set(self.book_list))
        self.table_name_list = list(set(self.table_name_list))
        self.table_column_list = list(set(self.table_column_list))
        self.table_action_list = list(set(self.table_action_list))
        self.value_old_list = list(set(self.value_old_list))
        self.value_new_list = list(set(self.value_new_list))
        self.other_list = list(set(self.other_list))

        self.timestamp_list.sort()
        self.book_list.sort()
        self.table_name_list.sort()
        self.table_column_list.sort()
        self.table_action_list.sort()
        self.value_old_list.sort()
        self.value_new_list.sort()
        self.other_list.sort

        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        tip = "<p style='white-space:wrap'>Undo/Reverse log line items for the selected book.  Only Deletes and Updates, plus Insert or Replaces for comments and some custom columns, may be Undeleted/Reversed. "

        self.layout_frame = QVBoxLayout()
        self.setLayout(self.layout_frame)
        self.layout_frame.setAlignment(Qt.AlignCenter)
        self.setToolTip(tip)

        font = QFont()

        font.setBold(False)
        font.setPointSize(10)

        #-----------------------------------------------------
        tip = "<p style='white-space:wrap'>Undo/Reversal actions may be performed for only a single book at one time.  The Audit Log Detail rows displayed may be filtered by their TimeStamp."

        self.layout_detail = QHBoxLayout()
        self.layout_frame.addLayout(self.layout_detail)
        self.layout_detail.setAlignment(Qt.AlignLeft)
        self.setToolTip(tip)

        timestamp_tip = "<p style='white-space:wrap'>The Audit Log Detail rows displayed may be filtered by their TimeStamp.."

        self.available_timestamps_combobox_label = QLabel("TimeStamp Filter:")
        self.available_timestamps_combobox_label.setFont(font)
        self.available_timestamps_combobox_label.setMaximumWidth(100)
        self.available_timestamps_combobox_label.setToolTip(timestamp_tip)
        self.layout_detail.addWidget(self.available_timestamps_combobox_label)

        self.available_timestamps_combobox = QComboBox()
        self.available_timestamps_combobox.setEditable(False)
        self.available_timestamps_combobox.setFont(font)
        self.available_timestamps_combobox.setMaximumWidth(150)
        self.available_timestamps_combobox.setToolTip(timestamp_tip)
        self.layout_detail.addWidget(self.available_timestamps_combobox)

        self.default_timestamp = "TimeStamp: any"
        self.available_timestamps_combobox.addItem(self.default_timestamp)

        for t in self.timestamp_list:
            self.available_timestamps_combobox.addItem(t)
        #END FOR
        self.available_timestamps_combobox.setCurrentIndex(0)
        self.current_timestamp = self.default_timestamp
        self.available_timestamps_combobox.currentIndexChanged.connect(self.event_filter_timestamp_changed)

        book_tooltip = "<p style='white-space:wrap'>Undo/Reversal actions may be performed for only a single book at one time.  The Audit Log Detail rows displayed are always filtered by your chosen book.  The choices displayed are obtained from the Calibre books that you had selected prior to clicking the menu item for Undo/Reverse.  The choices are further refined to include only books that actually have Audit Log Details to display."

        self.selected_books_combobox_label = QLabel("          Book ID & Title/Author Filter:")
        self.selected_books_combobox_label.setFont(font)
        self.selected_books_combobox_label.setMaximumWidth(300)
        self.selected_books_combobox_label.setToolTip(book_tooltip)
        self.layout_detail.addWidget(self.selected_books_combobox_label)

        self.selected_books_combobox = QComboBox()
        self.selected_books_combobox.setEditable(False)
        self.selected_books_combobox.setFont(font)
        self.selected_books_combobox.setMinimumWidth(400)
        self.selected_books_combobox.setMaximumWidth(800)
        self.selected_books_combobox.setToolTip(book_tooltip)
        self.layout_detail.addWidget(self.selected_books_combobox)

        for b in self.book_list:
            title,authors = self.get_title_authors(b)
            s = b + " [" + title + " - " + authors + "]"
            if len(s) > 100:
                s = s[0:100].strip() + "....."
            self.selected_books_combobox.addItem(s)
        #END FOR
        self.selected_books_combobox.setCurrentIndex(-1)
        self.selected_books_combobox.currentIndexChanged.connect(self.event_filter_book_changed)

        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        self.n_log_rows = len(audit_log_detail_list)
        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("Time +0000 UTC")
        column_label_list.append("Book")
        column_label_list.append("Table Name")
        column_label_list.append("Table Column")
        column_label_list.append("Table Action")
        column_label_list.append("Value Old")
        column_label_list.append("Value New")
        column_label_list.append("Other")
        column_label_list.append("AL ID")

        self.n_total_cols = 9

        self.logtable = QTableWidget(self.n_log_rows,self.n_total_cols)

        self.logtable.setSortingEnabled(False)

        self.logtable.setHorizontalHeaderLabels(column_label_list)

        self.resize_all_columns()

        self.logtable.clearContents()
        #--------------------------------------------------

        r = 0
        for row in audit_log_detail_list:
            try:
                #---------------------------
                #---------------------------
                id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other = row

                id = str(id)
                book = str(book)
                time_stamp = str(time_stamp)

                table_column = table_column.strip()  #correct V1.0.2 triggers for 'lang_code '.

                if table_name == "comments":  #Calibre does INSERT OR REPLACE, not DELETE then INSERT, and not UPDATE...
                    value_old = value_old[0:25].strip()
                    value_new = value_new[0:25].strip()
                    if len(value_old) > 0:
                        value_old = value_old + "....."
                    if len(value_new) > 0:
                        value_new = value_new + "....."

                if not value_old:
                    value_old = ""

                if not value_new:
                    value_new = ""

                if not other:
                    other = ""

                if "custom_column_" in table_name:
                    other,label = self.get_custom_column_name(other,table_name)
                    table_column = "#" + str(label)
                #---------------------------
                #---------------------------
                id_ = QTableWidgetItem(id)
                time_stamp_ = QTableWidgetItem(time_stamp)
                book_ = QTableWidgetItem(book)
                table_name_ = QTableWidgetItem(table_name)
                table_column_ = QTableWidgetItem(table_column)
                table_action_ = QTableWidgetItem(table_action)
                value_old_ = QTableWidgetItem(value_old)
                value_new_ = QTableWidgetItem(value_new)
                other_ = QTableWidgetItem(other)
                #---------------------------
                #---------------------------
                self.logtable.setItem(r,0,time_stamp_)
                self.logtable.setItem(r,1,book_)
                self.logtable.setItem(r,2,table_name_)
                self.logtable.setItem(r,3,table_column_)
                self.logtable.setItem(r,4,table_action_)
                self.logtable.setItem(r,5,value_old_)
                self.logtable.setItem(r,6,value_new_)
                self.logtable.setItem(r,7,other_)
                self.logtable.setItem(r,8,id_)

                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("class AuditLogDialog(SizePersistedDialog):", str(e))
                return
        #END FOR

        self.n_total_rows = r

        self.layout_frame.addWidget(self.logtable)

        self.logtable.setSortingEnabled(True)
        self.logtable.sortByColumn(0, Qt.DescendingOrder)

        self.resize_all_columns()

        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        self.layout_frame.addWidget(self.bottom_buttonbox)
        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.setToolTip("<p style='white-space:wrap'>Change the current column widths to fixed values, regardless of their longest values.")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,0)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.setToolTip("<p style='white-space:wrap'>Change the current column widths based on their longest values.")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,0)

        self.push_button_save_column_widths = QPushButton(" ", self)
        self.push_button_save_column_widths.setText("Save Optimization")
        self.push_button_save_column_widths.setToolTip("<p style='white-space:wrap'>Save the current column widths as your new default.")
        self.push_button_save_column_widths.clicked.connect(self.save_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_save_column_widths,0)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Export to Clipboard")
        self.push_button_copy_to_clipboard.setToolTip("<p style='white-space:wrap'>The Audit Log in its entirety will be copied to the Clipboard in a tab-delimited format.  Paste Special into a Spreadsheet, or Paste into a Text Document.")
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_log_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,0)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()


        tip = "<p style='white-space:wrap'>Only the following standard metadata may be Undone/Reversed:  title; pubdate; series; publisher; comments; identifiers; languages. \
                                                                  <br><br>All custom column metadata is supported."

        #-----------------------------------------------------
        self.push_button_undo_specified_changes = QPushButton(" ", self)
        self.push_button_undo_specified_changes.setText("Undo/Reverse Selected Row")
        self.push_button_undo_specified_changes.setToolTip(tip)
        self.push_button_undo_specified_changes.clicked.connect(self.undo_specified_changes)
        self.bottom_buttonbox.addButton(self.push_button_undo_specified_changes,0)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.old_book = None
        self.current_book = None
        self.new_book = None
        self.selected_books_combobox.setCurrentIndex(0)   # change from -1 to trigger an event...

        self.logtable.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.logtable.setSelectionMode(QAbstractItemView.SingleSelection)

        self.supported_tables_list = []
        self.supported_tables_list.append("books")
        self.supported_tables_list.append("identifiers")
        self.supported_tables_list.append("books_languages_link")
        self.supported_tables_list.append("books_publishers_link")
        self.supported_tables_list.append("books_series_link")
        self.supported_tables_list.append("comments")
        self.supported_tables_list.append("custom_column_")

        self.supported_columns_list = []
        self.supported_columns_list.append("type")  # identifiers
        self.supported_columns_list.append("title")
        self.supported_columns_list.append("pubdate")
        self.supported_columns_list.append("series")
        self.supported_columns_list.append("publisher")
        self.supported_columns_list.append("lang_code")
        self.supported_columns_list.append("text")   # comments
        self.supported_columns_list.append("value")  # custom columns

        self.supported_actions = []
        self.supported_actions.append("delete")
        self.supported_actions.append("update")

        self.books_to_update_list = []

    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_title_authors(self,book):
        title = "unknown"
        author_sort = "unknown"

        book = str(book)
        try:
            book = int(book)
        except:
            return title, author_sort

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return title,author_sort

        mysql = "SELECT title,author_sort FROM books WHERE id = ?"
        my_cursor.execute(mysql,([book]))
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        else:
            for row in tmp_rows:
                title,author_sort = row
            #END FOR

        my_db.close()


        if not author_sort:
            author_sort = "unknown"
        if not title:
            title = "unknown"

        return title,author_sort
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_custom_column_name(self,other,table_name):

        label = ""

        table_name = table_name.replace("books","")
        table_name = table_name.replace("custom_column","")
        table_name = table_name.replace("link","")
        table_name = table_name.replace("_","")
        id = str(table_name.strip())
        if id in self.custom_column_name_dict:
            id,label,name = self.custom_column_name_dict[id]           #     id = id,label,name; also in list...
            other = other + " " + name
        else:
            pass

        #~ if DEBUG: print(other)

        return other,label
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
    def resize_all_columns(self):

        n = prefs['COLUMN__0_WIDTH']
        n = int(n)
        if n == 1:       #config.py default value
            self.logtable.resizeColumnsToContents()
        else:               #previously saved by the user
            column__0_width = int(prefs['COLUMN__0_WIDTH'])
            column__1_width = int(prefs['COLUMN__1_WIDTH'])
            column__2_width = int(prefs['COLUMN__2_WIDTH'])
            column__3_width = int(prefs['COLUMN__3_WIDTH'])
            column__4_width = int(prefs['COLUMN__4_WIDTH'])
            column__5_width = int(prefs['COLUMN__5_WIDTH'])
            column__6_width = int(prefs['COLUMN__6_WIDTH'])
            column__7_width = int(prefs['COLUMN__7_WIDTH'])
            column__8_width = int(prefs['COLUMN__8_WIDTH'])

            if column__0_width < 125: column__0_width = 125
            if column__1_width < 50: column__1_width = 50
            if column__2_width < 200: column__2_width = 200
            if column__3_width < 100: column__3_width = 100
            if column__4_width < 125: column__4_width = 125
            if column__5_width < 125: column__5_width = 125
            if column__6_width < 125: column__6_width = 125
            if column__7_width < 100: column__7_width = 100
            if column__8_width < 50: column__8_width = 50

            self.logtable.setColumnWidth(0, column__0_width)
            self.logtable.setColumnWidth(1, column__1_width)
            self.logtable.setColumnWidth(2, column__2_width)
            self.logtable.setColumnWidth(3, column__3_width)
            self.logtable.setColumnWidth(4, column__4_width)
            self.logtable.setColumnWidth(5, column__5_width)
            self.logtable.setColumnWidth(6, column__6_width)
            self.logtable.setColumnWidth(7, column__7_width)
            self.logtable.setColumnWidth(8, column__8_width)
    #-----------------------------------------------------
    def save_column_widths(self):

        column__0_width = self.logtable.columnWidth(0)
        column__1_width = self.logtable.columnWidth(1)
        column__2_width = self.logtable.columnWidth(2)
        column__3_width = self.logtable.columnWidth(3)
        column__4_width = self.logtable.columnWidth(4)
        column__5_width = self.logtable.columnWidth(5)
        column__6_width = self.logtable.columnWidth(6)
        column__7_width = self.logtable.columnWidth(7)
        column__8_width = self.logtable.columnWidth(8)

        prefs['COLUMN__0_WIDTH'] = unicode(column__0_width)
        prefs['COLUMN__1_WIDTH'] = unicode(column__1_width)
        prefs['COLUMN__2_WIDTH'] = unicode(column__2_width)
        prefs['COLUMN__3_WIDTH'] = unicode(column__3_width)
        prefs['COLUMN__4_WIDTH'] = unicode(column__4_width)
        prefs['COLUMN__5_WIDTH'] = unicode(column__5_width)
        prefs['COLUMN__6_WIDTH'] = unicode(column__6_width)
        prefs['COLUMN__7_WIDTH'] = unicode(column__7_width)
        prefs['COLUMN__8_WIDTH'] = unicode(column__8_width)

        prefs

        self.save_audit_log_dialog_geometry()
    #-----------------------------------------------------
    def optimize_column_widths(self):

        self.logtable.resizeColumnsToContents()

        self.save_audit_log_dialog_geometry()
    #-----------------------------------------------------
    def deoptimize_column_widths(self):

        self.logtable.setColumnWidth(0, 125)
        self.logtable.setColumnWidth(1,  50)
        self.logtable.setColumnWidth(2, 200)
        self.logtable.setColumnWidth(3, 100)
        self.logtable.setColumnWidth(4, 125)
        self.logtable.setColumnWidth(5, 125)
        self.logtable.setColumnWidth(6, 125)
        self.logtable.setColumnWidth(7, 100)
        self.logtable.setColumnWidth(8, 50)
        self.save_audit_log_dialog_geometry()
    #-----------------------------------------------------
    def copy_log_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.logtable.selectAll()

        s = ''
        s = s + "\t".join([str(self.logtable.horizontalHeaderItem(i).text()) for i in xrange(0, self.n_total_cols)])
        s = s +  '\n'
        for r in xrange(0, self.n_total_rows):
            for c in xrange(0, self.n_total_cols):
                try:
                    s = s + str(self.logtable.item(r,c).text()) + "\t"
                except AttributeError:
                    s = s + "\t"
            s = s[:-1] + "\n"        #eliminate last '\t'
        self.clip.setText(s)

        self.save_audit_log_dialog_geometry()
    #-----------------------------------------------------
    def save_audit_log_dialog_geometry(self):
        self.dialog_closing(None)
    #-----------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_connect_to_current_library_path(self):

        my_db = self.guidb
        path = my_db.library_path
        path = os.path.join(path,'metadata.db')
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(str(e))
            msg = str("Audit Log cannot connect to the path that you selected:" + path + " - " + str(e) )
            self.gui.status_bar.showMessage(msg)
            return None,None,False
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,True
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#~ Events
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_book_changed(self,event):
        self.new_book = self.selected_books_combobox.currentText()
        l = self.new_book.split("[")
        self.new_book = l[0].strip()
        if self.new_book <> self.current_book:
            self.current_book = self.new_book
            self.filter_audit_log_by_book()
        else:
            pass
#---------------------------------------------------------------------------------------------------------------------------------------
    def event_filter_timestamp_changed(self,event):
        # loop through the audit log detail qt logtable, and delete any rows where timestamp <> selected timestamp
        self.new_timestamp = self.available_timestamps_combobox.currentText()
        if self.current_timestamp == self.new_timestamp:
            return
        self.filter_audit_log_by_timestamp()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def filter_audit_log_by_book(self):
        #hide all logtable rows with book <> self.current_book
        if "[" in self.current_book:
            l = self.current_book.split("[")
            self.current_book = l[0].strip()
        for r in range(0,self.n_log_rows):
            item = self.logtable.item(r,1)
            book = item.text()  # column 1 is book
            if str(book) <> str(self.current_book):
                self.logtable.setRowHidden(r,True)
            else:
                self.logtable.setRowHidden(r,False)
        #END FOR

        if self.current_timestamp == self.default_timestamp:
            pass
        else:
            self.filter_audit_log_by_timestamp(source="books")
    #-----------------------------------------------------
    def filter_audit_log_by_timestamp(self,source="timestamp"):
        # row have already been filtered by book, so never unhide a hidden row, just hide an unhidden row...
        self.current_timestamp = self.new_timestamp
        for r in range(0,self.n_log_rows):
            if source == "timestamp":  # user just filtered by timestamp without changing the current book...so don't unhide a hidden bad book row...
                item = self.logtable.item(r,1)
                book = item.text()  # column 1 is book
                if str(book) <> str(self.current_book):
                    continue
                else:  # current book for current row, which may be hidden due to previous timestamp filtering...
                    item = self.logtable.item(r,0)
                    timestamp = item.text()  # column 0 is timestamp
                    if timestamp <> self.current_timestamp:
                        if self.current_timestamp <> self.default_timestamp:
                            self.logtable.setRowHidden(r,True)
                        else:
                            self.logtable.setRowHidden(r,False)
                    else:
                        self.logtable.setRowHidden(r,False)
            elif source == "books":  # so only the current book is shown at the moment, so only hide bad timestamps...
                is_hidden = self.logtable.isRowHidden(r)
                if is_hidden:
                    continue
                item = self.logtable.item(r,0)
                timestamp = item.text()  # column 0 is timestamp
                if timestamp <> self.current_timestamp:
                    if self.current_timestamp <> self.default_timestamp:
                        self.logtable.setRowHidden(r,True)
            else:
                return
        #END FOR
    #-----------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
    def undo_specified_changes(self):

        selected_indexes_list = self.logtable.selectedIndexes()
        if len(selected_indexes_list) == 9:   # 9 columns
            timestamp = self.logtable.model().data(selected_indexes_list[0])
            book = self.logtable.model().data(selected_indexes_list[1])
            table_name = self.logtable.model().data(selected_indexes_list[2])
            table_column = self.logtable.model().data(selected_indexes_list[3])
            table_action = self.logtable.model().data(selected_indexes_list[4])
            value_old = self.logtable.model().data(selected_indexes_list[5])
            value_new = self.logtable.model().data(selected_indexes_list[6])
            other = self.logtable.model().data(selected_indexes_list[7])
            audit_log_table_id = self.logtable.model().data(selected_indexes_list[8])

            del selected_indexes_list

            is_valid_request = True

            if table_action in self.supported_actions:
                if table_name in self.supported_tables_list:
                    if table_column in self.supported_columns_list:
                        pass
                    else:
                        if "#" in table_column:
                            pass
                        else:
                            reason = "Unsupported table_column:    " + table_column
                            is_valid_request = False
                else:
                    if "custom_column"in table_name:
                        pass
                    else:
                        reason = "Unsupported table_name:    " + table_name
                        is_valid_request = False
            else:
                reason = "Unsupported table_action:    " + table_action
                is_valid_request = False

            if not is_valid_request:
                msg = 'Sorry, AL cannot Reverse/Undo the selected line. <br><br><b>' + reason
                return error_dialog(self, _('AL: Undo '),_(msg), show=True)

            if table_action == "delete":
                auditlogaction = "Add the deleted '[VALUE_OLD]' back for '[TABLE_COLUMN]'"
                auditlogaction = auditlogaction.replace("[VALUE_OLD]",value_old)
                auditlogaction = auditlogaction.replace("[TABLE_COLUMN]",table_column)
            elif table_action == "update":
                auditlogaction = "Change the value back to '[VALUE_OLD]' for '[TABLE_COLUMN]'"
                auditlogaction = auditlogaction.replace("[VALUE_OLD]",value_old)
                auditlogaction = auditlogaction.replace("[TABLE_COLUMN]",table_column)
            else:
                return error_dialog(self, _('AL: Undo '),_('Sorry, AL cannot currently Reverse/Undo the selected line; Canceled.'), show=True)

            msg = "Please confirm that you wish Audit Log to REVERSE/UNDO the following single log row: \
                        <br>Timestamp: [TIMESTAMP]\
                        <br>Book: [BOOK]\
                        <br>Table Name: [TABLE_NAME]\
                        <br>Table Column: [TABLE_COLUMN]\
                        <br>Table Action: [TABLE_ACTION]\
                        <br>Value Old: [VALUE_OLD]\
                        <br>Value New: [VALUE_NEW]\
                        <br>Other: [OTHER]\
                        <br>\
                        <br>Audit Log's action will be: <br><b>[AUDITLOGACTION]"

            msg = msg.replace("[TIMESTAMP]",timestamp)
            msg = msg.replace("[BOOK]",book)
            msg = msg.replace("[TABLE_NAME]",table_name)
            msg = msg.replace("[TABLE_COLUMN]",table_column)
            msg = msg.replace("[TABLE_ACTION]",table_action)
            msg = msg.replace("[VALUE_OLD]",value_old)
            msg = msg.replace("[VALUE_NEW]",value_new)
            msg = msg.replace("[OTHER]",other)
            msg = msg.replace("[AUDITLOGACTION]",auditlogaction)

            was_success = False

            if question_dialog(self, "UNDO", msg):

                self.logtable.clearSelection()

                if str(table_name) == str("identifiers"):
                    was_success = self.update_identifiers(book,table_action,value_old,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Identifiers metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif str(table_name) == str("books_publishers_link"):
                    was_success = self.update_publishers(book,table_action,value_old,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Publisher metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif str(table_name) == str("books_series_link"):
                    was_success = self.update_series(book,table_action,value_old,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Publisher metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif str(table_name) == str("books_languages_link"):
                    was_success = self.update_languages(book,table_action,value_old,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Language metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif str(table_name) == str("books") and str(table_column) == str("pubdate"):
                    was_success = self.update_pubdate(book,table_action,value_old,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Published Date metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif str(table_name) == str("books") and str(table_column) == str("title"):
                    was_success = self.update_title(book,table_action,value_old,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Title metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif str(table_name) == str("comments") and str(table_column) == str("text"):
                    was_success = self.update_comments(book,table_action,value_old,other,audit_log_table_id)  # audit_log_table_id to get the original _audit_log table value_old
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Comments metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                elif "custom_column" in table_name:
                    was_success = self.update_custom_column(book,table_name,table_column,table_action,value_old,value_new,other)
                    if was_success:
                        return info_dialog(self, _("Audit Log Reverse/Undo"), _("Success: Custom Column metadata value was changed for specified book."), show=True)
                    else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a technical database update error...Nothing done.'), show=True)

                else:
                        return error_dialog(self, _('AL: Undo '),_('Sorry, There was a program logic error [1] ...Nothing done.'), show=True)
            else:
                pass
        else:
            if DEBUG: print("selected_indexes_list does not have exactly 9 rows...")
            del selected_indexes_list

    #-----------------------------------------------------
    def update_identifiers(self,book,table_action,value_old,other):
        book = int(book)
        new_value = value_old
        mi_field = "identifiers"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_languages(self,book,table_action,value_old,other):
        book = int(book)
        new_value = value_old
        mi_field = "languages"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_publishers(self,book,table_action,value_old,other):
        book = int(book)
        new_value = value_old
        mi_field = "publisher"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_pubdate(self,book,table_action,value_old,other):
        book = int(book)
        new_value = str(value_old)
        mi_field = "pubdate"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_title(self,book,table_action,value_old,other):
        book = int(book)
        new_value = value_old
        mi_field = "title"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_series(self,book,table_action,value_old,other):
        book = int(book)
        new_value = value_old
        mi_field = "series"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_comments(self,book,table_action,value_old,other,audit_log_table_id):
        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return False
        book = int(book)
        audit_log_table_id = int(audit_log_table_id)
        mysql = "SELECT id,value_old FROM _audit_log WHERE book = ? AND id = ? AND table_name = 'comments' "
        my_cursor.execute(mysql,(book,audit_log_table_id))
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            my_db.close()
            return
        for row in tmp_rows:
            id,value_old = row
            actual_value_old = value_old
        #END FOR
        my_db.close()

        book = int(book)
        new_value = actual_value_old
        mi_field = "comments"
        was_success = self.update_standard_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    def update_custom_column(self,book,table_name,table_column,table_action,value_old,value_new,other):
        mi_field = table_column  # should be a '#label'
        new_value = value_old
        was_success = self.update_custom_metadata_using_standard_calibre(book,mi_field,new_value,other)
        return was_success
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
    def update_standard_metadata_using_standard_calibre(self,book,mi_field,new_value,other):
        # class Metadata in:  src>calibre>ebooks>metadata>book>base.py
        # all standard metadata fields are defined in:  src>calibre>ebooks>metadata>book>__init__.py

        #~ if DEBUG: print("standard metadata update params: ", str(book), str(mi_field),str(new_value),str(other))

        self.books_to_update_list[:] = []  #empties it only
        self.books_to_update_list.append(int(book))

        payload = self.books_to_update_list

        language_list = []

        if mi_field == "languages":
            language_list.append(new_value)

        identifiers_dict = {}

        if mi_field == "identifiers":
            identifiers_dict[new_value] = other      #  {isbn:"9781478976677"}

        id_map = {}
        for book in self.books_to_update_list:
            mi = Metadata(_('Unknown'))
            #~ ------------------------

            if mi.title:
                mi.title = None
            if mi.pubdate:
                mi.pubdate = None
            if mi.publisher:
                mi.publisher = None
            if mi.languages:
                mi.languages = None
            if mi.series:
                mi.series = None
            if mi.series_index:
                mi.series_index = None
            if mi.tags:
                mi.tags = None
            if mi.identifiers:
                mi.identifiers = None
            if mi.comments:
                mi.comments = None

            if mi_field == "pubdate":
                new_value = str(new_value[0:10])  #~ 2017-04-24 19:41:01+00:00
                mi.pubdate = datetime.strptime(new_value, "%Y-%m-%d")
                #~ if DEBUG: print(mi.pubdate)
            elif mi_field == "publisher":
                mi.publisher = new_value
            elif mi_field == "title":
                mi.title = new_value
            elif mi_field == "languages":
                mi.languages =  language_list
            elif mi_field == "identifiers":
                mi.identifiers = identifiers_dict
                #~ ---------------------------------------------------
                #~ There is a bug in Calibre V2.83 edit_metadata.apply_mi() .  Identifiers are all deleted and immediately added-back even if mi.identifiers = None,
                #~ but ONLY if the "Automerge" checkbox in Preferences > Adding Books is checked and being used to merge, and not set to be ignored.
                        #~ self.edit_metadata.apply_mi(....):
                            #~ ....
                            #~ idents = db.get_identifiers(book_id, index_is_id=True)
                            #~ if mi.identifiers:
                                #~ idents.update(mi.identifiers)
                            #~ mi.identifiers = idents       <<<< mi.identifiers is being changed from None to the current values regardless of user action (e.g. Edit Metadata Single Book) and intent (they did not touch Identifiers).
                #~ ---------------------------------------------------
            elif mi_field == "series":
                mi.series =  new_value
            elif mi_field == "comments":
                mi.comments =  new_value
            else:
                continue
            id_map[book] = mi
            break
        #END FOR

        #~ if DEBUG: print(str(id_map[book]))

        edit_metadata_action = self.maingui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map, callback=None,merge_tags=False,merge_comments=False)

        del language_list
        del identifiers_dict
        del id_map
        del mi

        return True
    #-----------------------------------------------------
    def update_custom_metadata_using_standard_calibre(self,book,mi_field,new_value,other):

        if not mi_field.startswith("#"):
            return False

        self.books_to_update_list[:] = []  #empties it only
        self.books_to_update_list.append(int(book))

        payload = self.books_to_update_list

        custom_columns = self.maingui.current_db.field_metadata.custom_field_metadata()

        id_map = {}

        for row in self.books_to_update_list:
            book = int(book)
            mi = Metadata(_('Unknown'))
            custcol = custom_columns[mi_field]   # should be a '#label'
            custcol['#value#'] = new_value
            mi.set_user_metadata(mi_field, custcol)
            id_map[book] = mi
        #END FOR
        edit_metadata_action = self.maingui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map, callback=None)

        del id_map
        del mi
        del custom_columns

        return True

#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------

#END OF audit_log_dialog.py