#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals, division, absolute_import, print_function
from contextlib import contextmanager
from subprocess import Popen, PIPE
from datetime import datetime, timedelta
import xml.etree.ElementTree as ET
import sys, os, locale, re, urllib, tempfile, shutil, codecs, socket, json, zipfile, time, pyperclip
from epub_utils import epub_zip_up_book_contents
iswindows = sys.platform.startswith('win')
isosx = sys.platform.startswith('darwin')
islinux = sys.platform.startswith('linux')

if iswindows:
    os_encoding = locale.getpreferredencoding()
else:
    os_encoding = 'UTF-8'

# macOS workaround (https://www.mobileread.com/forums/showpost.php?p=3869646&postcount=241)
if isosx:
    try:
        import certifi
        os.environ["SSL_CERT_FILE"] = certifi.where()
    except:
        pass

# make sure the plugin will work with Python 2 AND Python 3
PY2 = sys.version_info[0] == 2
if PY2:
    from urllib import urlopen, urlretrieve
else:
    from urllib.request import urlopen, urlretrieve

# get JVM bitness (https://stackoverflow.com/questions/2062020)
def get_arch(java_path, encoding):
    ''' returns the JVM bitness '''
    arch = '64'
    args = [java_path, '-XshowSettings:properties', '-version']
    _, stderr = jar_wrapper(*args)
    arch_pattern = re.compile(r'sun.arch.data.model = (\d+)')
    arch_info = arch_pattern.search(stderr.decode(encoding))
    if arch_info is not None:
        if len(arch_info.groups()) == 1:
            arch = arch_info.group(1)
    return arch

def jar_wrapper(*args):
    ''' wrapper for executing epubcheck.jar '''
    process = Popen(args, stdout=PIPE, stderr=PIPE, shell=False)
    ret = process.communicate()
    return ret

# code provided by DiapDealer
@contextmanager
def make_temp_directory():
    ''' creates a temporary folder '''
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)

# code provided by DiapDealer
def is_connected():
    ''' tests Internet connectivity '''
    try:
        sock = socket.create_connection(('8.8.8.8', 53), 1)
        sock.close()
        return True
    except:
        pass

def get_epc_version(epc_path):
    ''' returns the epubcheck.jar version number '''
    version = ''

    # make sure that epubcheck.jar actually exists
    if os.path.exists(epc_path):

        # read .jar file as zip file
        archive = zipfile.ZipFile(epc_path)

        # make sure that pom.xml exists
        if 'META-INF/maven/org.w3c/epubcheck/pom.xml' in archive.namelist():
            pom_data = archive.read('META-INF/maven/org.w3c/epubcheck/pom.xml')
            archive.close()

            # parse pom.xml as ElementTree
            root = ET.fromstring(pom_data)
            tag = root.find("*//{http://maven.apache.org/POM/4.0.0}tag")
            # look for <tag>
            if tag is not None:
                version = tag.text
            else:
                # look for <version>
                project_version = root.find("{http://maven.apache.org/POM/4.0.0}version")
                if project_version is not None:
                    version = 'v' + project_version.text
        else:
            print('pom.xml not found!')

    else:
        print('epubcheck.jar not found!')

    return version

def latest_epc_version(github_url):
    ''' returns the latest Github EPUBCheck version '''
    latest_version = ''
    browser_download_url = ''
    if is_connected():
        try:
            response = urlopen(github_url).read().decode('utf-8')
            parsed_json = json.loads(response)
            latest_version = parsed_json[0]['tag_name']
            browser_download_url = parsed_json[0]['assets'][0]['browser_download_url']
        except Exception as ex:
            import ssl
            print('*** PYTHON ERROR ***\nSSL version:', ssl.OPENSSL_VERSION)
            exception_info = "An exception of type {0} occurred.\nArguments:\n{1!r}".format(
                type(ex).__name__, ex.args)
            print(exception_info)
            browser_download_url = '*** PYTHON ERROR *** ' + exception_info.replace('\n', ' ')
            # Wait for 2 seconds
            time.sleep(2)

    return latest_version, browser_download_url

# code provided by DiapDealer
def string_to_date(datestring):
    ''' converts dates to strings '''
    return datetime.strptime(datestring, "%Y-%m-%d %H:%M:%S.%f")

# code provided by KevinH
def generate_line_offsets(s):
    ''' returns line offsets '''
    offlst = [0]
    i = s.find('\n', 0)
    while i >= 0:
        offlst.append(i)
        i = s.find('\n', i + 1)
    # single-line files
    if s.find('\n', 0) == -1:
        offlst.append(len(s) - 1)
    return offlst

# code provided by KevinH
def charoffset(line, col, offlst):
    ''' returns character offsets  '''
    coffset = None
    if iswindows:
        coffset = offlst[line-1]  + 2 + (col - 1) - line
    else:
        coffset = offlst[line-1]  + 1 + (col - 1)
    if line == 1:
        coffset -= 1
    return coffset

# taken from navprocessor.py
def xmlencode(data):
    ''' escapes text to make it xml safe '''
    if data is None:
        return ''
    newdata = data
    newdata = newdata.replace('&', '&amp;')
    newdata = newdata.replace('<', '&lt;')
    newdata = newdata.replace('>', '&gt;')
    newdata = newdata.replace('"', '&quot;')
    return newdata

# on macOS the file path contains dots instead of slashes
# e.g. 'OEBPS.Text.Section0001.xhtml'
def get_mac_filepath(filepath):
    ''' returns the macOS file path '''
    fparts = filepath.split('.') 
    fname = ".".join(fparts[-2:])
    del fparts[-2:]
    fparts.append(fname)
    bookpath = "/".join(fparts)
    return bookpath

def run(bk):
    ''' main routine'''

    # get GUI language
    sigil_lang = 'en'
    try:
        sigil_lang = bk.sigil_ui_lang[0:2]
    except:
        pass

    # get EpubCheck plugin path
    plugin_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name)

    # get local lib/epubcheck.jar paths
    epc_path = os.path.join(plugin_path, 'epubcheck.jar')
    epc_lib_dir = os.path.join(plugin_path, 'lib')

    # get current version number
    version = get_epc_version(epc_path)

    # get/set preference
    prefs = bk.getPrefs()

    # write initial JSON file
    if prefs == {}:
        prefs['update_check'] = True
        prefs['clipboard_copy'] = False
        prefs['usage'] = False
        prefs['github'] = True
        prefs['last_time_checked'] = str(datetime.now() - timedelta(days=7))
        prefs['check_interval'] = 7
        prefs['java_path'] = 'java'
        bk.savePrefs(prefs)

    # get preferences
    debug = prefs.get('debug', False)
    update_check = prefs.get('update_check', True)
    clipboard_copy = prefs.get('clipboard_copy', False)
    usage = prefs.get('usage', False)
    github = prefs.get('github', True)
    last_time_checked = prefs.get('last_time_checked', str(datetime.now() - timedelta(days=7)))
    check_interval = prefs.get('check_interval', 7)
    java_path = prefs.get('java_path', 'java').replace('\\\\', '/').replace('\\', '/')

    #------------------------------
    # check for 32bit JVM
    #------------------------------
    if 'is32bit' not in prefs:
        arch = get_arch(java_path, os_encoding)
        if arch == '32':
            is32bit = True
        else:
            is32bit = False
        prefs['is32bit'] = is32bit
        bk.savePrefs(prefs)
    else:
        is32bit = prefs.get('is32bit', False)

    # save preferences
    prefs['update_check'] = update_check
    prefs['clipboard_copy'] = clipboard_copy
    prefs['usage'] = usage
    prefs['github'] = github
    prefs['last_time_checked'] = last_time_checked
    prefs['check_interval'] = check_interval
    prefs['java_path'] = java_path
    bk.savePrefs(prefs)

    # get epubcheck language preference
    locale = None
    if 'locale' in prefs and prefs['locale'] in [
            'en',
            'de',
            'es',
            'fr',
            'it',
            'ja',
            'nl',
            'pt-BR',
            'ko-KR',
            'zh_TW',
            'da']:
        locale = prefs['locale']

    #===============================
    # run update check
    #===============================

    # get the version number of the latest EPUBCheck release
    if github:

        # make sure we have an Internet connection
        if is_connected():

            # compare current date against last update check date
            time_delta = (datetime.now() - string_to_date(last_time_checked)).days
            if time_delta >= check_interval:

                # display update check message
                if sigil_lang == 'fr':
                    print('Recherche de mises à jour...\n')
                elif sigil_lang == 'it':
                    print('Verifica aggiornamenti...\n')
                elif sigil_lang == 'de':
                    print('Update wird gesucht...\n')
                elif sigil_lang == 'es':
                    print('Buscando actualización...\n')
                elif sigil_lang == 'pt':
                    print('Verificando se há atualização...\n')
                else:
                    print('Running update check...\n')

                # update time stamp
                prefs['last_time_checked'] = str(datetime.now())
                bk.savePrefs(prefs)

                # get latest version and download URL
                github_url = 'https://api.github.com/repos/w3c/epubcheck/releases'
                latest_version, browser_download_url = latest_epc_version(github_url)

                if latest_version != version and latest_version != '':

                    # display New EPUBCheck version found message
                    if sigil_lang == 'fr':
                        print('Mise à jour trouvée : EPUBCheck {}\n'.format(latest_version))
                    elif sigil_lang == 'it':
                        print('Aggiornamento trovato: EPUBCheck {}\n'.format(latest_version))
                    elif sigil_lang == 'de':
                        print('Update gefunden: EPUBCheck {}\n'.format(latest_version))
                    elif sigil_lang == 'es':
                        print('Se encontró una actualización: EPUBCheck {}\n'.format(latest_version))
                    elif sigil_lang == 'pt':
                        print('Atualização foi encontrada: EPUBCheck {}\n'.format(latest_version))
                    else:
                        print('Update found: EPUBCheck {}\n'.format(latest_version))

                    base_name = os.path.basename(browser_download_url)
                    root_path = os.path.splitext(base_name)[0]

                    # create temp folder
                    with make_temp_directory() as td:
                        zip_file_name = os.path.join(td, base_name)
                        print('Downloading', browser_download_url, '...')
                        urlretrieve(browser_download_url, zip_file_name)

                        # make sure the file was actually downloaded
                        if os.path.exists(zip_file_name):
                            print(browser_download_url, ' downloaded.\n')

                            # read zip file (https://stackoverflow.com/a/19618531)
                            archive = zipfile.ZipFile(zip_file_name)
                            files = archive.namelist()
                            files_to_extract = [m for m in files \
                            if (m.startswith(root_path + '/lib') \
                            or m == root_path + '/epubcheck.jar')]
                            archive.extractall(td, files_to_extract)
                            archive.close()

                            # epubcheck.jar and /lib folder temp paths
                            temp_epc_path = os.path.join(td, root_path + '/epubcheck.jar')
                            temp_epc_lib_dir = os.path.join(td, root_path + '/lib')

                            # make sure the files were actually extracted
                            if os.path.isdir(temp_epc_lib_dir) and os.path.isfile(temp_epc_path):

                                # delete /lib folder
                                if os.path.isdir(epc_lib_dir):
                                    shutil.rmtree(epc_lib_dir)

                                # delete epubcheck.jar
                                if os.path.exists(epc_path):
                                    os.remove(epc_path)

                                # move new files to the plugin folder
                                shutil.move(temp_epc_lib_dir, plugin_path)
                                shutil.move(temp_epc_path, plugin_path)

                                # ensure you have execute rights for unix based platforms
                                if isosx or islinux:
                                    os.chmod(epc_path, 0o744)

                                # update version number
                                version = latest_version

                                # display update successful message
                                if sigil_lang == 'fr':
                                    print('EPUBCheck a été mise à jour vers la version {}.\n'.format(version))
                                elif sigil_lang == 'it':
                                    print('EPUBCheck è stata aggiornata alla versione {}.\n'.format(version))
                                elif sigil_lang == 'de':
                                    print('EPUBCheck wurde erfolgreich auf die Version {} aktualisiert.\n'.format(version))
                                elif sigil_lang == 'es':
                                    print('EPUBCheck se ha actualizado correctamente a la versión {}.\n'.format(version))
                                elif sigil_lang == 'pt':
                                    print('EPUBCheck foi atualizado com êxito para a versão {}.\n'.format(version))
                                else:
                                    print('EPUBCheck updated to {}.\n'.format(version))

                            else:

                                print('EPUBCheck files couldn\'t be extracted.\n')
                        else:
                            # display Github download failed message
                            print('Github download failed.\n')
                            # Wait for 2 seconds
                            time.sleep(2)

                else:

                    if latest_version != '':

                        # display no update found message
                        if sigil_lang == 'fr':
                            print('Aucune mise à jour trouvée.\n')
                        elif sigil_lang == 'it':
                            print('Non è stato trovato alcun aggiornamento.\n')
                        elif sigil_lang == 'de':
                            print('Kein Update gefunden.\n')
                        elif sigil_lang == 'es':
                            print('No se encontró ninguna actualización.\n')
                        elif sigil_lang == 'pt':
                            print('Nenhuma atualização foi encontrada.\n')
                        else:
                            print('No new EPUBCheck version found.\n')

                    else:
                        # display update check failed message
                        print('Internal error: update check failed.\n')
                        # Wait for 2 seconds
                        time.sleep(2)

                        # display Python error messages;
                        # (browser_download_url contains the exception details)
                        if browser_download_url != '':
                            bk.add_result('error', None, None, browser_download_url)

        else:
            # display update skipped message
            print('Update check skipped: no Internet.\n')
            # Wait for 2 seconds
            time.sleep(2)

    #======================
    # run epubcheck
    #======================

    #-------------------------------------------------
    # get opf file name and contents
    #-------------------------------------------------

    # get the default opf path and the opf file name
    if bk.launcher_version() >= 20190927:
        opf_path = os.path.join(bk._w.ebook_root, bk.get_opfbookpath()) 
        opf_name = os.path.basename(opf_path)
    else:
        opf_name = bk._w.opfname
        opf_path = os.path.join(bk._w.ebook_root, 'OEBPS', opf_name)

    # read the original opf file
    if opf_path:
        if PY2:
            f = codecs.open(opf_path, 'r', encoding='utf8')
        else:
            f = open(opf_path, 'r', encoding='utf-8')
        opf_contents = f.read()
        f.close()

    else:
        print('Internal error: opf path not found!', opf_path)
        return -1

    #-------------------------------------------------------------
    # copy current epub to temp epub
    #-------------------------------------------------------------
    with make_temp_directory() as temp_dir:

        # copy book files
        bk.copy_book_contents_to(temp_dir)

        # create mimetype file
        with open(os.path.join(temp_dir, "mimetype"), "w") as mimetype:
            mimetype.write("application/epub+zip")

        # zip up the epub folder and save the epub in the plugin folder
        epub_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'temp.epub')
        if os.path.isfile(epub_path):
            os.remove(str(epub_path))
        epub_zip_up_book_contents(temp_dir, epub_path)
        #if debug: print('\nepub_path', epub_path)

    #----------------------------------------------------------------------
    # define epubcheck command line parameters
    #----------------------------------------------------------------------

    epc_path = os.path.join(plugin_path, 'epubcheck.jar')
    if is32bit:
        args = [java_path, '-Xss1024k', '-jar', epc_path, epub_path, '--version']
    else:
        args = [java_path, '-jar', epc_path, epub_path, '--version']

    # change message language
    if locale:
        args.extend(['--locale', locale])

    # display usage messages
    if usage:
        args.append('--usage')

    # display status message
    if sigil_lang == 'fr':
        print('EPUBCheck {} en cours d\'exécution ... veuillez patienter.'.format(version))
    elif sigil_lang == 'it':
        print('EPUBCheck {} en corso ... attendere.'.format(version))
    elif sigil_lang == 'de':
        print('EPUBCheck {} wird ausgeführt ... bitte warten.'.format(version))
    elif sigil_lang == 'es':
        print('EPUBCheck {} en ejecución ...  espere un unos instantes.'.format(version))
    elif sigil_lang == 'pt':
        print('Executando EPUBCheck {} ... por favor aguarde.'.format(version))
    else:
        print('Running EPUBCheck {}... please wait.'.format(version))

    #--------------------------------
    # run epubcheck
    #--------------------------------
    if debug: print('\nargs', *args)
    result = jar_wrapper(*args)
    stdout = result[0].decode(os_encoding)
    stderr = result[1].decode(os_encoding)
    #if debug: print('stderr', stderr)

    # check for StackOverflowError error
    if stderr.find('java.lang.StackOverflowError') != -1 or stderr.find('Exception in thread') != -1:
        print('EPUBCheck Java error.\n', stderr)
        return -1

    # delete temp epub from plugin folder
    os.remove(str(epub_path))

    # get each line of the messages
    if usage:
        stderr += stdout

    # replace Windows file separator with '/'
    if iswindows:
        epub_path = epub_path.replace(os.sep, '/')

    # delete epub path from EPUBCheck messages
    #stderr = stderr.replace(epub_path + '/', '')
    if isosx:
        stderr = re.sub('\: .*?\.temp\.epub\.', ':', stderr)
    else:
        stderr = re.sub('\: .*?/temp.epub/', ':', stderr)

    # copy error messages to clipboard
    if clipboard_copy:
        pyperclip.copy(stderr+stdout)

    # split into lines
    epc_errors = stderr.splitlines()

    # used for coffset calculation
    lastfilename = None

    #------------------------------------------------------
    # parse EPUBCheck error messages
    #------------------------------------------------------
    for line in epc_errors:

        # process only errors, warnings and usage/info messages
        if line.startswith(('ERROR', 'WARNING', 'FATAL', 'INFO', 'USAGE')):

            # split message by colons
            err_list = line.split(':')

            # check for colons in error message
            if len(err_list) > 3:
                # merge list items
                err_list[2:len(err_list)] = [':'.join(err_list[2:len(err_list)])]

            # get error code e.g. FATAL(RSC-016) or ERROR(RSC-005)
            err_code = err_list[0]

            # get message (needs to be escaped)
            message = xmlencode(err_list[2].strip())

            # get file name & line/column numbers
            linenumber = None
            colnumber = None
            line_pos = re.search(r'\((-*\d+),(-*\d+)*\)', err_list[1])

            if line_pos:
                # get file name
                filepath = re.sub(r'\(-*\d+,-*\d+\)', '', err_list[1])

                # get line/column numbers
                if int(line_pos.group(1)) != -1:
                    linenumber = line_pos.group(1)
                if int(line_pos.group(2)) != -1:
                    colnumber = line_pos.group(2)
            else:
                # get file name only
                filepath = err_list[1]

            # fix macOS Python/Java bug with filenames with periods
            if isosx:
                filepath = get_mac_filepath(filepath)
                # double-check the file path
                if bk.launcher_version() >= 20190927:
                    if not bk.bookpath_to_id(filepath):
                        print('\n*** Invalid macOS file path returned! ***\n')
                        if not filepath.endswith('opf'):
                            base_name = os.path.basename(filepath)
                            manifest_id = bk.basename_to_id(base_name)
                            if manifest_id:
                                filepath = bk.id_to_bookpath(manifest_id)
                            else:
                                print('\n*** Book path not resolved! ***\n')
                                return -1
                        else:
                            filepath = bk.get_opfbookpath()

            # get file name without path
            filename = os.path.basename(filepath)

            # calculate line and character offsets (requires Sigil 0.9.7 or higher)
            coffset = None
            if colnumber:
                if bk.launcher_version() >= 20160909:

                    # calculate line offsets
                    if filename != lastfilename and not filename.endswith('.epub'):
                        text = None
                        offlst = [0]

                        if bk.basename_to_id(filename):
                            text = bk.readfile(bk.basename_to_id(filename))
                        elif filename == opf_name or filename.endswith('opf'):
                            text = opf_contents
                        else:
                            href = 'Text/' + filename
                            if bk.href_to_id(href):
                                text = bk.readfile(bk.href_to_id(href))
                            else:
                                bk.add_result('error', xmlencode(filename), None, \
                                xmlencode('*** Internal error. bk.href_to_id() failed for: ' + filename))

                        if text:
                            offlst = generate_line_offsets(text)
                        lastfilename = filename

                    # calculate character offset
                    if offlst != [0] and linenumber:
                        coffset = charoffset(int(linenumber), int(colnumber), offlst)
                        if iswindows and filename == opf_name:
                            coffset += int(linenumber) - 1

            # define message type
            if err_code.startswith(('ERROR', 'FATAL')):
                restype = 'error'
            elif err_code.startswith('WARNING'):
                restype = 'warning'
            else:
                restype = 'info'

            # epubcheck USAGE bug workaround
            # (some Usage messages are reported without a file name)
            if not filepath.endswith('.epub'):

                #---------------------------------------------------
                # add message to validation pane
                #---------------------------------------------------
                
                # older Sigil versions don't require the full file path
                if bk.launcher_version() < 20190927:
                    filepath = filename
                
                if coffset:
                    bk.add_extended_result(
                        restype,
                        xmlencode(filepath),
                        linenumber,
                        coffset,
                        'Col: ' + colnumber + ': ' + err_code + ': ' + message)
                else:
                    if colnumber:
                        # if this message is displayed,
                        # bk.readfile() has failed!!!
                        bk.add_result(
                            restype,
                            xmlencode(filepath),
                            linenumber,
                            '*** Col: ' + colnumber + ': ' + err_code + ': ' + message)
                    else:
                        bk.add_result(
                            restype,
                            xmlencode(filepath),
                            linenumber,
                            err_code + ':  ' + message)

    #=============================
    # run plugin version check
    #=============================

    if update_check:

        # make sure we have an Internet connection
        if is_connected():
            time_delta = (datetime.now() - string_to_date(last_time_checked)).days
            if time_delta >= check_interval:

                href = 'http://www.mobileread.com/forums/showpost.php?p=2950625&postcount=1'
                _latest_pattern = re.compile(r'Current Version:\s*&quot;([^&]*)&')
                plugin_xml_path = os.path.abspath(os.path.join(bk._w.plugin_dir, 'EpubCheck', 'plugin.xml'))
                plugin_version = ET.parse(plugin_xml_path).find('.//version').text
                try:
                    latest_version = None
                    if PY2:
                        response = urllib.urlopen(href)
                    else:
                        response = urllib.request.urlopen(href)
                    m = _latest_pattern.search(response.read().decode('utf-8', 'ignore'))
                    if m:
                        latest_version = (m.group(1).strip())
                        if latest_version and latest_version != plugin_version:
                            restype = 'info'
                            filename = linenumber = None

                            if sigil_lang == 'fr':
                                message = 'Une nouvelle version du greffon est disponible, la version {}.'.format(latest_version)
                            elif sigil_lang == 'it':
                                message = 'È disponibile una versione più recente del plugin, versione {}.'.format(latest_version)
                            elif sigil_lang == 'de':
                                message = 'Eine neuere Plugin-Version ist verfügbar: Version {}.'.format(latest_version)
                            elif sigil_lang == 'es':
                                message = 'Hay disponible una versión más reciente del complemento: {}.'.format(latest_version)
                            elif sigil_lang == 'pt':
                                message = 'Uma versão mais recente do plug-in está disponível: {}.'.format(latest_version)
                            else:
                                message = 'A newer plugin version is available, version {}.'.format(latest_version)

                            bk.add_result(restype, filename, linenumber, message)
                except:
                    print('Internal error: latest plugin version not found.\n')

                # update last time checked time stamp
                last_time_checked = str(datetime.now())
                prefs['last_time_checked'] = last_time_checked
                bk.savePrefs(prefs)
        else:
            print('Plugin update check skipped: no Internet.')
            # Wait for 2 seconds
            time.sleep(2)

    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
