/*
 * Decompiled with CFR 0.152.
 */
package com.hughes.android.dictionary.parser;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class WikiTokenizer {
    private static final Pattern wikiTokenEvent = Pattern.compile("(\\{\\{|\\}\\}|\\[\\[|\\]\\]|\\||=|<!--|''|<pre>|<math>|<ref>|$)", 8);
    private static final String listChars = "*#:;";
    final String wikiText;
    final Matcher matcher;
    boolean justReturnedNewline = true;
    int lastLineStart = 0;
    int end = 0;
    int start = -1;
    final List<String> errors = new ArrayList<String>();
    final List<String> tokenStack = new ArrayList<String>();
    private String headingWikiText;
    private int headingDepth;
    private int listPrefixEnd;
    private boolean isPlainText;
    private boolean isMarkup;
    private boolean isComment;
    private boolean isFunction;
    private boolean isWikiLink;
    private boolean isHtml;
    private int firstUnescapedPipePos;
    private int lastUnescapedPipePos;
    private int lastUnescapedEqualsPos;
    private final List<String> positionArgs = new ArrayList<String>();
    private final Map<String, String> namedArgs = new LinkedHashMap<String, String>();
    private static final Pattern POSSIBLE_WIKI_TEXT = Pattern.compile("\\{\\{|\\[\\[|<!--|''|<pre>|<math>|<ref>|[\n]");

    public WikiTokenizer(String wikiText) {
        this(wikiText, true);
    }

    public WikiTokenizer(String wikiText, boolean isNewline) {
        wikiText = wikiText.replaceAll("\u2028", "\n");
        this.wikiText = wikiText = wikiText.replaceAll("\u0085", "\n");
        this.matcher = wikiTokenEvent.matcher(wikiText);
        this.justReturnedNewline = isNewline;
    }

    private void clear() {
        this.errors.clear();
        this.tokenStack.clear();
        this.headingWikiText = null;
        this.headingDepth = -1;
        this.listPrefixEnd = -1;
        this.isPlainText = false;
        this.isMarkup = false;
        this.isComment = false;
        this.isFunction = false;
        this.isWikiLink = false;
        this.isHtml = false;
        this.firstUnescapedPipePos = -1;
        this.lastUnescapedPipePos = -1;
        this.lastUnescapedEqualsPos = -1;
        this.positionArgs.clear();
        this.namedArgs.clear();
    }

    public static void dispatch(String wikiText, boolean isNewline, Callback callback) {
        if (!POSSIBLE_WIKI_TEXT.matcher(wikiText).find()) {
            callback.onPlainText(wikiText);
        } else {
            WikiTokenizer tokenizer = new WikiTokenizer(wikiText, isNewline);
            while (tokenizer.nextToken() != null) {
                if (tokenizer.isPlainText()) {
                    callback.onPlainText(tokenizer.token());
                    continue;
                }
                if (tokenizer.isMarkup()) {
                    callback.onMarkup(tokenizer);
                    continue;
                }
                if (tokenizer.isWikiLink()) {
                    callback.onWikiLink(tokenizer);
                    continue;
                }
                if (tokenizer.isNewline()) {
                    callback.onNewline(tokenizer);
                    continue;
                }
                if (tokenizer.isFunction()) {
                    callback.onFunction(tokenizer, tokenizer.functionName(), tokenizer.functionPositionArgs(), tokenizer.functionNamedArgs());
                    continue;
                }
                if (tokenizer.isHeading()) {
                    callback.onHeading(tokenizer);
                    continue;
                }
                if (tokenizer.isListItem()) {
                    callback.onListItem(tokenizer);
                    continue;
                }
                if (tokenizer.isComment()) {
                    callback.onComment(tokenizer);
                    continue;
                }
                if (tokenizer.isHtml()) {
                    callback.onHtml(tokenizer);
                    continue;
                }
                if (!tokenizer.errors.isEmpty()) continue;
                throw new IllegalStateException("Unknown wiki state: " + tokenizer.token());
            }
        }
    }

    public List<String> errors() {
        return this.errors;
    }

    public boolean isNewline() {
        return this.justReturnedNewline;
    }

    public void returnToLineStart() {
        this.end = this.start = this.lastLineStart;
        this.justReturnedNewline = true;
    }

    public boolean isHeading() {
        return this.headingWikiText != null;
    }

    public String headingWikiText() {
        assert (this.isHeading());
        return this.headingWikiText;
    }

    public int headingDepth() {
        assert (this.isHeading());
        return this.headingDepth;
    }

    public boolean isMarkup() {
        return this.isMarkup;
    }

    public boolean isComment() {
        return this.isComment;
    }

    public boolean isListItem() {
        return this.listPrefixEnd != -1;
    }

    public String listItemPrefix() {
        assert (this.isListItem());
        return this.wikiText.substring(this.start, this.listPrefixEnd);
    }

    public static String getListTag(char c) {
        if (c == '#') {
            return "ol";
        }
        return "ul";
    }

    public String listItemWikiText() {
        assert (this.isListItem());
        return this.wikiText.substring(this.listPrefixEnd, this.end);
    }

    public boolean isFunction() {
        return this.isFunction;
    }

    public String functionName() {
        assert (this.isFunction());
        if (this.firstUnescapedPipePos != -1) {
            return WikiTokenizer.trimNewlines(this.wikiText.substring(this.start + 2, this.firstUnescapedPipePos).trim());
        }
        int safeEnd = Math.max(this.start + 2, this.end - 2);
        return WikiTokenizer.trimNewlines(this.wikiText.substring(this.start + 2, safeEnd).trim());
    }

    public List<String> functionPositionArgs() {
        return this.positionArgs;
    }

    public Map<String, String> functionNamedArgs() {
        return this.namedArgs;
    }

    public boolean isPlainText() {
        return this.isPlainText;
    }

    public boolean isWikiLink() {
        return this.isWikiLink;
    }

    public String wikiLinkText() {
        assert (this.isWikiLink());
        if (this.lastUnescapedPipePos != -1) {
            return WikiTokenizer.trimNewlines(this.wikiText.substring(this.lastUnescapedPipePos + 1, this.end - 2));
        }
        assert (this.start + 2 < this.wikiText.length() && this.end >= 2) : this.wikiText;
        return WikiTokenizer.trimNewlines(this.wikiText.substring(this.start + 2, this.end - 2));
    }

    public String wikiLinkDest() {
        assert (this.isWikiLink());
        if (this.firstUnescapedPipePos != -1) {
            return WikiTokenizer.trimNewlines(this.wikiText.substring(this.start + 2, this.firstUnescapedPipePos));
        }
        return null;
    }

    public boolean isHtml() {
        return this.isHtml;
    }

    public boolean remainderStartsWith(String prefix) {
        return this.wikiText.startsWith(prefix, this.start);
    }

    public void nextLine() {
        int oldStart = this.start;
        while (this.nextToken() != null && !this.isNewline()) {
        }
        if (this.isNewline()) {
            --this.end;
        }
        this.start = oldStart;
    }

    public WikiTokenizer nextToken() {
        this.clear();
        this.start = this.end;
        if (this.justReturnedNewline) {
            this.lastLineStart = this.start;
        }
        try {
            boolean atNewline;
            int len = this.wikiText.length();
            if (this.start >= len) {
                return null;
            }
            boolean bl = atNewline = this.wikiText.charAt(this.end) == '\n' || this.wikiText.charAt(this.end) == '\u2028';
            if (atNewline) {
                this.justReturnedNewline = true;
                ++this.end;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.justReturnedNewline) {
                this.justReturnedNewline = false;
                char firstChar = this.wikiText.charAt(this.end);
                if (firstChar == '=') {
                    int headerStart = this.end;
                    while (++this.end < len && this.wikiText.charAt(this.end) == '=') {
                    }
                    int headerTitleStart = this.end;
                    this.headingDepth = headerTitleStart - headerStart;
                    if (this.end < len) {
                        int nextNewline = WikiTokenizer.safeIndexOf(this.wikiText, this.end, "\n", "\n");
                        int closingEquals = this.escapedFindEnd(this.end, "=");
                        this.end = this.wikiText.charAt(closingEquals - 1) == '=' ? closingEquals - 1 : nextNewline;
                    }
                    int headerTitleEnd = this.end;
                    this.headingWikiText = this.wikiText.substring(headerTitleStart, headerTitleEnd);
                    while (this.end < len && ++this.end < len && this.wikiText.charAt(this.end) == '=') {
                    }
                    int headerEnd = this.end;
                    if (headerEnd - headerTitleEnd != this.headingDepth) {
                        this.errors.add("Mismatched header depth: " + this.token());
                    }
                    WikiTokenizer wikiTokenizer = this;
                    return wikiTokenizer;
                }
                if (listChars.indexOf(firstChar) != -1) {
                    while (++this.end < len && listChars.indexOf(this.wikiText.charAt(this.end)) != -1) {
                    }
                    this.listPrefixEnd = this.end;
                    this.end = this.escapedFindEnd(this.start, "\n");
                    WikiTokenizer wikiTokenizer = this;
                    return wikiTokenizer;
                }
            }
            if (this.wikiText.startsWith("'''", this.start)) {
                this.isMarkup = true;
                this.end = this.start + 3;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("''", this.start)) {
                this.isMarkup = true;
                this.end = this.start + 2;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("[[", this.start)) {
                this.end = this.escapedFindEnd(this.start + 2, "]]");
                this.isWikiLink = this.errors.isEmpty();
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("{{", this.start)) {
                this.end = this.escapedFindEnd(this.start + 2, "}}");
                this.isFunction = this.errors.isEmpty();
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("<pre>", this.start)) {
                this.end = WikiTokenizer.safeIndexOf(this.wikiText, this.start, "</pre>", "\n");
                this.isHtml = true;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("<ref>", this.start)) {
                this.end = WikiTokenizer.safeIndexOf(this.wikiText, this.start, "</ref>", "\n");
                this.isHtml = true;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("<math>", this.start)) {
                this.end = WikiTokenizer.safeIndexOf(this.wikiText, this.start, "</math>", "\n");
                this.isHtml = true;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("<!--", this.start)) {
                this.isComment = true;
                this.end = WikiTokenizer.safeIndexOf(this.wikiText, this.start, "-->", "\n");
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.startsWith("}}", this.start) || this.wikiText.startsWith("]]", this.start)) {
                this.errors.add("Close without open!");
                this.end += 2;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.wikiText.charAt(this.start) == '|' || this.wikiText.charAt(this.start) == '=') {
                this.isPlainText = true;
                ++this.end;
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            if (this.matcher.find(this.start)) {
                this.end = this.matcher.start(1);
                this.isPlainText = true;
                if (this.end == this.start) {
                    this.errors.add("Empty group: " + this.matcher.group());
                    assert (false);
                }
                WikiTokenizer wikiTokenizer = this;
                return wikiTokenizer;
            }
            this.end = this.wikiText.length();
            WikiTokenizer wikiTokenizer = this;
            return wikiTokenizer;
        }
        finally {
            if (!this.errors.isEmpty()) {
                System.err.println("Errors: " + this.errors + ", token=" + this.token());
            }
        }
    }

    public String token() {
        String token = this.wikiText.substring(this.start, this.end);
        assert (token.equals("\n") || !token.endsWith("\n")) : "token='" + token + "'";
        return token;
    }

    /*
     * Enabled aggressive block sorting
     */
    private int escapedFindEnd(int start, String toFind) {
        assert (this.tokenStack.isEmpty());
        boolean insideFunction = toFind.equals("}}");
        int end = start;
        int firstNewline = -1;
        while (end < this.wikiText.length()) {
            if (!this.matcher.find(end)) {
                this.errors.add("Couldn't find: " + toFind + ", " + this.wikiText.substring(start));
                return WikiTokenizer.safeIndexOf(this.wikiText, start, "\n", "\n");
            }
            String matchText = this.matcher.group();
            int matchStart = this.matcher.start();
            assert (this.matcher.end() > end || matchText.length() == 0) : "Group=" + this.matcher.group();
            if (matchText.length() == 0) {
                assert (matchStart == this.wikiText.length() || this.wikiText.charAt(matchStart) == '\n') : String.valueOf(this.wikiText) + ", " + matchStart;
                if (firstNewline == -1) {
                    firstNewline = this.matcher.end();
                }
                if (this.tokenStack.isEmpty() && toFind.equals("\n")) {
                    return matchStart;
                }
                ++end;
            } else {
                if (this.tokenStack.isEmpty() && matchText.equals(toFind)) {
                    if (insideFunction) {
                        this.addFunctionArg(insideFunction, matchStart);
                    }
                    return this.matcher.end();
                }
                if (matchText.equals("[[") || matchText.equals("{{")) {
                    this.tokenStack.add(matchText);
                } else if (matchText.equals("]]") || matchText.equals("}}")) {
                    if (this.tokenStack.size() <= 0) {
                        this.errors.add("Pop too many error: " + this.wikiText.substring(start).replaceAll("\n", "\\\\n"));
                        return WikiTokenizer.safeIndexOf(this.wikiText, start, "\n", "\n");
                    }
                    String removed = this.tokenStack.remove(this.tokenStack.size() - 1);
                    if (removed.equals("{{") && !this.matcher.group().equals("}}")) {
                        this.errors.add("Unmatched {{ error: " + this.wikiText.substring(start));
                        return WikiTokenizer.safeIndexOf(this.wikiText, start, "\n", "\n");
                    }
                    if (removed.equals("[[") && !this.matcher.group().equals("]]")) {
                        this.errors.add("Unmatched [[ error: " + this.wikiText.substring(start));
                        return WikiTokenizer.safeIndexOf(this.wikiText, start, "\n", "\n");
                    }
                } else if (matchText.equals("|")) {
                    if (this.tokenStack.isEmpty()) {
                        this.addFunctionArg(insideFunction, matchStart);
                    }
                } else if (matchText.equals("=")) {
                    if (this.tokenStack.isEmpty()) {
                        this.lastUnescapedEqualsPos = matchStart;
                    }
                } else if (matchText.equals("<!--")) {
                    end = this.wikiText.indexOf("-->");
                    if (end == -1) {
                        this.errors.add("Unmatched <!-- error: " + this.wikiText.substring(start));
                        return WikiTokenizer.safeIndexOf(this.wikiText, start, "\n", "\n");
                    }
                } else if (!(matchText.equals("''") || matchText.startsWith("<") && matchText.endsWith(">"))) {
                    assert (false) : "Match text='" + matchText + "'";
                    throw new IllegalStateException();
                }
            }
            end = Math.max(end, this.matcher.end());
        }
        if (toFind.equals("\n") && this.tokenStack.isEmpty()) {
            return end;
        }
        if (firstNewline != -1) {
            this.errors.add("Couldn't find: " + toFind + ", " + this.wikiText.substring(start));
            return firstNewline;
        }
        return end;
    }

    private void addFunctionArg(boolean insideFunction, int matchStart) {
        if (this.firstUnescapedPipePos == -1) {
            this.firstUnescapedPipePos = this.lastUnescapedPipePos = matchStart;
        } else if (insideFunction) {
            if (this.lastUnescapedEqualsPos > this.lastUnescapedPipePos) {
                String key = this.wikiText.substring(this.lastUnescapedPipePos + 1, this.lastUnescapedEqualsPos);
                String value = this.wikiText.substring(this.lastUnescapedEqualsPos + 1, matchStart);
                this.namedArgs.put(WikiTokenizer.trimNewlines(key), WikiTokenizer.trimNewlines(value));
            } else {
                String value = this.wikiText.substring(this.lastUnescapedPipePos + 1, matchStart);
                this.positionArgs.add(WikiTokenizer.trimNewlines(value));
            }
        }
        this.lastUnescapedPipePos = matchStart;
    }

    static final String trimNewlines(String s) {
        while (s.startsWith("\n")) {
            s = s.substring(1);
        }
        while (s.endsWith("\n")) {
            s = s.substring(0, s.length() - 1);
        }
        return s.replaceAll("\n", " ");
    }

    static int safeIndexOf(String s, int start, String target, String backup) {
        int close = s.indexOf(target, start);
        if (close != -1) {
            return close + (target.equals("\n") ? 0 : target.length());
        }
        close = s.indexOf(backup, start);
        if (close != -1) {
            return close + (backup.equals("\n") ? 0 : backup.length());
        }
        return s.length();
    }

    public static String toPlainText(String wikiText) {
        WikiTokenizer wikiTokenizer = new WikiTokenizer(wikiText);
        StringBuilder builder = new StringBuilder();
        while (wikiTokenizer.nextToken() != null) {
            if (wikiTokenizer.isPlainText()) {
                builder.append(wikiTokenizer.token());
                continue;
            }
            if (wikiTokenizer.isWikiLink()) {
                builder.append(wikiTokenizer.wikiLinkText());
                continue;
            }
            if (wikiTokenizer.isNewline()) {
                builder.append("\n");
                continue;
            }
            if (!wikiTokenizer.isFunction()) continue;
            builder.append(wikiTokenizer.token());
        }
        return builder.toString();
    }

    public static StringBuilder appendFunction(StringBuilder builder, String name, List<String> args, Map<String, String> namedArgs) {
        builder.append(name);
        for (String string2 : args) {
            builder.append("|").append(string2);
        }
        for (Map.Entry entry : namedArgs.entrySet()) {
            builder.append("|").append((String)entry.getKey()).append("=").append((String)entry.getValue());
        }
        return builder;
    }

    public static interface Callback {
        public void onPlainText(String var1);

        public void onMarkup(WikiTokenizer var1);

        public void onWikiLink(WikiTokenizer var1);

        public void onNewline(WikiTokenizer var1);

        public void onFunction(WikiTokenizer var1, String var2, List<String> var3, Map<String, String> var4);

        public void onHeading(WikiTokenizer var1);

        public void onListItem(WikiTokenizer var1);

        public void onComment(WikiTokenizer var1);

        public void onHtml(WikiTokenizer var1);
    }

    public static class DoNothingCallback
    implements Callback {
        @Override
        public void onPlainText(String text) {
        }

        @Override
        public void onMarkup(WikiTokenizer wikiTokenizer) {
        }

        @Override
        public void onWikiLink(WikiTokenizer wikiTokenizer) {
        }

        @Override
        public void onNewline(WikiTokenizer wikiTokenizer) {
        }

        @Override
        public void onFunction(WikiTokenizer tokenizer, String functionName, List<String> functionPositionArgs, Map<String, String> functionNamedArgs) {
        }

        @Override
        public void onHeading(WikiTokenizer wikiTokenizer) {
        }

        @Override
        public void onListItem(WikiTokenizer wikiTokenizer) {
        }

        @Override
        public void onComment(WikiTokenizer wikiTokenizer) {
        }

        @Override
        public void onHtml(WikiTokenizer wikiTokenizer) {
        }
    }
}

