// Copyright (C) 2008 Simon Mendoza
// This file is part of gtk-sudoku.
//
// gtk-sudoku is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// gtk-sudoku is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with gtk-sudoku.  If not, see <http://www.gnu.org/licenses/>.

#include "sudoku.h"
#include "solver.h"
#include "generator.h"

void popup_msg(const char *title,
			   const char *message)
{
	GtkWidget *msg;
	GtkWidget *hbox;
	GtkWidget *label;

	msg = gtk_dialog_new_with_buttons(
			title,
			GTK_WINDOW(window),
			GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR, 
			GTK_STOCK_OK, GTK_RESPONSE_NONE, NULL);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(msg)->vbox), hbox);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 8);
	gtk_widget_show(hbox);

	label = gtk_label_new(message);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	g_signal_connect(G_OBJECT(msg), "response", 
			G_CALLBACK(gtk_widget_destroy), NULL);
	gtk_widget_show(msg);
}

gboolean on_click(GtkWidget      *widget,
				  GdkEventButton *event,
				  gpointer        data)
{
	char value_f[50];

	if(!designing_sudoku && !game_started) return FALSE;

	if(sudoku.last_square_clicked != NULL
	   && sudoku.last_square_clicked->editable) {
		gtk_widget_hide(sudoku.last_square_clicked->entry);
		gtk_widget_show(sudoku.last_square_clicked->label);
	}
	switch(event->button) {
	case 1: // left-click
		if(SUDOKU_SQUARE(data)->editable) {
			gtk_widget_hide(SUDOKU_SQUARE(data)->label);
			gtk_widget_show(SUDOKU_SQUARE(data)->entry);
			gtk_widget_grab_focus(SUDOKU_SQUARE(data)->entry);
			// when focused, entry is automatically selected;
			// this deselects and places cursor after a digit
			gtk_editable_select_region(
					GTK_EDITABLE(SUDOKU_SQUARE(data)->entry), 1, 1);
			gtk_widget_modify_font((SUDOKU_SQUARE(data)->label), pango_font_description_from_string ("sans 20"));
                        gtk_widget_modify_font((SUDOKU_SQUARE(data)->entry), pango_font_description_from_string ("sans 20"));

		}
		sudoku.last_square_clicked = data;
		break;
	case 2: // middle-click
		if(!SUDOKU_SQUARE(data)->editable) return FALSE;
		gtk_entry_set_text(GTK_ENTRY(SUDOKU_SQUARE(data)->entry), "");
		gtk_label_set_text(GTK_LABEL(SUDOKU_SQUARE(data)->label), "");
		gtk_widget_modify_font((SUDOKU_SQUARE(data)->label), pango_font_description_from_string ("sans 20"));
		gtk_widget_modify_font((SUDOKU_SQUARE(data)->entry), pango_font_description_from_string ("sans 20"));
		++empty_squares;
		break;
	case 3: // right-click
		if(!SUDOKU_SQUARE(data)->editable || sudoku.last_square_clicked == NULL)
			return FALSE;
		gtk_entry_set_text(GTK_ENTRY(SUDOKU_SQUARE(data)->entry),
		gtk_entry_get_text(GTK_ENTRY(sudoku.last_square_clicked->entry)) );
		snprintf(value_f, 50, 
			"<span foreground=\"#009999\">%s</span>",
			gtk_label_get_text(GTK_LABEL(sudoku.last_square_clicked->label)));
		gtk_label_set_markup(GTK_LABEL(SUDOKU_SQUARE(data)->label), value_f);
		gtk_widget_modify_font((SUDOKU_SQUARE(data)->label), pango_font_description_from_string ("sans 20"));
		gtk_widget_modify_font((SUDOKU_SQUARE(data)->entry), pango_font_description_from_string ("sans 20"));
		break;
	}
	return FALSE;
}

gboolean on_enter(GtkWidget *widget, 
				  GdkEvent  *event, 
				  gpointer   data)
{
	gtk_widget_set_state(data, GTK_STATE_PRELIGHT);
	gtk_widget_set_state(widget, GTK_STATE_PRELIGHT);
	return FALSE;
}

gboolean on_leave(GtkWidget *widget, 
				  GdkEvent  *event, 
				  gpointer   data)
{
	gtk_widget_set_state(data, GTK_STATE_NORMAL);
	gtk_widget_set_state(widget, GTK_STATE_NORMAL);
	return FALSE;
}

void on_close(GtkWidget *widget, 
			  gpointer   data)
{
	register size_t row;

	if(status_bar_msg != NULL) {
		gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 1);
		g_free(status_bar_msg);
		g_mutex_unlock(mutex);
	}

	for(row = 0; row < GRID_SIZE; ++row)
		free(sudoku.square[row]);
	free(sudoku.square);
	sudoku.last_square_clicked = NULL;
	gtk_main_quit();
}

void verify_quit(GtkWidget *widget, gpointer data)
{
    GtkWidget   *dialog = NULL;

    dialog = gtk_message_dialog_new( GTK_WINDOW(window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_QUESTION,
                                     GTK_BUTTONS_YES_NO,
                                     "Are you sure you want to quit?" );
    gtk_window_set_deletable( GTK_WINDOW(dialog), FALSE );
    gint result;
    result = gtk_dialog_run( GTK_DIALOG(dialog) );
    if (result == GTK_RESPONSE_YES) {
        gtk_widget_destroy( dialog );
        gtk_widget_destroy( window );
	}
    else {
        gtk_widget_destroy( dialog );
        }
}

void on_backspace(GtkEntry *entry, 
				  gpointer  data)
{
	gtk_label_set_text(GTK_LABEL(SUDOKU_SQUARE(data)->label), "");
	++empty_squares;
}

bool rows_check_out(bool *values)
{
	register int row;
	register int col;
	const char *val;
	for(row = 0; row < GRID_SIZE; ++row) {
		for(col = 0; col < GRID_SIZE; ++col)
			values[col] = false;
		for(col = 0; col < GRID_SIZE; ++col) {
			val = gtk_label_get_text(GTK_LABEL(sudoku.square[row][col].label));
			if(*val == '\0')
				return false;
			switch(BLOCK_SIZE) {
			case 3: values[*val-'1'] = true; break;
			case 4: if(isdigit(*val))
						values[*val-'0'] = true;
					else
						values[*val-'A'+10] = true;
					break;
			}
		}
		for(col = 0; col < GRID_SIZE; ++col)
			if(!values[col])
				return false;
	}
	return true;
}

bool cols_check_out(bool *values)
{
	register int row;
	register int col;
	const char *val;
	for(col = 0; col < GRID_SIZE; ++col) {
		for(row = 0; row < GRID_SIZE; ++row)
			values[row] = false;
		for(row = 0; row < GRID_SIZE; ++row) {
			val = gtk_label_get_text(GTK_LABEL(sudoku.square[row][col].label));
			if(*val == '\0')
				return false;
			switch(BLOCK_SIZE) {
			case 3: values[*val-'1'] = true; break;
			case 4: if(isdigit(*val))
						values[*val-'0'] = true;
					else
						values[*val-'A'+10] = true;
					break;
		} }
		for(row = 0; row < GRID_SIZE; ++row)
			if(!values[row])
				return false;
	}
	return true;
}

bool correct_sudoku(void)
{
	bool *values = (bool *)malloc(GRID_SIZE * sizeof(bool));
	if(values == NULL) {
		fputs("  * Out of memory!\n", stderr);
		return false;
	}

	if(!rows_check_out(values))
		return false;
	if(!cols_check_out(values))
		return false;

	free(values);
	return true;
}

void on_insert(GtkEditable *editable, 
			   gchar       *number, 
			   gint         length, 
			   gint        *position,
			   gpointer     data)
{
	char number_f[40];          // html-formatted number to go into label

	if(BLOCK_SIZE == 3 && (*number == '0' || !isdigit(*number))) {
		*number = '\0'; return;
	}
	else if(!isxdigit(*number = toupper(*number))) {
		*number = '\0'; return;
	}

	snprintf(number_f, 40, "<span foreground=\"#009999\">%s</span>", number);
	gtk_label_set_markup(GTK_LABEL(SUDOKU_SQUARE(data)->label), number_f);
	gtk_widget_modify_font((SUDOKU_SQUARE(data)->label), pango_font_description_from_string ("sans 20"));
	gtk_widget_modify_font((SUDOKU_SQUARE(data)->entry), pango_font_description_from_string ("sans 20"));


	if(*position == 1) {
		*position = 0;
		gtk_editable_delete_text(editable, 0, 1);
	}
	else if(game_started) {
		--empty_squares;
		if(empty_squares == 0 && correct_sudoku()) {
			game_started = false;
			popup_msg("A Winner!!","It's solved!  Congratulations! \\o/");
	} }
}

void save_game(GtkWidget *toolbutton, 
			   gpointer   data)
{
	if(*save_file == '\0') {
		save_as(toolbutton, NULL);
		return;
	}
	if(designing_sudoku) {
		popup_msg("Whoa There!", "Finish up your sudoku before saving. =P");
		return;
	}
	FILE *sudoku_file = fopen(save_file, "w");
	if(sudoku_file != NULL) {
		register int row;
		register int col;
		const char *strval;

		for(row = 0; row < GRID_SIZE; ++row)
		for(col = 0; col < GRID_SIZE; ++col) {
			if(sudoku.square[col][row].editable) {
				strval = gtk_entry_get_text(
						GTK_ENTRY(sudoku.square[col][row].entry));
				if(*strval == '\0')
					fputc('.', sudoku_file);
				else
					fputc(*strval, sudoku_file);
			}
			else {
				strval = gtk_label_get_text(
						GTK_LABEL(sudoku.square[col][row].label));
				if(isdigit(*strval))
					fputc(*strval - '0' + 'K', sudoku_file);
				else
					fputc(*strval - 'A' + 'U', sudoku_file);
		} }
		fclose(sudoku_file);

		// display status bar message
		g_mutex_lock(mutex);
		status_bar_msg = g_strdup_printf("Saving to %s...", save_file);
		gtk_statusbar_push(GTK_STATUSBAR(statusbar), 1, status_bar_msg);
		g_thread_create(clear_statusbar_msg, (gpointer)3000000, FALSE, NULL);
	}
	else
		popup_msg("Error!", "Could not save to file...");
}

void save_as(GtkWidget *widget, 
			 gpointer   data)
{
	GtkWidget *file_chooser;

	if(designing_sudoku) {
		popup_msg("Whoa There!", "Finish up your sudoku before saving. =P");
		return;
	}

	file_chooser = gtk_file_chooser_dialog_new(
			"Save As...",
			GTK_WINDOW(window),
			GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
	
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(file_chooser),"/media/mmcblk0p1/Programs/_sudoku/games/");
	gtk_file_chooser_set_do_overwrite_confirmation(
			GTK_FILE_CHOOSER(file_chooser), TRUE);

	// if a file was chosen
	if(gtk_dialog_run(GTK_DIALOG(file_chooser)) == GTK_RESPONSE_ACCEPT) {
		register int row;
		register int col;
		FILE        *file;
		const char  *value;
		gchar       *sudoku_file;

		sudoku_file = 
			gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_chooser));

		file = fopen(sudoku_file, "w");
		// if opening the specified file was successful
		if(file != NULL) {
			strncpy(save_file, sudoku_file, FILE_SIZE-1);

			for(row = 0; row < GRID_SIZE; ++row)
			for(col = 0; col < GRID_SIZE; ++col) {
				if(sudoku.square[col][row].editable) {
					value = gtk_entry_get_text(
							GTK_ENTRY(sudoku.square[col][row].entry));
					if(*value == '\0')
						fputc('.', file);
					else
						fputc(*value, file);
				}
				else {
					value = gtk_label_get_text(
							GTK_LABEL(sudoku.square[col][row].label));
					if(isdigit(*value))
						fputc(*value - '0' + 'K', file);
					else
						fputc(*value - 'A' + 'U', file);
			} }
			fclose(file);

			// add message to statusbar
			g_mutex_lock(mutex);
			status_bar_msg = g_strdup_printf("Saving to %s...", save_file);
			gtk_statusbar_push(GTK_STATUSBAR(statusbar), 1, status_bar_msg);
			g_thread_create(clear_statusbar_msg, (gpointer)3000000,FALSE, NULL);
		}
		// if opening the specified file was not successful
		else {
			popup_msg("Error!", "Could not save to file...");
		}
		g_free(sudoku_file);
	}
	gtk_widget_destroy(file_chooser);
}

void open_game(GtkWidget *toolbutton, 
			   GtkWidget *done_btn)
{
	GtkWidget *file_chooser;

	file_chooser = gtk_file_chooser_dialog_new(
			"Load Sudoku Puzzle",
			GTK_WINDOW(window), 
			GTK_FILE_CHOOSER_ACTION_OPEN, 
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
			GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);

	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(file_chooser),"/media/mmcblk0p1/Programs/_sudoku/games/");

	if(gtk_dialog_run(GTK_DIALOG(file_chooser)) == GTK_RESPONSE_OK) {
		if(designing_sudoku) {
			g_thread_create(clear_statusbar_msg, (gpointer)0, FALSE, NULL);
			gtk_widget_set_sensitive(done_btn, FALSE);
			designing_sudoku = false;
		}
		gchar *sudoku_file;
		sudoku_file = 
			gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_chooser));
		load_file(sudoku_file);
		g_free(sudoku_file);
	}

	gtk_widget_destroy(file_chooser);
}

gboolean load_file(char *input_file)
{
	FILE *sudoku_file;
	char number[2];         // sudoku square value as string
	char number_f[40];      // same sudoku square value formatted for label
	register size_t row;
	register size_t col;

	if( (sudoku_file = fopen(input_file, "r")) == NULL) {
		char *message;
		message = g_strdup_printf("Unable to open %s!", input_file);
		popup_msg("?", message);
		g_free(message);
		return FALSE;
	}

	strncpy(save_file, input_file, FILE_SIZE-1);

	empty_squares = 0;
	game_started = false;
	// clear previous values (if any)
	for(row = 0; row < GRID_SIZE; ++row)
	for(col = 0; col < GRID_SIZE; ++col) {
		gtk_entry_set_text(GTK_ENTRY(sudoku.square[row][col].entry), "");
		gtk_label_set_text(GTK_LABEL(sudoku.square[row][col].label), "");
		sudoku.square[row][col].editable = TRUE;
	}

	// NUMBER will be treated as a string to place within sudoku square entry
	number[1] = '\0';

	for(row = 0; row < GRID_SIZE; ++row)
	for(col = 0; col < GRID_SIZE; ++col) {
		if( (*number = fgetc(sudoku_file)) == EOF ) {
			if(BLOCK_SIZE == 4)
				popup_msg("!!", "Your file's too short, dude.\n"
						"It's probably a 9x9 sudoku.");
			else
				popup_msg("!!", "Your file's missing some values, dude.");
			if(fclose(sudoku_file) != 0)
				g_print("  * Unable to close %s.\n", input_file);
			return FALSE;
		}
		// color format NUMBER_F depending on NUMBER value
		// also change NUMBER appropriately if it flags a non-editable square
		switch(*number) {
			case '1': case '2': case '3':
			case '4': case '5': case '6':
			case '7': case '8': case '9':
				snprintf(number_f, 40,
						"<span foreground=\"#009999\">%c</span>", *number);
				break;
			case '0': case 'A': case 'B':
			case 'C': case 'D': case 'E':
			case 'F':
				if(BLOCK_SIZE != 4) {
					popup_msg("!!", "Incorrect file format");
					return FALSE;
				}
				snprintf(number_f, 40,
						"<span foreground=\"#009999\">%c</span>", *number);
				break;
			case 'L': case 'M': case 'N':
			case 'O': case 'P': case 'Q':
			case 'R': case 'S': case 'T':
				*number = *number - 'L' + '1';
				snprintf(number_f, 40,
						"<span foreground=\"#000000\">%c</span>", *number);
				sudoku.square[col][row].editable = FALSE;
				break;
			case 'K':
				if(BLOCK_SIZE != 4) {
					popup_msg("!!", "Incorrect file format");
					return FALSE;
				}
				*number = '0';
				snprintf(number_f, 40,
						"<span foreground=\"#000000\">%c</span>", *number);
				sudoku.square[col][row].editable = FALSE;
				break;
			case 'U': case 'V': case 'W':
			case 'X': case 'Y': case 'Z':
				*number = *number - 'U' + 'A';
				snprintf(number_f, 40,
						"<span foreground=\"#000000\">%c</span>", *number);
				sudoku.square[col][row].editable = FALSE;
				break;
			// treat all other input as a blank square
			default:
				++empty_squares;
				continue;
		}
		gtk_entry_set_text(GTK_ENTRY(sudoku.square[col][row].entry), number);
		gtk_label_set_markup(GTK_LABEL(sudoku.square[col][row].label),number_f);
	}

	game_started = true;

	if(fclose(sudoku_file) != 0)
		g_print("  * Unable to close %s.\n", input_file);
	
	g_mutex_lock(mutex);
	status_bar_msg = g_strdup_printf("Opening %s...", save_file);
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), 1, status_bar_msg);
	g_thread_create(clear_statusbar_msg, (gpointer)3000000, FALSE, NULL);

	return TRUE;
}

gboolean make_sudoku_board(GtkBox *box)
{
	GtkWidget *board;           // table in which to place the sudoku squares
	GtkWidget *frame;           // placed around each sudoku square
	GtkWidget *vbox;
	register size_t row;        // sudoku square locations
	register size_t col;

	// set up table
	board = gtk_table_new(GRID_SIZE, GRID_SIZE, TRUE);
	gtk_box_pack_start(GTK_BOX(box), board, TRUE, TRUE, 0);
	for(row = BLOCK_SIZE-1; row < GRID_SIZE; row += BLOCK_SIZE) {
		gtk_table_set_row_spacing(GTK_TABLE(board), row, 2);
		gtk_table_set_col_spacing(GTK_TABLE(board), row, 2);
	}
	gtk_widget_show(board);

	// allocate mem for sudoku squares
	sudoku.square = malloc(GRID_SIZE * sizeof(Square *));
	if(sudoku.square == NULL) {
		fputs("  * Out of memory!\n", stderr);
		return FALSE;
	}
	for(row = 0; row < GRID_SIZE; ++row) {
		sudoku.square[row] = (Square*)malloc(GRID_SIZE * sizeof(Square));
		if(sudoku.square[row] == NULL) {
			fputs("  * Out of memory!\n", stderr);
			return FALSE;
	} }

	// initialize sudoku squares
	for(row = 0; row < GRID_SIZE; ++row)
	for(col = 0; col < GRID_SIZE; ++col) {
		// set up the bordering frame for each sudoku square
		frame = gtk_frame_new(NULL);
		GTK_FRAME(frame)->shadow_type = GTK_SHADOW_IN;
		gtk_widget_set_size_request(frame, SQUARE_SIZE, SQUARE_SIZE);
		gtk_table_attach(GTK_TABLE(board), frame, row, row+1, col, col+1,
				GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
		gtk_widget_show(frame);

		// set up the eventbox
		sudoku.square[row][col].eventbox = gtk_event_box_new();
		gtk_widget_set_name(sudoku.square[row][col].eventbox, "sudoku_square");
		gtk_widget_set_events(sudoku.square[row][col].eventbox,
				GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
		gtk_container_add(GTK_CONTAINER(frame), 
				sudoku.square[row][col].eventbox);

		// set up vbox
		vbox = gtk_vbox_new(TRUE, 0);
		gtk_container_add(
				GTK_CONTAINER(sudoku.square[row][col].eventbox), vbox);
		gtk_widget_show(vbox);

		// set up label
		sudoku.square[row][col].label = gtk_label_new(NULL);
		gtk_widget_set_name(sudoku.square[row][col].label, "sudoku_label");
		gtk_box_pack_start(
				GTK_BOX(vbox), sudoku.square[row][col].label, TRUE, TRUE, 0);
		gtk_widget_show(sudoku.square[row][col].label);

		// set up the entry
		sudoku.square[row][col].entry = gtk_entry_new();
		gtk_widget_set_name(sudoku.square[row][col].entry, "sudoku_entry");
		gtk_entry_set_has_frame(
				GTK_ENTRY(sudoku.square[row][col].entry), FALSE);
		gtk_entry_set_width_chars(GTK_ENTRY(sudoku.square[row][col].entry), 1);
		gtk_entry_set_max_length(GTK_ENTRY(sudoku.square[row][col].entry), 1);
		gtk_entry_set_alignment(GTK_ENTRY(sudoku.square[row][col].entry), 0.5);
		gtk_box_pack_start(GTK_BOX(vbox), sudoku.square[row][col].entry, 
				TRUE, TRUE, 0);

		// set up signals now that eventbox and entry widgets have been created
		g_signal_connect(G_OBJECT(sudoku.square[row][col].entry),
				"backspace", G_CALLBACK(on_backspace), 
				SUDOKU_SQUARE(&sudoku.square[row][col]));
		g_signal_connect(G_OBJECT(sudoku.square[row][col].entry), 
				"insert-text", G_CALLBACK(on_insert), 
				SUDOKU_SQUARE(&sudoku.square[row][col]));
		g_signal_connect(G_OBJECT(sudoku.square[row][col].eventbox),
				"button_press_event", G_CALLBACK(on_click), 
				SUDOKU_SQUARE(&sudoku.square[row][col]));
		g_signal_connect(G_OBJECT(sudoku.square[row][col].eventbox), 
				"enter_notify_event", G_CALLBACK(on_enter), 
				sudoku.square[row][col].entry);
		g_signal_connect(G_OBJECT(sudoku.square[row][col].eventbox), 
				"leave_notify_event", G_CALLBACK(on_leave), 
				sudoku.square[row][col].entry);
		g_signal_connect(G_OBJECT(sudoku.square[row][col].entry), 
				"enter_notify_event", G_CALLBACK(on_enter), 
				sudoku.square[row][col].eventbox);
		g_signal_connect(G_OBJECT(sudoku.square[row][col].entry),
				"leave_notify_event", G_CALLBACK(on_leave), 
				sudoku.square[row][col].eventbox);

		gtk_widget_show(sudoku.square[row][col].eventbox);
		// do not show sudoku.square[row][col].entry

		sudoku.square[row][col].editable = TRUE;
	}
	// by default no square has focus
	sudoku.last_square_clicked = NULL;
	return TRUE;
}

gpointer clear_statusbar_msg(gpointer time)
{
	g_usleep((gulong)time);
	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 1);
	g_free(status_bar_msg);
	status_bar_msg = NULL;
	g_mutex_unlock(mutex);
	g_thread_exit(NULL);
	return NULL;
}

void reset_game(GtkToolButton *toolbutton, 
				gpointer       data)
{
	register int row;
	register int col;

	if(designing_sudoku) {
		for(row = 0; row < GRID_SIZE; ++row)
		for(col = 0; col < GRID_SIZE; ++col) {
			sudoku.square[row][col].editable = TRUE;
			gtk_label_set_text(GTK_LABEL(sudoku.square[row][col].label), "");
			gtk_entry_set_text(GTK_ENTRY(sudoku.square[row][col].entry), "");
		}
		empty_squares = GRID_SIZE * GRID_SIZE;
		return;
	}

	game_started = false;
	empty_squares = 0;
	for(row = 0; row < GRID_SIZE; ++row)
	for(col = 0; col < GRID_SIZE; ++col) {
		if(sudoku.square[row][col].editable) {
			gtk_label_set_text(GTK_LABEL(sudoku.square[row][col].label), "");
			gtk_entry_set_text(GTK_ENTRY(sudoku.square[row][col].entry), "");
			++empty_squares;
	} }
	game_started = true;
}

void about_game(GtkWidget *toolbutton, 
				gpointer   data)
{
	GtkWidget *about_dialog;
	GtkWidget *vbox;
	GdkPixbuf *pix;
	GError *error = NULL;
	GtkWidget *logo;
	PangoFontDescription *font;
	GtkWidget *program_version;
	GtkWidget *author_label;

	about_dialog = gtk_dialog_new_with_buttons(
			"About gtk-sudoku", GTK_WINDOW(window),
			GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL,
			GTK_STOCK_CLOSE, GTK_RESPONSE_NONE,
			NULL);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_box_pack_start(
			GTK_BOX(GTK_DIALOG(about_dialog)->vbox), vbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 8);
	gtk_widget_show(vbox);

	pix = gdk_pixbuf_new_from_file("images/sudoku-about.png", &error);
	if(error == NULL) {         // no error loading image
		logo = gtk_image_new_from_pixbuf(pix);
		gtk_box_pack_start(GTK_BOX(vbox), logo, TRUE, TRUE, 0);
		gtk_widget_show(logo);
	}

	font = pango_font_description_new();
	pango_font_description_set_family(font, "Romeodn");
	pango_font_description_set_size(font, 13 * PANGO_SCALE);
	pango_font_description_set_weight(font, PANGO_WEIGHT_BOLD);

	program_version = gtk_label_new("gtk-sudoku 0.7.2");
	gtk_widget_modify_font(program_version, font);
	gtk_box_pack_start(GTK_BOX(vbox), program_version, TRUE, TRUE, 0);
	gtk_widget_show(program_version);

	pango_font_description_set_family(font, "Andale Mono");
	pango_font_description_set_size(font, 7 * PANGO_SCALE);
	pango_font_description_set_weight(font, PANGO_WEIGHT_NORMAL);

	author_label = gtk_label_new("2007-2008 Simon Mendoza");
	gtk_widget_modify_font(author_label, font);
	gtk_box_pack_start(GTK_BOX(vbox), author_label, TRUE, TRUE, 0);
	gtk_widget_show(author_label);

	gtk_dialog_run(GTK_DIALOG(about_dialog));
	gtk_widget_destroy(about_dialog);
}

void user_done_creating(GtkButton *button,
						gpointer   data)
{
	register size_t row;
	register size_t col;
	char number_f[40];    // formatted sudoku square value to become uneditable

	if(sudoku.last_square_clicked != NULL) {
		gtk_widget_hide(sudoku.last_square_clicked->entry);
		gtk_widget_show(sudoku.last_square_clicked->label);
	}

	// format the new label to be black (thereby visually marking it as 
	// uneditable)
	strcpy(number_f, "<span foreground=\"#000000\">q</span>");

	for(row = 0; row < GRID_SIZE; ++row)
	for(col = 0; col < GRID_SIZE; ++col)
	if(*gtk_entry_get_text(GTK_ENTRY(sudoku.square[row][col].entry)) != '\0' ) {
		// set location of q (27) to the current value
		number_f[27] = 
				*gtk_entry_get_text(GTK_ENTRY(sudoku.square[row][col].entry));
		gtk_label_set_markup(GTK_LABEL(sudoku.square[row][col].label),number_f);
		sudoku.square[row][col].editable = FALSE;
	}

	g_thread_create(clear_statusbar_msg, (gpointer)500000, FALSE, NULL);
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(button), "");
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	designing_sudoku = false;
	game_started = true;
}

void create_my_own(GtkMenuItem *menuitem,
				   GtkWidget    *done_btn)
{
	if(designing_sudoku) {
		popup_msg(";-)", "You're already designing one sudoku, silly!");
		return;
	}
	designing_sudoku = true;
	game_started = false;
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(done_btn), "All done?");
	gtk_widget_set_sensitive(done_btn, TRUE);

	reset_game(NULL, NULL);

	g_mutex_lock(mutex);
	status_bar_msg = g_strdup_printf("Click when finished -->");
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), 1, status_bar_msg);
}

void make_toolbar(GtkBox *box, GtkWidget *done_btn)
{
	GtkWidget   *toolbar;
	GtkToolItem *new_btn;
	GtkWidget   *new_menu;
	GtkWidget   *new_generate_stock, *new_generate_img, *new_generate;
	GtkWidget   *new_create_stock, *new_create_img, *new_create;
	GtkToolItem *load_btn;
	GtkToolItem *save_btn;
	GtkWidget   *save_menu;
	GtkWidget   *save;
	GtkWidget   *save_as_item;
	GtkToolItem *separator;
	GtkToolItem *verify_btn;
	GtkToolItem *clear_btn;
	GtkToolItem *solve_btn;
	GtkToolItem *about_btn;
	GtkToolItem *close_btn;

	toolbar = gtk_toolbar_new();
	gtk_widget_set_name(toolbar, "toolbar");
	gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), GTK_TOOLBAR_BOTH);
	gtk_toolbar_set_orientation (GTK_TOOLBAR (toolbar), GTK_ORIENTATION_HORIZONTAL);
  	gtk_container_set_border_width (GTK_CONTAINER (toolbar), 5);

	gtk_box_pack_start(box, toolbar, FALSE, FALSE, 0);

	// new menu ----------------
	new_btn = gtk_menu_tool_button_new_from_stock(GTK_STOCK_NEW);
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(new_btn), "New Game");
	g_signal_connect(G_OBJECT(new_btn), "clicked",
			G_CALLBACK(generate_game), done_btn);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), new_btn, 0);
	gtk_widget_show(GTK_WIDGET(new_btn));

	new_menu = gtk_menu_new();
	gtk_menu_tool_button_set_menu(GTK_MENU_TOOL_BUTTON(new_btn), new_menu);

	// stock item from which to take image
	new_generate_stock = 
		gtk_image_menu_item_new_from_stock(GTK_STOCK_NEW, NULL);
	new_generate_img = 
		gtk_image_menu_item_get_image(GTK_IMAGE_MENU_ITEM(new_generate_stock));
	// increase reference count by 1 to survive transition from stock item to
	// my own item
	g_object_ref(new_generate_img);
	// remove image from stock item
	gtk_container_remove(GTK_CONTAINER(new_generate_stock), new_generate_img);
	new_generate = gtk_image_menu_item_new_with_mnemonic("_Generate New");
	g_signal_connect(G_OBJECT(new_generate), "activate",
			G_CALLBACK(generate_game), NULL);
	gtk_image_menu_item_set_image(
			GTK_IMAGE_MENU_ITEM(new_generate), new_generate_img);
	// bring count down by 1
	g_object_unref(new_generate_img);
	gtk_menu_shell_append(GTK_MENU_SHELL(new_menu), new_generate);
	gtk_widget_show(new_generate);

	// do the same for this image menu item
	new_create_stock = gtk_image_menu_item_new_from_stock(GTK_STOCK_NEW, NULL);
	new_create_img = gtk_image_menu_item_get_image(
			GTK_IMAGE_MENU_ITEM(new_create_stock));
	g_object_ref(new_create_img);
	gtk_container_remove(GTK_CONTAINER(new_create_stock), new_create_img);
	new_create = gtk_image_menu_item_new_with_mnemonic("_Create My Own");
	g_signal_connect(G_OBJECT(new_create), "activate",
			G_CALLBACK(create_my_own), done_btn);
	gtk_image_menu_item_set_image(
			GTK_IMAGE_MENU_ITEM(new_create), new_create_img);
	g_object_unref(new_create_img);
	gtk_menu_shell_append(GTK_MENU_SHELL(new_menu), new_create);
	gtk_widget_show(new_create);
	// --------------------

	load_btn = gtk_tool_button_new_from_stock(GTK_STOCK_OPEN);
	g_signal_connect(G_OBJECT(load_btn), "clicked", 
			G_CALLBACK(open_game), done_btn);
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(load_btn), "Open Game");
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), load_btn, 1);
	gtk_widget_show(GTK_WIDGET(load_btn));

	// save menu ----------------
	save_btn = gtk_menu_tool_button_new_from_stock(GTK_STOCK_SAVE);
	g_signal_connect(G_OBJECT(save_btn), "clicked",
			G_CALLBACK(save_game), NULL);
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(save_btn), "Save Game");
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), save_btn, 2);
	gtk_widget_show(GTK_WIDGET(save_btn));

	save_menu = gtk_menu_new();
	gtk_menu_tool_button_set_menu(GTK_MENU_TOOL_BUTTON(save_btn), save_menu);

	save = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(save_menu), save);
	g_signal_connect(G_OBJECT(save), "activate",
			G_CALLBACK(save_game), NULL);
	gtk_widget_show(save);

	save_as_item = gtk_image_menu_item_new_from_stock(GTK_STOCK_SAVE_AS, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(save_menu), save_as_item);
	g_signal_connect(G_OBJECT(save_as_item), "activate",
			G_CALLBACK(save_as), NULL);
	gtk_widget_show(save_as_item);
	// --------------------------

	separator = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(separator), TRUE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), separator, 3);
	gtk_widget_show(GTK_WIDGET(separator));

	verify_btn = gtk_tool_button_new_from_stock(GTK_STOCK_EXECUTE);
	g_signal_connect(G_OBJECT(verify_btn), "clicked",
			G_CALLBACK(verify_game), NULL);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(verify_btn), "Verify");
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(verify_btn), "Check Game");
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), verify_btn, 4);
	gtk_widget_show(GTK_WIDGET(verify_btn));

	clear_btn = gtk_tool_button_new_from_stock(GTK_STOCK_CLEAR);
	g_signal_connect(G_OBJECT(clear_btn), "clicked", 
			G_CALLBACK(reset_game), NULL);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(clear_btn), "Reset");
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(clear_btn), "Reset Game");
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), clear_btn, 5);
	gtk_widget_show(GTK_WIDGET(clear_btn));

	solve_btn = gtk_tool_button_new_from_stock(GTK_STOCK_APPLY);
	g_signal_connect(G_OBJECT(solve_btn), "clicked", 
			G_CALLBACK(solve_game), NULL);
	gtk_tool_button_set_label(GTK_TOOL_BUTTON(solve_btn), "Solve!");
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(solve_btn), "Give up?");
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), solve_btn, 6);
	gtk_widget_show(GTK_WIDGET(solve_btn));

        close_btn = gtk_tool_button_new_from_stock(GTK_STOCK_QUIT);
        g_signal_connect(G_OBJECT(close_btn), "clicked",
                        G_CALLBACK(verify_quit), NULL);
        gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(close_btn), "Quit");
        gtk_toolbar_insert(GTK_TOOLBAR(toolbar), close_btn, 7);
        gtk_widget_show(GTK_WIDGET(close_btn));
        gtk_widget_show(toolbar);

	about_btn = gtk_tool_button_new_from_stock(GTK_STOCK_ABOUT);
	g_signal_connect(G_OBJECT(about_btn), "clicked",
			G_CALLBACK(about_game), NULL);
	gtk_tool_item_set_tooltip_text(GTK_TOOL_ITEM(about_btn), "About");
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), about_btn, 7);
	gtk_widget_show(GTK_WIDGET(about_btn));

	gtk_widget_show(toolbar);
}

void regular_or_super_sudoku(GdkPixbuf *icon)
{
	GtkWidget *difficulty_dialog;
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *regular;
	GtkWidget *super;

	difficulty_dialog = gtk_dialog_new_with_buttons(
						"Level", NULL, GTK_DIALOG_NO_SEPARATOR,
						GTK_STOCK_OK, GTK_RESPONSE_NONE, NULL);
	if(icon != NULL)
		gtk_window_set_icon(GTK_WINDOW(difficulty_dialog), icon);

	vbox = gtk_vbox_new(FALSE, 3);
	gtk_box_pack_start(
			GTK_BOX(GTK_DIALOG(difficulty_dialog)->vbox), vbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	gtk_widget_show(vbox);

	label = gtk_label_new("Select the level:");
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	regular = gtk_radio_button_new_with_label(
			NULL, "Regular sudoku (9x9)");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(regular), TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), regular, TRUE, TRUE, 0);
	gtk_widget_show(regular);

	super = gtk_radio_button_new_with_label_from_widget(
			GTK_RADIO_BUTTON(regular), "Super sudoku   (16x16)");
	gtk_box_pack_start(GTK_BOX(vbox), super, TRUE, TRUE, 0);
	gtk_widget_show(super);

	gtk_dialog_run(GTK_DIALOG(difficulty_dialog));

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(regular))) {
		BLOCK_SIZE = 3; GRID_SIZE = 9; SQUARE_SIZE = 41;
	}
	else {
		BLOCK_SIZE = 4; GRID_SIZE =16; SQUARE_SIZE = 31;
	}

	gtk_widget_destroy(difficulty_dialog);
}

int main(int   argc, 
		 char *argv[])
{
	GdkPixbuf *icon;
	GError *error = NULL;          // picks up image-loading errors
	GtkWidget *vbox, *hbox;
	GtkWidget *eventbox_statusbar; // for customization purposes (thru rc file)
	GtkWidget *eventbox_hbox;      // also for customization
	GtkToolItem *done;             // signals that user is done creating puzzle

	g_thread_init(NULL);
	gtk_init(&argc, &argv);

	gtk_rc_parse("gtk-sudoku.rc");

	icon = gdk_pixbuf_new_from_file("images/sudoku-icon.png", &error);
	if(error == NULL)
		regular_or_super_sudoku(icon);
	else
		regular_or_super_sudoku(NULL);

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	g_signal_connect(
			G_OBJECT(window), "destroy", G_CALLBACK(on_close), NULL);
	gtk_window_set_title(GTK_WINDOW(window), "gtk-sudoku");
	if(error == NULL)
		gtk_window_set_icon(GTK_WINDOW(window), icon);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(window), vbox);
	gtk_widget_show(vbox);

	statusbar = gtk_statusbar_new();
	done = gtk_tool_button_new_from_stock(GTK_STOCK_JUMP_TO);
	g_signal_connect(G_OBJECT(done), "clicked",
			G_CALLBACK(user_done_creating), NULL);
	gtk_widget_set_sensitive(GTK_WIDGET(done), FALSE);
	gtk_box_pack_end(GTK_BOX(statusbar), GTK_WIDGET(done), FALSE, FALSE, 0);
	gtk_widget_show(GTK_WIDGET(done));

	make_toolbar(GTK_BOX(vbox), GTK_WIDGET(done));

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);
	eventbox_hbox = gtk_event_box_new();
	gtk_widget_set_name(eventbox_hbox, "board");
	gtk_container_add(GTK_CONTAINER(eventbox_hbox), hbox);
	gtk_box_pack_start(GTK_BOX(vbox), eventbox_hbox, TRUE, TRUE, 0);
	gtk_widget_show(eventbox_hbox);
	gtk_widget_show(hbox);

	if(!make_sudoku_board(GTK_BOX(hbox))) {
		gtk_widget_destroy(window);
		exit(EXIT_FAILURE);
	}

	eventbox_statusbar = gtk_event_box_new();
	gtk_widget_set_name(eventbox_statusbar, "statusbar");
	gtk_container_add(GTK_CONTAINER(eventbox_statusbar), statusbar);
	gtk_box_pack_start(GTK_BOX(vbox), eventbox_statusbar, FALSE, FALSE, 0);
	gtk_widget_show(eventbox_statusbar);
	gtk_widget_show(statusbar);

	gtk_widget_show(window);

	mutex = g_mutex_new();
	gtk_main();
	g_mutex_free(mutex);

	return 0;
}
