# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2018,2019 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.76"  #Technical Tweaks after compiling with Python 3.8

import os, sys, apsw, re
from difflib import SequenceMatcher

from PyQt5.Qt import (Qt, QDialog, QGroupBox, QVBoxLayout, QGridLayout, QMargins,
                                       QTextOption, QDialogButtonBox, QPushButton, QTextEdit, QSize)

from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from polyglot.builtins import iteritems, unicode_type

#---------------------------------------------------------------------------------------------------------------------------------------
class CalibreSpySQLQueryBase(QDialog):
    def __init__(self, parent):
        QDialog.__init__(self, parent, Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint | Qt.WindowCloseButtonHint | Qt.WindowMinMaxButtonsHint)
    def resize_dialog(self):
        size = QSize()
        size.setWidth(700)
        size.setHeight(700)
        self.resize(size)
#---------------------------------------------------------------------------------------------------------------------------------------
class CalibreSpySQLQuery(CalibreSpySQLQueryBase):

    def __init__(self,icon,font,style_text,library_metadatadb_path,display_sqlquery_search_results,sql_prefs,return_new_local_pref_value):
        parent = None
        CalibreSpySQLQueryBase.__init__(self, parent)

        self.icon = icon
        self.library_metadatadb_path = library_metadatadb_path
        self.display_sqlquery_search_results = display_sqlquery_search_results
        self.sql_prefs = sql_prefs
        self.return_new_local_pref_value = return_new_local_pref_value

        self.setWindowTitle('CalibreSpy: SQL Query Search')
        self.setWindowIcon(icon)

        self.build_custom_column_dicts()

        tmp_list = []
        #~ for label,id in self.custom_column_label_dict.iteritems():
        for label,id in iteritems(self.custom_column_label_dict):
            if id in self.custom_column_datatype_dict:
                if id in self.custom_column_normalized_dict:
                    datatype = self.custom_column_datatype_dict[id]
                    normalized = self.custom_column_normalized_dict[id]
                    s =  "<br>" + bytes(id) + "---" + label + "---" + datatype
                    if normalized == 1:
                        s = s + "---" + "Normalized"
                    tmp_list.append(s)
        #END FOR
        tmp_list.sort()
        s = "<br>"
        for row in tmp_list:
            s = s + row
        #END FOR
        s1 = "<p style='white-space:wrap'>This is used to execute raw 'SELECT bookid' SQL.  Those books are then displayed by CalibreSpy."
        s1 = s1 + "The custom column information from table custom_columns for this library is shown below.  \
                        <br><br><b>Normalized</b>:          Table custom_column_N  has the values.  Columns:  id,value.   Table books_custom_column_N_link has the link of the value to the book.  Columns: id,book,value where value is the id of the value from table custom_column_N.\
                        <br><br><b>Not Normalized</b>:   Table custom_column_N  has both the value and the link to the book.  Columns:  id,book,value."
        s_tooltip = s1 + s
        del tmp_list
        del s
        del s1

        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        font.setBold(False)
        font.setPointSize(10)

        self.setStyleSheet(style_text)
        #~ self.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")
        #-----------------------------------------------------
        self.layout_frame = QGridLayout()
        self.layout_frame.setSpacing(0)
        self.layout_frame.setContentsMargins(QMargins(0,0,0,0));
        self.setLayout(self.layout_frame)

        self.setToolTip(s_tooltip)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.raw_sql_groupbox = QGroupBox('')
        self.raw_sql_groupbox.setToolTip(s_tooltip)
        self.raw_sql_groupbox.setMinimumWidth(690)
        self.raw_sql_groupbox.setMaximumWidth(690)
        self.raw_sql_groupbox.setMinimumHeight(200)
        self.layout_frame.addWidget(self.raw_sql_groupbox)

        self.raw_sql_layout = QVBoxLayout()
        self.raw_sql_layout.setAlignment(Qt.AlignCenter)
        self.raw_sql_groupbox.setLayout(self.raw_sql_layout)

        font.setPointSize(10)
        #-----------------------------------------------------
        self.top_buttonbox = QDialogButtonBox()
        self.top_buttonbox.setCenterButtons(True)
        self.raw_sql_layout.addWidget(self.top_buttonbox)
        #-----------------------------------------------------

        self.push_button_save_current_raw_sql_query = QPushButton(" ", self)
        self.push_button_save_current_raw_sql_query.setText("Save Current SQL")
        self.push_button_save_current_raw_sql_query.setToolTip("<p style='white-space:wrap'>Save the current SQL")
        self.push_button_save_current_raw_sql_query.clicked.connect(self.save_current_raw_sql_query)
        self.top_buttonbox.addButton(self.push_button_save_current_raw_sql_query,0)

        self.push_button_swap_raw_sql_query = QPushButton(" ", self)
        self.push_button_swap_raw_sql_query.setText("Swap Current/Saved SQL")
        self.push_button_swap_raw_sql_query.setToolTip("<p style='white-space:wrap'>Swap the current SQL and the saved SQL")
        self.push_button_swap_raw_sql_query.clicked.connect(self.swap_raw_sql_query)
        self.top_buttonbox.addButton(self.push_button_swap_raw_sql_query,0)

        self.push_button_load_saved_raw_sql_query = QPushButton(" ", self)
        self.push_button_load_saved_raw_sql_query.setText("Load Saved SQL")
        self.push_button_load_saved_raw_sql_query.setToolTip("<p style='white-space:wrap'>Load the previously saved SQL")
        self.push_button_load_saved_raw_sql_query.clicked.connect(self.load_saved_raw_sql_query)
        self.top_buttonbox.addButton(self.push_button_load_saved_raw_sql_query,0)

        self.push_button_example_raw_sql_query = QPushButton(" ", self)
        self.push_button_example_raw_sql_query.setText("Load Example SQL")
        self.push_button_example_raw_sql_query.setToolTip("<p style='white-space:wrap'>Load the example SQL")
        self.push_button_example_raw_sql_query.clicked.connect(self.load_example_raw_sql_query)
        self.top_buttonbox.addButton(self.push_button_example_raw_sql_query,0)

        font.setPointSize(12)
        font.setBold(False)

        self.raw_sql_qtextedit =  QTextEdit("")
        self.raw_sql_qtextedit.setAcceptRichText(True)
        self.raw_sql_qtextedit.setReadOnly(False)
        self.raw_sql_qtextedit.setFont(font)
        self.raw_sql_qtextedit.setWordWrapMode(QTextOption.WordWrap)
        self.raw_sql_qtextedit.clear()

        self.raw_sql_text_default = "SELECT book FROM comments WHERE text REGEXP '^.+$' "

        self.raw_sql_text = self.sql_prefs['SQL_QUERY_RAW_LAST_CURRENT']
        if not self.raw_sql_text > " ":
            self.raw_sql_text = self.raw_sql_text_default

        if not self.sql_prefs['SQL_QUERY_RAW_LAST_SAVED'] > " ":
            self.sql_prefs['SQL_QUERY_RAW_LAST_SAVED'] = self.raw_sql_text_default

        self.raw_sql_qtextedit.setPlainText(self.raw_sql_text)

        self.raw_sql_layout.addWidget(self.raw_sql_qtextedit)

        self.raw_sql_qtextedit.setToolTip("<p style='white-space:wrap'>Raw SQL.  The Rules: \
                                                                                                            <br><br>[1] Must start with SELECT \
                                                                                                            <br><br>[2] The very first column selected must be an integer book id (e.g. 'books.id' or 'books_authors_link.book') \
                                                                                                            <br><br>[3] May not contain SQLite Keywords that add, change, or delete data or objects \
                                                                                                            <br><br>[4] May not contain semi-colons \
                                                                                                            <br><br>[5] May not contain question marks \
                                                                                                            <br><br>[6] May not contain double quotes \
                                                                                                            <br><br>[7] Must otherwise comply with the syntax in: https://www.sqlite.org/lang_corefunc.html \
                                                                                                            <br><br> Suggestion: create a .txt file to keep your SQL history for future copy-and-paste ease. \
                                                                                                            <br><br>Refer to the window's ToolTips for a list of the current library's custom columns. \
                                                                                                            <br><br>SQLite Syntax questions?  Answer them at http://www.w3schools.com/sql/ or http://www.sqlcourse.com/.")

        #-----------------------------------------------------
        #-----------------------------------------------------
        self.push_button_initiate_raw_sql_query = QPushButton(" ", self)
        self.push_button_initiate_raw_sql_query.setText("Execute the Raw SQL Query [All Books]")
        self.push_button_initiate_raw_sql_query.clicked.connect(self.initiate_raw_sql_query)
        self.raw_sql_layout.addWidget(self.push_button_initiate_raw_sql_query)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.push_button_exit = QPushButton(" ", self)
        self.push_button_exit.setText("Exit")
        self.push_button_exit.clicked.connect(self.save_and_exit)
        self.raw_sql_layout.addWidget(self.push_button_exit)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize_dialog()
        #-----------------------------------------------------

        self.example_sql_text = \
        """
        SELECT id FROM books WHERE id IN (SELECT book FROM books_authors_link WHERE author IN (SELECT id FROM authors WHERE sort LIKE '%hugo%victor%' ) OR author IN (SELECT id FROM authors WHERE name LIKE '%victor%hugo%' )                                                 OR author IN (SELECT id FROM authors WHERE sort LIKE '%martin%george%r%r%' ) )                                                  AND has_cover = 0
        AND id IN (SELECT book FROM custom_column_19 WHERE value > 0  AND value < 10) AND id NOT IN (SELECT book FROM books_custom_column_4_link                                                                                                WHERE value IN (SELECT id FROM custom_column_4                                                                                                                                         WHERE value LIKE '%star wars%' AND value NOT NULL) )                                                 AND id IN (SELECT book FROM custom_column_8 WHERE value REGEXP '^.+$' )
        /* */
        /* */
        /* */
        /*Comments:  These are all of the valid initial SELECT statements:
        SELECT id
        SELECT book
        SELECT ALL id
        SELECT ALL book
        SELECT DISTINCT id
        SELECT DISTINCT book
        */
        /*Comments: The above REGEXP function will be evaluated after converting both 'value' and the regular expression itself to Unicode text. The re parameters to be used are re.IGNORECASE, re.DOTALL, re.MULTILINE, and re.escape. */
        /* */
        /*Comments: An example that finds all Series Names within a Title:  SELECT id FROM books CROSS JOIN (SELECT name FROM series) WHERE books.title LIKE '%'||name||'%'*/
        /* */
        /*Comments: An example that queries a non-Normalized custom column:  SELECT book FROM custom_column_11 WHERE value NOT NULL   */
        /* */
         /*Comments: An example that finds all books with Author Names that are a certain length:  SELECT id FROM books WHERE id IN (SELECT book FROM books_authors_link WHERE author IN (SELECT id FROM authors WHERE length(name) < 7 ))*/
         /* */
        /*Comments: An example that finds all books with Hugo Awards using the Multi-Column Search (MCS) plug-in special search tables: SELECT book FROM _mcs_authors_by_book,_mcs_book_awards WHERE award LIKE '%hugo%' AND subbytes(authorname,1,3) = subbytes(author,1,3)  AND SIMILARTO(authorname,author) > 0.80 AND _mcs_authors_by_book.book IN ( SELECT id FROM books WHERE books.id = _mcs_authors_by_book.book AND SIMILARTO(books.title,_mcs_book_awards.title) > 0.80) */
        /* */
        /*Comments: An example that finds ISBNs within Tags using the Multi-Column Search (MCS) plug-in special search tables:  SELECT book FROM _mcs_tags_by_book WHERE tagname LIKE '%978%' OR tagname LIKE '%045%' OR tagname LIKE '%isbn%' OR tagname REGEXP '^[0-9][-]*[0-9][-]*[0-9][-]*[0-9][-]*[0-9][-]*[0-9][-]*[0-9][-]*[0-9][-]*[0-9][-]*[0-9]+$'*/
        /* */
        """

        self.example_sql_text = self.example_sql_text.strip()

#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
    def save_current_raw_sql_query(self):
        self.sql_prefs['SQL_QUERY_RAW_LAST_CURRENT'] =  self.raw_sql_qtextedit.toPlainText()
        self.return_new_local_pref_value('SQL_QUERY_RAW_LAST_CURRENT',self.raw_sql_qtextedit.toPlainText(),update_db_now=False)
#--------------------------------------------------------------------------------------------------
    def load_saved_raw_sql_query(self):
        self.raw_sql_text = self.sql_prefs['SQL_QUERY_RAW_LAST_SAVED']
        self.raw_sql_qtextedit.setPlainText(self.raw_sql_text)
#--------------------------------------------------------------------------------------------------
    def swap_raw_sql_query(self):
        current = self.raw_sql_qtextedit.toPlainText()
        saved = self.sql_prefs['SQL_QUERY_RAW_LAST_SAVED']
        self.raw_sql_qtextedit.setPlainText(saved)
        self.sql_prefs['SQL_QUERY_RAW_LAST_CURRENT'] = saved
        self.return_new_local_pref_value('SQL_QUERY_RAW_LAST_CURRENT',saved,update_db_now=False)
        self.sql_prefs['SQL_QUERY_RAW_LAST_SAVED'] = current
        self.return_new_local_pref_value('SQL_QUERY_RAW_LAST_SAVED',current,update_db_now=False)
#--------------------------------------------------------------------------------------------------
    def load_example_raw_sql_query(self):
        self.raw_sql_qtextedit.setPlainText(self.example_sql_text)
        self.sql_prefs['SQL_QUERY_RAW_LAST_CURRENT'] =  self.raw_sql_qtextedit.toPlainText()
        self.return_new_local_pref_value('SQL_QUERY_RAW_LAST_CURRENT',self.raw_sql_qtextedit.toPlainText(),update_db_now=False)
#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
    def initiate_raw_sql_query(self):

        sql = self.raw_sql_qtextedit.toPlainText()

        lc_sql = bytes(sql.lower())
        if lc_sql.count("select") == 0:
            error_dialog(self, _('CalibreSpy'),_(("SQL must be for a SELECT query.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return
        elif lc_sql.count(" create ") > 0 or lc_sql.count(" drop ") > 0 or lc_sql.count(" insert ") > 0 or lc_sql.count(" update ") > 0 \
                                                          or lc_sql.count(" delete ") > 0 or lc_sql.count(" replace ") > 0 or lc_sql.count(" reindex ") > 0 \
                                                          or lc_sql.count(" pragma ") > 0 or lc_sql.count(" alter ") > 0 or lc_sql.count(" vacuum ") > 0 \
                                                          or lc_sql.count(" create ") > 0 or lc_sql.count(" begin ") > 0 or lc_sql.count(" commit ") > 0:
            error_dialog(self, _('CalibreSpy'),_(("SQL must be for a SELECT query.  CREATE, DROP, INSERT, UPDATE, REPLACE, DELETE and so forth are strictly forbidden.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return
        elif lc_sql.count("?") > 0:
            error_dialog(self, _('CalibreSpy'),_(("SQL may not contain a ? (Question Mark).<br><br>\
                                                                Execution Canceled. ")), show=True)
            return
        elif lc_sql.count('"') > 0:
            error_dialog(self, _('CalibreSpy'),_(("SQL may not contain Double Quotes.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return
        elif sql.count(";") > 0:
            error_dialog(self, _('CalibreSpy'),_(("SQL may not contain semi-colons.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return

        sql = sql.replace("select "," SELECT ")
        sql = sql.replace(" from "," FROM ")
        sql = sql.replace(" where "," WHERE ")
        sql = sql.replace(" all "," ALL ")
        sql = sql.replace(" by "," BY ")
        sql = sql.replace(" in "," IN ")
        sql = sql.replace(" like "," LIKE ")
        sql = sql.replace(" as "," AS ")
        sql = sql.replace(" and "," AND ")
        sql = sql.replace(" not "," NOT ")
        sql = sql.replace(" or "," OR ")
        sql = sql.replace(" null "," NULL ")
        sql = sql.replace(" exists "," EXISTS ")
        sql = sql.replace(" union "," UNION ")
        sql = sql.replace(" join "," JOIN ")
        sql = sql.replace(" inner "," INNER ")
        sql = sql.replace(" outer "," OUTER ")
        sql = sql.replace(" cross "," CROSS ")
        sql = sql.replace(" using "," USING ")
        sql = sql.replace(" group "," GROUP ")
        sql = sql.replace(" order "," ORDER ")
        sql = sql.replace(" distinct "," DISTINCT ")
        sql = sql.replace(" having "," HAVING ")
        sql = sql.replace(" between "," BETWEEN ")
        sql = sql.replace(" case "," CASE ")
        sql = sql.replace(" when "," WHEN ")
        sql = sql.replace(" end "," END ")
        sql = sql.replace(" else "," ELSE ")
        sql = sql.replace(" then "," THEN ")
        sql = sql.replace(" cast "," CAST ")
        sql = sql.replace(" regexp "," REGEXP ")
        sql = sql.replace(" similarto "," SIMILARTO ")
        sql = sql.replace(" similarto"," SIMILARTO")

        sql = sql.strip()

        self.raw_sql_qtextedit.setPlainText(sql)
        self.update()

        nl = sql.count("(")
        nr = sql.count(")")
        if nl != nr:
            error_dialog(self, _('CalibreSpy'),_(("SQL must have an equal number of left and right parentheses.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return

        if (not sql.startswith("SELECT id ")) and (not sql.startswith("SELECT book ")) \
            and (not sql.startswith("SELECT ALL id ")) and (not sql.startswith("SELECT ALL book ")) \
            and (not sql.startswith("SELECT DISTINCT id ")) and (not sql.startswith("SELECT DISTINCT book ")):
            error_dialog(self, _('CalibreSpy'),_(("SQL must start with: SELECT and the column to be returned must be either the integer 'id' or the integer 'book'.<br><br>\
                                                                Execution Canceled. ")), show=True)
            return

        self.sql_prefs['SQL_QUERY_RAW_LAST_CURRENT'] =  sql
        self.return_new_local_pref_value('SQL_QUERY_RAW_LAST_CURRENT',sql,update_db_now=False)

        self.return_new_local_pref_value('SQL_QUERY_RAW_LAST_SAVED',self.sql_prefs['SQL_QUERY_RAW_LAST_SAVED'],update_db_now=False)

        self.raw_sql_query_control(sql)
#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
    def raw_sql_query_control(self,mysql):

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        if mysql.count("REGEXP") > 0:
            func = "REGEXP"
            is_valid = self.apsw_create_user_functions(my_db,my_cursor,func)
        else:
            is_valid = True

        if mysql.count("SIMILARTO") > 0:
            func = "SIMILARTO"
            is_valid = self.apsw_create_user_functions(my_db,my_cursor,func)
        else:
            is_valid = True

        if is_valid:
            found_list = self.execute_raw_sql_query(my_db,my_cursor,mysql)
        else:
            found_list = []

        found_set = set(found_list)

        del found_list

        self.display_sqlquery_search_results(found_set)

        del found_set

#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
    def apsw_connect_to_library(self):
        try:
            my_db = apsw.Connection(self.library_metadatadb_path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", self.library_metadatadb_path)
            if DEBUG: print("error: ", bytes(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 25000;"      #milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
#--------------------------------------------------------------------------------------------------
    def execute_raw_sql_query(self,my_db,my_cursor,mysql):

        found_list = []

        try:
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                tmp_rows = []
            for row in tmp_rows:
                for col in row:
                    id = int(col)
                    found_list.append(id)
                    break
                #END FOR
            #END FOR
            my_db.close()
            return found_list
        except Exception as e:
            my_db.close()
            msg = "Raw SQL Query Fatal Error:   " + bytes(e)
            error_dialog(self, _('CalibreSpy'),_((msg)), show=True)
            dummy_list = []
            return dummy_list
#--------------------------------------------------------------------------------------------------
    def apsw_create_user_functions(self,my_db,my_cursor,func):
        try:
            if func == "REGEXP":
                my_db.createscalarfunction("regexp", self.apsw_user_function_regexp)
                #~ if DEBUG: print("Create_SQLite_User_Function 1 was successful...")
                return True
            if func == "SIMILARTO":
                my_db.createscalarfunction("similarto", self.apsw_user_function_similarto)
                #~ if DEBUG: print("Create_SQLite_User_Function 2 was successful...")
                return True
        except Exception as e:
            if DEBUG: print("Create_SQLite_User_Function [n] failed...cannot proceed...")
            if DEBUG: print(bytes(e))
            return False
#--------------------------------------------------------------------------------------------------
    def apsw_user_function_regexp(self,regexpr,avalue):
        #http://www.sqlite.org/lang_expr.html:  The "X REGEXP Y" operator will be implemented as a call to "regexp(Y,X)"
        #---------------------------------------------------------------------------------------------------------------------------------------
        #mysql = 'SELECT id FROM custom_column_8 WHERE value REGEXP '^.+$'
        #---------------------------------------------------------------------------------------------------------------------------------------
        if regexpr:
            if avalue:
                try:
                    s_string = unicode_type(avalue)
                    re_string = unicode_type(regexpr)
                    re.escape("\\")
                    p = re.compile(re_string, re.IGNORECASE|re.DOTALL|re.MULTILINE)
                    match = p.search(s_string)
                    if match:
                        return True
                    else:
                        return False
                except Exception as e:
                    if DEBUG: print(bytes(e))
                    return False
#--------------------------------------------------------------------------------------------------
    def apsw_user_function_similarto(self,a,b):
        #----------------------------------------------------------------
        #returns the probability the string a is the same as string b
        #~ SELECT book FROM _mcs_authors_by_book,_mcs_book_awards WHERE award LIKE '%hugo%' AND subbytes(authorname,1,3) = subbytes(author,1,3)  AND SIMILARTO(authorname,author) > 0.80 AND _mcs_authors_by_book.book IN ( SELECT id FROM books WHERE books.id = _mcs_authors_by_book.book AND SIMILARTO(books.title,_mcs_book_awards.title) > 0.80)
        #----------------------------------------------------------------
        try:
            p = SequenceMatcher(None, a, b).ratio()
            if p:
                return p
            else:
                return 0.0000
        except Exception as e:
            if DEBUG: print(bytes(e))
            return  0.0000

#--------------------------------------------------------------------------------------------------
    def save_and_exit(self):
        self.save_current_raw_sql_query()
        self.close()
#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
    def build_custom_column_dicts(self):
        self.custom_column_label_dict = {}                 #  [label] = id
        self.custom_column_datatype_dict = {}          #  [id] = datatype
        self.custom_column_normalized_dict = {}      #  [id] = is_normalized

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Library.'), show=True)

        mysql = "SELECT id,label,datatype,normalized FROM custom_columns"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        for row in tmp_rows:
            id,label,datatype,normalized = row
            label = '#' + label
            self.custom_column_label_dict[label] = id
            self.custom_column_datatype_dict[id] = datatype
            self.custom_column_normalized_dict[id] = normalized
        #END FOR

        my_db.close()
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
