# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2018,2019 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.76"  #Technical Tweaks after compiling with Python 3.8

from PyQt5.Qt import (Qt, QImage, QPalette, QPixmap, QTextOption, QSize, QIcon,
                                        QApplication, QLabel, QVBoxLayout, QDialog, QTextEdit, QTextBrowser, QDesktopServices,
                                        QHBoxLayout, QDialog, QPushButton, QAbstractScrollArea,
                                        QPushButton, QMessageBox, QScrollArea, QSizePolicy,
                                        QFrame, QSplitter, QSlider, QAction, QMenu, QKeySequence)

from polyglot.builtins import iteritems, range, unicode_type

IMAGE_DEFAULT_WIDTH = 450
IMAGE_DEFAULT_HEIGHT = 680
METADATA_DEFAULT_WIDTH = IMAGE_DEFAULT_WIDTH
METADATA_DEFAULT_HEIGHT = IMAGE_DEFAULT_HEIGHT
COMMENTS_DEFAULT_WIDTH = 300
COMMENTS_DEFAULT_HEIGHT = IMAGE_DEFAULT_HEIGHT
#---------------------------------------------------------------------------------------------------------------------------------------
class LibraryBrowserDialog(QDialog):

    #---------------------------------------------------------------------------------------------------------------------------------------
    def __init__(self,debug,icon_image,font,normal_fontsize,style_text,push_button_library_browser,
                        apsw_connect_to_library,row_to_bookid_dict,
                        images_dir,icon_viewer,icon_book,icon_editor,icon_email,icon_directory,
                        myos,matrix,n_matrix_rows,n_matrix_columns,details_column_label_list,
                        view_current_book,open_current_book,edit_current_book,email_current_book,
                        save_copy_single, path_column,library_path,is_cli,library_browser_is_active,
                        marked_bookids_color_dict,context_authors_link,search_engine_url):

        super(LibraryBrowserDialog, self).__init__()

        self.debug = debug
        self.icon = icon_image
        self.push_button_library_browser = push_button_library_browser
        self.apsw_connect_to_library = apsw_connect_to_library
        self.row_to_bookid_dict = row_to_bookid_dict
        self.images_dir = images_dir
        self.icon_viewer = icon_viewer
        self.icon_book = icon_book
        self.icon_editor = icon_editor
        self.icon_email = icon_email
        self.icon_directory = icon_directory
        self.os = myos
        self.matrix = matrix
        self.n_matrix_rows = n_matrix_rows
        self.n_matrix_columns = n_matrix_columns
        self.details_column_label_list = details_column_label_list
        self.view_current_book = view_current_book
        self.open_current_book = open_current_book
        self.edit_current_book = edit_current_book
        self.email_current_book = email_current_book
        self.save_copy_single = save_copy_single
        self.path_column = path_column
        self.library_path = library_path
        self.is_cli = is_cli
        self.library_browser_is_active = library_browser_is_active
        self.marked_bookids_color_dict = marked_bookids_color_dict
        self.context_authors_link = context_authors_link
        self.search_engine_url = search_engine_url

        try:
            self.search_engine_url = '"' + self.search_engine_url + '[VALUE]">'
            self.search =  '<a href=' + self.search_engine_url
        except:
            #user syntax error in Customization; probably quoted incorrectly.
            if DEBUG: print("'Library Browser' User Error:  Customized Search Engine URL is invalid.  Default will be used until corrected in CalibreSpy Customization.")
            self.search_engine_url = "https://www.google.com/search?q="
            self.search =  '<a href=' + self.search_engine_url

        font.setPointSize(normal_fontsize + 2)
        self.font = font
        self.normal_fontsize = normal_fontsize
        self.setStyleSheet(style_text)

        self.setWindowFlags(Qt.Window | Qt.WindowTitleHint | Qt.WindowMinMaxButtonsHint)
        self.setModal(False)
        self.setWindowIcon(icon_image)
        #~ title = "Library Browser: " + bytes(n_matrix_rows) + " Books"
        title = "Library Browser: " + unicode_type(n_matrix_rows) + " Books"
        self.setWindowTitle(title)
        #~ ---------------------------
        self.layout = QHBoxLayout()
        self.layout.setAlignment(Qt.AlignCenter)
        #~ ---------------------------
        self.image_label = QLabel()
        self.image_label.setBackgroundRole(QPalette.Base)
        self.image_label.setScaledContents(False)  #scale pixmap itself instead
        self.image_label.setMinimumWidth(METADATA_DEFAULT_WIDTH)
        self.image_label.setMaximumWidth(METADATA_DEFAULT_HEIGHT)
        self.image_label.setToolTip("<p style='white-space:wrap'>Book's Cover image in the Library.")

        self.finished_updating = True

        self.pixmap_default_cover = "tbd"

        self.update_details_image()
        #~ ---------------------------
        self.metadata_qtextbrowser = QTextBrowser(None)
        self.metadata_qtextbrowser.setAcceptRichText(True)
        self.metadata_qtextbrowser.setFont(font)
        self.metadata_qtextbrowser.setReadOnly(True)
        self.metadata_qtextbrowser.setOpenExternalLinks(True)
        self.metadata_qtextbrowser.setWordWrapMode(QTextOption.WordWrap)
        self.metadata_qtextbrowser.setSizeAdjustPolicy(QAbstractScrollArea.AdjustToContents)
        self.metadata_qtextbrowser.clear()
        self.metadata_qtextbrowser.setMinimumWidth(METADATA_DEFAULT_WIDTH)
        self.metadata_qtextbrowser.setToolTip("<p style='white-space:wrap'>Metadata currently displayed in CalibreSpy per Customization.<br><br>The 'Author Link' shortcut is 'Control+L'.")

        self.update_details_metadata()
        #~ ---------------------------
        self.comments_qtextedit = QTextEdit("")
        self.comments_qtextedit.setFont(font)
        self.comments_qtextedit.setReadOnly(True)
        self.comments_qtextedit.setWordWrapMode(QTextOption.WordWrap)
        self.comments_qtextedit.setSizeAdjustPolicy(QAbstractScrollArea.AdjustToContents)
        self.comments_qtextedit.clear()
        self.comments_qtextedit.setMinimumWidth(COMMENTS_DEFAULT_WIDTH)
        self.comments_qtextedit.setMaximumWidth(COMMENTS_DEFAULT_WIDTH)
        self.comments_qtextedit.setMinimumHeight(COMMENTS_DEFAULT_HEIGHT)
        self.comments_qtextedit.setToolTip("<p style='white-space:wrap'>Comments.")

        self.update_details_comments()
        #~ ---------------------------
        self.horizontal_splitter = QSplitter(Qt.Horizontal)
        self.horizontal_splitter.addWidget(self.image_label)
        self.horizontal_splitter.addWidget(self.metadata_qtextbrowser)
        self.horizontal_splitter.addWidget(self.comments_qtextedit)

        self.horizontal_splitter.setSizes([IMAGE_DEFAULT_WIDTH,METADATA_DEFAULT_WIDTH,COMMENTS_DEFAULT_WIDTH])
        #~ ---------------------------
        self.bottom_frame = QFrame()
        self.bottom_frame.setFrameShape(QFrame.StyledPanel)
        self.bottom_frame.setFrameShadow(QFrame.Sunken)

        self.bottom_frame_layout = QHBoxLayout()
        self.bottom_frame_layout.setAlignment(Qt.AlignCenter)
        self.bottom_frame.setLayout(self.bottom_frame_layout)

        self.bottom_slider = QSlider(Qt.Horizontal)
        self.bottom_slider.setMinimum(1)
        self.bottom_slider.setMaximum(self.n_matrix_rows)
        self.bottom_slider.setTickPosition(QSlider.NoTicks)
        self.bottom_slider_step = int(self.n_matrix_rows/100)
        self.bottom_slider.setTickInterval(self.bottom_slider_step)
        self.bottom_slider.setSingleStep(1)
        self.bottom_slider.setPageStep(100)
        self.bottom_slider.setValue(1)
        self.bottom_slider.setTracking(False)
        self.bottom_slider.setMinimumWidth(IMAGE_DEFAULT_WIDTH - 100)
        self.bottom_slider.setMaximumWidth(IMAGE_DEFAULT_WIDTH - 100)
        self.bottom_slider.setToolTip("<p style='white-space:wrap'>User actions are: Click; Drag and Release; Page Up; Page Down; Left Arrow; and Right Arrow.\
        <br><br>Clicking any text of the 'Metadata' or the 'Comments' transfers away the cursor's focus to those areas, so the actions above will apply to that text and not this slider.")
        self.bottom_frame_layout.addWidget(self.bottom_slider)

        self.slider_signals_blocked = False
        self.bottom_slider.valueChanged.connect(self.event_slider)

        icon_fast_previous_path = self.os.path.join(self.images_dir,"fast_previous.png")
        icon_fast_previous_path = icon_fast_previous_path.replace(self.os.sep,'/')

        if self.os.path.exists(icon_fast_previous_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_fast_previous_path)
            self.icon_fast_previous = QIcon(pixmap)
            del pixmap
        else:
            self.icon_fast_previous = None

        icon_previous_path = self.os.path.join(self.images_dir,"previous.png")
        icon_previous_path = icon_previous_path.replace(self.os.sep,'/')

        if self.os.path.exists(icon_previous_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_previous_path)
            self.icon_previous = QIcon(pixmap)
            del pixmap
        else:
            self.icon_previous = None

        icon_next_path = self.os.path.join(self.images_dir,"next.png")
        icon_next_path = icon_next_path.replace(self.os.sep,'/')

        if self.os.path.exists(icon_next_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_next_path)
            self.icon_next = QIcon(pixmap)
            del pixmap
        else:
            self.icon_next = None

        icon_fast_next_path = self.os.path.join(self.images_dir,"fast_next.png")
        icon_fast_next_path = icon_fast_next_path.replace(self.os.sep,'/')

        if self.os.path.exists(icon_fast_next_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_fast_next_path)
            self.icon_fast_next = QIcon(pixmap)
            del pixmap
        else:
            self.icon_fast_next = None

        self.fast_previous_pushbutton = QPushButton(self.icon_fast_previous,"",self)
        self.fast_previous_pushbutton.setToolTip("<p style='white-space:wrap'>Skip from the current book to a previous book.")
        self.fast_previous_pushbutton.clicked.connect(self.fast_previous)
        self.bottom_frame_layout.addWidget(self.fast_previous_pushbutton)

        self.previous_pushbutton = QPushButton(self.icon_previous,"",self)
        self.previous_pushbutton.setToolTip("<p style='white-space:wrap'>Move from the current book to the previous book.")
        self.previous_pushbutton.clicked.connect(self.previous)
        self.bottom_frame_layout.addWidget(self.previous_pushbutton)

        self.next_pushbutton = QPushButton(self.icon_next,"",self)
        self.next_pushbutton.setToolTip("<p style='white-space:wrap'>Move from the current book to the following book.")
        self.next_pushbutton.setDefault(True)
        self.next_pushbutton.clicked.connect(self.next)
        self.bottom_frame_layout.addWidget(self.next_pushbutton)

        self.fast_next_pushbutton = QPushButton(self.icon_fast_next,"",self)
        self.fast_next_pushbutton.setToolTip("<p style='white-space:wrap'>Skip from the current book to a following book.")
        self.fast_next_pushbutton.clicked.connect(self.fast_next)
        self.bottom_frame_layout.addWidget(self.fast_next_pushbutton)

        self.bottom_spacer_1_label = QLabel("")
        self.bottom_spacer_1_label.setMinimumWidth(50)
        self.bottom_spacer_1_label.setMaximumWidth(50)
        self.bottom_frame_layout.addWidget(self.bottom_spacer_1_label)

        self.read_pushbutton = QPushButton(self.icon_book,"",self)
        self.read_pushbutton.setToolTip("<p style='white-space:wrap'>Read the current book using the default ebook reading application for your computer.")
        self.read_pushbutton.clicked.connect(self.read)
        self.bottom_frame_layout.addWidget(self.read_pushbutton)

        self.view_pushbutton = QPushButton(self.icon_viewer,"",self)
        self.view_pushbutton.setToolTip("<p style='white-space:wrap'>View the current book using the Viewer specified in Customization.")
        self.view_pushbutton.clicked.connect(self.view)
        self.bottom_frame_layout.addWidget(self.view_pushbutton)

        self.save_copy_pushbutton = QPushButton(self.icon_directory,"",self)
        self.save_copy_pushbutton.setToolTip("<p style='white-space:wrap'>Save the current book to a specified target directory.")
        self.save_copy_pushbutton.clicked.connect(self.save_copy)
        self.bottom_frame_layout.addWidget(self.save_copy_pushbutton)

        self.edit_pushbutton = QPushButton(self.icon_editor,"",self)
        self.edit_pushbutton.setToolTip("<p style='white-space:wrap'>Edit the current book using the Editor specified in Customization.")
        self.edit_pushbutton.clicked.connect(self.edit)
        self.bottom_frame_layout.addWidget(self.edit_pushbutton)

        self.email_pushbutton = QPushButton(self.icon_email,"",self)
        self.email_pushbutton.setToolTip("<p style='white-space:wrap'>Email the current book using the SMTP command file specified in Customization.")
        self.email_pushbutton.clicked.connect(self.email_smtp)
        self.bottom_frame_layout.addWidget(self.email_pushbutton)

        self.bottom_spacer_2_label = QLabel("")
        self.bottom_spacer_2_label.setMinimumWidth(50)
        self.bottom_spacer_2_label.setMaximumWidth(50)
        self.bottom_frame_layout.addWidget(self.bottom_spacer_2_label)

        icon_exit_path = self.os.path.join(self.images_dir,"window_close.png")
        icon_exit_path = icon_exit_path.replace(self.os.sep,'/')

        if self.os.path.exists(icon_exit_path):
            pixmap = QPixmap(QSize(20,20))
            pixmap.load(icon_exit_path)
            self.icon_exit = QIcon(pixmap)
            del pixmap
        else:
            self.icon_exit = None

        self.exit_pushbutton = QPushButton(self.icon_exit,"",self)
        self.exit_pushbutton.setToolTip("<p style='white-space:wrap'>Close this window.")
        self.exit_pushbutton.clicked.connect(self.exit)
        self.bottom_frame_layout.addWidget(self.exit_pushbutton)
        #~ ---------------------------
        self.vertical_splitter = QSplitter(Qt.Vertical)
        self.vertical_splitter.addWidget(self.horizontal_splitter)
        self.vertical_splitter.addWidget(self.bottom_frame)
        #~ ---------------------------
        self.layout.addWidget(self.vertical_splitter)
        #~ ---------------------------
        self.create_author_link_shortcut()
        #~ ---------------------------
        self.setLayout(self.layout)
        #~ ---------------------------
        self.bottom_slider.setFocus()
        #~ ---------------------------
        if self.is_cli:
            import gc
            self.gc = gc
            del gc
            self.gc.collect()
#---------------------------------------------------------------------------------------------------------------------------------------
    def keyPressEvent(self, event):
        if event.key() == Qt.Key_Escape: #~ https://doc.qt.io/qt-5/qt.html#Key-enum
            self.exit()
        elif event.key() == Qt.Key_PageUp:
            if not self.metadata_qtextbrowser.hasFocus() and not self.comments_qtextedit.hasFocus():
                self.bottom_slider.setFocus()
            elif not self.metadata_qtextbrowser.copyAvailable() and not self.comments_qtextedit.copyAvailable():
                self.bottom_slider.setFocus()
            QDialog.keyPressEvent(self, event)
        elif event.key() == Qt.Key_PageDown:
            if not self.metadata_qtextbrowser.hasFocus() and not self.comments_qtextedit.hasFocus():
                self.bottom_slider.setFocus()
            elif not self.metadata_qtextbrowser.copyAvailable() and not self.comments_qtextedit.copyAvailable():
                self.bottom_slider.setFocus()
            QDialog.keyPressEvent(self, event)
        elif event.key() == Qt.Key_Up:
            if not self.metadata_qtextbrowser.hasFocus() and not self.comments_qtextedit.hasFocus():
                self.bottom_slider.setFocus()
            elif not self.metadata_qtextbrowser.copyAvailable() and not self.comments_qtextedit.copyAvailable():
                self.bottom_slider.setFocus()
            QDialog.keyPressEvent(self, event)
        elif event.key() == Qt.Key_Down:
            if not self.metadata_qtextbrowser.hasFocus() and not self.comments_qtextedit.hasFocus():
                self.bottom_slider.setFocus()
            elif not self.metadata_qtextbrowser.copyAvailable() and not self.comments_qtextedit.copyAvailable():
                self.bottom_slider.setFocus()
            QDialog.keyPressEvent(self, event)
        else:
            QDialog.keyPressEvent(self, event)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def event_slider(self,value=0):

        if self.slider_signals_blocked:
            return

        if not isinstance(value,int):
            return

        r = value - 1   #matrix rows are zero-based; slider is not.

        if r < 0:
            r = 0
        elif r > self.n_matrix_rows - 1:
            r = self.n_matrix_rows - 1

        self.finished_updating = False
        self.matrix.setCurrentCell(r,0)
        self.matrix.setFocus()
        self.update_details_image()
        self.update_details_comments()
        self.update_details_metadata()
        self.finished_updating = True
    #---------------------------------------------------------------------------------------------------------------------------------------
    def previous(self,source=None):
        if not self.finished_updating:
            return
        r = self.matrix.currentRow()
        if r > 0:
            r = r - 1
            if source == "fast":
                r = r - 50
                if r < 0:
                    r = 0
            self.matrix.setCurrentCell(r,0)
            self.slider_signals_blocked = True
            self.bottom_slider.setValue(r)
            self.slider_signals_blocked = False
            self.matrix.setFocus()
            self.update_details_image()
            self.update_details_comments()
            self.update_details_metadata()
        self.finished_updating = True
        if self.is_cli:
            self.gc.collect()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def fast_previous(self):
        if not self.finished_updating:
            return
        self.previous(source="fast")
    #---------------------------------------------------------------------------------------------------------------------------------------
    def next(self,source=None):
        if not self.finished_updating:
            return
        r = self.matrix.currentRow()
        if r + 1 <= self.n_matrix_rows - 1:
            r = r + 1
            if source == "fast":
                r = r + 50
                if r > self.n_matrix_rows - 1:
                    r = self.n_matrix_rows - 1
            self.matrix.setCurrentCell(r,0)
            self.slider_signals_blocked = True
            self.bottom_slider.setValue(r)
            self.slider_signals_blocked = False
            self.matrix.setFocus()
            self.update_details_image()
            self.update_details_comments()
            self.update_details_metadata()
        self.finished_updating = True
        if self.is_cli:
            self.gc.collect()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def fast_next(self):
        if not self.finished_updating:
            return
        self.next(source="fast")
    #---------------------------------------------------------------------------------------------------------------------------------------
    def view(self):
        self.view_current_book()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def read(self):
        self.open_current_book()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def edit(self):
        self.edit_current_book()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def email_smtp(self):
        self.email_current_book()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def save_copy(self):
        self.save_copy_single()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def exit(self):
        try:
            self.push_button_library_browser.setChecked(False)
            self.push_button_library_browser.setDown(False)
            self.push_button_library_browser.update()
            self.library_browser_is_active = False
            self.matrix.setFocus()
        except:
            pass
        self.done
        self.close()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def update_details_metadata(self):
        r = self.matrix.currentRow()
        if r is not None:
            if r > -1:
                f = "    "
                m = ""
                for c in range(0,(self.n_matrix_columns)):
                    item = self.matrix.item(r,c)
                    if item is not None:
                        value = item.text()
                        value = value.strip()
                    else:
                        value = None
                    if value is not None:
                        if value > " ":
                            val = value
                            column,label = self.details_column_label_list[c]
                            if label.lower() == "author(s)":
                                if not "&" in value:
                                    val = "book author " + val
                                    value = self.search + value + "</a>"
                                    value = value.replace("[VALUE]",val)
                                else:
                                    s_split = value.split("&")
                                    value = ""
                                    for auth in s_split:
                                        a = auth
                                        a = "author " + a
                                        auth = self.search + auth + "</a>"
                                        auth = auth.replace("[VALUE]",a)
                                        value = value + auth + " & "
                                    #END FOR
                                    value = value[0:-3]
                            elif label.lower() == "title":
                                val = "book title " + val
                                value = self.search + value + "</a>"
                                value = value.replace("[VALUE]",val)
                            elif label.lower() == "series":
                                val = "book series " + val
                                value = self.search + value + "</a>"
                                value = value.replace("[VALUE]",val)
                            elif label.lower() == "identifiers":
                                if not "," in value:
                                    if val.startswith("lccn") or val.startswith("lc_auth"):
                                        val = "library of congress " + val
                                        val = val.replace("_"," ")
                                    value = self.search + value + "</a>"
                                    val = val.replace(":", " ")
                                    value = value.replace("[VALUE]",val)
                                else:
                                    s_split = value.split(",")
                                    value = ""
                                    for id in s_split:
                                        v = id
                                        id = self.search + id + "</a>"
                                        v = v.replace(":", " ")
                                        if "lccn" in v or "lc_auth" in v:
                                            v = "library of congress " + v
                                            v = v.replace("_"," ")
                                        id = id.replace("[VALUE]",v)
                                        value = value + id + ", "
                                    #END FOR
                                    value = value[0:-2]
                            m = m + "<b>" + label + ":</b>" + f + value + "<br><br>"
                #END FOR
                color = ""
                if r in self.row_to_bookid_dict:
                    bookid = self.row_to_bookid_dict[r]
                    if bookid in self.marked_bookids_color_dict:
                        color = self.marked_bookids_color_dict[bookid]
                label = "Current Row"
                #~ value = bytes(r+1)
                value = unicode_type(r+1)
                if color > " ":
                    s =  '<a style="background-color:[BGCOLOR];"><FONT COLOR=[COLOR]><b>' + 'Marked' + '</b></FONT></a><b> ➝ '+ label + ':</b>' + f + value + '<br><br>'
                    if color == "cyan" or color == "yellow":
                        s = s.replace("[BGCOLOR]","dimgrey")
                    else:
                        s = s.replace("[BGCOLOR]","white")
                    s = s.replace("[COLOR]",color)
                    m = m + s
                else:
                    m = m + "<b>" + label + ":</b>" + f + value + '<br><br>'
                self.metadata_qtextbrowser.setHtml(m)
            else:
                self.metadata_qtextbrowser.setHtml("")
        else:
            self.metadata_qtextbrowser.setHtml("")
    #---------------------------------------------------------------------------------------------------------------------------------------
    def update_details_image(self):
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:  #e.g. has no path.
            path = None
        else:
            bookid = self.row_to_bookid_dict[r]
            item = self.matrix.item(r,self.path_column)
            value = item.text()
            value = value.strip()
            path = self.os.path.join(self.library_path,value)
            path = path.replace("\\","/")
            head,tail = self.os.path.split(path)
            path = self.os.path.join(head,"cover.jpg")
            path = path.replace("\\","/")
            if self.os.path.isfile(path):
                pass
            else:
                path = None

        if path is not None:
            image = QImage(path)
            if image.isNull():
                image = None
        else:
            image = None

        if image is None:
            self.load_default_cover()
            if self.pixmap_default_cover is not None:
                self.image_label.setPixmap(self.pixmap_default_cover.scaled(IMAGE_DEFAULT_WIDTH, IMAGE_DEFAULT_HEIGHT, Qt.KeepAspectRatio, Qt.SmoothTransformation))
            else:
                self.image_label.setPixmap(None)
        else:
            image = self.create_scaled_pixmap(image)
            self.image_label.setPixmap(image)

        self.image_label.update()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def update_details_comments(self):
        comments = None
        r = self.matrix.currentRow()
        if not r in self.row_to_bookid_dict:  #e.g. has no path.
            comments = "<b>Comments:</b><br><br> None."
            self.comments_qtextedit.setHtml(comments)
            self.comments_qtextedit.update()
            return
        bookid = self.row_to_bookid_dict[r]

        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
            self.comments_qtextedit.setHtml("<b>Comments:</b><br><br> None.")
            return error_dialog(None, _('CalibreSpy'),_('Database Connection Error.  Cannot Connect to the Chosen Library.'), show=True)

        mysql = "SELECT book,text FROM comments WHERE book = ?"
        for book,text in my_cursor.execute(mysql,([bookid])):
            if bookid == book:
                comments = text
        #END FOR
        my_db.close()
        if not comments:
            comments = "None."
        comments = comments.strip()
        comments = "<b>Comments:</b><br><br>" + comments
        self.comments_qtextedit.setHtml(comments)
        self.comments_qtextedit.update()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def load_default_cover(self):
        if not self.pixmap_default_cover == "tbd":
            return

        pixmap_default_cover_path = self.os.path.join(self.images_dir,"default_cover.png")
        pixmap_default_cover_path = pixmap_default_cover_path.replace(self.os.sep,'/')

        if self.os.path.exists(pixmap_default_cover_path):
            pixmap = QPixmap(QSize(IMAGE_DEFAULT_WIDTH,IMAGE_DEFAULT_HEIGHT))
            pixmap.load(pixmap_default_cover_path)
            self.pixmap_default_cover = QPixmap(pixmap)
            del pixmap
        else:
            self.pixmap_default_cover = None
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_scaled_pixmap(self,image):
        image = QPixmap.fromImage(image)
        size = QSize(IMAGE_DEFAULT_WIDTH,IMAGE_DEFAULT_HEIGHT)
        image = image.scaled(size, Qt.KeepAspectRatio, Qt.SmoothTransformation)
        return image
    #---------------------------------------------------------------------------------------------------------------------------------------
    def create_author_link_shortcut(self):
        self.lb_view_author_links_action = QAction(self.icon_directory,"View Author Link",None)
        self.lb_view_author_links_action.setShortcut(QKeySequence("Ctrl+L"))
        self.lb_view_author_links_action.triggered.connect(self.context_authors_link)
        self.metadata_qtextbrowser.addAction(self.lb_view_author_links_action)
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
#~ # end of library_browser_dialog.py