# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2018,2019 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.76"  #Technical Tweaks after compiling with Python 3.8

import ast,codecs,os

from PyQt5.Qt import (Qt, QDialog, QVBoxLayout, QHBoxLayout, QComboBox, QPushButton, QLabel, QLineEdit)
from calibre.constants import DEBUG

from polyglot.builtins import as_bytes, as_unicode, iteritems, range, unicode_type

FTP_COMBOBOX_HEADER_VALUE = "?"

class FTPTargetHostSelectionDialog(QDialog):

    def __init__(self,parent,icon,font,style_text,ftp_prefs,return_new_local_pref_value):

        QDialog.__init__(self, parent)

        self.setWindowFlags( Qt.Window | Qt.WindowTitleHint | Qt.WindowSystemMenuHint )

        self.setWindowTitle('CalibreSpy:  FTP Target Host Selection')
        self.setWindowIcon(icon)

        self.ftp_prefs = ftp_prefs

        self.font = font

        self.setStyleSheet(style_text)
        #~ self.setStyleSheet("QToolTip { color: #000000; background-color: #ffffcc; border: 1px solid white; }")

        self.return_new_local_pref_value = return_new_local_pref_value

        # main dialog always sends encoded values
        raw_in = ftp_prefs['CALIBRESPY_FTP_USERID']
        raw_in = bytes(raw_in)
        userid = codecs.decode(raw_in, 'rot13')
        ftp_prefs['CALIBRESPY_FTP_USERID'] = userid

        raw_in = ftp_prefs['CALIBRESPY_FTP_PASSWORD']
        raw_in = bytes(raw_in)
        password = codecs.decode(raw_in, 'rot13')
        ftp_prefs['CALIBRESPY_FTP_PASSWORD'] = password

        self.ftp_combobox_decoded = 0

        self.setModal(True)

        self.layout = QVBoxLayout()
        self.setLayout(self.layout)

        self.font.setBold(False)
        self.font.setPointSize(12)

        self.setToolTip("<p style='white-space:wrap'>Choose the FTP Target Host to be used in the currently in-progress FTP Action.")

        self.signals_blocked = False

        minwidth = 500

        self.ftp_target_location_combobox = QComboBox()
        self.ftp_target_location_combobox.setEditable(False)
        self.ftp_target_location_combobox.setFrame(True)
        self.ftp_target_location_combobox.setDuplicatesEnabled(False)
        self.ftp_target_location_combobox.setMaxCount(20)
        self.ftp_target_location_combobox.setMaxVisibleItems(20)
        self.ftp_target_location_combobox.setMinimumWidth(minwidth)
        self.ftp_target_location_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.ftp_target_location_combobox.setFont(self.font)
        self.layout.addWidget(self.ftp_target_location_combobox)

        device_settings_list = self.ftp_prefs['CALIBRESPY_FTP_TARGET_DEVICES_LIST']
        device_settings_list = bytes(device_settings_list)
        device_settings_list = ast.literal_eval(device_settings_list)
        if not isinstance(device_settings_list,list):
            if DEBUG: print("ERROR: device_settings_list is NOT a list...reset to default value")
            device_settings_list = []

        device_settings_list = list(set(device_settings_list))
        device_settings_list.sort()

        self.ftp_target_location_combobox.clear()
        self.ftp_target_location_combobox.addItem(FTP_COMBOBOX_HEADER_VALUE)

        for row in device_settings_list:
            if isinstance(row,tuple):
                if len(row) == 5:
                    host = row[0]
                    port = row[1]
                    host_directory = row[2]
                    userid = row[3]
                    password = row[4]

                    raw_in = userid
                    #~ raw_in = bytes(raw_in)
                    raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
                    raw_in = as_bytes(raw_in, encoding='ascii')
                    userid = codecs.decode(raw_in, 'rot13')
                    raw_in = password
                    #~ raw_in = bytes(raw_in)
                    raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
                    raw_in = as_bytes(raw_in, encoding='ascii')
                    password = codecs.decode(raw_in, 'rot13')

                    s = host + "," + port + "," + host_directory + "," + userid + "," + password
                    self.ftp_target_location_combobox.addItem(s)
        #END FOR

        self.ftp_target_location_combobox.update()

        self.layout_ftp = QHBoxLayout()
        self.layout_ftp.setAlignment(Qt.AlignLeft)
        self.layout.addLayout(self.layout_ftp)

        self.ftp_host_label = QLabel("FTP:")
        self.ftp_host_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>These FTP parameters control the FTP GUI Tool."
        self.ftp_host_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_host_label)

        ftp_width = 100

        host = self.ftp_prefs['CALIBRESPY_FTP_HOST']

        self.ftp_host_lineedit = QLineEdit(self)
        self.ftp_host_lineedit.setMinimumWidth(ftp_width)
        self.ftp_host_lineedit.setMaximumWidth(ftp_width)
        t = "<p style='white-space:wrap'>FTP Host.  Not TLS Encrypted."
        self.ftp_host_lineedit.setToolTip(t)
        self.ftp_host_lineedit.setText(host)
        self.layout_ftp.addWidget(self.ftp_host_lineedit)

        self.ftp_port_label = QLabel("Port:")
        self.ftp_port_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Port.  Usually 21.  Specify an integer."
        self.ftp_port_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_port_label)

        self.ftp_port_lineedit = QLineEdit(self)
        self.ftp_port_lineedit.setToolTip(t)
        self.ftp_port_lineedit.setMinimumWidth(50)
        self.ftp_port_lineedit.setMaximumWidth(50)
        port = self.ftp_prefs['CALIBRESPY_FTP_HOST_PORT']
        port = unicode_type(port)
        self.ftp_port_lineedit.setText(port)
        self.layout_ftp.addWidget(self.ftp_port_lineedit)

        self.ftp_directory_label = QLabel("Dir:")
        self.ftp_directory_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Host Directory into which the selected book formats will be uploaded.<br>Include a leading and trailing '/'.  <br>Example:  /myfolder/ "
        self.ftp_directory_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_directory_label)

        self.ftp_directory_lineedit = QLineEdit(self)
        self.ftp_directory_lineedit.setMinimumWidth(ftp_width)
        self.ftp_directory_lineedit.setMaximumWidth(ftp_width)
        self.ftp_directory_lineedit.setToolTip(t)
        self.ftp_directory_lineedit.setText(self.ftp_prefs['CALIBRESPY_FTP_HOST_DIRECTORY'])
        self.layout_ftp.addWidget(self.ftp_directory_lineedit)

        self.ftp_userid_label = QLabel("UserID:")
        self.ftp_userid_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Host User ID.  This userid is only very lightly encoded within CalibreSpy, and will be completely visible in each Library's metadata.db table _calibrespy_settings."
        self.ftp_userid_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_userid_label)

        self.ftp_userid_lineedit = QLineEdit(self)
        self.ftp_userid_lineedit.setMinimumWidth(ftp_width)
        self.ftp_userid_lineedit.setMaximumWidth(ftp_width)
        self.ftp_userid_lineedit.setToolTip(t)
        self.ftp_userid_lineedit.setText(self.ftp_prefs['CALIBRESPY_FTP_USERID'])
        self.layout_ftp.addWidget(self.ftp_userid_lineedit)

        self.ftp_password_label = QLabel("PW:")
        self.ftp_password_label.setAlignment(Qt.AlignRight)
        t = "<p style='white-space:wrap'>FTP Host Password.  This password is only very lightly encoded within CalibreSpy, and will be completely visible in each Library's metadata.db table _calibrespy_settings."
        self.ftp_password_label.setToolTip(t)
        self.layout_ftp.addWidget(self.ftp_password_label)

        self.ftp_password_lineedit = QLineEdit(self)
        self.ftp_password_lineedit.setText(self.ftp_prefs['CALIBRESPY_FTP_PASSWORD'])
        self.ftp_password_lineedit.setMinimumWidth(ftp_width)
        self.ftp_password_lineedit.setMaximumWidth(ftp_width)
        self.ftp_password_lineedit.setToolTip(t)

        self.layout_ftp.addWidget(self.ftp_password_lineedit)

        self.push_button_add_item_qpushbutton = QPushButton("Add Specified Settings", self)
        self.push_button_add_item_qpushbutton.clicked.connect(self.add_specified_settings_to_list)
        self.push_button_add_item_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_add_item_qpushbutton.setFont(self.font)
        self.push_button_add_item_qpushbutton.setDefault(True)
        self.push_button_add_item_qpushbutton.setToolTip("<p style='white-space:wrap'>Add the collective settings shown just above to the dropdown list.")
        self.layout.addWidget(self.push_button_add_item_qpushbutton)

        self.push_button_remove_item_qpushbutton = QPushButton("Remove Selected Settings", self)
        self.push_button_remove_item_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_remove_item_qpushbutton.clicked.connect(self.remove_selected_settings_from_list)
        self.push_button_remove_item_qpushbutton.setFont(self.font)
        self.push_button_remove_item_qpushbutton.setDefault(False)
        self.push_button_remove_item_qpushbutton.setToolTip("<p style='white-space:wrap'>Remove the selected item from the dropdown list.")
        self.layout.addWidget(self.push_button_remove_item_qpushbutton)

        self.push_button_exit_return_qpushbutton = QPushButton("Return to CalibreSpy", self)
        self.push_button_exit_return_qpushbutton.setMinimumWidth(minwidth)
        self.push_button_exit_return_qpushbutton.clicked.connect(self.exit_return_to_calibrespy)
        self.push_button_exit_return_qpushbutton.setFont(self.font)
        self.push_button_exit_return_qpushbutton.setDefault(False)
        self.push_button_exit_return_qpushbutton.setToolTip("<p style='white-space:wrap'>Finalize any changes, and return to CalibreSpy.")
        self.layout.addWidget(self.push_button_exit_return_qpushbutton)

        self.resize(self.sizeHint())

        self.ftp_target_location_combobox.currentIndexChanged.connect(self.event_current_text_changed)

        self.add_specified_settings_to_list()  #to add the current settings line to the combobox if needed, and to show it if not.

        self.show()
    #-----------------------------------------------------
    def keyPressEvent(self, event):
        if event.key() == Qt.Key_Escape:
            self.exit_return_to_calibrespy()
        else:
            QDialog.keyPressEvent(self, event)
    #-----------------------------------------------------
    def event_current_text_changed(self,event=None):
        #everything is decoded

        if self.signals_blocked:
            return

        chosen_settings = self.ftp_target_location_combobox.currentText()
        if chosen_settings == FTP_COMBOBOX_HEADER_VALUE:
            return
        #~ if DEBUG: print("chosen_settings", chosen_settings)     # chosen_settings = host + "," + port + "," + host_directory + "," + userid + "," + password
        s_split = chosen_settings.split(",")
        if len(s_split) == 5:
            host = s_split[0]
            port = s_split[1]
            host_directory = s_split[2]
            userid = s_split[3]
            password = s_split[4]
        else:
            return

        self.ftp_host_lineedit.setText(host)
        self.ftp_port_lineedit.setText(port)
        self.ftp_directory_lineedit.setText(host_directory)
        self.ftp_userid_lineedit.setText(userid)
        self.ftp_password_lineedit.setText(password)

        self.ftp_prefs['CALIBRESPY_FTP_HOST'] = host
        self.ftp_prefs['CALIBRESPY_FTP_HOST_PORT'] = port
        self.ftp_prefs['CALIBRESPY_FTP_HOST_DIRECTORY'] = host_directory
        self.ftp_prefs['CALIBRESPY_FTP_USERID'] = userid
        self.ftp_prefs['CALIBRESPY_FTP_PASSWORD'] = password

        self.save_all_items()
    #-----------------------------------------------------
    def remove_selected_settings_from_list(self):
        text = self.ftp_target_location_combobox.currentText()
        if text == FTP_COMBOBOX_HEADER_VALUE:
            return
        index = self.ftp_target_location_combobox.currentIndex()
        if index > -1:
            self.signals_blocked = True
            self.ftp_target_location_combobox.removeItem(index)
            self.signals_blocked = False
            self.ftp_target_location_combobox.setCurrentIndex(0)
            self.ftp_target_location_combobox.update()
            self.save_all_items()
    #-----------------------------------------------------
    def add_specified_settings_to_list(self):

        host = self.ftp_host_lineedit.text()
        port = self.ftp_port_lineedit.text()
        host_directory = self.ftp_directory_lineedit.text()
        userid = self.ftp_userid_lineedit.text()
        password = self.ftp_password_lineedit.text()

        self.ftp_prefs['CALIBRESPY_FTP_HOST'] = host
        self.ftp_prefs['CALIBRESPY_FTP_HOST_PORT'] = port
        self.ftp_prefs['CALIBRESPY_FTP_HOST_DIRECTORY'] = host_directory
        self.ftp_prefs['CALIBRESPY_FTP_USERID'] = userid
        self.ftp_prefs['CALIBRESPY_FTP_PASSWORD'] = password

        #~ chosen_settings = host,port,host_directory,userid,password
        chosen_settings = unicode_type(host + "," + port + "," + host_directory + "," + userid + "," + password)

        i = self.ftp_target_location_combobox.findText(chosen_settings)
        if i < 0:
            self.ftp_target_location_combobox.addItem(chosen_settings)
            self.ftp_target_location_combobox.setCurrentText(chosen_settings)
            self.ftp_target_location_combobox.update()
            self.save_all_items()
        else:
            self.ftp_target_location_combobox.setCurrentText(chosen_settings)
    #-----------------------------------------------------
    def save_all_items(self):

        device_settings_list = []

        nitems = self.ftp_target_location_combobox.count()

        for i in range(0,nitems):
            s = self.ftp_target_location_combobox.itemText(i)
            if s != FTP_COMBOBOX_HEADER_VALUE:
                if not s in device_settings_list:
                    s_split = s.split(",")
                    if len(s_split) == 5:
                        host = s_split[0]
                        port = s_split[1]
                        host_directory = s_split[2]
                        userid = s_split[3]
                        password = s_split[4]
                        raw_in = userid
                        #~ raw_in = bytes(raw_in)
                        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
                        raw_in = as_bytes(raw_in, encoding='ascii')
                        userid = codecs.encode(raw_in, 'rot13')
                        raw_in = password
                        #~ raw_in = bytes(raw_in)
                        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
                        raw_in = as_bytes(raw_in, encoding='ascii')
                        password = codecs.encode(raw_in, 'rot13')
                        row = host,port,host_directory,userid,password
                        device_settings_list.append(row)
                    else:
                        if DEBUG: print("FTP Target Host Selection:  save_all_items:  split error: ", bytes(s))
        #END FOR

        device_settings_list.sort()
        device_settings_list = unicode_type(device_settings_list)
        self.ftp_prefs['CALIBRESPY_FTP_TARGET_DEVICES_LIST'] = device_settings_list
        #~ if DEBUG: print("FTP Selection--save_all_items: currently saved device_settings_list = ",device_settings_list)
        del device_settings_list
    #-----------------------------------------------------
    def exit_return_to_calibrespy(self):
        #~ calibrespy_dialog self.local_prefs for FTP userid and password always remain encoded, so this function must return only encoded values.

        self.event_current_text_changed(event=None)
        self.save_all_items()

        userid = self.ftp_prefs['CALIBRESPY_FTP_USERID']
        raw_in = userid
        #~ raw_in = bytes(raw_in)
        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
        raw_in = as_bytes(raw_in, encoding='ascii')
        userid = codecs.encode(raw_in, 'rot13')
        self.ftp_prefs['CALIBRESPY_FTP_USERID'] = userid

        password = self.ftp_prefs['CALIBRESPY_FTP_PASSWORD']
        raw_in = password
        #~ raw_in = bytes(raw_in)
        raw_in = as_unicode(raw_in, encoding='utf-8', errors='replace')
        raw_in = as_bytes(raw_in, encoding='ascii')
        password = codecs.encode(raw_in, 'rot13')
        self.ftp_prefs['CALIBRESPY_FTP_PASSWORD'] = password

        #~ for k,v in self.ftp_prefs.iteritems():
        for k,v in iteritems(self.ftp_prefs):
            self.return_new_local_pref_value(k,v,True)
        #END FOR

        self.hide()
        self.close()
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
    #-----------------------------------------------------
#END OF ftp_target_device_selection_dialog.py