#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

import os, os.path, sys, shutil, inspect, re
import options
from PIL import Image
from decimal import *  
import tkinter as tk
import tkinter.messagebox as mbox

try:
    from sigil_bs4 import BeautifulSoup, Comment, Tag
except:
    from bs4 import BeautifulSoup, Comment, Tag
    
    
    
___all__=["processAllTasks", "copyTextFiles2Dir", "copyCSSFiles2Dir", "copyImageFiles2Dir", "writeFiles2Epub", "writeFiles2CSS", "show_msgbox", "prettifyCSS", "cleanExit", "prettifyCSS1", "prettifyCSS2", "addMediaQueries2CSS", "formatImages", "dualFormatHTMLImages", "prettifyXHTMLFile", "getImageSize", "reformatImageLayout", "svgAttributes2CamelCase", "dualFormatSVGImages", "LogSVGChanges"]



def processAllTasks(bk, wdir, t_ids, t_fnames, s_ids, s_fnames):
    print('\n -- Processing automatic tasks...')
    for fname in t_fnames:
        if 'cover.xhtml' in fname or 'TOC.xhtml' in fname:
            continue
        else:   
            print('\n -- Processing next file...' + fname + '\n')
            fname = os.path.join(wdir, fname)
            reformatImageLayout(wdir, fname)    # put div/img formatting into a standard single format
            dualFormatHTMLImages(wdir, fname)   # dual format html <img> images for Kindle KF7 and KF8 devices
            dualFormatSVGImages(wdir, fname)    # dual format <svg> images for Kindle KF7 and KF8 devices
            LogSVGChanges(wdir, fname)

            # exit on error
            if options.SYS_EXIT == True:
                return(0)
    
            # reformats the svg attributes to appropriate camel case
            svgAttributes2CamelCase(wdir, fname)
            prettifyXHTMLFile(wdir, fname)     # prettify the html
            
    for file in s_fnames:
        if file == options.SET_CSS_FNAME:      
            file = os.path.join(wdir, file)
            prettifyCSS1(wdir, file)
            addMediaQueries2CSS(wdir, file)    # add the appropriate media queries to the selected css
            prettifyCSS2(wdir, file)  
    
    # write files back to the epub         
    writeFiles2Epub(bk, wdir, t_ids, t_fnames)
    writeFiles2CSS(bk, wdir, s_ids, s_fnames)        
    return(0)
    
def copyTextFiles2Dir(bk, wdir):

    print('\n -- In copyTextFiles2Dir()...')
    t_ids = list()
    t_hrefs = list()
    t_fnames = list()
    
    for (id, href) in bk.text_iter():
        t_ids.append(id)
        t_hrefs.append(href)
        t_fnames.append(os.path.basename(href))
    
    # copy all xhtml files to the working dir    
    file = str()
    t_fnames_r = list()
    t_ids_r = list()
    i = 0      
    for id in t_ids:
        file = os.path.join(wdir, t_fnames[i])
        if 'cover.xhtml' in file or \
            'cover.html' in file or \
            'cover.htm' in file or \
            'TOC.xhtml' in file or \
            'contents.xhtml' in file or \
            'titlepage.xhtml' in file:
            i = i + 1
            continue    
        
        print(file)
        with open(file, 'wt', encoding='utf-8') as outfp:
            data = bk.readfile(id)
            t_fnames_r.append(t_fnames[i])
            t_ids_r.append(id)
            outfp.writelines(data)
            i = i + 1
    
    return(t_ids_r, t_fnames_r)     
                
def copyCSSFiles2Dir(bk, wdir):

    print(' -- In copyCSSFiles2Dir()...')
    s_ids = list()
    s_hrefs = list()
    s_fnames = list()
    
    # get the associate fiids from the epub
    for (i, h) in bk.css_iter():
        s_ids.append(i)
        s_hrefs.append(h)
        s_fnames.append(os.path.basename(h))
        
    j = 0    
    for sid in s_ids:
        file = os.path.join(wdir, s_fnames[j])
        print(file)
        with open(file, 'wt', encoding='utf-8') as outfp: 
            data = bk.readfile(sid)
            outfp.writelines(data)             
            j = j + 1     
            
    return(s_ids, s_fnames)

def copyImageFiles2Dir(bk, wdir):
    print(' ')
    i_ids = list()
    i_hrefs = list()
    i_fnames = list()
    
    print(' -- In copyImages2Dir()...')
    for (i, h, m) in bk.image_iter():
        i_ids.append(i)
        i_hrefs.append(h)
        i_fnames.append(os.path.basename(h))
        
    j = 0    
    for iid in i_ids:
        file = os.path.join(wdir, i_fnames[j])
        print(file)
        with open(file, 'wb') as outfp: 
            data = bk.readfile(iid)
            outfp.write(data)                
            j = j + 1     
            
    return(i_ids, i_fnames) 
    
    
def writeFiles2Epub(bk, wdir, ids, fnames):

    print('\n -- In writeFiles2Epub()...' )       
    for file in fnames:    
       print(file)    

    count = len(fnames)
    if count == 0: 
        return(0)
    
    i = 0
    for file in fnames:
        file = os.path.join(wdir, file)
        with open(file, 'rt', encoding='utf-8') as fp:
            data = fp.read()           
            bk.writefile(ids[i], data)
            i = i + 1
                                

def writeFiles2CSS(bk, wdir, ids, s_fnames): 
    # no css files with imported html docs
  
    print('\n -- In writefile2CSS()...')
    count = len(s_fnames)
    if count == 0: 
        return(0) 
        
    i = 0
    for file in s_fnames:
        if os.path.getsize(os.path.join(wdir, file)) < 5:
            continue        
        prettifyCSS2(wdir, file) 
        print(str(file))        
        with open(os.path.join(wdir, file), 'rt', encoding='utf-8') as fp:              # input file is epub 
            data = fp.read()           
            bk.writefile(ids[i], data)
            i = i + 1
                        
    return(0)                  
    
def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
        
def prettifyCSS(wdir, css):
    output = os.path.join(wdir, 'link_rel.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(css, 'rt', encoding='utf8') as infp:      
        for line in infp:
            if re.match(r'^\s*$', line):
                continue 
            if line.strip() == '':
                continue   
            if line.strip() == ';':
                continue 
            if '{' in line:
                line = line.replace(' ', '')
                line = line.replace('{', '  {')                
            outfp.write(line.strip() + '\n')                
    
    outfp.close()
    os.remove(css)
    os.rename(output, css)
    return(0)  
 
def cleanExit(wdir):
    shutil.rmtree(wdir, ignore_errors=True)
    return(0)
 
def prettifyCSS1(wdir, css):
    output = os.path.join(wdir, 'reformat1.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(css, 'rt', encoding='utf8') as infp:      
        for line in infp:
           
            if ':' in line and ': ' not in line and not line.strip().startswith('a:'):
                line = line.replace(':', ': ')  

            if ':' in line and ': ' not in line:
                line = line.replace(':', ': ')            
           
           # format attribute from line to stacked format
            if '{' in line and ':' in line and '}' in line:
                line = line.replace('{', '  {\n')
                line = line.replace(';', ';\n')
                line = line.replace('}', '\n}\n')                
            outfp.write(line.strip() +'\n')
    
    outfp.close()
    os.remove(css)
    os.rename(output, css)    
    
    output = os.path.join(wdir, 'reformat2.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(css, 'rt', encoding='utf8') as infp:      
        for line in infp:
            if line.strip() == '':
                continue
        
           # ensure all style properties end in ';'
            if '{' not in line and '}' not in line and ',' not in line:
                if not line.strip().endswith(';'):
                    line = line + ';\n'                
            outfp.write(line.strip() + '\n')
    
    outfp.close()
    os.remove(css)
    os.rename(output, css)    
    return(0)
    
def prettifyCSS2(wdir, css):
    css = os.path.join(wdir, css)
    output = os.path.join(wdir, 'link_rel.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(css, 'rt', encoding='utf8') as infp:      
        for line in infp:
            if 'font-family:' not in line and '{' not in line and 'src: url' not in line and 'src:url' not in line:
                line = line.lower()    
        
            if re.match(r'^\s*$', line):
                continue 
            
            if line.strip() == '':
                continue   
            
            if line.strip() == ';':
                continue        
                
            outfp.write(line)
    
    outfp.close()
    os.remove(css)
    os.rename(output, css)    
    return(0)
    
def addMediaQueries2CSS(wdir, css):  
    """ Adds the appropriate media queries to
        the css to control KF7/KF8 ebook image 
        displays    
    """
    
    print('\n -- Add media queries to stylesheet...')
    
    output = os.path.join(wdir, 'link_rel.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(css, 'rt', encoding='utf8') as infp:      
        for line in infp:
            outfp.write(line)
        data = '@media amzn-mobi {\n'
        data += ' .kf8only {\n'
        data += '  display: none;\n' 
        data += '  }\n'
        data += ' .mobionly {\n'
        data += '  display: inline;\n' 
        data += '  }\n'     
        data += '}\n'       
     
        data += '@media not amzn-mobi {\n'
        data += ' .kf8only {\n'
        data += '  display: inline;\n' 
        data += '  }\n'
        data += ' .mobionly {\n'
        data += '  display: none;\n' 
        data += '  }\n'     
        data += '}\n'            
        outfp.write(data) 
    outfp.close()
    os.remove(css)
    os.rename(output, css)    
    return(0)
    
def formatImages(wdir, line):
    """ Dual formats all html <img> images, providing 
        KF8 image displays as a % of current screen width 
        as well as providing KF7 B & W displays using 
        absolute pixels values.
    """        
    
    # needed for later dual format processing
    original_line = line
    
    # find the image name & location
    soup = BeautifulSoup(line, 'html.parser')
    tag = soup.find('img')    
    if tag.has_attr('src'):
        text = tag['src']      
    else:
        return(line)  
    
    #get the img file name from the img path     
    file_name = os.path.basename(text)
    file_path = os.path.join(wdir, file_name)
     
    # get the image dimensions 
    width, height = getImageSize(file_path)                             # uses the PIL 'image' method to get image dimensions from the bitmap
    perc_width = round(width/int(options.BASE_WIDTH_VALUE) * 100)       # calculates width as a percentage of screen width
    perc_height = round(height/1200 * 100)                              # calculates height as a percentage of screen height
    
    # reduce image widths to 100% screen
    # width for all outsize images
    if perc_width >= 100:
        perc_width = 100
        
    if perc_height >= 100:
        perc_height = 100
        
    # precaution in case image html name contains spaces(not allowed in epubs)    
    file_name = os.path.split(file_name)[1]
    file_name = file_name.replace(' ', '_')
    
    ### Now dual format the image line ###
    
    # insert the height and width image values for KF8 devices
    soup = BeautifulSoup(line, 'html.parser')
    img = soup.img
    if img.has_attr('class'):
        del img['class']                      # remove all classes from the img tags 
    if not img.has_attr('style'):             
        if img.has_attr('width'):
            del img['width']
        if img.has_attr('height'):
            del img['height']            
        img['style'] = 'width: ' + str(perc_width) + '%;height: auto;'     # add the calculated % values into the inline style
    else:
        img['style'] = 'width: ' + str(perc_width) + '%;height: auto;'     # delete then re-add the calculated % values into the inline style   
    line = str(soup)
    
    # format % image values for kf8 devices only using inline styling
    img1 = soup.img
    if img1.has_attr('class'):
        del img1['class']            # remove any classes
        img1['class'] = 'kf8only'    # add the appropriate @media class
    else:    
        img1['class'] = 'kf8only'    # add the appropriate @media class if there are no classes present
        
    if img1.has_attr('id'):          # remove ids from the image tag -- not required(avoids id duplication errors on Epubcheck)
        del img1['id']    
        
    l = str(soup)
    line2 = l.strip() + '\n'
    
    # format pixel image values for mobi(KF7) only using 
    # img tag html attributes for height and width in pixels
    soup = BeautifulSoup(original_line, 'html.parser')
    img2 = soup.img
    
    # remove all inline styling from the img tag
    if img2.has_attr('style'):
        del img2['style']
    
    # remove and then adds the appropriate @media classes for mobi 
    if img2.has_attr('class'):
        del img2['class']
        img2['class'] = 'mobionly'
    else:
        img2['class'] = 'mobionly'
    
    # remove and add the appropriate 'px' values for h nd w
    # in pixels for mobi version only 
    if img2.has_attr('width'):
        del img2['width'] 
        img2['width'] = str(width) + 'px'
    else: 
        img2['width'] = str(width) + 'px'   
    
    if img2.has_attr('height'):
        del img2['height']
        img2['height'] = str(height) + 'px'
    else:
        img2['height'] = str(height) + 'px'    

    if img2.has_attr('id'):
        del img2['id']            
    
    # now join these two line as one(they already contain newline end chars)
    line1 = str(soup).strip()
    line = line1 + '\n' + line2
    
    return(line)    
    
def dualFormatHTMLImages(wdir, file):
    """ Dual formats all html <img> images, providing 
        KF8 image displays as a % of current screen width 
        as well as providing KF7 displays using pixels 
        values.
    """        
    print(' -- In dualFormatHTMLImages()...\n')   
    # inserts and reformats all ebook images 
    outfile = wdir + os.sep + 'images.html'
    infp = open(file, 'rt', encoding=('utf-8'))
    outfp = open(outfile, 'wt', encoding=('utf-8'))
    for line in infp:
        if '<img' in line: 
            if 'mobionly' not in line and 'kf8only' not in line:
                line = formatImages(wdir, line)
                print(line)
                outfp.write(line)
            else:
                msg = "Media queries have been detected in your code. You already have the relevant media queries in this epub."
                show_msgbox('Warning', msg, msgtype="warning")
                options.SYS_EXIT = True
                return(0)                
        else:
            outfp.write(line) 
        
    outfp.close()                
    infp.close() 
    os.remove(file) 
    os.rename(outfile, file)
    return(0)                       
        
def prettifyXHTMLFile(wdir, file):
    """ Prettifies the html """
    
    # reformat and prettify the XHTML file
    outfile= os.path.join(wdir, 'final_one.css')
    infp = open(file, 'rt', encoding='utf-8')
    outfp = open(outfile, 'wt', encoding='utf-8') 
    
    for line in infp:
            
        # prettify the html file
        if line.strip().startswith('<?xml') or \
            line.strip().startswith('<!DOCTYPE') or \
            line.strip().startswith('<html') or \
            line.strip().startswith('<head>') or \
            line.strip().startswith('<meta')or \
            line.strip().startswith('<title>') or \
            line.strip().startswith('<link') or \
            line.strip().startswith('</head>') or \
            line.strip().startswith('<body') or \
            line.strip().startswith('<body>'):
            line = line.strip()
            if not line:
                continue
            if line.startswith('<meta') or \
                line.startswith('<title>') or \
                line.startswith('<link'):
                line = '  ' + line      
            if line.startswith('<body'):
                line = '\n' + line
            if line.startswith('</body>'):
                outfp.write('\n' + line.rstrip() + '\n')
            else:
                outfp.write(line.rstrip() + '\n')   
        else:
            line = line.strip() 
            if not line or line == ';':
                continue
            if line.strip().startswith('<img'):
                line = line.strip().replace('/> <img', '/>\n\n<img')   
                line = line.strip().replace('<img', '  <img')
            if '<image' in line:
                line = line.strip().replace('<image', '\n  <image')                
            if line.strip().startswith('<p'):
                line = '  ' + line.strip()           
            outfp.write('\n' + line + '\n')    
            
    infp.close()
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)
    
    # reformat svg tags
    outfile= os.path.join(wdir, 'final.css')
    outfp = open(outfile, 'wt', encoding='utf-8') 
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
            if line.strip().startswith('<svg'):
                line = line.replace('<svg', '  <svg')
            outfp.write(line)
            
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)    
    return(0)                 
             
def getImageSize(image):
    """ Uses PIL to get image dimensions 
    """
    
    # error warning if images file name has spaces
    base = os.path.basename(image)
    print('IMAGE NAME...' + base)
    if '%20' in base or ' ' in base:
        msg = 'The epub image file name - "' + base + '" - contains spaces which are illegal in an epub.' \
              ' Replace all spaces with underscores and try again.'         
        show_msgbox('Error', msg, 'error')
        options.SYS_EXIT = True
        return(0)
        
    
    image = image.replace('\n', '')
    im = Image.open(image)
    ht = im.size[1]        
    wd = im.size[0]        
    return(wd, ht)
                    
def reformatImageLayout(wdir, file):
    """ This function just converts an img tag containing inline div tags 
        to div tags on separate lines(with newline characters). So this function 
        just converts all the different img tag/div tag formats that are possible 
        to a single standard format for ease of handing later on in the processing.     
    """
    print('-- In reformatImageLayout()...\n')
    outfile= os.path.join(wdir, 'final_one.css')
    infp = open(file, 'rt', encoding='utf-8')
    outfp = open(outfile, 'wt', encoding='utf-8') 
    with open(file, 'rt', encoding='utf-8') as infp:
        # if img tag has an inline div then reformat it to put 
        # the div tags on separate lines above and below the img tag           
        for line in infp:   
            soup = BeautifulSoup(line, 'html.parser')
            if soup.find('div') != None and soup.find('img') != None:
                line = line.replace('<img', '\n  <img')
                line = line.replace('<div', '\n<div')
                line = line.replace('</div>', '\n</div>')
                outfp.write(line)
            else:
                outfp.write(line)                   

    infp.close()
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)
    return(0)                 
                      
def svgAttributes2CamelCase(wdir, file):
    """ I wrote this function because of the problems and frustrations caused by 
        bs4 automatically converting all svg attributes to lowercase, causing 
        various epub svg errors. I also tried using bs4's "xml" parser as a remedy 
        and doing this gave me an even bigger mess. So, in the end, I wrote 
        this simple function which just converts certain svg attributes back 
        to proper camel case.
    """       
    
    print(' -- svgAttributes2CamelCase()...')
    file = os.path.join(wdir, file)
    file = os.path.join(wdir, os.path.basename(file))       
    output = os.path.join(wdir, 'reformat.html')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()

    soup = BeautifulSoup(html, 'html.parser')
    
    for svg in soup.find_all('svg'):
    
        # reformat the par to camel case
        if svg.has_attr('preserveaspectratio'):
            par = svg['preserveaspectratio']
            del svg['preserveaspectratio']
            svg['preserveAspectRatio'] = par
        
        # reformat the vb to camel case
        if svg.has_attr('viewbox'):
            vb = svg['viewbox']
            del svg['viewbox']
            svg['viewBox'] = vb
            
        if not svg.has_attr('xmlns:xlink'):
            svg['xmlns:xlink'] = "http://www.w3.org/1999/xlink"        
               
    outfp.writelines(str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
    outfp.close()
    os.remove(file)
    os.rename(output, file)        
    return(0)        
    
def dualFormatSVGImages(wdir, file):
    """ Leaves SVG formatting as is for KF8 and
        formats the image for KF7 in pixels.
    """    
   
    ### dual format with <img> line
    print(' -- In dualFormatSVGImages()...\n')
    
    outfile= os.path.join(wdir, 'svg2.css')
    outfp = open(outfile, 'wt', encoding='utf-8') 
    html = open(file, 'rt', encoding='utf-8').read()
    
    soup = BeautifulSoup(html, 'html.parser')
    
    # add the "kf8only" class to the kf8 div tag
    for div in soup.find_all('div'):
        if div.svg != None:
            div['class'] = div.get('class', []) + ['kf8only']          
    
    # add the "mobionly" class to the new kf7 div tag
    div_strng = str()
    strng = str()
    for div_tag in soup.find_all('div'):
        if div_tag.svg != None:
            href = div_tag.svg.image['xlink:href']
            width = div_tag.svg.image['width'] 
            div_strng = str(div_tag)            
         
            #get the img file name from the img path     
            file_name = os.path.basename(href)
            file_path = os.path.join(wdir, file_name)
             
            # get the image dimensions
            width, height = getImageSize(file_path)                             
            
            # create the new <img> line for mobi only
            img_line = '\n<div class="mobionly">' + \
                       '\n<img alt="" src="' + href + '" width="' + str(width) + \
                       'px" height="' + str(height) + 'px" />\n' + \
                       '</div>\n'
             
            div_tag.insert_after(soup.new_string(img_line))
            LogSVGChanges(str(div_tag), img_line)
            
    outfp.writelines(str(soup).replace('&lt;', '<'). replace('&gt;', '>'))
    outfp.close()    
    os.remove(file)
    os.rename(outfile, file)
    return(0)            
    
def LogSVGChanges(svg_line, img_line):
       
    soup = BeautifulSoup(svg_line, 'html.parser')
    
    svg = soup.find('svg')
    if svg == None:
        return
    else:    
        # reformat the par to camel case
        if svg.has_attr('preserveaspectratio'):
            par = svg['preserveaspectratio']
            del svg['preserveaspectratio']
            svg['preserveAspectRatio'] = par
        
        # reformat the vb to camel case
        if svg.has_attr('viewbox'):
            vb = svg['viewbox']
            del svg['viewbox']
            svg['viewBox'] = vb
            
        if not svg.has_attr('xmlns:xlink'):
            svg['xmlns:xlink'] = "http://www.w3.org/1999/xlink"       
    
    strng = str(soup)
    strng = strng.replace('&lt;', '<').replace('&gt;', '>')
    print(strng)
    print(img_line)
    print('------------------------------------' + '\n' )                           
               
    return(0)
    
                