import ctypes
import errno
import os
import platform
import string
import subprocess

import zipfile


from calibre_plugins.manga.manga_api.chapter_container import ChapterContainer

APP_NAME = "MangaFinder"
APP_BOOK_DIR_NAME = "Books"

class BuildFormats:

    CBZ = "cbz"
    EPUB = "epub"

    def __init__(self):
        pass

    @staticmethod
    def get_build_function(build_format):
        # type: (BuildFormats) -> Callable[[List[ChapterContainer], AnyStr, AnyStr], None]

        switcher = {
            "cbz": Utils.build_cbz,
            "epub": Utils.build_epub
                }

        return switcher.get(build_format, None)

class Utils:

    @staticmethod
    def make_filename_safe(s):
        valid_chars = "-_.() %s%s[]" % (string.ascii_letters, string.digits)
        filename = ''.join(c for c in s if c in valid_chars)
        # filename = filename.replace(' ', '_')  # I don't like spaces in filenames.

        return filename

    @staticmethod
    def create_dir_if_not_exist(dir_path):
        if not os.path.exists(dir_path):
            try:
                os.makedirs(dir_path)
            except OSError as exc:  # Guard against race condition
                print(exc)
                if exc.errno != errno.EEXIST:
                    raise

    @staticmethod
    def build_cbz(chapter_containers, filename=None, location=""):
        # type: (List[ChapterContainer], str, str) -> str
        if len(chapter_containers) <= 0:
            return None

        manga_container = chapter_containers[0].manga_container
        output_dir = Utils.get_sub_folder_book(location)
        cbz_filename = Utils.make_filename_safe(filename if filename else manga_container.name)
        cbz_filename = os.path.join(output_dir, "{}.cbz".format(cbz_filename))

        Utils.create_dir_if_not_exist(output_dir)
        try:
            if os.path.exists(cbz_filename):
                os.remove(cbz_filename)
            zf = zipfile.ZipFile(cbz_filename, 'a')
            try:
                p_index_total = 0;
                for c_index, chapter in enumerate(chapter_containers):
                    page_list = chapter.pages
                    for p_index, page in enumerate(page_list):
                        filename, file_extension = os.path.splitext(page)

                        zf.write(page, arcname="{}{}".format(format(p_index_total, '05d'), file_extension))

                        p_index_total += 1
            finally:
                zf.close()
        except IOError as e:
            print(e)

        print(os.path.abspath(cbz_filename))
        return os.path.abspath(cbz_filename)

    @staticmethod
    def build_epub(chapter_containers, filename=None, location=""):
        # type: (List[ChapterContainer], str, str) -> str

        if len(chapter_containers) <= 0:
            return None

        manga_container = chapter_containers[0].manga_container
        epub_filename = os.path.join(location, "{}.epub".format(filename if filename else manga_container.name))
        epub_filename = Utils.make_filename_safe(epub_filename)

        book = epub.EpubBook()
        book.set_identifier(filename)
        book.set_title(filename)
        book.set_language('en')

        book.add_author(manga_container.author)
        # book.set_cover("image.jpg", open('cover.jpg', 'rb').read())

        book.spine = ['nav']
        nav = epub.EpubNav()
        nav.is_linear = False

        style = '''
        body { margin: 0em; padding: 0em; }
        img {
            display: block;
            margin-left: auto;
            margin-right: auto;
            height: 100%;
        }
        '''

        css = epub.EpubItem(uid="style_default", file_name="style/default.css", media_type="text/css", content=style)
        book.add_item(css)

        chapter_links = []
        pages_html = []
        pages_image = []

        for c_index, chapter in enumerate(chapter_containers):
            page_list = chapter.pages
            for p_index, page in enumerate(page_list):
                filename, file_extension = os.path.splitext(page)

                output_filename_xhtml = "{}-{}.xhtml".format(chapter.c_number, format(p_index, '05d'))
                if p_index == 0:
                    chapter_links.append(epub.Link(href=output_filename_xhtml, title=chapter.name, uid=chapter.c_number))
                output_filename_image = "{}-{}{}".format(chapter.c_number, format(p_index, '05d'), file_extension)
                page_image = epub.EpubCover(uid=output_filename_image, file_name=output_filename_image)
                with open(page, "r+b") as fs:
                    page_image.content = fs.read()
                book.add_item(page_image)
                page_html = epub.EpubHtml(uid=output_filename_xhtml, file_name=output_filename_xhtml, title="")
                page_html.content = u'<div><img src="{}" alt="" /></div>'.format(output_filename_image)
                page_html.add_item(css)
                book.add_item(page_html)

                pages_html.append(page_html)
                pages_image.append(page_image)

        print(len(chapter_links))

        book.toc = tuple(chapter_links)

        # book.add_item(epub.EpubNcx())
        for i in pages_html:
            nav.add_item(i)
        book.add_item(nav)

        for i in pages_html:
            book.spine.append(i)

        # print epub_filename
        epub.write_epub(epub_filename, book, {})

        return os.path.abspath(epub_filename)

    @staticmethod
    def get_content_folder():
        return os.path.join(Utils.get_download_folder(), APP_NAME)

    @staticmethod
    def get_sub_folder_book(location=None):
        if not location:
            location = Utils.get_content_folder()
        return os.path.join(location, APP_BOOK_DIR_NAME)

    if os.name == 'nt':
        @staticmethod
        def get_download_folder():
            return _get_known_folder_path(FOLDERID_Download)
    else:
        @staticmethod
        def get_download_folder():
            home = os.path.expanduser("~")
            return os.path.join(home, "Downloads")

    @staticmethod
    def show_in_explorer(path):
        if platform.system() == "Windows":
            os.startfile(path)
        elif platform.system() == "Darwin":
            subprocess.Popen(["open", path])
        else:
            subprocess.Popen(["xdg-open", path])


if os.name == 'nt':
    import ctypes
    from ctypes import windll, wintypes
    from uuid import UUID

    # ctypes GUID copied from MSDN sample code
    class GUID(ctypes.Structure):
        _fields_ = [
            ("Data1", wintypes.DWORD),
            ("Data2", wintypes.WORD),
            ("Data3", wintypes.WORD),
            ("Data4", wintypes.BYTE * 8)
        ]

        def __init__(self, uuidstr):
            uuid = UUID(uuidstr)
            ctypes.Structure.__init__(self)
            self.Data1, self.Data2, self.Data3, \
                self.Data4[0], self.Data4[1], rest = uuid.fields
            for i in range(2, 8):
                self.Data4[i] = rest >> (8-i-1) * 8 & 0xff

    SHGetKnownFolderPath = windll.shell32.SHGetKnownFolderPath
    SHGetKnownFolderPath.argtypes = [
        ctypes.POINTER(GUID), wintypes.DWORD,
        wintypes.HANDLE, ctypes.POINTER(ctypes.c_wchar_p)
    ]

    def _get_known_folder_path(uuidstr):
        pathptr = ctypes.c_wchar_p()
        guid = GUID(uuidstr)
        if SHGetKnownFolderPath(ctypes.byref(guid), 0, 0, ctypes.byref(pathptr)):
            raise ctypes.WinError()
        return pathptr.value

    FOLDERID_Download = '{374DE290-123F-4565-9164-39C4925E467B}'