import json
import os

from calibre_plugins.manga.thread_task.download_manager import DownloadManager
from calibre_plugins.manga.thread_task.download_page_task import DownloadPageTask
from calibre_plugins.manga.thread_task.parallel_task import ParallelTask
from calibre_plugins.manga.utils import Utils

try:
    from urllib2 import urlopen, Request
except:
    from urllib.request import urlopen, Request

from PyQt5 import QtCore

from calibre_plugins.manga.thread_task.base_task import BaseTask, BaseTaskRunnable

from calibre_plugins.manga.manga_api.chapter_container import ChapterContainer

mutex = QtCore.QMutex()


class RequestChapterContentTask(BaseTask):

    _parallel_limit = 0

    def __init__(self, chapter_container, download_if_needed=True):
        # type: (ChapterContainer) -> None
        BaseTask.__init__(self)
        self.chapter_container = chapter_container
        self.download_if_needed = download_if_needed

    def get_runnable(self):
        return _RequestChapterContentTaskRunnable(self)

    def on_parallel_task_completed(self, task, results):
        self.create_metadata(json.dumps(results))
        self.chapter_container.pages = list(results)
        self.results = list(results)
        self.emit_completed()
        DownloadManager.emit_chapter_complete_event(self.chapter_container)

    def on_parallel_task_update(self, task, current, total):
        # print float(current)/total
        if current < total:
            self.increment_progress()

    def on_parallel_task_error(self, task, error):
        self.on_error.emit(task, error)

    def get_metadata_filename(self):
        return os.path.join(self.get_output_dir(), "metadata.json")

    def get_metadata_version(self):
        return 1

    def create_metadata(self, data):
        metadata = {"metadata_version": self.get_metadata_version(),
                    "module": self.chapter_container.manga_container.module.get_module_id(),
                    "m_id": self.chapter_container.manga_container.m_id,
                    "c_id": self.chapter_container.c_id,
                    "name": self.chapter_container.name,
                    "c_number": str(self.chapter_container.c_number),
                    "last_updated": self.chapter_container.last_updated,
                    "pages": json.loads(data)}
        metadata_file = self.get_metadata_filename()
        with open(metadata_file, "w") as fs:
            fs.write(json.dumps(metadata, ensure_ascii=False))

    def get_metadata(self):
        metadata_file = self.get_metadata_filename()
        if os.path.exists(metadata_file) and os.path.isfile(metadata_file):
            with open(metadata_file, "r") as fs:
                content = fs.read()
                metadata = json.loads(content)
                if metadata["metadata_version"] >= self.get_metadata_version():
                    return metadata

        return None

    def get_output_dir(self):
        location = Utils.get_content_folder()
        manga_container = self.chapter_container.manga_container
        chapter_dir_name = "[{}]-[{}]".format(format(self.chapter_container.c_number, '04f'),
                                              Utils.make_filename_safe(self.chapter_container.name))

        return os.path.join(location, manga_container.module.get_module_id(), manga_container.m_id, chapter_dir_name)

    def __eq__(self, other):
        if isinstance(other, RequestChapterContentTask):
            return self.chapter_container == other.chapter_container
        return False


class _RequestChapterContentTaskRunnable(BaseTaskRunnable):

    def execute(self):
        output_dir = self.task.get_output_dir()
        Utils.create_dir_if_not_exist(output_dir)

        metadata = self.task.get_metadata()

        if metadata is not None:
            self.load_local_content(metadata)
        elif self.task.download_if_needed:
            self.download_chapter()
        else:
            self.task.results = None
            self.task.emit_completed()

    def load_local_content(self, metadata):
        pages = metadata['pages']
        self.task.chapter_container.pages = pages
        self.task.results = pages
        self.task.emit_completed()
        pass

    def download_chapter(self):
        module = self.task.chapter_container.manga_container.module
        chapter_container = self.task.chapter_container

        output_dir = self.task.get_output_dir()
        manga_container = chapter_container.manga_container

        page_urls = module.get_page_list(chapter_container)
        # print page_urls
        self.task.set_progress_total(len(page_urls))
        # print("number of pages:{}".format(len(page_urls)))
        # print("current:{}".format(self.task.get_progress()))
        chapter_container.pages = None
        parallel_task = ParallelTask()
        for index, url in enumerate(page_urls):
            task = DownloadPageTask(url, output_dir, index)
            parallel_task.add_task(task)

        parallel_task.setObjectName("RequestChapterContentTask")
        parallel_task.on_error.connect(self.task.on_parallel_task_error)
        parallel_task.on_completed.connect(self.task.on_parallel_task_completed)
        parallel_task.on_process_update.connect(self.task.on_parallel_task_update)
        parallel_task.start()

    def on_completed(self):
        pass
