import os
import struct

try:
    from urllib2 import urlopen, Request
except:
    from urllib.request import urlopen, Request

from PyQt5 import QtCore
from PyQt5.QtCore import QByteArray, QBuffer
from PyQt5.QtGui import QImage
from enum import Enum

from calibre_plugins.manga.thread_task.base_task import BaseTask, BaseTaskRunnable
mutex = QtCore.QMutex()


def _download(url):
    req = Request(url)
    req.add_header("User-Agent", "")
    response = urlopen(req)

    return response.read()


def _parse_mri_data_to_webp_buffer(data):
    size_list = [0] * 4
    size = len(data)
    header_size = size + 7

    # little endian byte representation
    # zeros to the right don't change the value
    for i, byte in enumerate(struct.pack("<I", header_size)):
        size_list[i] = byte

    buffer = [
        82,  # R
        73,  # I
        70,  # F
        70,  # F
        size_list[0],
        size_list[1],
        size_list[2],
        size_list[3],
        87,  # W
        69,  # E
        66,  # B
        80,  # P
        86,  # V
        80,  # P
        56,  # 8
    ]

    for bit in data:
        buffer.append(101 ^ bit)

    return buffer


def _convert_To_JPEG(image_data):
    img = QImage()
    img.loadFromData(bytearray(image_data))
    ba = QByteArray()
    buffer = QBuffer(ba)
    img.save(buffer, "jpeg")
    return ba


class ConvertFormat(Enum):
    DEFAULT = 0
    WEBP = 1
    JPEG = 2
    PNG = 3


class DownloadPageTask(BaseTask):

    _parallel_limit = 1

    def __init__(self, page_url, output_dir, output_filename, convert_format=ConvertFormat.JPEG):
        # type: (AnyStr) -> None
        BaseTask.__init__(self)
        self.page_url = page_url
        self.output_dir = output_dir
        self.output_filename = output_filename
        self.format = convert_format

    def get_runnable(self):
        return _DownloadPageTaskRunnable(self)


class _DownloadPageTaskRunnable(BaseTaskRunnable):

    def execute(self):
        url = self.task.page_url
        output_dir = self.task.output_dir
        output_filename = self.task.output_filename
        convert_format = self.task.format

        filename, file_extension = os.path.splitext(url)
        image_data = bytearray(_download(url))

        if file_extension == ".mri":
            image_data = _parse_mri_data_to_webp_buffer(image_data)
            file_extension = ".webp"

        if convert_format is not None:
            if convert_format is ConvertFormat.JPEG:
                image_data = _convert_To_JPEG(image_data)
                file_extension = ".jpg"

        output_filename = os.path.join(output_dir, "{}{}".format(format(output_filename, '04d'), file_extension))

        with open(output_filename, "w+b") as fs:
            fs.write(bytearray(image_data))

        self.task.results = output_filename

        # print("downloaded:{}".format(url))




