import traceback

from PyQt5 import QtCore
from PyQt5.Qt import QObject
from PyQt5.QtCore import pyqtSignal, QRunnable
from PyQt5.QtWidgets import QApplication

from calibre_plugins.manga.thread_task.task_manager import TaskManager

try:
    from calibre_plugins.manga.manga_api import mangarock
except ImportError:
    from calibre_plugins.manga.manga_api import mangarock_API

import time


mutex = QtCore.QMutex()


class BaseTask(QObject):

    on_completed = pyqtSignal(object, object)
    on_error = pyqtSignal(object, str)

    on_process_update = pyqtSignal(object, int, int, str)

    _parallel_limit = 0

    def __init__(self):
        # type: () -> None
        QObject.__init__(self)

        self._is_running = False
        self._error = None
        self._is_completed = False

        self._chain_tasks = []

        self._current_progress_index = 0
        self._number_of_progress = 1

        self.start_time = 0
        self.end_time = 0

        self.runnable = None
        self.results = None

        self.moveToThread(TaskManager.get_instance())

    def init_runnable(self):
        self.runnable = self.get_runnable()

    def get_runnable(self):
        pass

    def start(self, single_only=False):
        TaskManager.get_instance().add_task(self, single_only)

    def stop(self):
        TaskManager.get_instance().remove_task(self)

    def get_parallel_limit(self):
        return self._parallel_limit

    def chain(self, task):
        self._chain_tasks.append(task)
        self.on_completed.connect(task.start)

    def is_running(self):
        return self._is_running

    def get_error(self):
        return self._error

    def is_completed(self):
        return self._is_completed

    def get_results(self):
        return self.results

    def get_progress_total(self):
        return self._number_of_progress

    def set_progress_total(self, num_of_process):
        self._number_of_progress = num_of_process

    def get_progress(self):
        return self._current_progress_index, self._number_of_progress

    # def add_progress(self, value, verbal=""):
    #     self._current_process_index += value
    #     self.update_progress(self._current_process_index, verbal)

    def on_completed_progress(self, verbal=""):
        self._current_progress_index = self._number_of_progress
        self.update_progress(self._current_progress_index)

    def increment_progress(self, verbal=""):
        self._current_progress_index += 1
        self.update_progress(self._current_progress_index, verbal)
        # print self._current_process_index

    def update_progress(self, current_process_index, verbal=""):
        self._current_progress_index = current_process_index
        if self._current_progress_index > self._number_of_progress:
            raise Exception('current progress greater than total progress!')
        self.on_process_update.emit(self, current_process_index, self._number_of_progress, verbal)

    def emit_completed(self):
        # print("complete:{}".format(self))
        if not self._is_completed:
            self._is_running = False
            self._is_completed = True
            # time.sleep(0.1)
            self.on_completed_progress()
            self.on_completed.emit(self, self.results)

class BaseTaskRunnable(QRunnable):

    def __init__(self, task):
        # type: (BaseTask) -> None
        QRunnable.__init__(self)
        self.task = task

    def run(self):
        self.task.start_time = int(round(time.time() * 1000))
        self.task._is_running = True
        # print(type(self.task))
        try:
            self.execute()
            self.task.end_time = int(round(time.time() * 1000))
            self.on_completed()
        except Exception as e:
            traceback.print_exc()
            self.task._error = e
            self.task.on_error.emit(self.task, str(e))

    def on_completed(self):
        self.task._is_running = False
        self.task._is_completed = True
        self.task.on_completed_progress()
        self.task.on_completed.emit(self.task, self.task.results)

    def execute(self):
        pass
