from __future__ import absolute_import, division, print_function, unicode_literals

import os
import zipfile

from PyQt5 import QtCore
from PyQt5.QtWidgets import QDialog, QLineEdit, QListView, QHBoxLayout, QVBoxLayout, \
    QGroupBox, QTabWidget, QWidget, QApplication, QComboBox

from calibre_plugins.manga.manga_api.module_manager import ModuleManager
from calibre_plugins.manga.view.download_page import DownloadPage
from calibre_plugins.manga.view.manga_page import MangaPage

try:
    from calibre_plugins.manga.config import prefs
except ImportError:
    prefs = None
    pass

from calibre_plugins.manga.thread_task.request_manga_quick_list_task import RequestMangaQuickListTask

from calibre_plugins.manga.view.manga_list_delegate import MangaListDelegate
from calibre_plugins.manga.model.manga_list_model import MangaListModel

mutex = QtCore.QMutex()


try:
    import calibre
    running_in_calibre = True
except ImportError:
    running_in_calibre = False


class MangaFinderDialog(QDialog):

    def __init__(self, gui=None, icon=None, do_user_config=None):
        QDialog.__init__(self, gui)

        self.gui = gui
        self.do_user_config = do_user_config
        self.selected_chapters = []
        self.latest_search_time = 0

        if gui is not None:
            self.db = gui.current_db # type: calibre.db.cache.Cache

        self.setWindowTitle('Manga Finder')
        if icon is not None:
            self.setWindowIcon(icon)

        self.root_layout = QHBoxLayout()

        self.create_manga_quick_search()
        self.root_layout.addWidget(self.quick_search_bar)

        self.manga_page = MangaPage(self)
        self.manga_page.complete_build.connect(self.on_complete_build)
        self.root_layout.addWidget(self.manga_page)

        search_page = QWidget()
        search_page.setLayout(self.root_layout)

        # self.setLayout(self.root_layout)

        test_layout = QHBoxLayout()
        self.setLayout(test_layout)
        tabWidget = QTabWidget()
        tabWidget.addTab(search_page, "Search")

        self.download_page = DownloadPage()
        tabWidget.addTab(self.download_page, "Downloads")

        test_layout.addWidget(tabWidget)

        self.resize(960, 720)

    def create_manga_quick_search(self):
        self.quick_search_bar = QGroupBox("Search:")
        self.quick_search_bar.setFixedWidth(200)
        layout = QVBoxLayout()

        self.module_Combo_box = QComboBox()
        self.module_Combo_box.setEditable(False)
        for i, module in enumerate(ModuleManager.get_module_list()):
            self.module_Combo_box.addItem(module.get_module_id())
        self.selected_module = ModuleManager.get_module_list()[0]
        self.module_Combo_box.currentIndexChanged.connect(self.on_module_change)
        layout.addWidget(self.module_Combo_box)

        self.textbox = QLineEdit(self)
        self.textbox.move(20, 20)
        self.textbox.resize(280, 40)
        self.textbox.textChanged.connect(self.update_search_results)
        layout.addWidget(self.textbox)

        self.manga_list_view = QListView(self)
        self.manga_list_model = MangaListModel()
        self.manga_list_delegate = MangaListDelegate()
        self.manga_list_view.setModel(self.manga_list_model)
        self.manga_list_view.selectionModel().selectionChanged.connect(self.on_manga_selected)
        self.manga_list_view.setItemDelegate(self.manga_list_delegate)

        layout.addWidget(self.manga_list_view)

        self.quick_search_bar.setLayout(layout)

    def on_module_change(self, index):
        self.selected_module = ModuleManager.get_module_list()[index]
        self.update_search_results(self.textbox.text())

    def on_manga_selected(self, selected):
        indexes = selected.indexes()
        if len(indexes) > 0:
            self.manga = self.manga_list_model.get_data_at(indexes[0].row())
            self.manga_page.set_manga(self.manga)

    def update_search_results(self, search_string):
        self.manga_list_view.clearSelection()
        self.manga_list_model.clear()
        task = RequestMangaQuickListTask(self.selected_module, search_string)
        task.on_completed.connect(self.update_manga_list_view)
        task.start(True)

    def update_manga_list_view(self, task):
        if task.start_time > self.latest_search_time:
            self.latest_search_time = task.start_time
            search_results = task.get_results()
            self.manga_list_model.clear()
            for manga in search_results:
                self.manga_list_model.add(manga)

    def on_complete_build(self, task, results):
        if running_in_calibre:
            manga = task.chapter_containers[0].manga_container
            filename , ext = os.path.splitext(os.path.basename(results))
            self.add_book(results, filename, manga.authors)

    def add_book(self, book_file_path, title, authors=None):
        from calibre.ebooks.metadata import MetaInformation
        mi = MetaInformation(title, authors)
        filename, file_extension = os.path.splitext(book_file_path)
        dirpath = os.path.abspath(book_file_path)
        book_id = self.db.new_api.create_book_entry(mi, self.read_first_book_image(book_file_path))
        successful = self.db.new_api.add_format(book_id, file_extension, dirpath)

        self.refresh_gui(1, set_current_row=0)

    def read_first_book_image(self, book_file_path):
        with zipfile.ZipFile(book_file_path) as zip:
            with zip.open(zip.infolist()[0].filename) as f:
                return f.read()

    def refresh_gui(self, num, set_current_row=-1, recount=True):
        self.gui.library_view.model().books_added(num)
        if set_current_row > -1:
            self.gui.library_view.set_current_row(0)
        self.gui.refresh_cover_browser()
        if recount:
            self.gui.tags_view.recount()

        self.gui.library_view.model().refresh()


if __name__ == '__main__':
    app = QApplication([])
    window = MangaFinderDialog()
    window.show()
    app.exec_()
        