import re

from lxml import etree, html

from calibre_plugins.manga.manga_api.api_module import APIModule

try:
    from urllib2 import urlopen, Request
    from urllib import quote
except:
    from urllib.request import urlopen, Request
    from urllib.parse import quote


import json


from .chapter_container import ChapterContainer
from .manga_container import MangaContainer
from decimal import Decimal
from datetime import datetime, time


class MangafoxAPI(APIModule):

    def get_module_id(self):  # type: () -> AnyStr
        return "mangafox.me"

    def get_host(self):
        return "fanfox.net"

    def get_base_url(self):
        return "http://fanfox.net"

    def _get_quick_search_without_details(self, search_string):
        if len(search_string) < 3:
            return []

        search_string = quote(search_string)
        url = "{}/search?title={}".format(self.get_base_url(), search_string)

        req = Request(url)
        req.add_header("User-Agent", "")
        req.add_header('Cookie', 'isAdult=1')
        response = urlopen(req)
        tree = html.fromstring(response.read())
        nodes = tree.xpath('/html/body/div[2]/div/div/ul/li')

        m_ids = []
        for i in nodes:
            title = i[0].attrib["title"]
            href = i[0].attrib["href"]
            m_id = href.replace("/manga/", "").replace("/", "")
            m_ids.append(m_id)

        return m_ids

    def get_quick_search(self, search_string):
        if len(search_string) >= 3:
            search_results = self._get_quick_search_without_details(search_string)
            series_details = self.get_manga_list(search_results)
            return series_details

        return []

    def get_manga(self, m_id):
        url = "{}/manga/{}/".format(self.get_base_url(), m_id)

        req = Request(url)
        req.add_header("Accept", "*/*")
        req.add_header("User-Agent", "")
        req.add_header('Cookie', 'isAdult=1')
        response = urlopen(req).read()
        tree = html.fromstring(response)

        manga = MangaContainer(self, m_id)
        manga.name = tree.xpath("//span[@class='detail-info-right-title-font']")[0].text
        manga.authors = map(lambda x: x.text, tree.xpath("/html/body/div[5]/div/div[2]/p[2]/a"))
        manga.cover = tree.xpath("//img[@class='detail-info-cover-img']")[0].attrib["src"]
        manga.description = tree.xpath("//p[@class='fullcontent']")[0].text
        chapter_nodes = tree.xpath("//ul[@class='detail-main-list']/li/a")

        manga.chapters = []
        for c in chapter_nodes:
            c_id = c.attrib["href"].replace("/manga/{}/".format(m_id), '').replace("/1.html", "")
            c_id = re.sub(r'v.*?\/', '', c_id)
            c_id = c_id.replace("c", "")
            chapter = ChapterContainer(manga, c_id)
            chapter.name = c.attrib["title"].replace("{} ".format(manga.name), "")
            chapter.c_number = Decimal(c_id)
            try:
                date = datetime.strptime(c[0][1].text, '%b %d,%Y ')
            except:
                date = datetime.now()
            chapter.last_updated = (date - datetime(1970, 1, 1)).total_seconds()

            manga.chapters.append(chapter)

        return manga

    def get_manga_list(self, manga_ids):
        # type: (List[string]) -> List[MangaContainer]

        manga_list = []
        for m_id in manga_ids:
            manga = self.get_manga(m_id)
            manga_list.append(manga)

        return manga_list

    def get_chapter_list(self, manga_container):
        # type: (MangaContainer) -> List[ChapterContainer]
        manga = self.get_manga(manga_container.m_id)
        manga_container.chapters = manga.chapters
        for ch in manga_container.chapters:
            ch.manga_container = manga_container

        return manga.chapters

    def _get_url_data(self, url):
        req = Request(url)
        req.add_header("Host", self.get_host())
        req.add_header("User-Agent", "")
        req.add_header("Referer", url)
        req.add_header("Cache-Control", "no-cache")
        return urlopen(req).read()

    def _get_page_list(self, c_id, internal_c_id, page_start_index=1):
        url = "{}/manga/chapterfun.ashx?cid={}&page={}".format(self.get_base_url(), internal_c_id, page_start_index)
        req = Request(url)
        req.add_header("Host", self.get_host())
        req.add_header("User-Agent", "")
        req.add_header("Referer", self.get_base_url())
        req.add_header("Cookie", "image_time_cookie={}|0|0".format(internal_c_id))
        req.add_header("Cache-Control", "no-cache")
        response = urlopen(req).read()

        t = re.findall(r'\}\(.*{}\)', response)[0]
        t = "[{}]".format(t[2:-1])

        def swap_quotes(s):
            s = s.replace("\"", "'")
            return "\"{}\"".format(s[1:-1])

        t = re.sub('\'.*?\'', lambda x: swap_quotes(x.group()), t)

        def split(s):
            return ",{}".format(json.dumps(s[1:-1].split('|')))

        t = re.sub(r'(?:\,)(\".*\")(?:\.split\(\"\|\"\))', lambda x: split(x.group(1)), t)

        param_list = json.loads(t)
        js = MangafoxAPI._parse(*param_list)

        url_base = re.findall(r'(?:(pix=))\'(.*?)\'', js)[0][1]
        url_page_list = re.findall(r'(?:(pvalue=))(\[.*?\])', js)[0][1]
        url_page_list = json.loads(url_page_list.replace("'", "\""))
        for i, url in enumerate(url_page_list):
            url_page_list[i] = "http:{}{}".format(url_base, url)

        return url_page_list

    @staticmethod
    def _base36encode(number, alphabet='0123456789abcdefghijklmnopqrstuvwxyz'):
        """Converts an integer to a base36 string."""
        if not isinstance(number, (int, long)):
            raise TypeError('number must be an integer')

        base36 = ''
        sign = ''

        if number < 0:
            sign = '-'
            number = -number

        if 0 <= number < len(alphabet):
            return sign + alphabet[number]

        while number != 0:
            number, i = divmod(number, len(alphabet))
            base36 = alphabet[i] + base36

        return sign + base36

    @staticmethod
    def _parse(p, a, c, k, e, d):

        def e(c):
            part1 = ""
            if c < a:
                part1 = ""
            else:
                part1 = e(int(c / a))
            part2 = ""
            c = c % a
            if c > 35:
                part2 = chr(c + 29)
            else:
                part2 = MangafoxAPI._base36encode(c)

            return "{}{}".format(part1, part2)

        while c:
            c = c - 1
            d[e(c)] = k[c] or e(c)

        def k1(e):
            if e == '':
                return ''
            return d[e]

        k = [k1]

        def e1():
            return '\\w+'

        c = 1

        while not c == 0:
            c = c - 1
            if k[c]:
                p = re.sub(r'[\w]*', lambda x: k[c](x.group()), p)

        return p

    def get_chapter_url(self, chapter_container):
        m_id = chapter_container.manga_container.m_id
        c_id = chapter_container.c_id
        return "{}/manga/{}/c{}/1.html".format(self.get_base_url(), m_id, c_id)

    def get_page_list(self, chapter_container):
        m_id = chapter_container.manga_container.m_id
        c_id = chapter_container.c_id
        url = self.get_page_list_url(chapter_container)
        data = self._get_url_data(url)
        internal_c_id = int(re.search(r'(?:var\ chapterid\ =)(\d*)', data).group(1))

        page_list_1 = self._get_page_list(c_id, internal_c_id)

        if len(page_list_1) > 1:
            page_list_2 = self._get_page_list(c_id, internal_c_id, 2)
            for i in page_list_2:
                if i not in page_list_1:
                    page_list_1.append(i)
            page_list_1.remove(page_list_1[-1])

        return page_list_1