import re

# from PyQt5.QtQml import QJSEngine
from lxml import etree, html

from calibre_plugins.manga.manga_api.api_module import APIModule

try:
    from urllib2 import urlopen, Request
    from urllib import quote
except:
    from urllib.request import urlopen, Request
    from urllib.parse import quote


import json


from .chapter_container import ChapterContainer
from .manga_container import MangaContainer
from decimal import Decimal
from datetime import datetime, time


class MangafoxAPI(APIModule):

    def get_module_id(self):  # type: () -> AnyStr
        return "mangafox.me"

    def _get_quick_search_without_details(self, search_string):
        if len(search_string) < 3:
            return []

        search_string = quote(search_string)
        url = "http://fanfox.net/search?title={}".format(search_string)

        req = Request(url)
        req.add_header("User-Agent", "")
        req.add_header('Cookie', 'isAdult=1')
        response = urlopen(req)
        tree = html.fromstring(response.read())
        nodes = tree.xpath('/html/body/div[2]/div/div/ul/li')

        m_ids = []
        for i in nodes:
            title = i[0].attrib["title"]
            href = i[0].attrib["href"]
            m_id = href.replace("/manga/", "").replace("/", "")
            m_ids.append(m_id)

        return m_ids

    def get_quick_search(self, search_string):
        if len(search_string) >= 3:
            search_results = self._get_quick_search_without_details(search_string)
            series_details = self.get_manga_list(search_results)
            return series_details

        return []

    def get_manga(self, m_id):
        url = "http://fanfox.net/manga/{}/".format(m_id)

        req = Request(url)
        req.add_header("Accept", "*/*")
        req.add_header("User-Agent", "")
        req.add_header('Cookie', 'isAdult=1')
        response = urlopen(req).read()
        tree = html.fromstring(response)

        manga = MangaContainer(self, m_id)
        manga.name = tree.xpath("//span[@class='detail-info-right-title-font']")[0].text
        manga.authors = map(lambda x: x.text, tree.xpath("/html/body/div[5]/div/div[2]/p[2]/a"))
        manga.cover = tree.xpath("//img[@class='detail-info-cover-img']")[0].attrib["src"]
        manga.description = tree.xpath("//p[@class='fullcontent']")[0].text
        chapter_nodes = tree.xpath("//ul[@class='detail-main-list']/li/a")

        manga.chapters = []
        for c in chapter_nodes:
            c_id = c.attrib["href"].replace("/manga/{}/".format(m_id), '').replace("/1.html", "")
            c_id = re.sub(r'v.*?\/', '', c_id)
            c_id = c_id.replace("c", "")
            chapter = ChapterContainer(manga, c_id)
            chapter.name = c.attrib["title"].replace("{} ".format(manga.name), "")
            chapter.c_number = Decimal(c_id)
            date = datetime.strptime(c[0][1].text, '%b %d,%Y ')
            chapter.last_updated = (date - datetime(1970, 1, 1)).total_seconds()

            manga.chapters.append(chapter)

        return manga

    def get_manga_list(self, manga_ids):
        # type: (List[string]) -> List[MangaContainer]

        manga_list = []
        for m_id in manga_ids:
            manga = self.get_manga(m_id)
            manga_list.append(manga)

        return manga_list

    def get_chapter_list(self, manga_container):
        # type: (MangaContainer) -> List[ChapterContainer]
        manga = self.get_manga(manga_container.m_id)
        manga_container.chapters = manga.chapters
        for ch in manga_container.chapters:
            ch.manga_container = manga_container

        return manga.chapters

    def _get_url_data(self, url):
        req = Request(url)
        req.add_header("Host", "fanfox.net")
        req.add_header("User-Agent", "")
        req.add_header("Referer", url)
        req.add_header("Cache-Control", "no-cache")
        return urlopen(req).read()

    def _get_page_list(self, c_id, internal_c_id, page_start_index=1):
        url = "http://fanfox.net/manga/chapterfun.ashx?cid={}&page={}".format(internal_c_id, page_start_index)
        req = Request(url)
        req.add_header("Host", "fanfox.net")
        req.add_header("User-Agent", "")
        req.add_header("Referer", "http://fanfox.net/manga/naruto/c{}/".format(c_id))
        req.add_header("Cookie", "image_time_cookie={}|0|0".format(internal_c_id))
        req.add_header("Cache-Control", "no-cache")
        response = urlopen(req).read()

        engine = QJSEngine()
        engine.evaluate(response)

        js_page_array = engine.globalObject().property("dm5imagefun").call()
        length = js_page_array.property("length").toInt()
        page_list = []
        for i in range(length):
            page_url = "http:{}".format(js_page_array.property(i).toString())
            page_list.append(page_url)

        return page_list

    def get_page_list(self, chapter_container):
        m_id = chapter_container.manga_container.m_id
        c_id = chapter_container.c_id
        url = "http://fanfox.net/manga/{}/c{}/1.html".format(m_id, c_id)
        data = self._get_url_data(url)
        js_str_ch_id = re.search(r'(var( )*chapterid( )*\=[0-9]*\;)', data).group(0)

        engine = QJSEngine()
        engine.evaluate(js_str_ch_id)

        internal_c_id = engine.globalObject().property("chapterid").toInt()
        page_list_1 = self._get_page_list(c_id, internal_c_id)
        if len(page_list_1) > 1:
            page_list_2 = self._get_page_list(c_id, internal_c_id, 2)
            for i in page_list_2:
                if i not in page_list_1:
                    page_list_1.append(i)
            page_list_1.remove(page_list_1[-1])

        return page_list_1