#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import os.path
import zipfile
import re
from functools import partial

# PyQt libraries
from PyQt5.Qt import QMenu, QIcon, QPixmap, QVBoxLayout

# Calibre libraries
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import config_dir
from calibre_plugins.Skoob_Sync.config import prefs
from calibre_plugins.Skoob_Sync.main import SkoobSyncTools


class InterfacePlugin(InterfaceAction):

    name = 'Skoob Sync'

    # Declare the main action associated with this plugin
    # The keyboard shortcut can be None if you dont want to use a keyboard
    # shortcut. Remember that currently calibre has no central management for
    # keyboard shortcuts, so try to use an unusual/unused shortcut.

    action_spec = ('Skoob Sync', None, None, None)

    def get_icon(self, icon_name):
        # Check to see whether the icon exists as a Calibre resource
        # This will enable skinning if the user stores icons within a folder like:
        # ...\AppData\Roaming\calibre\resources\images\Plugin Name\
        icon_path = os.path.join(config_dir, 'resources', 'images', 'Skoob Sync',
                                 icon_name.replace('images/', ''))
        if os.path.exists(icon_path):
            pixmap = QPixmap()
            pixmap.load(icon_path)
            return QIcon(pixmap)
        # As we did not find an icon elsewhere, look within our zip resources
        return get_icons(icon_name)

    def genesis(self):
        # This method is called once per plugin, do initial setup here

        # Set the icon for this interface action
        # The get_icons function is a builtin function defined for all your
        # plugin code. It loads icons from the plugin zip file. It returns
        # QIcon objects, if you want the actual data, use the analogous
        # get_resources builtin function.
        #
        # Note that if you are loading more than one icon, for performance, you
        # should pass a list of names to get_icons. In this case, get_icons
        # will return a dictionary mapping names to QIcons. Names that
        # are not found in the zip file will result in null QIcons.
        self.key_themes_idx = prefs['themes_idx'] + 1
        icon = self.get_icon('images/theme' + str(self.key_themes_idx) + '/icon.png')

        # Extract resource files to use internally
        plugin_zip = os.path.join(config_dir, 'plugins', 'Skoob Sync.zip')
        plugin_dir = os.path.join(config_dir, 'plugins', 'Skoob Sync')
        with zipfile.ZipFile(plugin_zip, 'r') as zf:
            zf.extract('images/user.png', plugin_dir)
            zf.extract('images/password.png', plugin_dir)
            zf.extract('cookie/cookie.txt', plugin_dir)

        # Menu creation
        # Only the toolbar menu is created here. All the others are created after the main GUI initialization
        # This allows us to change menus on the fly
        self.tools = SkoobSyncTools(self.gui)
        self.menu = QMenu(self.gui)
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(icon)

        # This action is configurable by the user. It is actioned by clicking on the main plugin button
        self.main_button_action = self.qaction

        # Define main action based on the user choice
        self.key_main_button = prefs['main_action']
        if self.key_main_button == _('Download from Skoob'):
            sync_option = 'from_skoob'
        else:
            sync_option = 'to_skoob'

        self.main_button_action.triggered.connect(partial(self.tools.update_status, sync_option=sync_option, shelf=None))

    def initialization_complete(self):
        # This method is called once per action, after the initialization of the main GUI
        # This is right place to create changing menus
        m = self.menu
        m.clear()

        m.aboutToShow.connect(self.about_to_show_menu)

        # Main menu itens
        self.has_skoob_id_item = m.addAction(self.get_icon('images/theme' + str(self.key_themes_idx) + '/skoob.png'),
                                             _('&List books with Skoob ID'))
        self.has_skoob_id_item.triggered.connect(self.tools.has_skoob_id)
        m.addSeparator()
        self.download_from_skoob = m.addAction(self.get_icon('images/download_from_skoob.png'), _('&Download from Skoob'))
        self.download_from_skoob.triggered.connect(partial(self.tools.update_status, sync_option='from_skoob', shelf=None))
        self.upload_to_skoob = m.addAction(self.get_icon('images/upload_to_skoob.png'), _('&Upload to Skoob'))
        self.upload_to_skoob.triggered.connect(partial(self.tools.update_status, sync_option='to_skoob', shelf=None))
        m.addSeparator()
        self.add_to_skoob = m.addAction(self.get_icon('images/add_to_skoob.png'), _('&Add to Skoob'))
        self.add_to_skoob.triggered.connect(partial(self.tools.update_status, sync_option='add_to_skoob', shelf=None))
        self.remove_from_skoob = m.addAction(self.get_icon('images/remove_from_skoob.png'), _('&Remove from Skoob'))
        self.remove_from_skoob.triggered.connect(partial(self.tools.update_status, sync_option='remove_from_skoob', shelf=None))
        m.addSeparator()
        self.add_to_shelf = m.addMenu(self.get_icon('images/shelf.png'), _('Add to &shelf'))

        # Add to shelf submenus
        self.shelf_read = self.add_to_shelf.addAction(self.get_icon('images/read.png'), _('&Read'))
        self.shelf_read.triggered.connect(partial(self.tools.update_status, sync_option='add_to_skoob', shelf=1))
        self.shelf_reading = self.add_to_shelf.addAction(self.get_icon('images/reading.png'), _('R&eading'))
        self.shelf_reading.triggered.connect(partial(self.tools.update_status, sync_option='add_to_skoob', shelf=2))
        self.shelf_to_read = self.add_to_shelf.addAction(self.get_icon('images/to_read.png'), _('&Want to read'))
        self.shelf_to_read.triggered.connect(partial(self.tools.update_status, sync_option='add_to_skoob', shelf=3))
        self.shelf_rereading = self.add_to_shelf.addAction(self.get_icon('images/rereading.png'), _('Rerea&ding'))
        self.shelf_rereading.triggered.connect(partial(self.tools.update_status, sync_option='add_to_skoob', shelf=4))
        self.shelf_abandoned = self.add_to_shelf.addAction(self.get_icon('images/abandoned.png'), _('A&bandoned'))
        self.shelf_abandoned.triggered.connect(partial(self.tools.update_status, sync_option='add_to_skoob', shelf=5))

        m.addSeparator()
        # Add to Reading Goal
        self.add_to_reading_goal = m.addAction(self.get_icon('images/add_to_goal.png'), _('Add to Reading &Goal'))
        self.add_to_reading_goal.triggered.connect(partial(self.tools.update_status, sync_option='add_to_goal', shelf=None))

        # Get read date
        self.get_read_date = m.addAction(self.get_icon('images/read_date.png'), _('&Get Read Date'))
        self.get_read_date.triggered.connect(partial(self.tools.update_status, sync_option='get_read_date', shelf=None))

        # Import Skoob Library
        m.addSeparator()
        self.import_library = m.addAction(self.get_icon('images/import.png'), _('&Import Skoob library'))
        self.import_library.setToolTip(_('Create new entries on your calibre library based on your Skoob library'))
        self.import_library.triggered.connect(self.tools.skoob_library)

        # Config
        m.addSeparator()
        self.config_item = m.addAction(self.get_icon('images/config.png'), _('&Customize plugin'))
        self.config_item.triggered.connect(self.do_config)

    def show_dialog(self):
        # The base plugin object defined in __init__.py
        base_plugin_object = self.interface_action_base_plugin
        # Show the config dialog
        # The config dialog can also be shown from within
        # Preferences->Plugins, which is why the do_user_config
        # method is defined on the base plugin class
        do_user_config = base_plugin_object.do_user_config

        # self.gui is the main calibre GUI. It acts as the gateway to access
        # all the elements of the calibre user interface, it should also be the
        # parent of the dialog
        d = SkoobSyncDialog(self.gui, self.qaction.icon(), do_user_config)
        d.show()

    def do_config(self):
        from calibre.gui2.widgets2 import Dialog
        from calibre.gui2.tweak_book import tprefs
        from PyQt5.Qt import QDialogButtonBox
        from calibre_plugins.Skoob_Sync.config import ConfigWidget
        tool = self

        class ConfigDialog(Dialog):

            def __init__(self, gui):
                self.gui = gui
                Dialog.__init__(self, _('Options'), 'plugin-skoob-sync-config-dialog', parent=tool.gui, prefs=tprefs)

            def setup_ui(self):
                self.box = QVBoxLayout(self)
                self.widget = ConfigWidget(self, self.gui)
                self.box.addWidget(self.widget)
                self.button = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
                self.box.addWidget(self.button)
                self.button.accepted.connect(self.accept)
                self.button.rejected.connect(self.reject)

            def accept(self):
                self.widget.save_settings()
                Dialog.accept(self)

        d = ConfigDialog(self)
        d.exec_()

    def apply_settings(self):
        from calibre_plugins.Skoob_Sync.config import prefs
        # In an actual non trivial plugin, you would probably need to
        # do something based on the settings in prefs
        prefs

    def about_to_show_menu(self):
        self.key_user = prefs['user']
        self.key_password = prefs['password'].decode('base64')

        # Check if credentials are filled
        if all(hasattr(self, attr) for attr in ['download_from_skoob', 'upload_to_skoob', 'add_to_skoob',
                                                'remove_from_skoob', 'add_to_shelf', 'add_to_reading_goal',
                                                'get_read_date', 'import_library']):
            if self.key_user in ('user', '') or self.key_password in ('password', ''):
                _enabled = False
            else:
                _enabled = True
            self.download_from_skoob.setEnabled(_enabled)
            self.upload_to_skoob.setEnabled(_enabled)
            self.add_to_skoob.setEnabled(_enabled)
            self.remove_from_skoob.setEnabled(_enabled)
            self.add_to_shelf.setEnabled(_enabled)
            self.add_to_reading_goal.setEnabled(_enabled)
            self.get_read_date.setEnabled(_enabled)
            self.import_library.setEnabled(_enabled)
