#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import os.path
import re
import cookielib
import urllib
import urllib2
import datetime
import time
import threading
import Queue
import json
from timeit import default_timer as timer
from lxml.html import fromstring, tostring

# PyQt libraries
from PyQt5.Qt import Qt, QApplication, QTreeWidget, QTreeWidgetItem,\
    QVBoxLayout, QTreeWidgetItemIterator, QAbstractItemView, QDialogButtonBox
from PyQt5 import QtCore, QtGui

# Calibre libraries
from calibre.utils.config import config_dir
from calibre_plugins.Skoob_Sync.config import prefs
from calibre.utils.cleantext import clean_ascii_chars
from calibre.gui2 import error_dialog, info_dialog, warning_dialog, question_dialog
from calibre.library.comments import sanitize_comments_html
from calibre.ebooks.metadata import MetaInformation
from calibre.gui2.actions.show_quickview import get_quickview_action_plugin
from calibre.gui2.widgets2 import Dialog

# Starting credentials. Used to check if it is changed.
KEY_USER = prefs['user']
KEY_PASSWORD = prefs['password'].decode('base64')


class SkoobSyncTools:
    def __init__(self, gui):
        self.gui = gui

    def has_skoob_id(self):
        ''' Show books with Skoob Books IDs '''

        self.gui.search.setEditText('identifiers:"=skoob:"')
        self.gui.search.do_search()

    def update_status(self, sync_option, shelf):
        '''Sync book metatada with Skoob.'''

        self.shelf = shelf
        self.sync_option = sync_option

        # Get preferences
        self.key_user = prefs['user']
        self.key_password = prefs['password'].decode('base64')
        self.key_reading_progress_column_idx = prefs['reading_progress_column_idx']
        self.key_reading_progress_column = prefs['reading_progress_column']
        self.key_status_date_column_idx = prefs['status_date_column_idx']
        self.key_status_date_column = prefs['status_date_column']
        self.key_review_column_idx = prefs['review_column_idx']
        self.key_review_column = prefs['review_column']
        self.key_rating_column_idx = prefs['rating_column_idx']
        self.key_rating_column = prefs['rating_column']
        self.key_page_count_column_idx = prefs['page_count_column_idx']
        self.key_page_count_column = prefs['page_count_column']
        self.key_read_books_column_idx = prefs['read_books_column_idx']
        self.key_read_books_column = prefs['read_books_column']
        self.key_tags_column_idx = prefs['tags_column_idx']
        self.key_tags_column = prefs['tags_column']
        self.key_physical_column_idx = prefs['physical_column_idx']
        self.key_physical_column = prefs['physical_column']
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        self.key_sync_review = prefs['sync_review']
        self.key_sync_rating = prefs['sync_rating']
        self.key_sync_page_count = prefs['sync_page_count']
        self.key_sync_read_books = prefs['sync_read_books']
        self.key_sync_update_added = prefs['sync_update_added']
        self.key_sync_tags = prefs['sync_tags']

        # Check if credentials are set
        if self.key_user in ('user', '') or self.key_password in ('password', ''):
            return warning_dialog(self.gui, _('Invalid Credentials'), _('You must first setup your Skoob account'),
                               show_copy_button=False, show=True)

        # Check if no sync option is checked
        sync = [self.key_sync_reading_progress, self.key_sync_review, self.key_sync_rating,
                self.key_sync_page_count]
        if not any(sync):
            return warning_dialog(self.gui, _('Nothing to sync'), _('No sync actions selected'),
                               show_copy_button=False, show=True)

        # Check if the reading progress column is set
        if self.key_sync_reading_progress:
            if len(self.key_reading_progress_column) == 0:
                return warning_dialog(self.gui, _('Configuration needed'),
                                   _('You must choose the reading progress custom column'),
                                   show_copy_button=False, show=True)

        # Check if the review column is set
        if self.key_sync_review:
            if len(self.key_review_column) == 0:
                return warnig_dialog(self.gui, _('Configuration needed'),
                                   _('You must choose the review custom column'), show_copy_button=False, show=True)

        # Check if the page count column is set
        if self.key_sync_page_count:
            if len(self.key_page_count_column) == 0:
                return warning_dialog(self.gui, _('Configuration needed'),
                                   _('You must choose the page count custom column'), show_copy_button=False, show=True)

        # Check if the option to mark read book is selected
        if self.key_sync_reading_progress and self.key_sync_read_books:
            if len(self.key_read_books_column) == 0:
                return warning_dialog(self.gui, _('Configuration needed'),
                                      _('You must choose the read books custom column'),
                                      show_copy_button=False, show=True)

        # Get currently selected books
        rows = self.gui.library_view.selectionModel().selectedRows()

        # Check if selection is empty
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, _('Empty selection'),
                                _('No books selected'), show_copy_button=False, show=True)

        # Check if selected books have a skoob id
        is_selection_valid = self.are_selected_books_linked()
        if is_selection_valid is False:
            if len(rows) == 1:
                return error_dialog(self.gui, _('Selection invalid'),
                                    _('Selected book does not have a Skoob ID'), show_copy_button=False, show=True)
            else:
                return error_dialog(self.gui, 'Selection invalid',
                                    _('All selected books must have a Skoob ID'), show_copy_button=False, show=True)

        # Map the rows to book ids
        ids = list(map(self.gui.library_view.model().id, rows))
        db = self.gui.current_db.new_api

        # Create list of altered books
        self.books_not_updated = []
        self.books_updated = []
        self.books_not_in_shelf = []
        self.bad_skoob_id = []

        # Check if too many books are selected at once
        if len(ids) <= 50:
            if len(ids) > 20:
                if question_dialog(self.gui, _('Caution'),
                             _('You selected %d books. Selecting too many books may cause calibre to crash. '
                               'Are you sure you want to continue?') % len(ids), show_copy_button=False):
                    pass
                else:
                    return

            start = timer()

            # Login to Skoob.
            self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
            QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)
            login_status = self.login()
            if login_status is False or login_status is None:
                return self.gui.status_bar.clearMessage()
            self.gui.status_bar.clearMessage()

            # THIS IS REALLY IMPORTANT! This code allows the plugin to run simultaneous checks. Nice!
            aux_list = []
            for book_id in ids:
                aux_list.append((book_id,))
            self.run_parallel_in_threads(self.update_routine, aux_list)

            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            # Refresh the GUI, so the Book Info panel shows the new info
            self.gui.library_view.model().refresh_ids(ids)
            self.gui.library_view.model().refresh_ids(ids, current_row=self.gui.library_view.currentIndex().row())

            # Final message to the user, informing how many books were updated
            if len(self.books_not_updated) > 0:
                if len(self.books_updated) > 0:
                    warning_dialog(self.gui, _('Update status'),
                                   _('Updated %d book(s). ') % len(self.books_updated) +
                                     _('No updates for %d book(s).') % len(self.books_not_updated),
                                        show_copy_button=False, show=True)
                else:
                    warning_dialog(self.gui, _('Nothing to update'), _('No updates for %d book(s)')
                                    % len(self.books_not_updated), show_copy_button=False, show=True)
            else:
                if len(self.books_updated) > 0:
                    info_dialog(self.gui, _('Update status'), _('Updated %d book(s)')
                                    % len(self.books_updated), show_copy_button=False, show=True)
                # else:
                #     error_dialog(self.gui, _('Not in shelf'), _('%d book(s) not in any shelf'
                #                     % len(self.books_not_in_shelf)), show_copy_button=False, show=True)
            if len(self.books_not_in_shelf) > 0:
                book_list = []
                for book_id in self.books_not_in_shelf:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                error_dialog(self.gui, _('Not in shelf'), _('%d book(s) not in any shelf')
                            % len(self.books_not_in_shelf) + ':\n' + book_list_str, show_copy_button=False, show=True)

            if len(self.bad_skoob_id) > 0:
                book_list = []
                for book_id in self.bad_skoob_id:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                if sync_option in ('add_to_skoob', 'add_go_toal'):
                    error_dialog(self.gui, _('Not added'), _('%d book(s) with unknown Skoob IDs')
                                    % len(self.bad_skoob_id) + ':\n' + book_list_str,
                                    show_copy_button=False, show=True)
                else:
                    error_dialog(self.gui, _('Not removed'), _('%d book(s) are not on your shelves '
                                        'or have invalid IDs') % len(self.bad_skoob_id) + ':\n' + book_list_str,
                                        show_copy_button=False, show=True)

        else:
            # To avoid overloading the server, we must update no more than 50 books at a time. This is just a guess.
            # This plugin does not use Skoob API, since is has not been made public yet, according to the their FAQ.
            return error_dialog(self.gui, _('Too many books'), _('%d books selected. Try to select 50 books at most.')
                               % len(ids), show_copy_button=False, show=True)


    def run_parallel_in_threads(self, target, args_list):
        ''' Utility - spawn a thread to execute target for each args '''
        # https://stackoverflow.com/questions/3490173/how-can-i-speed-up-fetching-pages-with-urllib2-in-python/3491455

        result = Queue.Queue()

        # Wrapper to collect return value in a Queue
        def task_wrapper(*args):
            result.put(target(*args))

        threads = [threading.Thread(target=task_wrapper, args=args) for args in args_list]

        count = 0

        for t in threads:
            count += 1
            t.start()
            # Don't send all requests at once
            list_size = len(args_list)
            if (list_size - 20) > 20:
                if count == 20:
                    time.sleep(2)
                elif count == 40:
                    time.sleep(2)
            elif (list_size - 20) > 0:
                if count == 20:
                    time.sleep(1)
            time.sleep(0.1)
            QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)
        for t in threads:
            t.join()
        return result

    def update_routine(self, book_id):
        ''' This method does all the Skoob syncing activity'''

        # Get the current metadata for this book from the db
        db = self.gui.current_db.new_api
        mi = db.get_metadata(book_id)
        # Get skoob_id from calibre
        identifiers = mi.get_identifiers()
        skoob_id = identifiers['skoob']
        is_skoob_id = re.search('(ed)(.+)', skoob_id.lower())
        if is_skoob_id is None:
            return self.bad_skoob_id.append(book_id)
        else:
            skoob_id = re.search('(ed)(.+)', skoob_id.lower()).groups(0)[1]

        # Download FROM Skoob
        if self.sync_option == 'from_skoob':
            self.gui.status_bar.showMessage(_("Downloading from Skoob..."))
            self.download_from_skoob(skoob_id, book_id, db, mi)
            self.gui.status_bar.clearMessage()

        # Upload TO Skoob
        if self.sync_option == 'to_skoob':
            self.gui.status_bar.showMessage(_("Uploading to Skoob..."))
            self.upload_to_skoob(skoob_id, book_id, db, mi)
            self.gui.status_bar.clearMessage()

        # Add to Skoob / Add to shelf
        if self.sync_option == 'add_to_skoob':
            if self.shelf is not None:
                self.gui.status_bar.showMessage(_("Adding to shelf..."))
            else:
                self.gui.status_bar.showMessage(_("Adding to Skoob..."))
            valid_id, to_update = self.add_to_skoob(skoob_id, book_id, mi)
            # If update_added is checked, it will return True. This will prevent duplicating final message.
            if to_update:
                self.upload_to_skoob(skoob_id, book_id, db, mi)
            else:
                # If the Skoob ID is invalid, then abort
                if valid_id:
                    # Set the read column according to reading progress
                    read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']
                    if self.key_sync_read_books:
                        if len(self.key_read_books_column) > 0:
                            if read_percentage is None:
                                pass
                                # db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                            else:
                                if int(read_percentage) == 100:
                                    db.new_api.set_field(self.key_read_books_column, {book_id: True})
                                else:
                                    db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                else:
                    self.gui.status_bar.clearMessage()
                    return
            self.gui.status_bar.clearMessage()

        # Remove from Skoob
        if self.sync_option == 'remove_from_skoob':
            self.gui.status_bar.showMessage(_("Removing from Skoob..."))
            valid_id = self.remove_from_skoob(skoob_id, book_id, mi)
            # If the Skoob ID is invalid, then abort
            if valid_id:
                # Set the read column according to reading progress
                read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']
                if self.key_sync_read_books:
                    if len(self.key_read_books_column) > 0:
                        if read_percentage is None:
                            pass
                            # db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                        else:
                            if int(read_percentage) == 100:
                                db.new_api.set_field(self.key_read_books_column, {book_id: True})
                            else:
                                db.new_api.set_field(self.key_read_books_column, {book_id: ''})
            else:
                self.gui.status_bar.clearMessage()
                return
            self.gui.status_bar.clearMessage()

        # Add to Reading Goal
        if self.sync_option == 'add_to_goal':
            self.gui.status_bar.showMessage(_("Adding to Reading Goal..."))
            self.add_to_goal(skoob_id, book_id, mi)
            self.gui.status_bar.clearMessage()

        # Get Read Date
        if self.sync_option == 'get_read_date':
            self.gui.status_bar.showMessage(_("Updating Read Date..."))
            self.get_read_date(skoob_id, book_id, db, mi)
            self.gui.status_bar.clearMessage()

    def download_from_skoob(self, skoob_id, book_id, db, mi):
        # Download FROM Skoob
        try:
            details = self.get_details(skoob_id)
            if details is False:
                return self.books_not_in_shelf.append(book_id)
            status = details[0]  # Get reading progress and status date
            review = details[1]  # Get review title and text
            rating = details[2]  # Get rating
            page_count = details[3]  # Get the page count
            shelf = details[4]  # Get the book shelf
            tags = details[5]  # Get user tags
            physical = details[6]  # Get book media type
            result = [status[0], status[1], review[0], review[1], rating, page_count, shelf, tags, physical]

            if self.any(result):

                # -- Set Reading Progress --
                if self.key_sync_reading_progress:
                    status_date = status[0]
                    status_progress = status[1]

                    if status_progress:
                        if int(status_progress) > 100:
                            status_progress = '100'

                    # Set the read column according to reading progress
                    if self.key_sync_read_books:
                        if len(self.key_read_books_column) > 0:
                            if status_progress is not None:
                                if status_progress == '100':
                                    db.new_api.set_field(self.key_read_books_column, {book_id: True})
                                else:
                                    db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                            else:
                                if shelf == '1':
                                    db.new_api.set_field(self.key_read_books_column, {book_id: True})

                    if len(self.key_reading_progress_column) > 0:
                        db.new_api.set_field(self.key_reading_progress_column, {book_id: status_progress})
                    if len(self.key_status_date_column) > 0 and status_date is not None:
                        db.new_api.set_field(self.key_status_date_column, {book_id: status_date})

                # -- Set Review --
                if self.key_sync_review:
                    if review == (None, None):
                        pass
                    else:
                        if self.key_review_column == 'comments':
                            skoob_review = mi.comments
                        else:
                            skoob_review = mi.metadata_for_field(self.key_review_column)['#value#']
                        # Personalized entry to merge with existing comments, so it can be retrieved easier
                        review_open = '## Skoob ##'
                        review_end = '##########'
                        review_title = review[0]
                        if review_title is None:
                            review_title = ''
                        review_text = tostring(review[1], method='html')
                        review_text = sanitize_comments_html(review_text)

                        if len(review_text) > 0:
                            # Check if a Skoob Review is already set
                            if self.key_review_column == 'comments':
                                if skoob_review:
                                    existing_comments_review = re.search('## Skoob ##', skoob_review)
                                    if existing_comments_review:
                                        comments_clean = re.search('([\\s\\S]+)(## Skoob ##)',
                                                                   skoob_review).groups(0)[0]
                                        new_comments = comments_clean + review_open + '<h4>' + review_title + '</h4>'\
                                                       + review_text + review_end
                                    else:
                                        new_comments = skoob_review + review_open + '<h4>' + review_title + '</h4>'\
                                                       + review_text + review_end
                                else:
                                    new_comments = review_open + '<h4>' + review_title + '</h4>'\
                                                   + review_text + review_end
                            else:
                                new_comments = '<h4>' + review_title + '</h4>' + review_text
                            db.new_api.set_field(self.key_review_column, {book_id: new_comments})
                        else:
                            pass

                # -- Set Rating --
                if self.key_sync_rating and rating is not None:
                    rating_number = float(rating) * 2
                    if len(self.key_rating_column) > 0:
                        db.new_api.set_field(self.key_rating_column, {book_id: rating_number})

                # -- Set Page Count --
                if self.key_sync_page_count:
                    if len(self.key_page_count_column) > 0:
                        db.new_api.set_field(self.key_page_count_column, {book_id: page_count})

                # -- Set User Tags --
                if self.key_sync_tags:
                    tags = tags[0]
                    if tags is None:
                        pass
                    else:
                        if self.key_tags_column == 'tags':
                            existing_tags = mi.tags
                            for tag_item in existing_tags:
                                tags.append(tag_item)
                            db.new_api.set_field(self.key_tags_column, {book_id: tags})
                        else:
                            db.new_api.set_field(self.key_tags_column, {book_id: tags})

                # -- Set Physical Book --
                if len(self.key_physical_column) > 0:
                    db.new_api.set_field(self.key_physical_column, {book_id: physical})

                # Mark the book as updated
                self.books_updated.append(book_id)
            else:
                # Mark the book as NOT updated
                self.books_not_updated.append(book_id)
        except:
            return

    def upload_to_skoob(self, skoob_id, book_id, db, mi):
        success = False
        try:
            # Reading progress
            if self.key_sync_reading_progress and self.shelf not in [3, 4]:
                reading_progress_url = 'https://api.skoob.com.br/estante/s_historico_leitura/' + str(skoob_id)
                read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']
                if read_percentage > 100:
                    read_percentage = 100
                    db.new_api.set_field(self.key_reading_progress_column, {book_id: read_percentage})

                # Set the read column according to reading progress
                if self.key_sync_read_books:
                    if len(self.key_read_books_column) > 0:
                        if read_percentage is None:
                            pass
                            # db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                        else:
                            if int(read_percentage) == 100:
                                db.new_api.set_field(self.key_read_books_column, {book_id: True})
                            else:
                                db.new_api.set_field(self.key_read_books_column, {book_id: ''})

                # Check if book exists on shelves
                root = self.get_user_data(skoob_id)
                exists = root[2]
                if exists:
                    pass
                else:
                    return self.books_not_in_shelf.append(book_id)

                if read_percentage:

                    payload = {
                        'data[LendoHistorico][paginas]': read_percentage,
                        'data[LendoHistorico][tipo]': 1,
                    }
                    # Use urllib to encode the payload
                    data = urllib.urlencode(payload)

                    # Build our Request object (supplying 'data' makes it a POST)
                    reading_progress_req = urllib2.Request(reading_progress_url, data)

                    try:
                        reading_progress_open = urllib2.urlopen(reading_progress_req)
                        success = True

                    except urllib2.URLError:
                        error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                                     show_copy_button=False, show=True)
                        raise urllib2.URLError(_('No internet connection'))

            # Rating
            if self.key_sync_rating:
                if self.key_read_books_column == 'rating':
                    rating = mi.rating
                else:
                    rating = mi.metadata_for_field(self.key_rating_column)['#value#']

                if rating is not None:
                    rating_url = 'https://api.skoob.com.br/v1/book_rate/' + str(skoob_id) + '/' + str(rating/2) + '/'

                    # Build our Request object
                    rating_req = urllib2.Request(rating_url)

                    try:
                        rating_open = urllib2.urlopen(rating_req)
                        success = True

                    except urllib2.URLError:
                        error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                                     show_copy_button=False, show=True)
                        raise urllib2.URLError(_('No internet connection'))

            # Review
            if self.key_sync_review:
                review_title = review_text = None
                # Check which column holds the review
                if self.key_review_column == 'comments':
                    calibre_review = mi.comments
                else:
                    calibre_review = mi.metadata_for_field(self.key_review_column)['#value#']

                if calibre_review:

                    if self.key_review_column == 'comments':
                        # Check if the review is inside 'comments' column. If it is, then get it.
                        if calibre_review:
                            existing_comments_review = re.search('## Skoob ##', calibre_review)
                            if existing_comments_review:
                                review_title = re.search(
                                    '<p>## Skoob ##</p>[\\s\\S]*<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>\\s',
                                                         calibre_review).groups(0)[0]
                                review_text = re.search(
                                    '<p>## Skoob ##</p>[\\s\\S]*<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>\\s',
                                                        calibre_review).groups(0)[1]
                    else:
                        # Check if the review is on a custom column. If it is, then get it.
                        is_review_title = re.search('<h4.*>([\\s\\S]*)</h4>([\\s\\S]*)', calibre_review)
                        is_review_text = re.search('<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>', calibre_review)
                        if is_review_title:
                            review_title = re.search('<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>',
                                                     calibre_review).groups(0)[0]
                        if is_review_text:
                            review_text = re.search('<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>',
                                                    calibre_review).groups(0)[1]
                        else:
                            review_title = None
                            review_text = sanitize_comments_html(calibre_review)
                            clean_review_text = re.compile('<.*?>')
                            review_text = re.sub(clean_review_text, '', review_text)

                    if not review_title and not review_text:
                        pass
                    else:
                        review_url = 'https://api.skoob.com.br/estante/s_resenha/' + str(skoob_id)

                        if review_title:
                            payload = {
                                'data[Resenha][titulo_resenha]': review_title,
                                'data[Resenha][resenha]': review_text
                            }
                        else:
                            payload = {
                                'data[Resenha][resenha]': review_text
                            }

                        # Use urllib to encode the payload
                        data = urllib.urlencode(payload)

                        # Build our Request object (supplying 'data' makes it a POST)
                        review_req = urllib2.Request(review_url, data)

                        try:
                            review_open = urllib2.urlopen(review_req)
                            success = True

                        except urllib2.URLError:
                            error_dialog(self.gui, _('No internet connection'),
                                         _('Seems like you have no internet access'), show_copy_button=False, show=True)
                            raise urllib2.URLError(_('No internet connection'))

            # User tags
            if self.key_sync_tags:
                # Check which column holds the review
                if self.key_tags_column == 'tags':
                    calibre_tags = mi.tags
                else:
                    calibre_tags = mi.metadata_for_field(self.key_tags_column)['#value#']

                if calibre_tags:
                    calibre_tags = ','.join(calibre_tags)
                    tags_url = 'https://api.skoob.com.br/estante/s_editar/' + str(skoob_id)

                    tags_req = urllib2.Request(tags_url)
                    tags_info = urllib2.urlopen(tags_req)
                    tags = tags_info.read()
                    tags_decode = tags.decode('iso-8859-1', errors='replace')
                    root_t = fromstring(clean_ascii_chars(tags_decode))
                    tags, date_read = self.parse_tags(root_t)

                    if date_read == [[], [], []]:
                        payload = {
                            'data[MeusLivro][tags_publicas]': calibre_tags.encode("ISO-8859-1"),
                        }
                    else:
                        payload = {
                            'data[MeusLivro][tags_publicas]': calibre_tags.encode("ISO-8859-1"),
                            'data[MeusLivro][dia_leitura]': date_read[0][0],
                            'data[MeusLivro][mes_leitura]': date_read[1][0],
                            'data[MeusLivro][ano_leitura]': date_read[2][0]
                        }

                    # Use urllib to encode the payload
                    data = urllib.urlencode(payload)

                    # Build our Request object (supplying 'data' makes it a POST)
                    tags_req = urllib2.Request(tags_url, data)

                    try:
                        tags_open = urllib2.urlopen(tags_req)
                        success = True

                    except urllib2.URLError:
                        error_dialog(self.gui, _('No internet connection'),
                                     _('Seems like you have no internet access'), show_copy_button=False, show=True)
                        raise urllib2.URLError(_('No internet connection'))

            if success:
                self.books_updated.append(book_id)
            else:
                self.books_not_updated.append(book_id)

        except:
            return

    def add_to_skoob(self, skoob_id, book_id, mi):
        try:
            read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']

            # Add to the according shelf
            if self.shelf is None:
                if read_percentage == 100:
                    # Read shelf
                    add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/1/'
                else:
                    # Reading shelf
                    add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/2/'
            else:
                add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) +\
                                   '/' + str(self.shelf) + '/'

            # Build our Request object
            add_to_skoob_req = urllib2.Request(add_to_skoob_url)

            try:
                # For some reason, Skoob only recognizes the second attempt to add the book...
                add_to_skoob_open = urllib2.urlopen(add_to_skoob_req)
                add_to_skoob_open = urllib2.urlopen(add_to_skoob_req)

                # Check if the operation succeeded
                add_response = add_to_skoob_open.read()
                if '"success":true' in add_response:
                    valid_id = True
                else:
                    valid_id = False
                    self.bad_skoob_id.append(book_id)
                    return valid_id, False

                # If checked, updates the book status on Skoob after adding to shelf
                if self.key_sync_update_added:
                    to_update = True
                    return valid_id, to_update
                else:
                    self.books_updated.append(book_id)
                    return valid_id, False

            except urllib2.URLError:
                self.books_not_updated(book_id)
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
                raise urllib2.URLError(_('No internet connection'))

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s could not be added to Skoob.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def remove_from_skoob(self, skoob_id, book_id, mi):
        try:
            # Remove from Skoob
            remove_from_skoob_url = 'https://api.skoob.com.br/v1/shelf_del/' + str(skoob_id) + '/'

            # Build our Request object
            remove_from_skoob_req = urllib2.Request(remove_from_skoob_url)

            try:
                remove_from_skoob_open = urllib2.urlopen(remove_from_skoob_req)

                # Check if the operation succeeded
                remove_response = remove_from_skoob_open.read()
                if '"success":true' in remove_response:
                    self.books_updated.append(book_id)
                    return True
                else:
                    self.bad_skoob_id.append(book_id)
                    return False

            except urllib2.URLError:
                self.books_not_updated(book_id)
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
                raise urllib2.URLError(_('No internet connection'))

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not removed'), _('%s could not be removed from Skoob.\n'
                                                              'Please verify its Skoob ID.') % title,
                                                                show_copy_button=False, show=True)

    def add_to_goal(self, skoob_id, book_id, mi):
        try:
            add_to_goal_url = 'https://api.skoob.com.br/v1/meta/add/' + str(skoob_id) + '/'

            # Build our Request object
            add_to_goal_req = urllib2.Request(add_to_goal_url)

            try:
                add_to_goal_open = urllib2.urlopen(add_to_goal_req)

                # Check if the operation succeeded
                add_response = add_to_goal_open.read()
                if '"success":true' in add_response:
                    self.books_updated.append(book_id)
                else:
                    self.bad_skoob_id.append(book_id)

            except urllib2.URLError:
                self.books_not_updated(book_id)
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
                raise urllib2.URLError(_('No internet connection'))

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s could not be added to Reading Goal.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def get_read_date(self, skoob_id, book_id, db, mi):
        try:
            read_date_url = 'https://api.skoob.com.br/estante/s_editar/' + str(skoob_id)

            # Build our Request object
            read_date_req = urllib2.Request(read_date_url)

            try:
                read_date_open = urllib2.urlopen(read_date_req)
                read_date_read = read_date_open.read()
                read_date_decode = read_date_read.decode('iso-8859-1', errors='replace')

                if 'Não existe o livro' in read_date_decode:
                    return self.books_not_in_shelf.append(book_id)
                else:
                    pass

                root_rd = fromstring(clean_ascii_chars(read_date_decode))

            except urllib2.URLError:
                self.books_not_updated(book_id)
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
                raise urllib2.URLError(_('No internet connection'))

            try:
                tags, read_date = self.parse_tags(root_rd)
                if read_date == [[], [], []]:
                    status_date = None
                else:
                    status_date = read_date[0][0] + '/' + read_date[1][0] + '/' + read_date[2][0]

            except:
                status_date = None

            if len(self.key_status_date_column) > 0 and status_date is not None:
                db.new_api.set_field(self.key_status_date_column, {book_id: status_date})
                self.books_updated.append(book_id)
            else:
                self.books_not_updated.append(book_id)

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s has no Read Date.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def login(self):
        self.key_user = prefs['user']
        self.key_password = prefs['password'].decode('base64')

        # Login to Skoob. Session will keep open while checking multiple books.
        try:
            self.cookie_path = os.path.join(config_dir, 'plugins', 'Skoob Sync', 'cookie', 'cookie.txt')
            self.cj = cookielib.MozillaCookieJar(self.cookie_path)

            # Check if user credentials have changed
            if not (KEY_USER == self.key_user and KEY_PASSWORD == self.key_password):
                login_status = self.skoob_login()

            else:
                self.cj.load()
                self.opener = urllib2.build_opener(urllib2.HTTPCookieProcessor(self.cj))
                login_status = True

        except IOError:
            login_status = self.skoob_login()

        if login_status:
            return True
        elif login_status == False:
            open(self.cookie_path, 'w').close()
            error_dialog(self.gui, _('Invalid credentials'),
                                _('Username and/or password invalid. Can\'t connect to Skoob.'),
                                show_copy_button=False, show=True)
            return False
        else:
            return None

    def skoob_login(self):
        ''' Login to Skoob. Called once, during Calibre session. '''

        # Solution using urllib.
        # Found here: https://stackoverflow.com/questions/13925983/login-to-website-using-urllib2-python-2-7

        # Store the cookies and create an opener that will hold them.
        # Needed to make multiple requests, after logging in.

        try:
            print(_('Loading stored cookies failed, manually logging in...'))

            self.opener = urllib2.build_opener(urllib2.HTTPCookieProcessor(self.cj))

            # Install our opener (note that this changes the global opener to the one
            # we just made, but you can also just call opener.open() if you want)
            urllib2.install_opener(self.opener)

            # Authentication page
            authentication_url = 'https://api.skoob.com.br/login'

            # Credentials
            payload = {
                'data[Usuario][email]': self.key_user,
                'data[Usuario][senha]': self.key_password
            }

            # Use urllib to encode the payload
            data = urllib.urlencode(payload)

            # Build our Request object (supplying 'data' makes it a POST)
            login_req = urllib2.Request(authentication_url, data)

            login = urllib2.urlopen(login_req)

            # Save our cookie for future uses
            self.cj.save(self.cookie_path, ignore_discard=True, ignore_expires=True)

            # Test if credentials were accepted
            user_page_url = login.geturl()
            self.user_id = re.search('(https://api.skoob.com.br/usuario/)(.\\d+)', user_page_url)
            if self.user_id:
                self.user_id = re.search('(https://api.skoob.com.br/usuario/)(.\\d+)', user_page_url).groups(0)[1]
                prefs['user_id'] = self.user_id
            if user_page_url == 'https://api.skoob.com.br/login/':
                return False
            else:
                return True

        except urllib2.URLError:
            error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                         show_copy_button=False, show=True)
            return None

    def get_details(self, skoob_id):
        '''Main function to fetch data from Skoob'''

        try:
            root = self.get_user_data(skoob_id)

            if len(root) == 0:
                return

            exists = root[2]
            if exists:
                pass
            else:
                return False

            try:
                if self.key_sync_reading_progress:
                    status = self.parse_status(root[0])
                    shelf = self.parse_shelf(root[0])
                    type = self.parse_physical(root[0])
                    if type == '0':
                        physical = True
                    else:
                        physical = None
                else:
                    status = (None, None)
                    shelf = None
                    physical = None
            except:
                status = (None, None)
                shelf = None
                physical = None

            try:
                if self.key_sync_review:
                    review = self.parse_review(root[1])
                else:
                    review = (None, None)
            except:
                review = (None, None)

            try:
                if self.key_sync_rating:
                    rating = self.parse_rating(root[0])
                else:
                    rating = None
            except:
                rating = None

            try:
                if self.key_sync_page_count:
                    page_count = self.parse_page_count(root[0])
                else:
                    page_count = None
            except:
                page_count = None

            try:
                if self.key_sync_tags:
                    tags = self.parse_tags(root[3])
                else:
                    tags = None
            except:
                tags = None

        except:
            return error_dialog(self.gui, _('Error'), _('Error parsing book details'),
                                                        show_copy_button=False, show=True)

        return status, review, rating, page_count, shelf, tags, physical

    def get_user_data(self, skoob_id):
        root_rp = None
        root_p = None
        exists = False
        root_t = None
        # Reading progress and review pages
        status_url = 'https://api.skoob.com.br/estante/s_historico_leitura/' + str(skoob_id)
        review_url = 'https://api.skoob.com.br/estante/s_resenha/' + str(skoob_id)
        tags_url = 'https://api.skoob.com.br/estante/s_editar/' + str(skoob_id)

        # Build our Requests objects
        status_req = urllib2.Request(status_url)
        review_req = urllib2.Request(review_url)
        tags_req = urllib2.Request(tags_url)

        try:
            # Request status of the book
            if self.key_sync_reading_progress or self.key_sync_rating or self.key_sync_page_count:
                status_info = urllib2.urlopen(status_req)
                status = status_info.read()
                status_decode = status.decode('iso-8859-1', errors='replace')

                try:
                    root_rp = fromstring(clean_ascii_chars(status_decode))
                    exists = self.is_book_in_shelf(root_rp)
                    if exists:
                        pass
                    else:
                        return None, None, exists
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse reading progress'), show=True)

            # Request review of the book
            if self.key_sync_review:
                review_info = urllib2.urlopen(review_req)
                review = review_info.read()
                review_decode = review.decode('iso-8859-1', errors='replace')

                try:
                    root_p = fromstring(clean_ascii_chars(review_decode))
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse book review'),
                                 show_copy_button=False, show=True)

            # Request user tags for the book
            if self.key_sync_tags:
                tags_info = urllib2.urlopen(tags_req)
                tags = tags_info.read()
                tags_decode = tags.decode('iso-8859-1', errors='replace')

                try:
                    root_t = fromstring(clean_ascii_chars(tags_decode))
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse user tags'),
                                 show_copy_button=False, show=True)

        except urllib2.URLError:
            error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                         show_copy_button=False, show=True)
            return

        return root_rp, root_p, exists, root_t

    def is_book_in_shelf(self, root):
        in_shelf_node = root.xpath("id('wd_referente')/div[1]/div[1]/h2[1]")
        if in_shelf_node:
            return True
        else:
            return False

    def parse_status(self, root):
        date_node = root.xpath("id('wd_referente')/div[1]/div[2]/div[1]/div[1]")
        if date_node:
            date = date_node[0].text_content().strip()
        else:
            date = None

        progress_node = root.xpath("id('wd_referente')/div[1]/div[2]/div[1]/div[2]/div[2]")
        if progress_node:
            progress_text = progress_node[0].text_content().strip()
            progress = re.search('(.+)(?=%)', progress_text).groups(0)[0]
        else:
            progress = None

        return date, progress

    def parse_review(self, root):
        review_title_node = root.xpath("//input[@name='data[Resenha][titulo_resenha]']/@value")
        if review_title_node:
            review_title = review_title_node[0]
        else:
            review_title = None

        review_text_node = root.xpath("//textarea[@name='data[Resenha][resenha]']")
        if review_text_node:
            review_text = review_text_node[0]
            aux_review_text = review_text.text_content().strip()
            if aux_review_text == '':
                review_text = None
        else:
            review_text = None

        return review_title, review_text

    def parse_rating(self, root):
        rating_node = root.xpath("//star-rating/@rate")
        if rating_node:
            rating = rating_node[0]
        else:
            rating = None

        return rating

    def parse_page_count(self, root):
        page_count_node = root.xpath("id('wd_referente')/div[1]/div[1]/text()")
        if page_count_node:
            page_count_text = page_count_node[1]
            page_count = re.search('(/ )(.\\d+)', page_count_text).groups(0)[1]
        else:
            page_count = None

        return page_count

    def parse_shelf(self, root):
        shelf_node = root.xpath("//button-shelf/@status")
        if shelf_node:
            shelf = shelf_node[0]
        else:
            shelf = None

        return shelf

    def parse_tags(self, root):
        tags_shelf_node = root.xpath("//input[@name='data[MeusLivro][tags]']/@value")
        tags_public_node = root.xpath("//input[@name='data[MeusLivro][tags_publicas]']/@value")

        day_read = root.xpath("//select[@name='data[MeusLivro][dia_leitura]']/option[@selected]/text()")
        month_read = root.xpath("//select[@name='data[MeusLivro][mes_leitura]']/option[@selected]/text()")
        year_read = root.xpath("//select[@name='data[MeusLivro][ano_leitura]']/option[@selected]/text()")
        date_read = [day_read, month_read, year_read]

        tags = []
        if tags_shelf_node:
            for tag_node in tags_shelf_node:
                tag_list = tag_node.split(',')
                for tag_item in tag_list:
                    tags.append(tag_item)

        if tags_public_node:
            for tag_node in tags_public_node:
                tag_list = tag_node.split(',')
                for tag_item in tag_list:
                    tags.append(tag_item)

        return tags, date_read

    def parse_physical(self, root):
        physical_node = root.xpath("//div[@data-ng-controller='prateleiraCtrl']/@data-ng-init")
        if physical_node:
            physical = re.search('(\\d),(\\d)', physical_node[0]).groups(0)[1]
        else:
            physical = None

        return physical

    def parse_read_date(self, root):
        read_date_node = root.xpath("//div[@data-ng-controller='prateleiraCtrl']/@data-ng-init")
        if read_date_node:
            read_date = re.search('(\\d),(\\d)', read_date_node[0]).groups(0)[1]
        else:
            read_date = None

        return read_date

    def are_selected_books_linked(self):
        all_has_skoob_id = True
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return False
        db = self.gui.library_view.model().db
        for row in rows:
            calibre_id = db.id(row.row())
            identifiers = db.get_identifiers(calibre_id, index_is_id=True)
            if 'skoob' not in identifiers:
                all_has_skoob_id = False
                break

        return all_has_skoob_id

    # Method 'any' reimplemented, since it is marked to change, according to FUTURE warning
    def any(self, iterable):
        for element in iterable:
            if element is not None:
                return True
        return False

    def create_empty_books(self, skoob_id):

        self.ids = []

        # Get preferences
        self.key_reading_progress_column = prefs['reading_progress_column']
        self.key_status_date_column = prefs['status_date_column']
        self.key_rating_column = prefs['rating_column']
        self.key_page_count_column = prefs['page_count_column']
        self.key_read_books_column = prefs['read_books_column']
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        self.key_sync_rating = prefs['sync_rating']
        self.key_sync_page_count = prefs['sync_page_count']
        self.key_sync_read_books = prefs['sync_read_books']

        # Create mi object and include standard fields
        skoob_id = int(skoob_id)
        book_title = self.books[skoob_id]['title']
        if type(book_title) is int:
            book_title = str(book_title)
        book_authors = self.books[skoob_id]['authors']
        db = self.gui.library_view.model().db
        mi = MetaInformation(book_title, book_authors.split(','))
        skoob_id_complete = str(self.books[skoob_id]['book_id']) + 'ed' + str(skoob_id)
        mi.set_identifier('skoob', skoob_id_complete)
        mi.publisher = self.books[skoob_id]['publisher']
        mi.pubdate = datetime.datetime(int(self.books[skoob_id]['year']), 1, 2)
        cover_url = self.books[skoob_id]['cover_url']
        cover_path = os.path.join(config_dir, 'plugins', 'Skoob Sync', 'images', str(skoob_id) + 'cover.jpg')
        urllib.urlretrieve(cover_url, cover_path)
        mi.cover = cover_path

        # Create empty book (no format) based on the mi object
        book_id = db.import_book(mi, [])
        self.ids.append(book_id)

        # Update custom columns
        # Reading progress
        if self.key_sync_reading_progress:
            status_progress = self.books[skoob_id]['reading_progress']
            if status_progress:
                if status_progress > 100:
                    status_progress = 100
            db.new_api.set_field(self.key_reading_progress_column, {book_id: status_progress})
            # Set the status date
            if len(self.key_status_date_column) > 0:
                status_date = self.books[skoob_id]['status_date']
                if len(status_date) > 0:
                    # date_aux = [int(re.search('(\\d+)-(\\d+)-(\\d+)', status_date).groups(0)[0]),
                    #             int(re.search('(\\d+)-(\\d+)-(\\d+)', status_date).groups(0)[1]),
                    #             int(re.search('(\\d+)-(\\d+)-(\\d+)', status_date).groups(0)[2])]
                    # status_date = datetime.datetime(date_aux[0], date_aux[1], date_aux[2], tzinfo=utc_tz)
                    db.new_api.set_field(self.key_status_date_column, {book_id: status_date})
            # Set the read column according to reading progress
            if self.key_sync_read_books:
                if len(self.key_read_books_column) > 0:
                    if status_progress == 100 or self.books[skoob_id]['shelf'] == 1:
                        db.new_api.set_field(self.key_read_books_column, {book_id: True})
                    else:
                        db.new_api.set_field(self.key_read_books_column, {book_id: ''})
        # Rating
        if self.key_sync_rating:
            if self.key_rating_column == 'rating':
                mi.rating = self.books[skoob_id]['rating']*2
            else:
                db.new_api.set_field(self.key_rating_column, {book_id: self.books[skoob_id]['rating']*2})
        # Page count
        if self.key_sync_page_count:
            db.new_api.set_field(self.key_page_count_column, {book_id: self.books[skoob_id]['pages']})

    def skoob_library(self):

        start = timer()

        # Get preferences
        self.key_reading_progress_column = prefs['reading_progress_column']
        self.key_status_date_column = prefs['status_date_column']
        self.key_rating_column = prefs['rating_column']
        self.key_page_count_column = prefs['page_count_column']
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        self.key_sync_rating = prefs['sync_rating']
        self.key_sync_page_count = prefs['sync_page_count']

        # Login to Skoob.
        self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
        QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)
        login_status = self.login()
        if login_status is False or login_status is None:
            return self.gui.status_bar.clearMessage()
        self.gui.status_bar.clearMessage()

        if self.user_id:
            user_id = self.user_id
        else:
            user_id = prefs['user_id']

        self.gui.status_bar.showMessage(_("Reading Skoob Library..."))
        QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)

        json_path = os.path.join(config_dir, 'plugins', 'Skoob Sync', 'library.json')
        skoob_library_url = 'https://api.skoob.com.br/v1/bookcase/books/' + user_id + '/shelf_id:0/page:1/limit:10000/'
        skoob_library_req = urllib2.Request(skoob_library_url)
        skoob_library = urllib2.urlopen(skoob_library_req)
        skoob_library = skoob_library.read().strip()
        json_file = open(json_path, "w")
        json_file.write(json.dumps(json.loads(skoob_library), indent=4, sort_keys=True))
        json_file.close()

        if len(json.loads(skoob_library)) > 0:
            self.books = {}
            for book in json.loads(skoob_library)['response']:
                # Get book details
                skoob_id = book['edicao']['id']
                book_id = book['livro_id']
                book_title = book['edicao']['titulo']
                book_authors = book['edicao']['autor']
                book_publisher = book['edicao']['editora']
                book_year = book['edicao']['ano']
                book_pages = book['edicao']['paginas']
                book_read_pages = book['paginas_lidas']
                book_status_date = book['dt_leitura']
                if book_read_pages > 0:
                    book_reading_progress = book_read_pages/book_pages*100
                else:
                    book_reading_progress = None
                book_rating = book['ranking']
                book_shelf = book['tipo']
                book_cover_url = book['edicao']['capa_grande']
                # Create a nested dict with all the book entries
                self.books[skoob_id] = {
                                   'book_id': book_id,
                                   'title': book_title,
                                   'authors': book_authors,
                                   'publisher': book_publisher,
                                   'year': book_year,
                                   'pages': book_pages,
                                   'reading_progress': book_reading_progress,
                                   'status_date': book_status_date,
                                   'rating': book_rating,
                                   'shelf': book_shelf,
                                   'cover_url': book_cover_url}
            # print(json.dumps(self.books, sort_keys=True, indent=4))

            self.gui.status_bar.clearMessage()

            # Create a widget listing all books on library
            self.tree = QTreeWidget()
            self.tree.setSelectionMode(QAbstractItemView.ExtendedSelection)
            # self.tree.setRootIsDecorated(False)
            layout = QVBoxLayout()
            layout.addWidget(self.tree)
            self.tree.setHeaderLabels([_('Books'), _('Skoob ID'), _('Title'), _('Authors'),
                                       _('Publisher'), _('Year'), _('Pages')])

            self.read = QTreeWidgetItem(self.tree)
            self.read.setText(0, _('1. Read'))
            self.read.setFlags(self.read.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.read.setExpanded(True)

            self.reading = QTreeWidgetItem(self.tree)
            self.reading.setText(0, _('2. Reading'))
            self.reading.setFlags(self.reading.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.reading.setExpanded(True)

            self.want_read = QTreeWidgetItem(self.tree)
            self.want_read.setText(0, _('3. Want to read'))
            self.want_read.setFlags(self.want_read.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.want_read.setExpanded(True)

            self.rereading = QTreeWidgetItem(self.tree)
            self.rereading.setText(0, _('4. Rereading'))
            self.rereading.setFlags(self.rereading.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.rereading.setExpanded(True)

            self.abandoned = QTreeWidgetItem(self.tree)
            self.abandoned.setText(0, _('5. Abandoned'))
            self.abandoned.setFlags(self.abandoned.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.abandoned.setExpanded(True)

            # Add books to list widget
            index = 0
            shelves = [1, 2, 3, 4, 5]
            for shelf in shelves:
                for entry, values in self.books.items():
                    t_skoob_id = entry
                    t_info = values
                    t_title, t_authors, t_publisher, t_year, t_page_count, t_shelf = t_info['title'], t_info['authors'],\
                        t_info['publisher'], t_info['year'], t_info['pages'], t_info['shelf']

                    if t_shelf is not shelf:
                        pass
                    else:
                        index = int(index) + 1
                        index = "{0:0=3d}".format(index)
                        if t_shelf == 1:
                            parent = self.read
                        elif t_shelf == 2:
                            parent = self.reading
                        elif t_shelf == 3:
                            parent = self.want_read
                        elif t_shelf == 4:
                            parent = self.rereading
                        else:
                            parent = self.abandoned
                        child = QTreeWidgetItem(parent, [str(index), str(t_skoob_id), str(t_title), str(t_authors),
                                                              str(t_publisher), str(t_year), str(t_page_count)])
                        child.setFlags(child.flags() | Qt.ItemIsUserCheckable)
                        child.setCheckState(0, Qt.Unchecked)
                        # Select background color based on severity
                        if t_shelf == 1:
                            bg_color = QtGui.QBrush(QtGui.QColor(180, 255, 180))
                        elif t_shelf == 2:
                            bg_color = QtGui.QBrush(QtGui.QColor(240, 255, 170))
                        elif t_shelf == 3:
                            bg_color = QtGui.QBrush(QtGui.QColor(180, 230, 255))
                        elif t_shelf == 4:
                            bg_color = QtGui.QBrush(QtGui.QColor(255, 220, 150))
                        else:
                            bg_color = QtGui.QBrush(QtGui.QColor(120, 120, 120))
                        child.setBackground(0, QtGui.QColor(bg_color))
                        child.setBackground(1, QtGui.QColor(bg_color))
                        child.setBackground(2, QtGui.QColor(bg_color))
                        child.setBackground(3, QtGui.QColor(bg_color))
                        child.setBackground(4, QtGui.QColor(bg_color))
                        child.setBackground(5, QtGui.QColor(bg_color))
                        child.setBackground(6, QtGui.QColor(bg_color))
                        parent.addChild(child)

            # Auto adjust column sizes
            self.tree.resizeColumnToContents(5)
            self.tree.resizeColumnToContents(6)

            # Enable sorting
            self.tree.setSortingEnabled(True)
            self.tree.sortItems(0, Qt.AscendingOrder)
            self.tree.setColumnHidden(1, True)

            # Enable selection when clicking anywhere on the line
            # self.tree.itemClicked.connect(self.on_item_clicked)

            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            self.show_books(self.tree)

            # Create selected empty books
            if self.skoob_ids is None:
                return
            elif len(self.skoob_ids) > 0:
                start = timer()

                if self.key_sync_reading_progress:
                    if len(self.key_reading_progress_column) == 0:
                        return warning_dialog(self.gui, _('Configuration needed'),
                                              _('You must choose the reading progress custom column'),
                                              show_copy_button=False, show=True)

                if self.key_sync_page_count:
                    if len(self.key_page_count_column) == 0:
                        return warning_dialog(self.gui, _('Configuration needed'),
                                              _('You must choose the page count custom column'),
                                              show_copy_button=False, show=True)

                # Create empty books (each on a different thread)
                self.gui.status_bar.showMessage(_('Creating new entries to calibre library...'))
                QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)

                aux_list = []
                for skoob_id in self.skoob_ids:
                    aux_list.append((skoob_id,))
                self.run_parallel_in_threads(self.create_empty_books, aux_list)

                # Refresh the GUI, so the new books are displayed
                m = self.gui.library_view.model()
                m.refresh_ids(list(self.ids))
                m.refresh_ids(self.ids, current_row=self.gui.library_view.currentIndex().row())
                current = self.gui.library_view.currentIndex()
                self.gui.refresh_cover_browser()
                self.gui.tags_view.recount_with_position_based_index()
                qv = get_quickview_action_plugin()
                if qv:
                    qv.refresh_quickview(current)

                self.gui.status_bar.clearMessage()

                end = timer()
                time = end - start
                print(datetime.timedelta(seconds=time))

                info_dialog(self.gui, _('New Books'), _('%d new book(s) added to calibre library')
                            % len(self.skoob_ids), show_copy_button=False, show=True)
            else:
                return

        else:
            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            self.gui.status_bar.clearMessage()
            error_dialog(self.gui, _('Error'), _('Failed to read Skoob Library'),
                         show_copy_button=False, show=True)

    # def on_item_clicked(self):
    #     item = self.tree.currentItem()
    #     item.setCheckState(0, QtCore.Qt.Unchecked if item.checkState(0) == QtCore.Qt.Checked else QtCore.Qt.Checked)

    def show_books(self, tree_widget):
        tool = self
        _gui = self.gui

        class SkoobLibraryDialog(Dialog):

            def __init__(self):
                Dialog.__init__(self, _('Skoob Library'), 'plugin-skoob-sync-dialog', parent=tool.gui)

            def setup_ui(self):
                self.box = QVBoxLayout(self)
                self.widget = tree_widget
                self.box.addWidget(self.widget)
                self.button_box = QDialogButtonBox()
                self.button_box.addButton(_('  Create books  '), QDialogButtonBox.AcceptRole)
                self.button_box.addButton(_('Cancel'), QDialogButtonBox.RejectRole)
                self.box.addWidget(self.button_box)
                self.button_box.accepted.connect(self.handler)
                self.button_box.rejected.connect(self.reject)

            def handler(self):
                self.count = -1
                # Get selected books
                SkoobSyncTools.skoob_ids = []

                # Look for checked items
                iterator = QTreeWidgetItemIterator(d.widget, QTreeWidgetItemIterator.Checked)
                while iterator.value():
                    item = iterator.value()
                    if item.parent():
                        SkoobSyncTools.skoob_ids.append(item.text(1))
                        self.count += 1
                    iterator += 1

                # Look for selected items (so one could use shift/ctrl + arrow keys)
                self.new_count = 0
                iterator = QTreeWidgetItemIterator(d.widget, QTreeWidgetItemIterator.NotChecked |
                                                   QTreeWidgetItemIterator.Selected)
                while iterator.value():
                    item = iterator.value()
                    if item.parent():
                        SkoobSyncTools.skoob_ids.append(item.text(1))
                        self.new_count += 1
                    iterator += 1

                self.count += self.new_count + 1
                if self.count > 50:
                    SkoobSyncTools.skoob_ids = None
                    error_dialog(_gui, _('Too many books'),
                                    _('%d books selected. Try to select 50 books at most.') % self.count,
                                    show_copy_button=False, show=True)
                elif self.count == -1:
                    SkoobSyncTools.skoob_ids = None
                    error_dialog(_gui, _('Empty selection'), _('No books selected'), show_copy_button=False, show=True)
                else:
                    Dialog.accept(self)

            def reject(self):
                SkoobSyncTools.skoob_ids = None
                Dialog.reject(self)

        d = SkoobLibraryDialog()
        d.exec_()
