#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals, division, absolute_import, print_function
import xml.etree.ElementTree as ET
import os, os.path, sys, codecs, re
import subprocess, shutil
from subprocess import Popen, PIPE
from os.path import expanduser

if sys.version_info[0] == 2:
    import tkFileDialog as tkinter_filedialog
else:
    import tkinter.filedialog as tkinter_filedialog

try:
    from sigil_bs4 import BeautifulSoup
except:
    from bs4 import BeautifulSoup

#----------------------------------
# detect operating system
#----------------------------------
ISOSX = sys.platform.startswith('darwin')
ISLINUX = sys.platform.startswith('linux')
ISWINDOWS = sys.platform.startswith('win32')

def get_desktop():
    ''' returns the Desktop folder path '''
    # output directory
    home = expanduser('~')
    output_dir = home
    # look for Desktop folder
    for desktop_name in ['Desktop', 'Bureau', 'Escritorio']:
        new_output_dir = os.path.join(home, desktop_name)
        if os.path.isdir(new_output_dir):
            output_dir = new_output_dir
            break
    return output_dir

# display prince file selection dialog
def get_file_name(title):
    ''' displays a file dialog '''
    file_path = tkinter_filedialog.askopenfilename(title=title)
    return file_path

# find prince binary
def find_prince():
    ''' returns the prince file path '''
    prince_path = None

    if ISWINDOWS:
        # the default path is: C:\Program Files (x86)\Prince\engine\bin\prince.exe
        default_windows_path = os.path.join(os.getenv('programfiles(x86)'), 'Prince', 'engine', 'bin', 'prince.exe')
        if os.path.isfile(default_windows_path):
            prince_path = default_windows_path
    elif ISOSX:
        # the default path is: /usr/local/bin/prince
        default_osx_path = os.path.join('/usr', 'local', 'bin', 'prince')
        if os.path.isfile(default_osx_path):
            prince_path = default_osx_path
    else:
        # default paths are: /usr/sbin/prince or /usr/bin/prince 
        default_linux_path = shutil.which("prince")
        if os.path.isfile(default_linux_path):
            prince_path = default_linux_path

    # display select file dialog box
    if not prince_path:
        prince_path = get_file_name('Select Prince executable')
        if not prince_path or not os.path.basename(prince_path).startswith('prince'):
            prince_path = None

    return prince_path

# simple prince wrapper
def prince_wrapper(*args):
    ''' wrapper for running prince '''
    try:
        process = Popen(list(args), stdout=PIPE, stderr=PIPE)
        ret = process.communicate()
        returncode = process.returncode
        return ret, returncode
    except FileNotFoundError:
        print('Prince binary not found!')
        return -1

# main routine
def run(bk):
    ''' main routine '''

    # this plugin only works with standard file paths
    if bk.launcher_version() >= 20190927 and not bk.epub_is_standard():
        print('This plugin only works with standard epubs.\nClick OK to close the Plugin Runner window.')
        return 0

    prince_path = None
    output_dir = None

    # load preferences
    prefs = bk.getPrefs()

    # write initial preference file
    if prefs == {}:
        prince_path = find_prince()
        prefs['prince_path'] = prince_path
        output_dir = get_desktop()
        prefs['output_dir'] = output_dir
        bk.savePrefs(prefs)

    #-------------------------------------
    # get prince executable file path
    #-------------------------------------
    if 'prince_path' in prefs:
        prince_path = prefs['prince_path']
        if not os.path.isfile(prince_path):
            prince_path = find_prince()
            prefs['prince_path'] = prince_path
            bk.savePrefs(prefs)
    else:
        prince_path = find_prince()
        prefs['prince_path'] = prince_path
        bk.savePrefs(prefs)

    #-------------------------------------
    # get output folder
    #-------------------------------------
    if 'output_dir' in prefs:
        output_dir = prefs['output_dir']
        if not os.path.isdir(output_dir):
            output_dir = get_desktop()
            prefs['output_dir'] = output_dir
            bk.savePrefs(prefs)
    else:
        output_dir = get_desktop()
        prefs['output_dir'] = output_dir
        bk.savePrefs(prefs)

    #-----------------------------
    # check prince paths
    #-----------------------------
    if ISWINDOWS:
        if os.path.isfile(prince_path):
            args = [prince_path, '-v']
        else:
            print('prince.exe not found: ' + prince_path)
            return -1
    elif ISOSX:
        if os.path.isfile(prince_path):
            args = [prince_path, '-v']
        else:
            print('prince not found: ' + prince_path)
            return -1
    else:
        if shutil.which("prince"):
            args = ['prince', '-v']
        else:
            print('prince not found: ' + prince_path)
            return -1
            
    # ---------------------------------
    # get optional prince parameters
    #----------------------------------

    # Apply an external style sheet.
    stylesheet_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'style.css')
    if 'style' in prefs and prefs['style'] and os.path.isfile(stylesheet_path):
        args.append('--style=' + stylesheet_path)

    # Specify the media type (eg. print, screen).
    if 'media' in prefs:
        media = prefs['media']
        args.append('--media=' + media)

    # Specify the page size (eg. A4).
    if 'page_size' in prefs:
        page_size = prefs['page_size']
        args.append('--page-size=' + page_size)

    # Specify the page margin (eg. 20mm).
    if 'page_margin' in prefs:
        page_margin = prefs['page_margin']
        args.append('--page-margin=' + page_margin)

    # Ignore author style sheets.
    if 'no_author_style' in prefs and prefs['no_author_style']:
        args.append('--no-author-style')

    # Ignore default style sheets.
    if 'no_default_style' in prefs and prefs['no_default_style']:
        args.append('--no-default-style')

    # post processing with cpdf
    if 'cpdf' in prefs:
        cpdf = prefs['cpdf']
    else:
        cpdf = False
        prefs['cpdf'] = False
        bk.savePrefs(prefs)

    # remove annotations
    if 'remove_annotations' in prefs:
        remove_annotations = prefs['remove_annotations']
    else:
        remove_annotations = False

    #--------------------------------------------------
    # Get title, author and subject info from metadata
    #--------------------------------------------------

    # get metadata from content.opf
    temp_dir = bk._w.ebook_root

    # get the default opf path and the opf file name
    if bk.launcher_version() >= 20190927:
        opf_path = os.path.join(temp_dir, bk.get_opfbookpath())
    else:
        opf_path = os.path.join(temp_dir, 'OEBPS', 'content.opf')

    # parse content.opf
    tree = ET.parse(opf_path)
    #root = tree.getroot()

    # get book title
    dc_title = 'Untitled'
    meta_title = tree.find('.//{http://purl.org/dc/elements/1.1/}title')
    if hasattr(meta_title, 'text') and meta_title.text:
        dc_title = meta_title.text
        args.append('--pdf-title=' + dc_title)

    # get author
    dc_creator = 'Unknown'
    meta_creator = tree.find('.//{http://purl.org/dc/elements/1.1/}creator')
    if hasattr(meta_creator, 'text') and meta_creator.text:
        dc_creator = meta_creator.text
        args.append('--pdf-author=' + dc_creator)

    # get subject(s)
    dc_subject = None
    for meta_subject in tree.findall('.//{http://purl.org/dc/elements/1.1/}subject'):
        if hasattr(meta_subject, 'text') and meta_subject.text:
            if dc_subject:
                dc_subject += ', ' + meta_subject.text
            else:
                dc_subject = meta_subject.text
    if dc_subject:
        args.append('--pdf-subject=' + dc_subject)

    # get language
    dc_language = 'en'
    meta_language = tree.find('.//{http://purl.org/dc/elements/1.1/}language')
    if hasattr(meta_language, 'text') and meta_language.text:
        dc_language = meta_language.text
        if len(dc_language) != 2:
            dc_language = dc_language[0:2]

    # -------------------------
    # get list of file names
    # -------------------------

    file_list = list(bk.text_iter())
    file_name_list = None
    for (html_id, href) in file_list:

        # check for custom folders in sigil 1.x (and higher)
        if bk.launcher_version() >= 20190927:
            book_path = bk.id_to_bookpath(html_id)
            file_name = os.path.abspath(os.path.join(temp_dir, book_path))
        else:
            file_name = os.path.abspath(os.path.join(temp_dir, 'OEBPS', href))

        # the folder that contains the html file
        file_root_path = os.path.dirname(file_name)
        base_name = os.path.basename(href)

        #----------------------------------------
        # pre-process all xhtml files for prince
        #----------------------------------------
        html = bk.readfile(html_id)
        soup = BeautifulSoup(html, 'html.parser')
        orig_soup = str(soup)

        # update all href attributes
        tags = soup.find_all('a')
        for each_tag in tags:
            if each_tag.has_attr('href'):
                # "normalize" cross-documente links (required by prince xml)
                tag_href = each_tag['href']
                href_parts = href.split('#')
                href_file_name = href_parts[0]
                if os.path.isfile(os.path.join(file_root_path, os.path.basename(href_file_name))):
                    each_tag['href'] = os.path.basename(tag_href)
                else:
                    if not tag_href.startswith('http'):
                        print('Linked file {} not found!'.format(href_file_name))

        # add language attribute to <html> tag for proper hyphenation
        if dc_language in ['da', 'de', 'en', 'es', 'fi', 'fr', 'is', 'it', 'lt', 'pl', 'pt', 'ru', 'sl', 'sv']:
            tag = soup.find('html')
            if not tag.has_attr('xml:lang') and not tag.has_attr('lang'):
                tag['xml:lang'] = dc_language

        # save updated html files
        if str(soup) != orig_soup:
            with codecs.open(file_name, 'w', 'utf-8') as fp:
                fp.write(str(soup))

        #----------------------------
        # add file name to file list
        #----------------------------
        if file_name_list:
            file_name_list += base_name + '\n'
        else:
            file_name_list = base_name + '\n'

    # write file names to file list file
    file_list_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'file_list.txt')
    with codecs.open(file_list_path, 'w', 'utf-8') as fp:
        fp.write(file_name_list)
    args.append('-l')
    args.append(file_list_path)

    #--------------------------------------
    # add pdf file path to parameter list
    #--------------------------------------

    # define cross platform safe output file iname
    pdf_path = os.path.join(output_dir, re.sub(r'[/|\?|<|>|\\\\|:|\*|\||"|\^| ]+', '_', dc_title) + '.pdf')

    # add output file as last prince command line parameter
    args.append('-o')
    args.append(pdf_path)

    # debug: display command line arguments
    #print('command line:\n' + ' '.join(args) + '\n')

    #--------------------------------
    # run prince
    #--------------------------------

    # locate the Text folder ... and go there
    # (otherwise links won't work in the Windows version!!!)
    if bk.launcher_version() >= 20190927:
        file_root_path = os.path.dirname(file_name)
    else:
        file_root_path = os.path.join(temp_dir, 'OEBPS', 'Text')
    os.chdir(file_root_path)

    print('Running Prince...')
    try:   
        result, return_code = prince_wrapper(*args)
    except TypeError:
        print('\nClick OK to close the Plugin Runner window.')
        return -1

    # delete file name list file
    os.remove(file_list_path)

    # debug: display prince messages
    print(result[1].decode("utf-8"))

    if return_code != 0:
        print('Return code:', return_code)
        print('Prince failed!')
        return -1

    if os.path.isfile(pdf_path):
        print('pdf file written to: ' + pdf_path)
        # run cpdf
        if cpdf:
            squeezed_path = pdf_path + '.squeezed'
            if remove_annotations:
                args = ['cpdf', '-squeeze', '-remove-annotations', pdf_path, '1', '-o', squeezed_path]
            else:
                args = ['cpdf', '-squeeze', pdf_path, '-o', squeezed_path]
            try:
                print('\nRunning cpdf...')
                result, return_code = prince_wrapper(*args)
                stdout = result[0]
                sderr = result[1]
                print(stdout.decode('utf-8'))
                if os.path.isfile(squeezed_path):
                    os.remove(pdf_path)
                    os.rename(squeezed_path, pdf_path)
                else:
                    print('Return code:', return_code)
                    print('cpdf error:')
                    print(stdout.decode('utf-8'))
                    print(stderr.decode('utf-8'))
            except:
                print('\ncpdf failed or not found!!!')
        return 0
    else:
        print('No pdf file written to: ' + pdf_path)
        return -1

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
