#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()

from threading import Thread
from time import sleep

from calibre.gui2.threaded_jobs import ThreadedJob


try:
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory

except ImportError:
    from logging_factory import LoggingFactory


class LoggerWrapper(object):
    '''
    Extends a normal Logger as being call-able
    '''
    
    def __init__(self, logger):
        '''
        This LoggerWrapper redirects ThreadedJob logging to this plug-in's logging
        -> html and plain_text fields must be defined
        '''
        self.logger = logger
        self.html = ''
        self.plain_text = "See plug-in's log file"
        
    
    def __call__(self, msg, *args, **kwargs):
        self.logger.info(msg + args[0])
        
    
    def debug(self, msg, *args, **kwargs):
        self.logger.debug(
            msg + 
            (str(args[0]) if len(args) >= 1 else '') + 
            (str(args[1]) if len(args) >= 2 else ''), 
            **kwargs)
        
    
    def exception(self, msg, *args, **kwargs):
        self.logger.exception(msg, *args, **kwargs)
    

class PluginWorker(object):
    '''
    Class serving as a wrapper between the Action and the Invoker performing the actual work.
    This work can be done in the background leaving the UI free (not blocked).
    '''
    
    def __init__(self, gui):
        '''
        Initialization
        '''
        self.gui = gui
        
        
    def start_work(self, work, callback):
        '''
        Starts a piece of work
        @param work: a piece of work, represented by a function object
        @param callback: the callback to be invoked on completion
        '''
        self.work = work
        self.callback = callback
        
        log = LoggingFactory().getLogger(self)
        self.job = ThreadedJob(
            'Toc View Generator Worker', 
            _("Toc Extraction"), 
            self, [], {}, 
            self._callback, 
            killable = False,
            log = LoggerWrapper(log))
        
        self.gui.job_manager.run_threaded_job(self.job)
        
        
    def __call__(self, abort = None, log = None, notifications = None):
        '''
        Performs a piece of work, given by self.work
        '''
        self.notifications = notifications
        self.progress_thread = Thread(target = self._update_progress)
        self.progress_thread.start()
        
        if not self.work():
            self.job.failed = True
        
        notifications.put((1.0, _('Finished')))
    
    
    def _update_progress(self):
        '''
        Updates the Job progress
        '''
        progress = 0.0
        
        while True:
            progress += 0.01
            if progress >= 1.0:
                progress = 0.0
                
            notification = (progress, _('Just working'))
            self.notifications.put(notification);
            sleep(0.1)
    
    
    def _callback(self, job):
        '''
        Invokes the callback after a piece of work is done
        '''
        self.callback(not job.failed)
        