#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import os.path
import re
import cookielib
import urllib
import urllib2
from timeit import default_timer as timer
import datetime
import time
import threading
import Queue
from lxml.html import fromstring, tostring

# PyQt libraries
from PyQt5.Qt import Qt, QApplication
from PyQt5 import QtCore

# Calibre libraries
from calibre.utils.config import config_dir
from calibre_plugins.Skoob_Sync.config import prefs
from calibre.utils.cleantext import clean_ascii_chars
from calibre.gui2 import error_dialog, info_dialog, warning_dialog, question_dialog
from calibre.library.comments import sanitize_comments_html

# Starting credentials. Used to check if it is changed.
KEY_USER = prefs['user']
KEY_PASSWORD = prefs['password'].decode('base64')


class SkoobSyncTools:
    def __init__(self, gui):
        self.gui = gui

    def has_skoob_id(self):
        ''' Show books with Skoob Books IDs '''

        self.gui.search.setEditText('identifiers:"=skoob:"')
        self.gui.search.do_search()

    def update_status(self, sync_option, shelf):
        '''Sync book metatada with Skoob.'''

        start = timer()
        self.shelf = shelf
        self.sync_option = sync_option

        # Get preferences
        self.key_user = prefs['user']
        self.key_password = prefs['password'].decode('base64')
        self.key_reading_progress_column_idx = prefs['reading_progress_column_idx']
        self.key_reading_progress_column = prefs['reading_progress_column']
        self.key_status_date_column_idx = prefs['status_date_column_idx']
        self.key_status_date_column = prefs['status_date_column']
        self.key_review_column_idx = prefs['review_column_idx']
        self.key_review_column = prefs['review_column']
        self.key_rating_column_idx = prefs['rating_column_idx']
        self.key_rating_column = prefs['rating_column']
        self.key_page_count_column_idx = prefs['page_count_column_idx']
        self.key_page_count_column = prefs['page_count_column']
        self.key_read_books_column_idx = prefs['read_books_column_idx']
        self.key_read_books_column = prefs['read_books_column']
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        self.key_sync_review = prefs['sync_review']
        self.key_sync_rating = prefs['sync_rating']
        self.key_sync_page_count = prefs['sync_page_count']
        self.key_sync_read_books = prefs['sync_read_books']
        self.key_sync_update_added = prefs['sync_update_added']

        # Check if credentials are set
        if self.key_user in ('user', '') or self.key_password in ('password', ''):
            return warning_dialog(self.gui, _('Invalid Credentials'), _('You must first setup your Skoob account'),
                               show_copy_button=False, show=True)

        # Check if no sync option is checked
        sync = [self.key_sync_reading_progress, self.key_sync_review, self.key_sync_rating,
                self.key_sync_page_count]
        if not any(sync):
            return warning_dialog(self.gui, _('Nothing to sync'), _('No sync actions selected'),
                               show_copy_button=False, show=True)

        # Check if the reading progress and status date columns are set
        if self.key_sync_reading_progress:
            if len(self.key_reading_progress_column) == 0:
                return warning_dialog(self.gui, _('Configuration needed'),
                                   _('You must choose the reading progress custom column'),
                                   show_copy_button=False, show=True)

        # Check if the review column is set
        if self.key_sync_review:
            if len(self.key_review_column) == 0:
                return warnig_dialog(self.gui, _('Configuration needed'),
                                   _('You must choose the review custom column'), show_copy_button=False, show=True)

        # Check if the page count column is set
        if self.key_sync_page_count:
            if len(self.key_page_count_column) == 0:
                return warning_dialog(self.gui, _('Configuration needed'),
                                   _('You must choose the page count custom column'), show_copy_button=False, show=True)

        # Check if the option to mark read book is selected
        if self.key_sync_reading_progress and self.key_sync_read_books:
            if len(self.key_read_books_column) == 0:
                return warning_dialog(self.gui, _('Configuration needed'),
                                      _('You must choose the read books custom column'),
                                      show_copy_button=False, show=True)

        # Get currently selected books
        rows = self.gui.library_view.selectionModel().selectedRows()

        # Check if selection is empty
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, _('Empty selection'),
                                _('No books selected'), show=True)

        # Check if selected books have a skoob id
        is_selection_valid = self.are_selected_books_linked()
        if is_selection_valid is False:
            if len(rows) == 1:
                return error_dialog(self.gui, _('Selection invalid'),
                                    _('Selected book does not have a Skoob id'), show_copy_button=False, show=True)
            else:
                return error_dialog(self.gui, 'Selection invalid',
                                    _('All selected books must have a Skoob id'), show_copy_button=False, show=True)

        # Map the rows to book ids
        ids = list(map(self.gui.library_view.model().id, rows))
        db = self.gui.current_db.new_api

        # Create list of altered books
        self.books_not_updated = []
        self.books_updated = []
        self.books_not_in_shelf = []
        self.bad_skoob_id = []

        # Check if too many books are selected at once
        if len(ids) < 50:
            if len(ids) > 20:
                if question_dialog(self.gui, _('Caution'),
                             _('You selected %d books. Selecting too many books may cause calibre to crash. '
                               'Are you sure you want to continue?') % len(ids), show_copy_button=False):
                    pass
                else:
                    return

            # Login to Skoob. Session will keep open while checking multiple books.
            try:
                self.cookie_path = os.path.join(config_dir, 'plugins', 'Skoob Sync', 'cookie', 'cookie.txt')
                self.cj = cookielib.MozillaCookieJar(self.cookie_path)

                # Check if user credentials have changed
                if not (KEY_USER == self.key_user and KEY_PASSWORD == self.key_password):
                    self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
                    QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)
                    login_status = self.skoob_login()

                else:
                    self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
                    QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)
                    self.cj.load()
                    self.opener = urllib2.build_opener(urllib2.HTTPCookieProcessor(self.cj))
                    login_status = True

            except IOError:
                self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
                QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)
                login_status = self.skoob_login()

            finally:
                self.gui.status_bar.clearMessage()
                QApplication.processEvents(QtCore.QEventLoop.ExcludeUserInputEvents)

            if login_status:
                pass
            elif login_status == False:
                return error_dialog(self.gui, _('Invalid credentials'),
                                    _('Username and/or password invalid. Can\'t connect to Skoob.'),
                                    show_copy_button=False, show=True)
            else:
                return

            # THIS IS REALLY IMPORTANT! This code allows the plugin to run simultaneous checks. Nice!
            aux_list = []
            for book_id in ids:
                aux_list.append((book_id,))
            self.run_parallel_in_threads(self.update_routine, aux_list)

            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            # Refresh the GUI, so the Book Info panel shows the new info
            self.gui.library_view.model().refresh_ids(ids, current_row=self.gui.library_view.currentIndex().row())

            # Final message to the user, informing how many books were updated
            if len(self.books_not_updated) > 0:
                if len(self.books_updated) > 0:
                    warning_dialog(self.gui, _('Update status'),
                                   _('Updated %d book(s). ') % len(self.books_updated) +
                                     _('No updates for %d book(s).') % len(self.books_not_updated),
                                        show_copy_button=False, show=True)
                else:
                    warning_dialog(self.gui, _('Nothing to update'), _('No updates for %d book(s)')
                                    % len(self.books_not_updated), show_copy_button=False, show=True)
            else:
                if len(self.books_updated) > 0:
                    info_dialog(self.gui, _('Update status'), _('Updated %d book(s)')
                                    % len(self.books_updated), show_copy_button=False, show=True)
                # else:
                #     error_dialog(self.gui, _('Not in shelf'), _('%d book(s) not in any shelf'
                #                     % len(self.books_not_in_shelf)), show_copy_button=False, show=True)
            if len(self.books_not_in_shelf) > 0:
                book_list = []
                for book_id in self.books_not_in_shelf:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                error_dialog(self.gui, _('Not in shelf'), _('%d book(s) not in any shelf')
                            % len(self.books_not_in_shelf) + ':\n' + book_list_str, show_copy_button=False, show=True)

            if len(self.bad_skoob_id) > 0:
                book_list = []
                for book_id in self.bad_skoob_id:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                if sync_option == 'add_to_skoob':
                    error_dialog(self.gui, _('Not added'), _('%d book(s) with unknown Skoob IDs')
                                    % len(self.bad_skoob_id) + ':\n' + book_list_str,
                                    show_copy_button=False, show=True)
                else:
                    error_dialog(self.gui, _('Not removed'), _('%d book(s) are not on your shelves '
                                        'or have invalid IDs') % len(self.bad_skoob_id) + ':\n' + book_list_str,
                                        show_copy_button=False, show=True)

        else:
            # To avoid overloading the server, we must update no more than 50 books at a time. This is just a guess.
            # This plugin does not use Skoob API, since is has not been made public yet, according to the their FAQ.
            return error_dialog(self.gui, _('Too many books'), _('%d books selected. Try to select 50 books at most.')
                               % len(ids), show_copy_button=False, show=True)


    def run_parallel_in_threads(self, target, args_list):
        ''' Utility - spawn a thread to execute target for each args '''
        # https://stackoverflow.com/questions/3490173/how-can-i-speed-up-fetching-pages-with-urllib2-in-python/3491455

        result = Queue.Queue()

        # Wrapper to collect return value in a Queue
        def task_wrapper(*args):
            result.put(target(*args))

        threads = [threading.Thread(target=task_wrapper, args=args) for args in args_list]
        for t in threads:
            t.start()
            # Don't send all requests at once
            time.sleep(0.1)
        for t in threads:
            t.join()
        return result

    def update_routine(self, book_id):
        ''' This method does all the Skoob syncing activity'''

        # Get the current metadata for this book from the db
        db = self.gui.current_db.new_api
        mi = db.get_metadata(book_id)
        # Get skoob_id from calibre
        identifiers = mi.get_identifiers()
        skoob_id = identifiers['skoob']
        is_skoob_id = re.search('(ed)(.+)', skoob_id)
        if is_skoob_id is None:
            return self.bad_skoob_id.append(book_id)
        else:
            skoob_id = re.search('(ed)(.+)', skoob_id).groups(0)[1]

        # Download FROM Skoob
        if self.sync_option == 'from_skoob':
            self.gui.status_bar.showMessage(_("Downloading from Skoob..."))
            self.download_from_skoob(skoob_id, book_id, db, mi)
            self.gui.status_bar.clearMessage()

        # Upload TO Skoob
        if self.sync_option == 'to_skoob':
            self.gui.status_bar.showMessage(_("Uploading to Skoob..."))
            self.upload_to_skoob(skoob_id, book_id, db, mi)
            self.gui.status_bar.clearMessage()

        # Add to Skoob / Add to shelf
        if self.sync_option == 'add_to_skoob':
            if self.shelf is not None:
                self.gui.status_bar.showMessage(_("Adding to shelf..."))
            else:
                self.gui.status_bar.showMessage(_("Adding to Skoob..."))
            valid_id, to_update = self.add_to_skoob(skoob_id, book_id, mi)
            # If update_added is checked, it will return True. This will prevent duplicating final message.
            if to_update:
                self.upload_to_skoob(skoob_id, book_id, db, mi)
            else:
                # If the Skoob ID is invalid, then abort
                if valid_id:
                    # Set the read column according to reading progress
                    read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']
                    if self.key_sync_read_books:
                        if len(self.key_read_books_column) > 0:
                            if read_percentage is None:
                                db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                            else:
                                if int(read_percentage) == 100:
                                    db.new_api.set_field(self.key_read_books_column, {book_id: True})
                                else:
                                    db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                else:
                    self.gui.status_bar.clearMessage()
                    return
            self.gui.status_bar.clearMessage()

        # Remove from Skoob
        if self.sync_option == 'remove_from_skoob':
            self.gui.status_bar.showMessage(_("Removing from Skoob..."))
            valid_id = self.remove_from_skoob(skoob_id, book_id, mi)
            # If the Skoob ID is invalid, then abort
            if valid_id:
                # Set the read column according to reading progress
                read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']
                if self.key_sync_read_books:
                    if len(self.key_read_books_column) > 0:
                        if read_percentage is None:
                            db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                        else:
                            if int(read_percentage) == 100:
                                db.new_api.set_field(self.key_read_books_column, {book_id: True})
                            else:
                                db.new_api.set_field(self.key_read_books_column, {book_id: ''})
            else:
                self.gui.status_bar.clearMessage()
                return
            self.gui.status_bar.clearMessage()

    def download_from_skoob(self, skoob_id, book_id, db, mi):
        # Download FROM Skoob
        try:
            details = self.get_details(skoob_id)
            if details is False:
                return self.books_not_in_shelf.append(book_id)
            status = details[0]  # Get reading progress and status date
            review = details[1]  # Get review title and text
            rating = details[2]  # Get rating
            page_count = details[3]  # Get the page count
            result = [status[0], status[1], review[0], review[1], rating, page_count]

            if self.any(result):

                # -- Set Reading Progress --
                if self.key_sync_reading_progress:
                    status_date = status[0]
                    status_progress = status[1]

                    # Set the read column according to reading progress
                    if self.key_sync_read_books:
                        if len(self.key_read_books_column) > 0:
                            if status_progress == '100':
                                db.new_api.set_field(self.key_read_books_column, {book_id: True})
                            else:
                                db.new_api.set_field(self.key_read_books_column, {book_id: ''})

                    if len(self.key_reading_progress_column) > 0:
                        db.new_api.set_field(self.key_reading_progress_column, {book_id: status_progress})
                    if len(self.key_status_date_column) > 0:
                        db.new_api.set_field(self.key_status_date_column, {book_id: status_date})

                # -- Set Review --
                if self.key_sync_review:
                    if review == (None, None):
                        pass
                    else:
                        if self.key_review_column == 'comments':
                            skoob_review = mi.comments
                        else:
                            skoob_review = mi.metadata_for_field(self.key_review_column)['#value#']
                        # Personalized entry to merge with existing comments, so it can be retrieved easier
                        review_open = '## Skoob ##'
                        review_end = '##########'
                        review_title = review[0]
                        review_text = tostring(review[1], method='html')
                        review_text = sanitize_comments_html(review_text)

                        if len(review_text) > 0:
                            # Check if a Skoob Review is already set
                            if self.key_review_column == 'comments':
                                if skoob_review:
                                    existing_comments_review = re.search('## Skoob ##', skoob_review)
                                    if existing_comments_review:
                                        comments_clean = re.search('([\\s\\S]+)(## Skoob ##)',
                                                                   skoob_review).groups(0)[0]
                                        new_comments = comments_clean + review_open + '<p><b id="skoob_title">' +\
                                                       review_title + '</b></p><div id="skoob_review">'\
                                                       + review_text + '</div>' + review_end
                                    else:
                                        new_comments = skoob_review + review_open + '<p><b id="skoob_title">' +\
                                                       review_title + '</b></p><div id="skoob_review">'\
                                                       + review_text + '</div>' + review_end
                                else:
                                    new_comments = review_open + '<b id="skoob_title">' + review_title +\
                                                   '</b><div id="skoob_review">' + review_text + '</div>' + review_end
                            else:
                                new_comments = '<b id="skoob_title">' + review_title + '</b><div id="skoob_review">'\
                                               + review_text + '</div>'
                            db.new_api.set_field(self.key_review_column, {book_id: new_comments})
                        else:
                            pass

                # -- Set Rating --
                if self.key_sync_rating and rating is not None:
                    rating_number = float(rating) * 2
                    if len(self.key_rating_column) > 0:
                        db.new_api.set_field(self.key_rating_column, {book_id: rating_number})

                # -- Set Page Count --
                if self.key_sync_page_count:
                    if len(self.key_page_count_column) > 0:
                        db.new_api.set_field(self.key_page_count_column, {book_id: page_count})

                # Mark the book as updated
                self.books_updated.append(book_id)
            else:
                # Mark the book as NOT updated
                self.books_not_updated.append(book_id)
        except:
            return

    def upload_to_skoob(self, skoob_id, book_id, db, mi):
        success = False
        try:
            # Reading progress
            if self.key_sync_reading_progress and (self.shelf not in [3, 4]):
                reading_progress_url = 'https://www.skoob.com.br/estante/s_historico_leitura/' + str(skoob_id)
                read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']
                if read_percentage > 100:
                    read_percentage = 100
                    db.new_api.set_field(self.key_reading_progress_column, {book_id: read_percentage})

                # Set the read column according to reading progress
                if self.key_sync_read_books:
                    if len(self.key_read_books_column) > 0:
                        if read_percentage is None:
                            db.new_api.set_field(self.key_read_books_column, {book_id: ''})
                        else:
                            if int(read_percentage) == 100:
                                db.new_api.set_field(self.key_read_books_column, {book_id: True})
                            else:
                                db.new_api.set_field(self.key_read_books_column, {book_id: ''})

                # Check if book exists on shelves
                root = self.get_user_data(skoob_id)
                exists = root[2]
                if exists:
                    pass
                else:
                    return self.books_not_in_shelf.append(book_id)

                if read_percentage:

                    payload = {
                        'data[LendoHistorico][paginas]': read_percentage,
                        'data[LendoHistorico][tipo]': 1,
                    }
                    # Use urllib to encode the payload
                    data = urllib.urlencode(payload)

                    # Build our Request object (supplying 'data' makes it a POST)
                    reading_progress_req = urllib2.Request(reading_progress_url, data)

                    try:
                        reading_progress_open = urllib2.urlopen(reading_progress_req)
                        success = True

                    except urllib2.URLError:
                        error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                                     show_copy_button=False, show=True)
                        raise urllib2.URLError(_('No internet connection'))

            # Rating
            if self.key_sync_rating:
                if self.key_read_books_column == 'rating':
                    rating = mi.rating
                else:
                    rating = mi.metadata_for_field(self.key_rating_column)['#value#']

                if rating is None:
                    rating = 0
                else:
                    pass
                rating_url = 'https://www.skoob.com.br/v1/book_rate/' + str(skoob_id) + '/' + str(rating/2) + '/'

                # Build our Request object
                rating_req = urllib2.Request(rating_url)

                try:
                    rating_open = urllib2.urlopen(rating_req)
                    success = True

                except urllib2.URLError:
                    error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                                 show_copy_button=False, show=True)
                    raise urllib2.URLError(_('No internet connection'))

            # Review
            if self.key_sync_review:
                review_title = review_text = None
                # Check which column holds the review
                if self.key_review_column == 'comments':
                    calibre_review = mi.comments
                else:
                    calibre_review = mi.metadata_for_field(self.key_review_column)['#value#']

                if calibre_review:
                    review_root = fromstring(clean_ascii_chars(calibre_review))
                    existing_comments_review = review_root.xpath("id('skoob_review')")
                    if existing_comments_review:
                        review_title = review_root.xpath("id('skoob_title')")[0].text_content().strip()
                        review_text = review_root.xpath("id('skoob_review')")[0].text_content().strip()
                    else:
                        if self.key_review_column == 'comments':
                            pass
                        else:
                            review_title = None
                            review_text = calibre_review

                    if (not review_title) and (not review_text):
                        pass
                    else:
                        review_url = 'https://www.skoob.com.br/estante/s_resenha/' + str(skoob_id)

                        if review_title:
                            payload = {
                                'data[Resenha][titulo_resenha]': review_title,
                                'data[Resenha][resenha]': review_text
                            }
                        else:
                            payload = {
                                'data[Resenha][resenha]': review_text
                            }

                        # Use urllib to encode the payload
                        data = urllib.urlencode(payload)

                        # Build our Request object (supplying 'data' makes it a POST)
                        review_req = urllib2.Request(review_url, data)

                        try:
                            review_open = urllib2.urlopen(review_req)
                            success = True

                        except urllib2.URLError:
                            error_dialog(self.gui, _('No internet connection'),
                                         _('Seems like you have no internet access'), show_copy_button=False, show=True)
                            raise urllib2.URLError(_('No internet connection'))

            if success:
                self.books_updated.append(book_id)
            else:
                self.books_not_updated.append(book_id)

        except:
            return

    def add_to_skoob(self, skoob_id, book_id, mi):
        try:
            read_percentage = mi.metadata_for_field(self.key_reading_progress_column)['#value#']

            # Add to the according shelf
            if self.shelf is None:
                if read_percentage == 100:
                    # Read shelf
                    add_to_skoob_url = 'https://www.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/1/'
                else:
                    # Reading shelf
                    add_to_skoob_url = 'https://www.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/2/'
            else:
                add_to_skoob_url = 'https://www.skoob.com.br/v1/shelf_add/' + str(skoob_id) +\
                                   '/' + str(self.shelf) + '/'

            # Build our Request object
            add_to_skoob_req = urllib2.Request(add_to_skoob_url)

            try:
                # For some reason, Skoob only recognizes the second attempt to add the book...
                add_to_skoob_open = urllib2.urlopen(add_to_skoob_req)
                add_to_skoob_open = urllib2.urlopen(add_to_skoob_req)

                # Check if the operation succeeded
                add_response = add_to_skoob_open.read()
                if '"success":true' in add_response:
                    valid_id = True
                else:
                    valid_id = False
                    self.bad_skoob_id.append(book_id)
                    return valid_id, False

                # If checked, updates the book status on Skoob after adding to shelf
                if self.key_sync_update_added:
                    to_update = True
                    return valid_id, to_update
                else:
                    self.books_updated.append(book_id)
                    return valid_id, False

            except urllib2.URLError:
                self.books_not_updated(book_id)
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
                raise urllib2.URLError(_('No internet connection'))

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s could not be added to Skoob.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def remove_from_skoob(self, skoob_id, book_id, mi):
        try:
            # Remove from Skoob
            remove_from_skoob_url = 'https://www.skoob.com.br/v1/shelf_del/' + str(skoob_id) + '/'

            # Build our Request object
            remove_from_skoob_req = urllib2.Request(remove_from_skoob_url)

            try:
                remove_from_skoob_open = urllib2.urlopen(remove_from_skoob_req)

                # Check if the operation succeeded
                remove_response = remove_from_skoob_open.read()
                if '"success":true' in remove_response:
                    self.books_updated.append(book_id)
                    return True
                else:
                    self.bad_skoob_id.append(book_id)
                    return False

            except urllib2.URLError:
                self.books_not_updated(book_id)
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
                raise urllib2.URLError(_('No internet connection'))

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not removed'), _('%s could not be removed from Skoob.\n'
                                                              'Please verify its Skoob ID.') % title,
                                                                show_copy_button=False, show=True)

    def skoob_login(self):
        ''' Login to Skoob. Called once, during Calibre session. '''

        # Solution using urllib.
        # Found here: https://stackoverflow.com/questions/13925983/login-to-website-using-urllib2-python-2-7

        # Store the cookies and create an opener that will hold them.
        # Needed to make multiple requests, after logging in.

        try:
            print(_('Loading stored cookies failed, manually logging in...'))

            self.opener = urllib2.build_opener(urllib2.HTTPCookieProcessor(self.cj))

            # Install our opener (note that this changes the global opener to the one
            # we just made, but you can also just call opener.open() if you want)
            urllib2.install_opener(self.opener)

            # Authentication page
            authentication_url = 'https://www.skoob.com.br/login'

            # Credentials
            payload = {
                'data[Usuario][email]': self.key_user,
                'data[Usuario][senha]': self.key_password
            }

            # Use urllib to encode the payload
            data = urllib.urlencode(payload)

            # Build our Request object (supplying 'data' makes it a POST)
            login_req = urllib2.Request(authentication_url, data)

            login = urllib2.urlopen(login_req)

            # Save our cookie for future uses
            self.cj.save(self.cookie_path, ignore_discard=True, ignore_expires=True)

            # Test if credentials were accepted
            if login.geturl() == 'https://www.skoob.com.br/login/':
                return False
            else:
                return True

        except urllib2.URLError:
            error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                         show_copy_button=False, show=True)
            return None

    def get_details(self, skoob_id):
        '''Main function to fetch data from Skoob'''

        try:
            root = self.get_user_data(skoob_id)

            if len(root) == 0:
                return

            exists = root[2]
            if exists:
                pass
            else:
                return False

            try:
                if self.key_sync_reading_progress:
                    status = self.parse_status(root[0])
                else:
                    status = (None, None)
            except:
                status = (None, None)

            try:
                if self.key_sync_review:
                    review = self.parse_review(root[1])
                else:
                    review = (None, None)
            except:
                review = (None, None)

            try:
                if self.key_sync_rating:
                    rating = self.parse_rating(root[0])
                else:
                    rating = None
            except:
                rating = None

            try:
                if self.key_sync_page_count:
                    page_count = self.parse_page_count(root[0])
                else:
                    page_count = None
            except:
                page_count = None

        except:
            return error_dialog(self.gui, _('Error'), _('Error parsing book details'),
                                                        show_copy_button=False, show=True)

        return status, review, rating, page_count

    def get_user_data(self, skoob_id):
        root_rp = None
        root_p = None
        # Reading progress and review pages
        status_url = 'https://www.skoob.com.br/estante/s_historico_leitura/' + str(skoob_id)
        review_url = 'https://www.skoob.com.br/estante/s_resenha/' + str(skoob_id)

        # Build our Requests objects
        status_req = urllib2.Request(status_url)
        review_req = urllib2.Request(review_url)

        try:
            # Request status of the book
            if self.key_sync_reading_progress or self.key_sync_rating or self.key_sync_page_count:
                status_info = urllib2.urlopen(status_req)
                status = status_info.read()
                status_decode = status.decode('iso-8859-1', errors='replace')

                try:
                    root_rp = fromstring(clean_ascii_chars(status_decode))
                    exists = self.is_book_in_shelf(root_rp)
                    if exists:
                        pass
                    else:
                        return None, None, exists
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse reading progress'), show=True)

            # Request review of the book
            if self.key_sync_review:
                review_info = urllib2.urlopen(review_req)
                review = review_info.read()
                review_decode = review.decode('iso-8859-1', errors='replace')

                try:
                    root_p = fromstring(clean_ascii_chars(review_decode))
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse book review'),
                                 show_copy_button=False, show=True)

        except urllib2.URLError:
            error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                         show_copy_button=False, show=True)
            return

        return root_rp, root_p, exists

    def is_book_in_shelf(self, root):
        in_shelf_node = root.xpath("id('wd_referente')/div[1]/div[1]/h2[1]")
        if in_shelf_node:
            return True
        else:
            return False

    def parse_status(self, root):
        date_node = root.xpath("id('wd_referente')/div[1]/div[2]/div[1]/div[1]")
        if date_node:
            date = date_node[0].text_content().strip()
        else:
            date = None

        progress_node = root.xpath("id('wd_referente')/div[1]/div[2]/div[1]/div[2]/div[2]")
        if progress_node:
            progress_text = progress_node[0].text_content().strip()
            progress = re.search('(.+)(?=%)', progress_text).groups(0)[0]
        else:
            progress = None

        return date, progress

    def parse_review(self, root):
        review_title_node = root.xpath("//input[@name='data[Resenha][titulo_resenha]']/@value")
        if review_title_node:
            review_title = review_title_node[0]
        else:
            review_title = None

        review_text_node = root.xpath("//textarea[@name='data[Resenha][resenha]']")
        if review_text_node:
            review_text = review_text_node[0]
            aux_review_text = review_text.text_content().strip()
            if aux_review_text == '':
                review_text = None
        else:
            review_text = None

        return review_title, review_text

    def parse_rating(self, root):
        rating_node = root.xpath("//star-rating/@rate")
        if rating_node:
            rating = rating_node[0]
        else:
            rating = None

        return rating

    def parse_page_count(self, root):
        page_count_node = root.xpath("id('wd_referente')/div[1]/div[1]/text()")
        if page_count_node:
            page_count_text = page_count_node[1]
            page_count = re.search('(/ )(.\\d+)', page_count_text).groups(0)[1]
        else:
            page_count = None

        return page_count

    def are_selected_books_linked(self):
        all_has_skoob_id = True
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return False
        db = self.gui.library_view.model().db
        for row in rows:
            calibre_id = db.id(row.row())
            identifiers = db.get_identifiers(calibre_id, index_is_id=True)
            if 'skoob' not in identifiers:
                all_has_skoob_id = False
                break

        return all_has_skoob_id

    # Method 'any' reimplemented, since it is marked to change, according to FUTURE warning
    def any(self, iterable):
        for element in iterable:
            if element is not None:
                return True
        return False
