#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os

# PyQt libraries
from PyQt5.Qt import QWidget, QVBoxLayout, QLabel, QLineEdit, QGroupBox, QComboBox, QGridLayout, QCheckBox

# Calibre libraries
from calibre.utils.config import JSONConfig, config_dir

# This is where all preferences for this plugin will be stored
# Remember that this name (i.e. plugins/interface_demo) is also
# in a global namespace, so make it as unique as possible.
# You should always prefix your config file name with plugins/,
# so as to ensure you dont accidentally clobber a calibre config file
prefs = JSONConfig('plugins/Skoob_Sync')

# Set defaults
prefs.defaults['user'] = ''
prefs.defaults['password'] = ''
prefs.defaults['reading_progress_column_idx'] = 0
prefs.defaults['reading_progress_column'] = None
prefs.defaults['status_date_column_idx'] = 0
prefs.defaults['status_date_column'] = None
prefs.defaults['review_column_idx'] = 0
prefs.defaults['review_column'] = None
prefs.defaults['rating_column_idx'] = 0
prefs.defaults['rating_column'] = None
prefs.defaults['page_count_column_idx'] = 0
prefs.defaults['page_count_column'] = None
prefs.defaults['read_books_column_idx'] = 0
prefs.defaults['read_books_column'] = None
prefs.defaults['rating_column_idx'] = 0
prefs.defaults['rating_column'] = None
prefs.defaults['main_action_idx'] = 0
prefs.defaults['main_action'] = 'Download from Skoob'
prefs.defaults['themes_idx'] = 0
prefs.defaults['themes'] = None
prefs.defaults['sync_reading_progress'] = True
prefs.defaults['sync_review'] = True
prefs.defaults['sync_rating'] = True
prefs.defaults['sync_page_count'] = True
prefs.defaults['sync_read_books'] = True
prefs.defaults['sync_update_added'] = True

class ConfigWidget(QWidget):

    def __init__(self, gui, plugin_action):
        QWidget.__init__(self)
        self.gui = gui
        self.plugin_action = plugin_action
        self.layout = QGridLayout(self)
        self.setLayout(self.layout)

        # --- Credentials ---
        credentials_group_box = QGroupBox(_('Credentials:'), self)
        self.layout.addWidget(credentials_group_box, 0, 0, 1, 1)
        credentials_group_box_layout = QVBoxLayout()
        credentials_group_box.setLayout(credentials_group_box_layout)

        # User box
        self.user_label = QLabel()
        ico_src = os.path.join(config_dir, 'plugins', 'Skoob Sync', 'images', 'user.png')
        img_src = '<img src="' + ico_src + '"> '
        self.user_label.setText(img_src + _('&User'))
        self.layout.addWidget(self.user_label)
        credentials_group_box_layout.addWidget(self.user_label)
        self.user = QLineEdit(self)
        self.user.setMinimumWidth(260)  # 185
        self.user.setText(prefs['user'])
        self.layout.addWidget(self.user)
        self.user_label.setBuddy(self.user)
        credentials_group_box_layout.addWidget(self.user)

        # Password box
        self.password_label = QLabel()
        ico_src = os.path.join(config_dir, 'plugins', 'Skoob Sync', 'images', 'password.png')
        img_src = '<img src="' + ico_src + '"> '
        self.password_label.setText(img_src + _('&Password'))
        self.layout.addWidget(self.password_label)
        credentials_group_box_layout.addWidget(self.password_label)
        self.password = QLineEdit(self)
        self.password.setMinimumWidth(185)  # 185
        self.password.setEchoMode(QLineEdit.Password)
        self.password.setText(prefs['password'].decode('base64'))
        self.layout.addWidget(self.password)
        self.password_label.setBuddy(self.password)
        credentials_group_box_layout.addWidget(self.password)

        # --- Sync Options ---
        sync_options_group_box = QGroupBox(_('Sync Options:'), self)
        self.layout.addWidget(sync_options_group_box, 0, 1, 1, 2)
        sync_options_group_box_layout = QVBoxLayout()
        sync_options_group_box.setLayout(sync_options_group_box_layout)

        # Reading progress checkbox
        self.reading_progress_check = QCheckBox(_('&Reading progress'), self)
        self.reading_progress_check.setToolTip(_('Sync reading progress with Skoob'))
        sync_options_group_box_layout.addWidget(self.reading_progress_check)
        # Load the checkbox with the current preference setting
        self.reading_progress_check.setChecked(prefs['sync_reading_progress'])

        # Review checkbox
        self.review_check = QCheckBox(_('R&eview'), self)
        self.review_check.setToolTip(_('Sync review with Skoob'))
        sync_options_group_box_layout.addWidget(self.review_check)
        # Load the checkbox with the current preference setting
        self.review_check.setChecked(prefs['sync_review'])

        # Rating checkbox
        self.rating_check = QCheckBox(_('Rati&ng'), self)
        self.rating_check.setToolTip(_('Sync book rating with Skoob'))
        sync_options_group_box_layout.addWidget(self.rating_check)
        # Load the checkbox with the current preference setting
        self.rating_check.setChecked(prefs['sync_rating'])

        # Page count checkbox
        self.page_count_check = QCheckBox(_('Page &count'), self)
        self.page_count_check.setToolTip(_('Get page count from Skoob'))
        sync_options_group_box_layout.addWidget(self.page_count_check)
        # Load the checkbox with the current preference setting
        self.page_count_check.setChecked(prefs['sync_page_count'])

        # --- Custom Columns ---
        custom_columns_group_box = QGroupBox(_('Custom Columns:'), self)
        self.layout.addWidget(custom_columns_group_box, 1, 0, 1, 3)
        custom_columns_group_box_layout = QGridLayout()
        custom_columns_group_box.setLayout(custom_columns_group_box_layout)

        # Reading progress combobox
        self.reading_progress_box_label = QLabel(_('Rea&ding Progress:'), self)
        tooltip = _('Select a custom column to store the reading progress. Type: float/int.')
        self.reading_progress_box_label.setToolTip(tooltip)
        self.reading_progress_box = QComboBox()
        reading_progress_custom_columns = self.get_custom_columns(['float', 'int'])
        self._reading_progress_column_combo = CustomColumnComboBox(self, reading_progress_custom_columns,
                                                                   initial_items=[''])
        self._reading_progress_column_combo.setToolTip(tooltip)
        self._reading_progress_column_combo.setMinimumWidth(250)  # 120
        self.reading_progress_box_label.setBuddy(self._reading_progress_column_combo)
        custom_columns_group_box_layout.addWidget(self.reading_progress_box_label, 0, 0)
        custom_columns_group_box_layout.addWidget(self._reading_progress_column_combo, 0, 1)
        self._reading_progress_column_combo.setCurrentIndex(prefs['reading_progress_column_idx'])

        # Status date combobox
        self.status_date_box_label = QLabel(_('&Status date:'), self)
        tooltip = _('Select a custom column to store the reading progress date. Type: datetime.')
        self.status_date_box_label.setToolTip(tooltip)
        self.status_date_box = QComboBox()
        status_date_custom_columns = self.get_custom_columns(['datetime'])
        self._status_date_column_combo = CustomColumnComboBox(self, status_date_custom_columns,
                                                                   initial_items=[''])
        self._status_date_column_combo.setToolTip(tooltip)
        self._status_date_column_combo.setMinimumWidth(250)  # 120
        self.status_date_box_label.setBuddy(self._status_date_column_combo)
        custom_columns_group_box_layout.addWidget(self.status_date_box_label, 1, 0)
        custom_columns_group_box_layout.addWidget(self._status_date_column_combo, 1, 1)
        self._status_date_column_combo.setCurrentIndex(prefs['status_date_column_idx'])

        # Review combobox
        self.review_box_label = QLabel(_('Re&view:'), self)
        tooltip = _('Select a custom column to store the book review. Type: comments.')
        self.review_box_label.setToolTip(tooltip)
        self.review_box = QComboBox()
        review_custom_columns = self.get_custom_columns(['comments'])
        self._review_column_combo = CustomColumnComboBox(self, review_custom_columns,
                                                                   initial_items=['comments'])
        self._review_column_combo.setToolTip(tooltip)
        self._review_column_combo.setMinimumWidth(250)  # 120
        self.review_box_label.setBuddy(self._review_column_combo)
        custom_columns_group_box_layout.addWidget(self.review_box_label, 2, 0)
        custom_columns_group_box_layout.addWidget(self._review_column_combo, 2, 1)
        self._review_column_combo.setCurrentIndex(prefs['review_column_idx'])

        # Rating combobox
        self.rating_box_label = QLabel(_('Ra&ting:'), self)
        tooltip = _('Select a custom column to store the book rating. Type: rating.')
        self.rating_box_label.setToolTip(tooltip)
        self.rating_box = QComboBox()
        rating_custom_columns = self.get_custom_columns(['rating'])
        self._rating_column_combo = CustomColumnComboBox(self, rating_custom_columns,
                                                                   initial_items=['rating'])
        self._rating_column_combo.setToolTip(tooltip)
        self._rating_column_combo.setMinimumWidth(250)  # 120
        self.rating_box_label.setBuddy(self._rating_column_combo)
        custom_columns_group_box_layout.addWidget(self.rating_box_label, 3, 0)
        custom_columns_group_box_layout.addWidget(self._rating_column_combo, 3, 1)
        self._rating_column_combo.setCurrentIndex(prefs['rating_column_idx'])

        # Page count combobox
        self.page_count_box_label = QLabel(_('Pa&ge count:'), self)
        tooltip = _('Select a custom column to store the page count. Type: int.')
        self.page_count_box_label.setToolTip(tooltip)
        self.page_count_box = QComboBox()
        page_count_custom_columns = self.get_custom_columns(['int'])
        self._page_count_column_combo = CustomColumnComboBox(self, page_count_custom_columns,
                                                                   initial_items=[''])
        self._page_count_column_combo.setToolTip(tooltip)
        self._page_count_column_combo.setMinimumWidth(250)  # 120
        self.page_count_box_label.setBuddy(self._page_count_column_combo)
        custom_columns_group_box_layout.addWidget(self.page_count_box_label, 4, 0)
        custom_columns_group_box_layout.addWidget(self._page_count_column_combo, 4, 1)
        self._page_count_column_combo.setCurrentIndex(prefs['page_count_column_idx'])

        # Read books combobox
        self.read_books_box_label = QLabel(_('Read &books:'), self)
        tooltip = _('Select a custom column to store the read books (yes/no). Type: bool.')
        self.read_books_box_label.setToolTip(tooltip)
        self.read_books_box = QComboBox()
        read_books_custom_columns = self.get_custom_columns(['bool'])
        self._read_books_column_combo = CustomColumnComboBox(self, read_books_custom_columns,
                                                        initial_items=[''])
        self._read_books_column_combo.setToolTip(tooltip)
        self._read_books_column_combo.setMinimumWidth(250)  # 120
        self.read_books_box_label.setBuddy(self._read_books_column_combo)
        custom_columns_group_box_layout.addWidget(self.read_books_box_label, 5, 0)
        custom_columns_group_box_layout.addWidget(self._read_books_column_combo, 5, 1)
        self._read_books_column_combo.setCurrentIndex(prefs['read_books_column_idx'])

        # --- Customize interface ---
        customize_group_box = QGroupBox(_('Customize interface'))
        self.layout.addWidget(customize_group_box, 2, 0, 1, 3)
        customize_group_box_layout = QGridLayout()
        customize_group_box.setLayout(customize_group_box_layout)

        # Main action combobox
        self.main_action_box_label = QLabel(_('&Main button action:'), self)
        tooltip = _('Select which action will be used for the main button on calibre GUI. Must restart after changing.')
        self.main_action_box_label.setToolTip(tooltip)
        self.main_action_column_combo = QComboBox()
        self.main_action_column_combo.setToolTip(tooltip)
        self.main_action_column_combo.setMinimumWidth(120)
        self.main_action_column_combo.addItems({_('Download from Skoob'), _('Upload to Skoob')})
        self.main_action_column_combo.model().sort(0)
        self.main_action_box_label.setBuddy(self.main_action_column_combo)
        customize_group_box_layout.addWidget(self.main_action_box_label, 1, 0)
        customize_group_box_layout.addWidget(self.main_action_column_combo, 1, 1)
        self.main_action_column_combo.setCurrentIndex(prefs['main_action_idx'])

        # Themes combobox
        self.themes_box_label = QLabel(_('&Icons:'), self)
        tooltip = _('Change icons for: \'Main Button\' and \'List books with Skoob ID\'. Must restart after changing.')
        self.themes_box_label.setToolTip(tooltip)
        self.themes_column_combo = QComboBox()
        self.themes_column_combo.setToolTip(tooltip)
        self.themes_column_combo.setMinimumWidth(120)
        self.themes_column_combo.addItems({_('Blue'), _('Grey'), _('Grey and Yellow')})
        self.themes_column_combo.model().sort(0)
        self.themes_box_label.setBuddy(self.themes_column_combo)
        customize_group_box_layout.addWidget(self.themes_box_label, 2, 0)
        customize_group_box_layout.addWidget(self.themes_column_combo, 2, 1)
        self.themes_column_combo.setCurrentIndex(prefs['themes_idx'])

        # --- End options ---

        # Read books checkbox
        self.read_books_check = QCheckBox(_('&Automatically mark books as read'), self)
        self.read_books_check.setToolTip(_('Uses the \'read books\' column to mark books as finished'))
        self.layout.addWidget(self.read_books_check, 3, 0, 1, 3)
        # Load the checkbox with the current preference setting
        self.read_books_check.setChecked(prefs['sync_read_books'])
        # Enables / disables the read books check box
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        self.key_read_books_column = prefs['read_books_column']
        self.reading_progress_check.stateChanged.connect(self.refresh_widget)
        box_enabled = self.reading_progress_check.isChecked() and (self.key_read_books_column is not None)
        if box_enabled:
            self.read_books_check.setEnabled(box_enabled)
        else:
            self.read_books_check.setEnabled(box_enabled)

        # Update added checkbox
        self.update_added_check = QCheckBox(_('Update b&ooks added to Skoob'), self)
        self.update_added_check.setToolTip(_('Automatically update status of books added to Skoob'))
        self.layout.addWidget(self.update_added_check, 4, 0, 1, 3)
        # Load the checkbox with the current preference setting
        self.update_added_check.setChecked(prefs['sync_update_added'])

        self.resize(self.sizeHint())

    def refresh_widget(self):
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        box_enabled = self.reading_progress_check.isChecked()
        if box_enabled:
            self.read_books_check.setEnabled(box_enabled)
        else:
            self.read_books_check.setEnabled(box_enabled)
        self.update()

    def get_custom_columns(self, column_types):
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ in column_types:
                available_columns[key] = column
        return available_columns

    def save_settings(self):
        prefs['user'] = self.user.text()
        prefs['password'] = self.password.text().encode('base64')
        prefs['reading_progress_column_idx'] = self._reading_progress_column_combo.currentIndex()
        prefs['reading_progress_column'] = self._reading_progress_column_combo.get_selected_column()
        prefs['status_date_column_idx'] = self._status_date_column_combo.currentIndex()
        prefs['status_date_column'] = self._status_date_column_combo.get_selected_column()
        prefs['review_column_idx'] = self._review_column_combo.currentIndex()
        prefs['review_column'] = self._review_column_combo.get_selected_column()
        prefs['rating_column_idx'] = self._rating_column_combo.currentIndex()
        prefs['rating_column'] = self._rating_column_combo.get_selected_column()
        prefs['page_count_column_idx'] = self._page_count_column_combo.currentIndex()
        prefs['page_count_column'] = self._page_count_column_combo.get_selected_column()
        prefs['read_books_column_idx'] = self._read_books_column_combo.currentIndex()
        prefs['read_books_column'] = self._read_books_column_combo.get_selected_column()
        prefs['main_action_idx'] = self.main_action_column_combo.currentIndex()
        prefs['main_action'] = self.main_action_column_combo.currentText()
        prefs['themes_idx'] = self.themes_column_combo.currentIndex()
        prefs['themes'] = self.themes_column_combo.currentText()
        prefs['sync_reading_progress'] = self.reading_progress_check.isChecked()
        prefs['sync_review'] = self.review_check.isChecked()
        prefs['sync_rating'] = self.rating_check.isChecked()
        prefs['sync_page_count'] = self.page_count_check.isChecked()
        prefs['sync_read_books'] = self.read_books_check.isChecked()
        prefs['sync_update_added'] = self.update_added_check.isChecked()


class CustomColumnComboBox(QComboBox):

    def __init__(self, parent, custom_columns={}, selected_column='', initial_items=['']):
        QComboBox.__init__(self, parent)
        self.populate_combo(custom_columns, selected_column, initial_items)

    def populate_combo(self, custom_columns, selected_column, initial_items=['']):
        self.clear()
        self.column_names = list(initial_items)
        if len(initial_items) > 0:
            self.addItems(initial_items)
        selected_idx = 0
        for idx, value in enumerate(initial_items):
            if value == selected_column:
                selected_idx = idx
        for key in sorted(custom_columns.keys()):
            self.column_names.append(key)
            self.addItem('%s (%s)' % (key, custom_columns[key]['name']))
            if key == selected_column:
                selected_idx = len(self.column_names) - 1
        self.setCurrentIndex(selected_idx)

    def get_selected_column(self):
        return self.column_names[self.currentIndex()]
