package PAR::StrippedPARL::Dynamic;
use 5.006;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut

__DATA__

M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````Z`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!QTR5(-;)+
M&S6R2QLULDL;6JU!&SZR2QNVKD4;-[)+&UJM3QLWLDL;5ZU8&S&R2QM.KD<;
M-[)+&S6R2AM[LDL;RI)/&S:R2QL#E$$;-[)+&XVT31LTLDL;4FEC:#6R2QL`
M`````````%!%``!,`00`4,S?10``````````X``/`0L!!@``,````)``````
M``#L.0```!````!``````$```!`````0```$``````````0``````````-``
M```0```'QP$``P``````$```$``````0```0````````$```````````````
M=)0``&0`````L```T!$``````````````````````````````````"!!```<
M``````````````````````````````````````````````````````````!`
M```@`0``````````````````````````````````+G1E>'0````L*P```!``
M```P````$```````````````````(```8"YR9&%T80``-EH```!`````8```
M`$```````````````````$```$`N9&%T80```'`$````H````!````"@````
M``````````````!```#`+G)S<F,```#0$0```+`````@````L```````````
M````````0```P```````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`&A@04``:/XX0`!H1$%``.CD*```4.C8*```@\00PS/`PU6+[('L"`0``%-6
M5VB(04``Z*@!``")1?R-A?C[___'!"0`!```,]M04_\5$$!``(7`#X5,`0``
M_S40H$``BW4(5O\5A$!``%E9A<!6=!-H>$%``.CS`0``68O&6>E7`0``Z!XI
M``#_-12@0`"+/8Q`0`")1?C_=0S_UX/$#(OP._,/A!\!```X'G4%OG1!0``Y
M7?QT$U;_=?SHX"@``%F%P%D/A,L```!6Z-8H``"*5##_68L-$*!``#H1=0I(
MB!P&BE0&_^OLBU7X_W4(C400`3T`!```#XW-````45:-A?C[__]H;$%``%#_
M%91`0`#HX2<``%#H[2<``(LPZ-0G``!0Z-HG``"#Q!Q0C87X^___4.B^)P``
M4.C*)P``6?\P_U90@\0,A<!U2.BF)P``4.BL)P``9HM`!EEF)0#P9CT`@'4N
MZ(PG``!0Z)@G``"+,%F-A?C[__]J`5#H=2<``%#H@2<``%G_,/\6@\0,A<!T
M$/\U%*!``%/_UUE9Z?K^__^-A?C[__]0:'A!0`#HMP```(V%^/O__U#_%41`
M0`"#Q`SK$O]U"&AX04``Z)@```"+10A965]>6\G#58OLC44(4/]U".@$````
M65E=PU6+[%%35HMU"%>%]G1FBQV`0$``_].#.`!T68H.B\:$R70+@/D]=`:*
M2`%`Z_$KQHE%_/_3BS"+/H7_=#?_=?S_=0A7_Q5`0$``@\0,A<!U"8M%_(`\
M!SUT!8/&!.O8_],K,(M%#,'^`HDPBT7\C40'`>L",\!?7EO)PU6+[%%1BT4,
M@#@]=0/_10Q35E?_=0SH'B<``(M="(E%^(U%_%!3Z%K___^+-8!`0`"+^(/$
M#(7_=")7Z/DF```[1?A9#X++````BTT,B@&(!T=!A,!U]ND\`0``_]:+`#/_
M,\DY"'0&@\`$1^OV.0U0I$``=$#H$"8``%#H(B8``(L868T$O0@```!0_];_
M,.CV)0``4.@()@``6?\P_U,$@\0,B]C_UHD8_]:#.`!U5>FQ````QP50I$``
M`0```.C&)0``4.C8)0``BQA9C02]"````%#HL"4``%#HPB4``%G_,/\3B]A9
MA=M9='F+Q\'@`E#_UO\P4_\5?$!``(/$#/_6B1C_UHL`BUT(@V2X!`")??R*
M`XO[A,!T"CP]=`:*1P%'Z_+H7R4``%#H<24``(L0BT7X*_M9B54(C40'`E#H
M1"4``%#H5B4``%G_,(M%"/\068OX6?_6BP"+3?R%_XD\B'4%@\C_ZR[_UHL`
MBTW\BP2(B@M#A,F("'0(@/D]=`-`Z^^+50S&`#U`B@J("$!"A,EU]C/`7UY;
MR<.+1"0$5HOPB\B*`(3`=!>+%1"@0`"*$CK"=0.-<0&*00%!A,!U\8O&7L-6
MBW0D"(7V=&^`/@!T:E;H7"4``(U$,/]9BPT0H$``.\9V%HH1.!!U!4@[QG?W
M.\9V"3@0=`-(Z_4[QG4)B@`Z`70U4>LW2#O&=@:*$#H1=/4KQHU(`H'Y``0`
M`'8$,\!>PT!05KY(H$``5O\5>$!``(/$#(O&7L-H=$%``/\51$!``%E>PU6+
M[(/L)%-65VH(6;Y<0D``C7W<:$Q"0`#SI6:EZ`D!``#'!"1`0D``Z/T```"[
MB$%``%/H\@```+XT0D``5NCG````ORA"0`!7Z-P```!H'$)``.C2````:'A!
M0`#HR````&@00D``Z+X```!H!$)``.BT````:/!!0`#HHOS__X/$)(7`=`E0
M5^@B_?__65EHW$%``.B(_/__A<!9=`U0:-!!0`#H!OW__UE9:,!!0`#H;/S_
M_X7`670+4%/H[OS__UE9ZTYHK$%``.A2_/__A<!9=`105NOD:*1!0`#H!B0`
M`+^804``C5P%Z%?H]R,``%!74_\5=$!``(/$%(7`=1)7Z.$C```#PU!6Z)_\
M__^#Q`QHE$%``&A,0D``Z(W\__]965]>6\G#58OL48U%_%#_=0CH^_O__UF%
MP%ET'O\5@$!``(L`BTW\C0R(BU$$C4$$A=*)$734B\CK\,G#H1B@0`"#^/]U
M)6@T0D``Z*[[__^%P%ET#XH`A,!T"3PP=`5J`5CK`C/`HQB@0`##58OL@^P@
MN+A"0`!7B_A0B47@QT7DL$)``,=%Z*!"0`#'1>R80D``QT7PA$)``,=%](!"
M0`")??CH'",``(7`60^&Q````(U%X%-6B47\5^@^^___B_!9A?8/A(,```!6
MZ/0B``"%P%ET>/]U"%;_%81`0`!9A<!9=7/H+"(``%#H/B(``/\U%*!``(LX
MZ,<B``!6B]CHOR(``/]U"`/8Z+4B``"#Q!"-1`,"4.CZ(0``4.@,(@``6?\P
M_Q=6B_C_-12@0`#_=0AH;$%``%?_%91`0`!7_W7XZ$/[__^#Q"3K"_]U"%?H
M-?O__UE9@T7\!(M%_(LX5XE]^.A:(@``A<!9#X=&____7EM?R<-5B^RXX(``
M`.A)(@``@V7T`%-6NX!"0`!HB$%``,=%V$1#0`#'1=P\0T``QT7@=$%``(E=
MY,=%P-!!0`#'1<0T0T``QT7(+$-``,=%S"1#0`#'1=`@0T``B5W4QT7H&$-`
M`,=%[`Q#0`")7?#H%/K__XOP687V=!]6Z,XA``"%P%ET%%;H:O[__U;_%41`
M0`!96>ER!0``O@`$``!7B77XZ/D@``!0Z`LA``"+.%E6Z.H@``!0Z/P@``!9
M_S#_%UE9C4WX45#_%0!`0`#H\_?__X7`B47X=3Z-?>B+-U;H:2$``(7`678?
M5NB7^?__A<!9=`M0_Q5$0$``68E%^(/'!(-]^`!TU(-]^`!U"<=%^`1#0`#K
M1(MU^(`^`'0\OP<!``"A9$!``(,X`7X/#[X&5U#_%6A`0`!96>L1BPUL0$``
M#[X&BPEFBP1!(\>%P'4#Q@9?1H`^`'7)C47`B47\BT7\BS!6Z.$@``"%P%D/
MAJ````!6Z`OY__^+\%F%]@^$@0```.@4(```4.@@(```BSCH!R```%#H#2``
M`%E94%;H^!\``%#H!"```%G_,/]74(/$#(7`=4WHX!\``%#HYA\``&:+0`99
M9B4`\&8]`$!U,^C&'P``4.C2'P``BSA9:@)6Z+4?``!0Z,$?``!9_S#_%X/$
M#(7`=0M6_Q5$0$``68E%](-%_`2#??0`#X1,____@WWT``^%J0```(U%V(E%
M_(M%_(LP5N@=(```A<!9#X:/````Z&$?``!0Z&T?``"+..A4'P``4.A:'P``
M65E05NA%'P``4.A1'P``6?\P_U=0@\0,A<!U3>@M'P``4.@S'P``9HM`!EEF
M)0#P9CT`0'4SZ!,?``!0Z!\?``"+.%EJ`E;H`A\``%#H#A\``%G_,/\7@\0,
MA<!U"U;_%41`0`!9B47T@T7\!(-]]``/A%W_____=?CH?1\``/]U](OXZ',?
M``"^_$)```/X5NAF'P``4P/XZ%X?``"-O`<`!```Z*0>``!0Z+8>``"+$(/$
M%(E5_%?HD!X``%#HHAX``%G_,(M%_/\0_W7XB47\5HLUE$!``/\U$*!``/]U
M]&CP0D``4/_6_W7\_Q5P0$``:)A"0`#H./?__U#_%41`0`"+?0A0_S?H:/7_
M_X/$-(E%"(7`=06+!XE%".A0^___A<`/A2D"``#H'AX``%#H*AX``(LXQP0D
M`(```/]U".@''@``4.@3'@``6?\P_U<\@\0,B44(A<`/A/,!``#HZ!T``%#H
M]!T``%F+.&H":O]J[O]U".C1'0``4.C='0``6?\P_U<PZ,`=``!0Z,P=``"+
M.(/$&(V%('___VH&4/]U".BD'0``4.BP'0``6?\P_U=$@\00@+T@?___``^%
MM0```("](7___T,/A:@```"`O2)___]!#X6;````@+TC?___0P^%C@```("]
M)'___T@/A8$```"`O25___]%=7CH1AT``%#H4AT``%F+.&H":O]JQO]U".@O
M'0``4.@['0``6?\P_U<PZ!X=``!0Z"H=``"+.(/$&(V%('___VHI4/]U".@"
M'0``4.@.'0``6?\P_U=$C84@?___4U#_-1"@0`#_=?QHX$)``/]U_/_6@\0H
MZ2(!``"-A2#___]0Z#8!``#HPAP``%#HSAP``%E9BSB-A2!___]H`(```%#_
M=0CHI!P``%#HL!P``%G_,/]71(/$$(7`?B10C84@?___4(V%(/___U#H'@$`
M`.AX'```4.B$'```@\00Z[/H:!P``%#H=!P``(LX6?]U".A7'```4.AC'```
M6?\P_U<0C84@____4(U%K%#HNQ,``(U%@(/$$#/_B44(#[9$/:Q0:-A"0`#_
M=0C_UH-%"`*#Q`Q'@_\4?..-18!3@&6H`%#_-1"@0`#_=?QHX$)``.M!:)1!
M0`!H-$)``.A@]?__65E3Z.,;``!0Z/L;``"+..C6&P``4.CN&P``_S#_5VB#
MQ`Q0_S40H$``_W7\:,A"0`#_=?S_UH/$&/]U_&B(04``Z!GU____=?SH\?C_
M_XM%_(/$#%]>6\G#BT0D!#/)QP`!(T5GQT`$B:O-[\=`"/[<NIC'0`QV5#(0
MQT`0\.'2PXE(%(E(&(E(7,-5B^Q3BUT05HMU"%>+3A2-!-D[P7,#_T88B484
MB\/!Z!T!1AB+1ER%P'0T:D!?*_@[^WX"B_N+1EQ7_W4,C40&'%#H`!P```%]
M#`%^7(/$#"O?@WY<0'5+5NA*````68/[0'PLB_O![P:+Q_?8P>`&`]AJ0(U&
M'/]U#%#HQQL``(-%#$!6Z!X```"#Q!!/=>)3C48<_W4,4.BJ&P``@\0,B5Y<
M7UY;7<-5B^R![%0!``"+10A35E>-4!R-M:S^___'10@0````BPJ#P@2+^8O9
M@><``/\`P>L0"_N+V8'C`/\``,'A$`O9P>\(P>,("_N)/H/&!/]-"'7-:D"-
ME;3^__]>BTHL,TH8,TKX,PJ#P@2+^0/)P>\?"_E.B7HT=>.+>`2+4`B+<`R+
M"(O?(]?WTR/>B_$+VHO1P>H;P>8%"]:+<!`#\P/6B[6L_O__C90RF7F"6HOW
MP>8>P>\""_>+^??7(W@(B77L(_&+V@O^B_+!ZQO!Y@4+WHNUL/[__P/W`]Z+
M<`R-O#.9>8):B_&+V8E]^,'F'L'K`@OSB]KWTR-=[(EU_"/R"]Z+M;3^__^)
M70B+W\'K&\'G!0O?BWT(`_>+^L'G'L'J`@/>BW`("_J+5?B-M#.9>8):BUWX
M]](C5?PCWXE]](O^"].+WL'O&\'C!0O[BYVX_O__`]H#^XM5[(V<%YEY@EJ+
M??B+UXE=",'B'L'O`@O7B_[WUR-]](E5^"/6"_J+T\'J&\'C!0O3BYV\_O__
M`]^+??P#TXV4.IEY@EJ+_L'G'L'N`@O^BW4(]]8C=?B)??`C?0B+VL'C!0OW
MB_K![QL+^XN=P/[__P/>BW7T`_N-G#>9>8):BWT(B_>)7?S!YA[![P(+]XOZ
M]]<C??")=0@C\@O^B_/![AO!XP4+\XN=Q/[__P/?BWWX`_.-M#Z9>8):B_K!
MYQ[!Z@(+^HM5_/?2(U4(B7WL(WW\B][!XP4+UXO^P>\;"_N+G<C^__\#VHM5
M\`/[C9P7F7F"6HM]_(O7B5WXP>(>P>\""]>+_O?7(WWLB57\(]8+^HO3P>H;
MP>,%"].+G<S^__\#WXM]"`/3BUWXC90ZF7F"6HO^P><>P>X""_Z+=?CWUB-U
M_"/?B7WTB_H+\XO:P>\;P>,%"_N+G=#^__\#WHMU[`/[C9PWF7F"6HM]^(OW
MB5T(P>8>P>\""_>+^O?7(WWTB77X(_(+_HOSP>X;P>,%"_.+G=3^__\#WXM]
M_`/SC;0^F7F"6HOZP><>P>H""_J+50CWTB-5^(E]\"-]"(O>P>,%"]>+_L'O
M&PO[BYW8_O__`]J+5?0#^XV<%YEY@EJ+?0B+UXE=_,'B'L'O`@O7B_[WUR-]
M\(E5""/6"_J+T\'J&\'C!0O3BYW<_O__`]^+??@#TXV4.IEY@EJ+_L'G'L'N
M`@O^BW7\]]8C=0B)?>PC??R+VL'C!0OWB_K![QL+^XN=X/[__P/>BW7P`_N-
MG#>9>8):BWW\B_>)7?C!YA[![P(+]XOZ]]<C?>R)=?PC\@O^B_/![AO!XP4+
M\XN=Y/[__P/?BWT(`_.+7?B-M#Z9>8):B_K!YQ[!Z@(+^HM5^/?2(U7\B7WT
M(]^+_@O3B][![QO!XP4+^XN=Z/[__P/:BU7L`_N-G!>9>8):BWWXB]>)70C!
MXA[![P(+UXO^]]<C??2)5?@CU@OZB]/!ZAO!XP4+TXN=[/[__P/?BWW\`].-
ME#J9>8):B_[!YQ[![@(+_HMU"/?6(W7XB7WP(WT(B]K!XP4+]XOZP>\;"_N+
MG?#^__\#WHMU]`/[C9PWF7F"6HM]"(OWB5W\P>8>P>\""_>+^O?7(WWPB74(
M(_(+_HOSP>X;P>,%"_.+G?3^__\#WXM]^`/SC;0^F7F"6HOZP><>P>H""_J+
M5?SWTB-5"(E]["-]_`O7B_[![QN+WL'C!0O[BYWX_O__`]J+5?`#^XV<%YEY
M@EJ+??R+UXE=^,'B'L'O`@O7B_O![QO!XP4+^XM=[(E5_#/:BY7\_O__,]X#
MUXO^`]J+50C!YQ[![@(+_HV4$Z'KV6Z)??2+\HOZBYT`____P>X;P><%"_>+
M??@S??P#WHMU[#-]]`/[C9PWH>O9;HM]^(OWB5T(P>8>P>\""_>+^\'O&\'C
M!8EU^`O[BYT$____,_(S=?0#WXM]_`/SC;0^H>O9;HOZP><>P>H""_J+UHO>
MB7WPP>H;P>,%"].+7?@SWXN]"/___S-="`/ZBU7T`]^-G!.AZ]ENBWT(B5W\
MB]?!XA[![P(+UXO[P>\;P>,%"_N+7?")50@SVHN5#/___S/>`]>+_@/:BU7X
MP><>P>X"C903H>O9;@O^B_*+VL'N&\'C!0OSBUT(B7WL,]^+O1#___\S7?P#
M_HMU\`/?BWW\C9PSH>O9;HOWP>8>P>\""_>+^XE=^(EU_,'O&\'C!0O[BUWL
M,]Z+M13___\SV@/WB_H#WHMU",'G'L'J`@OZC;0SH>O9;HE]](O6B_Z+G1C_
M___!ZAO!YP4+UXM]^#-]_`/:BU7L,WWT`_N-G!>AZ]ENBWWXB]>)70C!XA[!
M[P(+UXE5^(O[,]8S5?3![QO!XP4+^XN='/___P/?BWW\`].-E#JAZ]ENB_[!
MYQ[![@(+_HORB]J)??#![AO!XP4+\XM=^#/?B[T@____,UT(`_Z+=?0#WXM]
M"(V<,Z'KV6Z+]\'F'L'O`@OWB_N)7?R)=0C![QO!XP4+^XM=\#/>B[4D____
M,]H#]P/>BW7XB_K!YQ[!Z@*-M#.AZ]EN"_J+UHO>P>H;P>,%"].+70B)?>PS
MWXN]*/___S-=_`/ZBU7P`]^+??R-G!.AZ]ENB]?!XA[![P(+UXO[B5WXB57\
MP>\;P>,%"_N+7>PSVHN5+/___S/>`]>+_@/:BU4(P><>P>X""_Z-E!.AZ]EN
MB7WTB_*+^HN=,/___\'N&\'G!0OWBWWX,WW\`]Z+=>PS??0#^XV<-Z'KV6Z+
M??B+]XE=",'F'L'O`@OWB_O![QO!XP6)=?@+^XN=-/___S/R,W7T`]^+??P#
M\XVT/J'KV6Z+^L'G'L'J`@OZB]:+WHE]\,'J&\'C!0O3BUWX,]^+O3C___\S
M70@#^HM5]`/?BWT(C9P3H>O9;HO7P>(>P>\""]>+^XE=_(E5",'O&\'C!0O[
MBUWP,]J+E3S___\SW@/7`]J+5?B-O!.AZ]ENB];!XA[![@(+UHOWB]_![AO!
MXP4+\XM="(E5[#/:BY5`____,UW\`]:+=?P#VHM5\(V<$Z'KV6Z+UL'B'L'N
M`@O6B_.)7?B)5?S![AO!XP4+\XM=[#/:BY5$____,]\#UHOW`]J+50C!YA[!
M[P(+]XV4$Z'KV6Z)=?2+\HOZBYU(____P>X;P><%"_>+??@S??P#WHMU[#-]
M]`/[C;0WH>O9;HM]^(EU"(OWP>8>P>\""_>+^@M]](M="(EU^"/^B_(C=?3!
MXP4+_HMU",'N&POSBYU,____`]^+??P#\XVT/MR\&X^+^L'G'L'J`@OZB]>)
M??`+50@C5?@C?0B+W@O7B_[![QO!XP4+^XN=4/___P/:BU7T`_N-G!?<O!N/
MBWT(B]>)7?S!XA[![P(+UXOZB54("_XCUB-]\`OZB]/!ZAO!XP4+TXN=5/__
M_P/?BWWX`].-E#K<O!N/B_[!YQ[![@(+_HO:B_>)?>P+=?PC??S!XP4C=0@+
M]XOZP>\;"_N+G5C___\#WHMU\`/[C9PWW+P;CXM]_(OWB5WXP>8>P>\""_>+
M_HEU_`OZ(_(C?>P+_HOSP>X;P>,%"_.+G5S___\#WXM]"`/SC;0^W+P;CXOZ
MP><>P>H""_J+5?R)??2+7?P+UR-5^"/?B_X+TXO>P>\;P>,%"_N+G6#___\#
MVHM5[`/[C907W+P;CXM]^(E5"(O7P>(>P>\""]>+_@M]](M="(E5^"/ZB]8C
M5?3!XP4+^HM5",'J&PO3BYUD____`]^+??P#TXV4.MR\&X^+_L'G'L'N`@O^
MB]J+]XE]\`MU""-]",'C!2-U^`OWB_K![QL+^XN=:/___P/>BW7T`_N-G#?<
MO!N/BWT(B_>)7?S!YA[![P(+]XO^B74("_HC\B-]\`O^B_/![AO!XP4+\XN=
M;/___P/?BWWX`_.-M#[<O!N/B_K!YQ[!Z@(+^HO>B]>)?>P+5?PC??S!XP4C
M50@+UXO^P>\;"_N+G7#___\#VHM5\`/[C9P7W+P;CXM]_(O7B5WXP>(>P>\"
M"]>+^HE5_`O^(]8C?>P+^HO3P>H;P>,%"].+G73___\#WXM]"`/3BUW\C90Z
MW+P;CXO^P><>P>X""_Z+=?P+]R/?(W7XB7WTB_H+\XO:P>\;P>,%"_N+G7C_
M__\#WHMU[`/[C;0WW+P;CXM]^(EU"(OWP>8>P>\""_>+^@M]](M="(EU^"/^
MB_(C=?3!XP4+_HMU",'N&POSBYU\____`]\#\XM]_(VT/MR\&X^+^L'G'L'J
M`@OZB]Z+UXE]\`M5""-]",'C!2-5^`O7B_[![QL+^XM=@`/:BU7T`_N-G!?<
MO!N/BWT(B]>)7?S!XA[![P(+UXOZB54("_XCUB-]\`OZB]/!ZAO!XP4+TXM=
MA`/?BWWX`].-E#K<O!N/B_[!YQ[![@(+_HO:B_>)?>P+=?PC??S!XP4C=0@+
M]XOZP>\;"_N+78@#WHMU\`/[C9PWW+P;CXM]_(OWB5WXP>8>P>\""_>+_HEU
M_`OZ(_(C?>P+_HOSP>X;P>,%"_.+78P#WXM]"`/SBUW\C;0^W+P;CXOZP><>
MP>H""_J+5?P+UR/?(U7XB7WTB_X+TXO>P>\;P>,%"_N+79`#VHM5[`/[C907
MW+P;CXM]^(E5"(O7P>(>P>\""]>+_@M]](M="(E5^"/ZB]8C5?3!XP4+^HM5
M",'J&PO3BUV4`]^+??P#TXV4.MR\&X^+_L'G'L'N`@O^B]J+]XE]\`MU""-]
M",'C!2-U^`OWB_K![QL+^XM=F`/>BW7T`_N-G#?<O!N/BWT(B_>)7?S!YA[!
M[P(+]XO[P>\;P>,%"_N+7?`SWHEU"#/:BW6<`_>+^@/>BW7XP><>P>H"C;0S
MUL%BR@OZB]:+WL'J&\'C!0O3BUT(,]^)?>R+?:`S7?P#^HM5\`/?BWW\C9P3
MUL%BRHO7P>(>P>\""]>+^XE=^(E5_,'O&\'C!0O[BUWL,]J+5:0SW@/7B_X#
MVHM5",'G'L'N`@O^C903UL%BRHE]](ORB_J+7:C![AO!YP4+]XM]^#-]_`/>
MBW7L,WWT`_N-G#?6P6+*BWWXB_>)70C!YA[![P(+]XO[P>\;P>,%B77X"_N+
M7:PS\C-U]`/?BWW\`_.-M#[6P6+*B_K!YQ[!Z@(+^HO6B]Z)??#!ZAO!XP4+
MTXM=^#/?BWVP,UT(`_J+5?0#WXM]"(V<$];!8LJ+U\'B'L'O`@O7B_N)7?R)
M50C![QO!XP4+^XM=\#/:BU6T,]X#UP/:BU7XB_[!YQ[![@*-E!/6P6+*"_Z+
M\HO:P>X;P>,%"_.+70B)?>PSWXM]N#-=_`/^BW7P`]^+??R-G#/6P6+*B_?!
MYA[![P(+]XO[B5WXB77\P>\;P>,%"_N+7>PSWHMUO#/:`_>+^@/>BW4(P><>
MP>H""_J-M#/6P6+*B7WTB]:+_L'J&\'G!0O7BWWXBUW`,WW\`]J+5>PS??0#
M^XV<%];!8LJ+??B+UXE=",'B'L'O`@O7B_O![QO!XP6)5?@+^XM=Q#/6,U7T
M`]^+??P#TXV4.M;!8LJ+_L'G'L'N`@O^B_*+VHE]\,'N&\'C!0OSBUWX,]^+
M?<@S70@#_HMU]`/?BWT(C9PSUL%BRHOWP>8>P>\""_>+^XE=_(EU",'O&\'C
M!0O[BUWP,]Z+=<PSV@/W`]Z+=?B+^L'G'L'J`HVT,];!8LH+^HO6B][!ZAO!
MXP4+TXM="(E][#/?BWW0,UW\`_J+5?`#WXV<$];!8LJ+??R)7?B+U\'B'L'O
M`@O7B_O![QO!XP4+^XM=[(E5_#/:BU74,]X#UXO^`]J+50C!YQ[![@(+_HV4
M$];!8LJ)??2+\HOZBUW8P>X;P><%"_>+??@S??P#WHMU[#-]]`/[C9PWUL%B
MRHM]^(OWB5T(P>8>P>\""_>+^\'O&\'C!8EU^`O[BUW<,_(S=?0#WXM]_`/S
MC;0^UL%BRHOZP><>P>H""_J+UHO>B7WPP>H;P>,%"].+7?@SWXM]X#-="`/Z
MBU7T`]^+?0B-G!/6P6+*B]?!XA[![P(+UXE=_(E5"(O[P>\;P>,%"_N+7?`S
MVHM5Y#/>`]<#VHM5^(O^P><>P>X"C903UL%BRHM="`O^B_*)5?0SW\'N&\'B
M!0ORBU7\,]H#S@/9BTWP`UWHBW7TC8P+UL%BRHD(BT@$`\Z)2`2+RL'A'L'J
M`@O*BU4(`T@(B4@(BT@,`\]?B4@,BT@0`\I>B4@06\G#58OL4U:+=0R+1AB+
M7A2)10R+P\'H`X/@/\9$,!R`0(/X.'XJ:D!9*\B-1#`<46H`4.@X"0``5NB-
M[?__:CB-1AQJ`%#H)0D``(/$'.L5:CA9*\B-1#`<46H`4.@."0``@\0,BT4,
M5HO(B$97P>D8B$Y4B\C!Z1"(3E6+R(O#_W4(P>@8B$98B\/!Z!"(1EF+P\'I
M",'H"(A.5HA&6HA>6^@&````65E>6UW#5HMT)`Q6Z!/M__^+1"0,68L.P>D8
MB`B+#L'I$(A(`8L.P>D(B$@"B@Z(2`.+5@2-3@3!ZAB(4`2+$<'J$(A0!8L1
MP>H(B%`&B@F(2`>+5@B-3@C!ZAB(4`B+$<'J$(A0"8L1P>H(B%`*B@F(2`N+
M5@R-3@S!ZAB(4`R+$<'J$(A0#8L1P>H(B%`.B@F(2`^+5A"-3A!>P>H8B%`0
MBQ'!ZA"(4!&+$<'J"(A0$HH)B$@3PX'L)`$``%.+G"0L`0``5E>%VP^$,0(`
M`%/HLP<``(7`60^$(@(``%/HI`<``(OP@<8"`0``Z.D&``!0Z/L&``"+.%E9
M5NC9!@``4.CK!@``6?\P_Q>+/91`0`"+\%-H8$-``%;_UXU$)"Q05O\52$!`
M`(/$'(/X_XE$)!0/A,<!``!5BRV$0$``C40D,&A`04``4/_5687`68U$)#!0
M=0Y3:%A#0`!6_]>#Q!#K"VA40T``5O_7@\0,NU!#0`!35O_5B_A9A?]9=!Y3
M5^CZ!@``687`670L:$Q#0`!7Z.D&``!9A<!9=!OV1"0<$'1<@[PD/`$```!T
M4FH!5NC__O__65F-1"0<4/]T)!S_%4Q`0`!9A<!9#X43`0``C40D,&A`04``
M4/_5687`68U$)#!0=5O_M"0\`0``:%A#0`!6_Q640$``@\00ZU*-1"0P4/\5
M"$!``(-D)!``BSU00$``B40D%%;_UX7`6726BT0D$/]$)!"#^`I]B8-\)!0`
M=`K_="04_Q4,0$``5NO8:%1#0`!6_Q640$``@\0,4U;_U8OX687_670F4U?H
M&P8``%F%P%D/A$G___]H3$-``%?H!@8``%F%P%D/A#3____V1"0<$'0.@[PD
M/`$````/A17___^-1"0P4/\5"$!``(-D)!``BSU00$``B40D%%;_UX7`60^$
M^?[__XM$)!#_1"00@_@*#XWH_O__@WPD%`!T"O]T)!3_%0Q`0`!6Z]#_="08
M_Q540$``_[0D/`$``/\5H$!``%E975]>6X'$)`$``,-35HMT)`Q75N@!X/__
MB_A7Z,_?__]9B]A9Z-?A__^%P'0TA?9T,%;H4@4``(7`670E:/Q"0`!3_Q6$
M0$``63O#6743:@%6Z&+]__]J`%?H6OW__X/$$%]>6\-1@V0D``!35E?H:`0`
M`%#HO`0``&H!6X@8QP0D3$)``.@XW?__A<!9=`N`.#%U!H!X`0!T!>@`X/__
M:@I3:,Q#0`#H+P0``%#H?00``(/$$(7`#X2=````BP`Y'4BD0`!5B_.+0""+
M>!!^7KT```0`C4;_:@!05^CZ`P``4.A"!```@\00A<!T0(L`BT@((\T[S74&
MBP"+`.L.4.C6`P``4.@8!```65E0H4RD0`#_-+#H:`0``%F%P%EU#?]$)!!&
M.S5(I$``?*>A2*1``%V-2/\Y3"0,=1T[PXOS?A=7Z)(#``!0Z,X#``!&63LU
M2*1``%E\Z6H$4VC(0T``Z'0#``!0Z,(#``"+\(/$$(7V=#6+!FB\0T``BT`@
M_S#H5`,``%#HB@,``(L&@\0,BT`@BP#V0`E`=`Y0Z#@#``!0Z&@#``!966H$
M4VBX0T``Z",#``!0Z'$#``"+\(/$$(7V=')H>$%``.CQV___A<!9=`-0ZTOH
M(]K__XOXA?]T.8L&5XM`(/\PZ.H"``!0Z"`#``#HWP(``%#H\0(``(LP@\00
M5^C.`@``4.C@`@``6?\P_U8(65GK'*%,I$``_S"+!HM`(/\PZ*L"``!0Z.$"
M``"#Q`SHG0(``%#H\0(``(`@`.B/`@``4.BY`@``@R``:(A!0`#H8]O__X/$
M#*-$H$``A<!U$?\U3*1``.C'X/__6:-$H$``Z%H"``!0Z'X"``"+,,<$).T!
M``#_-42@0`#H0`(``%#H9`(``%G_,/\6@\0,A<!?7EMT)8/X$70@@_C_=!M0
MH4RD0`#_-42@0`#_,&AH0T``Z"T"``"#Q!!9PU6+[(U%#%90C44(4/\5J$!`
M`&H`Z($"``"#Q`R`.`!U3.AN`@``A<"C0*!``'4CZ-`!``!0Z.@!``"+,%EJ
M`>C``0``4.C8`0``6?\P_U8(65G_-4"@0`#H,0(``.BB`0``4.@@`@``@R``
M65GHD@$``%#H"@(``(`(`NB$`0``4.C\`0``@`@!BT4(@\`%63W___\_678+
M:-Q#0`#HA@$``%F+10B-!(44````4.C)`0``6:-,I$``BTT,:@&+"8D(H4RD
M0`#'0`380T``H4RD0`#'0`@(1$``H4RD0`#'!4BD0``#````QT`,U$-``/\%
M2*1``%@Y10A^'HL-2*1``(M5#(LU3*1```/(BQ2"0(E4COP[10A\XJ%(I$``
MBTT(`\&+#4RD0`!J`(-D@?P`BT4(BPU(I$``_S5,I$``C40!_U!HMSA``/\U
M0*!``.@A`0``B_"#Q!2%]G4._S5`H$``Z`<!``!9B_#_-4"@0`#H\P```,<$
M)$!"0`#H:-G__X7`674DH42@0`"%P'44:(A!0`#H4-G__X7`6:-$H$``=`=0
MZ(G[__]9_S5`H$``Z*T```!9_Q740$``B\9>7</H/````%#H/M?__UFX@$)`
M`%!0:((T0`!H\$-``.@?````4.@3````@\004.@0````4.A&````@\0,P_\E
M&$%``/\E%$%``/\E$$%``/\E#$%``/\E"$%``/\E!$%``/\E`$%``/\E_$!`
M`/\E^$!``/\E]$!``/\E\$!``/\E[$!``/\EZ$!``/\EY$!``/\EX$!``/\E
MW$!``/\EV$!``/\ET$!``/\ES$!``/\ER$!``/\EQ$!``/\EP$!``/\EO$!`
M`/\EN$!``/\EM$!``/\EL$!``/\EK$!``,S,S,S,S,S,S,S,S/\ED$!``/\E
MB$!``,S,S,Q1/0`0``"-3"0(<A2!Z0`0```M`!```(4!/0`0``!S["O(B\2%
M`8OABPB+0`10P\S_)6!`0`#_)5Q`0`!5B^QJ_VAHE$``:"`[0`!DH0````!0
M9(DE`````(/L(%-65XEEZ(-E_`!J`?\56$!``%F##62D0`#_@PUHI$``__\5
M&$!``(L-8*1``(D(_Q4<0$``BPU<I$``B0BA($!``(L`HVRD0`#HP````(,]
M,*!```!U#&@>$$``_Q4D0$``6>B4````:`R@0`!H"*!``.A_````H5BD0`")
M1=B-1=A0_S54I$``C47@4(U%U%"-1>10_Q4L0$``:`2@0`!H`*!``.A,````
M_Q4P0$``BTW@B0C_=>#_==3_=>3H)?S__X/$,(E%W%#_%31`0`"+1>R+"(L)
MB4W04%'H#P```%E9PXMEZ/]UT/\5/$!``/\E.$!``/\E*$!``&@```,`:```
M`0#H#P```%E9P\/,S,S,S/\EF$!``/\EG$!`````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````%:8````````-)@``":8```0F````````/29``#DF0``U)D``,"9``"T
MF0``I)D``)29``",F0``?ID``':9``!>F0``5)D``$:9```ZF0``,)D``"*9
M```"F@``#ID```29``#TF```Z)@``-Z8``#4F```RI@``,"8``"VF```II@`
M`)R8``"2F```B)@``'Z8``!TF```%)H``"B:```8F0```````/*7``#<EP``
MSI<``+R7``"<EP``A)<``&Z7``!@EP``5)<``$27```XEP``)I<``!*7````
MEP``\)8``-R6``#,E@``O)8``*Z6``":E@``BI8``'*6``!@E@``4)8``$"6
M```LE@``&I8```:6``#XE0````````````!0S-]%``````(````W````````
M``#0```[````7````$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!P97)L
M>'-I+F,````E<R5S)7,``"X```!005)?4%)/1TY!344`````4$%27U1%35``
M````,0```%!!4E]#3$5!3CT``$-,14%.````4$%27T=,3T)!3%]#3$5!3@``
M``!005)?1TQ/0D%,7U1%35``4$%27U1-4$1)4@``4$%27T=,3T)!3%]435!$
M25(```!005)?1TQ/0D%,7T1%0E5'`````%!!4E]!4D=67S```%!!4E]!4D=#
M`````%!!4E]#04-(10```%!!4E]$14)51P```%!!4E]#3$5!3@```%!!4E]3
M4$%73D5$`%!!4E])3DE424%,25I%1`!?7T5.5E]005)?0TQ%04Y?7R`@("`@
M("`@("`@("`@(```````````1%E,1%],24)205)97U!!5$@```!0051(````
M`$Q)0E)!4EE?4$%42`````!,24)0051(`$Q$7TQ)0E)!4EE?4$%42``E<R5S
M=&5M<"TE=25S````)3`R>``````E<R5S8V%C:&4M)7,E<P``)7,E<R5S)7,`
M````<&%R+0````!365-414T``%5315).04U%`````%5315(`````5$U0`%1%
M35``````5$5-4$1)4@!435!$25(``"]T;7``````0SI<5$5-4`!<+BX`7"X`
M`"5S```E<UPE<P```"5S7"HN*@``)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@
M=&5M<&]R87)Y('-U8F1I<F5C=&]R>2`E<R!F86EL960@+2!A8F]R=&EN9R!W
M:71H("5I+@H````P````<&5R;"YE>&4`````&````$%21U8`````+2T``"UE
M``!P86YI8SH@;65M;W)Y('=R87```$EN=&5R;F%L<SHZ4$%2.CI"3T]4````
M`",A+W5S<B]B:6XO<&5R;`H*979A;"`G97AE8R`O=7-R+V)I;B]P97)L("`M
M4R`D,"`D>S$K(B1`(GTG"B`@("!I9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R
M('-O;64@<VAE;&P*"G!A8VMA9V4@7U]P87)?<&P["@HC("TM+2!4:&ES('-C
M<FEP="!M=7-T(&YO="!U<V4@86YY(&UO9'5L97,@870@8V]M<&EL92!T:6UE
M("TM+0HC('5S92!S=')I8W0["@HC;&EN92`Q-3D*"FUY("@D<&%R7W1E;7`L
M("1P<F]G;F%M92P@0'1M<&9I;&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%
M04Y]*2!["B`@("!U;FQI;FL@0'1M<&9I;&4["B`@("!R;61I<B`D<&%R7W1E
M;7`["B`@("`D<&%R7W1E;7`@/7X@<WM;7EQ<+UTJ6UQ<+UT_)'U[?3L*("`@
M(')M9&ER("1P87)?=&5M<#L*?2!]"@I"14=)3B!["B`@("!);G1E<FYA;',Z
M.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4
M.PH*("`@(&5V86P@>PH*7W!A<E]I;FET7V5N=B@I.PH*:68@*&5X:7-T<R`D
M14Y6>U!!4E]!4D=67S!](&%N9"`D14Y6>U!!4E]!4D=67S!]("D@>PH@("`@
M0$%21U8@/2!M87`@)$5.5GLB4$%27T%21U9?)%\B?2P@*#$@+BX@)$5.5GM0
M05)?05)'0WT@+2`Q*3L*("`@("0P(#T@)$5.5GM005)?05)'5E\P?3L*?0IE
M;'-E('L*("`@(&9O<B`H:V5Y<R`E14Y6*2!["B`@("`@("`@9&5L971E("1%
M3E9[)%]](&EF("]>4$%27T%21U9?+SL*("`@('T*?0H*;7D@)'%U:65T(#T@
M(21%3E9[4$%27T1%0E5'?3L*"B,@9FEX("1P<F]G;F%M92!I9B!I;G9O:V5D
M(&9R;VT@4$%42`IM>2`E0V]N9FEG(#T@*`H@("`@<&%T:%]S97`@("`@/3X@
M*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"B`@("!?97AE("`@("`@
M("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G
M(#H@)R<I+`H@("`@7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3
M,B\@/R`G7%PG(#H@)R\G*2P**3L*"E]S971?<')O9VYA;64H*3L*7W-E=%]P
M87)?=&5M<"@I.PH*(R!-86=I8R!S=')I;F<@8VAE8VMI;F<@86YD(&5X=')A
M8W1I;F<@8G5N9&QE9"!M;V1U;&5S('M[>PIM>2`H)'-T87)T7W!O<RP@)&1A
M=&%?<&]S*3L*>PH@("`@;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT[
M"@H@("`@(R!#:&5C:R!F:6QE('1Y<&4L(&=E="!S=&%R="!O9B!D871A('-E
M8W1I;VX@>WM["B`@("!O<&5N(%]&2"P@)SPG+"`D<')O9VYA;64@;W(@;&%S
M=#L*("`@(&)I;FUO9&4H7T9(*3L*"B`@("!M>2`D8G5F.PH@("`@<V5E:R!?
M1D@L("TX+"`R.PH@("`@<F5A9"!?1D@L("1B=68L(#@["B`@("!L87-T('5N
M;&5S<R`D8G5F(&5Q(")<;E!!4BYP;5QN(CL*"B`@("!S965K(%]&2"P@+3$R
M+"`R.PH@("`@<F5A9"!?1D@L("1B=68L(#0["B`@("!S965K(%]&2"P@+3$R
M("T@=6YP86-K*").(BP@)&)U9BDL(#(["B`@("!R96%D(%]&2"P@)&)U9BP@
M-#L*"B`@("`D9&%T85]P;W,@/2`H=&5L;"!?1D@I("T@-#L*("`@(",@?7U]
M"@H@("`@(R!%>'1R86-T:6YG(&5A8V@@9FEL92!I;G1O(&UE;6]R>2![>WL*
M("`@(&UY("5R97%U:7)E7VQI<W0["B`@("!W:&EL92`H)&)U9B!E<2`B1DE,
M12(I('L*("`@("`@("!R96%D(%]&2"P@)&)U9BP@-#L*("`@("`@("!R96%D
M(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["@H@("`@("`@(&UY("1F
M=6QL;F%M92`]("1B=68["B`@("`@("`@;W5T<RAQ<2A5;G!A8VMI;F<@9FEL
M92`B)&9U;&QN86UE(BXN+BDI.PH@("`@("`@(&UY("1C<F,@/2`H("1F=6QL
M;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["B`@
M("`@("`@;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ
M+RD_*"XJ*2A<+BXJ*7PI.PH*("`@("`@("!R96%D(%]&2"P@)&)U9BP@-#L*
M("`@("`@("!R96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["@H@
M("`@("`@(&EF("AD969I;F5D*"1E>'0I(&%N9"`D97AT("%^("]<+B@_.G!M
M?'!L?&EX?&%L*20O:2D@>PH@("`@("`@("`@("!M>2`H)&]U="P@)&9I;&5N
M86UE*2`](%]T96UP9FEL92@D97AT+"`D8W)C*3L*("`@("`@("`@("`@:68@
M*"1O=70I('L*("`@("`@("`@("`@("`@(&)I;FUO9&4H)&]U="D["B`@("`@
M("`@("`@("`@("!P<FEN="`D;W5T("1B=68["B`@("`@("`@("`@("`@("!C
M;&]S92`D;W5T.PH@("`@("`@("`@("`@("`@8VAM;V0@,#<U-2P@)&9I;&5N
M86UE.PH@("`@("`@("`@("!]"B`@("`@("`@("`@("1005(Z.DAE879Y.CI&
M=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*("`@("`@("`@("`@
M)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE
M.PH@("`@("`@('T*("`@("`@("!E;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO
M/W-H;&EB+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["B`@("`@
M("`@("`@(",@<VAO=6QD(&)E(&UO=F5D('1O(%]T96UP9FEL92@I"B`@("`@
M("`@("`@(&UY("1F:6QE;F%M92`]("(D14Y6>U!!4E]414U0?2\D8F%S96YA
M;64D97AT(CL*("`@("`@("`@("`@;W5T<R@B4TA,24(Z("1F:6QE;F%M95QN
M(BD["B`@("`@("`@("`@(&]P96X@;7D@)&]U="P@)SXG+"`D9FEL96YA;64@
M;W(@9&EE("0A.PH@("`@("`@("`@("!B:6YM;V1E*"1O=70I.PH@("`@("`@
M("`@("!P<FEN="`D;W5T("1B=68["B`@("`@("`@("`@(&-L;W-E("1O=70[
M"B`@("`@("`@?0H@("`@("`@(&5L<V4@>PH@("`@("`@("`@("`D<F5Q=6ER
M95]L:7-T>R1F=6QL;F%M97T@/0H@("`@("`@("`@("`D4$%2.CI(96%V>3HZ
M36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*("`@("`@("`@("`@("`@(&)U
M9B`]/B`D8G5F+`H@("`@("`@("`@("`@("`@8W)C(#T^("1C<F,L"B`@("`@
M("`@("`@("`@("!N86UE(#T^("1F=6QL;F%M92P*("`@("`@("`@("`@?3L*
M("`@("`@("!]"B`@("`@("`@<F5A9"!?1D@L("1B=68L(#0["B`@("!]"B`@
M("`C('U]?0H*("`@(&QO8V%L($!)3D,@/2`H<W5B('L*("`@("`@("!M>2`H
M)'-E;&8L("1M;V1U;&4I(#T@0%\["@H@("`@("`@(')E='5R;B!I9B!R968@
M)&UO9'5L92!O<B`A)&UO9'5L93L*"B`@("`@("`@;7D@)&9I;&5N86UE(#T@
M9&5L971E("1R97%U:7)E7VQI<W1[)&UO9'5L97T@?'P@9&\@>PH@("`@("`@
M("`@("!M>2`D:V5Y.PH@("`@("`@("`@("!F;W)E86-H("AK97ES("5R97%U
M:7)E7VQI<W0I('L*("`@("`@("`@("`@("`@(&YE>'0@=6YL97-S("]<421M
M;V1U;&5<120O.PH@("`@("`@("`@("`@("`@)&ME>2`]("1?.R!L87-T.PH@
M("`@("`@("`@("!]"B`@("`@("`@("`@(&1E;&5T92`D<F5Q=6ER95]L:7-T
M>R1K97E](&EF(&1E9FEN960H)&ME>2D["B`@("`@("`@?2!O<B!R971U<FX[
M"@H@("`@("`@("1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D9FEL96YA;64O
M)&UO9'5L92(["@H@("`@("`@(&EF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E
M9FEN960H)DE/.CI&:6QE.CIN97<I*2!["B`@("`@("`@("`@(&UY("1F:"`]
M($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@)"$["B`@("`@("`@("`@
M(&)I;FUO9&4H)&9H*3L*("`@("`@("`@("`@<')I;G0@)&9H("1F:6QE;F%M
M92T^>V)U9GT["B`@("`@("`@("`@('-E96LH)&9H+"`P+"`P*3L*("`@("`@
M("`@("`@<F5T=7)N("1F:#L*("`@("`@("!]"B`@("`@("`@96QS92!["B`@
M("`@("`@("`@(&UY("@D;W5T+"`D;F%M92D@/2!?=&5M<&9I;&4H)RYP;2<L
M("1F:6QE;F%M92T^>V-R8WTI.PH@("`@("`@("`@("!I9B`H)&]U="D@>PH@
M("`@("`@("`@("`@("`@8FEN;6]D92@D;W5T*3L*("`@("`@("`@("`@("`@
M('!R:6YT("1O=70@)&9I;&5N86UE+3Y[8G5F?3L*("`@("`@("`@("`@("`@
M(&-L;W-E("1O=70["B`@("`@("`@("`@('T*("`@("`@("`@("`@;W!E;B!M
M>2`D9F@L("<\)RP@)&YA;64@;W(@9&EE("0A.PH@("`@("`@("`@("!B:6YM
M;V1E*"1F:"D["B`@("`@("`@("`@(')E='5R;B`D9F@["B`@("`@("`@?0H*
M("`@("`@("!D:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86YN;W0@9FEN
M9"`D;6]D=6QE(5QN(CL*("`@('TL($!)3D,I.PH*("`@(",@3F]W(&QO860@
M86QL(&)U;F1L960@9FEL97,@>WM["@H@("`@(R!I;FET:6%L:7IE('-H87)E
M9"!O8FIE8W0@<')O8V5S<VEN9PH@("`@<F5Q=6ER92!84TQO861E<CL*("`@
M(')E<75I<F4@4$%2.CI(96%V>3L*("`@(')E<75I<F4@0V%R<#HZ2&5A=GD[
M"B`@("!R97%U:7)E($5X<&]R=&5R.CI(96%V>3L*("`@(%!!4CHZ2&5A=GDZ
M.E]I;FET7V1Y;F%L;V%D97(H*3L*"B`@("`C(&YO=R!L970G<R!T<GD@9V5T
M=&EN9R!H96QP97(@;6]D=6QE<R!F<F]M('=I=&AI;@H@("`@<F5Q=6ER92!)
M3SHZ1FEL93L*"B`@("`C(&QO860@<F5S="!O9B!T:&4@9W)O=7`@:6X*("`@
M('=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI
M<W0I6S!=*2!["B`@("`@("`@(VQO8V%L("1)3D-[)T-W9"YP;2=](#T@7U]&
M24Q%7U\@:68@)%Y/(&YE("=-4U=I;C,R)SL*("`@("`@("!U;FQE<W,@*"1)
M3D-[)&9I;&5N86UE?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*("`@
M("`@("`@("`@(R!R97%U:7)E(&UO9'5L97,L(&1O(&]T:&5R(&5X96-U=&%B
M;&4@9FEL97,*("`@("`@("`@("`@:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_
M)"]I*2!["B`@("`@("`@("`@("`@("!R97%U:7)E("1F:6QE;F%M93L*("`@
M("`@("`@("`@?0H@("`@("`@("`@("!E;'-E('L*("`@("`@("`@("`@("`@
M(",@4VMI<"!!8W1I=F53=&%T92=S('-I=&5C=7-T;VUI>F4N<&P@9FEL93H*
M("`@("`@("`@("`@("`@(&1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE
M(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PH@("`@("`@("`@("!]"B`@("`@
M("`@?0H@("`@("`@(&1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T[
M"B`@("!]"@H@("`@(R!]?7T*"B`@("!L87-T('5N;&5S<R`D8G5F(&5Q(")0
M2UPP,#-<,#`T(CL*("`@("1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L*
M?0HC('U]?0H*(R!!<F=U;65N="!P<F]C97-S:6YG('M[>PIM>2!`<&%R7V%R
M9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*
M"B1Q=6EE="`](#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*"B,@1&]N)W0@
M<W=A;&QO=R!A<F=U;65N=',@9F]R(&-O;7!I;&5D(&5X96-U=&%B;&5S('=I
M=&AO=70@+2UP87(M;W!T:6]N<PII9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=6
M6S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*("`@(&UY("5D
M:7-T7V-M9"`]('%W*`H@("`@("`@('`@("!B;&EB7W1O7W!A<@H@("`@("`@
M(&D@("!I;G-T86QL7W!A<@H@("`@("`@('4@("!U;FEN<W1A;&Q?<&%R"B`@
M("`@("`@<R`@('-I9VY?<&%R"B`@("`@("`@=B`@('9E<FEF>5]P87(*("`@
M("D["B`@("!W:&EL92`H0$%21U8I('L*("`@("`@("`D05)'5ELP72`]?B`O
M7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T.PH*("`@("`@("!I
M9B`H)#$@97$@)TDG*2!["B`@("`@("`@("`@('5N<VAI9G0@0$E.0RP@)#([
M"B`@("`@("`@?0H@("`@("`@(&5L<VEF("@D,2!E<2`G32<I('L*("`@("`@
M("`@("`@979A;"`B=7-E("0R(CL*("`@("`@("!]"B`@("`@("`@96QS:68@
M*"0Q(&5Q("=!)RD@>PH@("`@("`@("`@("!U;G-H:69T($!P87)?87)G<RP@
M)#(["B`@("`@("`@?0H@("`@("`@(&5L<VEF("@D,2!E<2`G3R<I('L*("`@
M("`@("`@("`@)&]U="`]("0R.PH@("`@("`@('T*("`@("`@("!E;'-I9B`H
M)#$@97$@)V(G*2!["B`@("`@("`@("`@("1B=6YD;&4@/2`G<VET92<["B`@
M("`@("`@?0H@("`@("`@(&5L<VEF("@D,2!E<2`G0B<I('L*("`@("`@("`@
M("`@)&)U;F1L92`]("=A;&PG.PH@("`@("`@('T*("`@("`@("!E;'-I9B`H
M)#$@97$@)W$G*2!["B`@("`@("`@("`@("1Q=6EE="`](#$["B`@("`@("`@
M?0H@("`@("`@(&5L<VEF("@D,2!E<2`G3"<I('L*("`@("`@("`@("`@;W!E
M;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92`B6%A8.B!#86YN;W0@;W!E;B!L
M;V<Z("0A(CL*("`@("`@("!]"B`@("`@("`@96QS:68@*"0Q(&5Q("=4)RD@
M>PH@("`@("`@("`@("`D8V%C:&5?;F%M92`]("0R.PH@("`@("`@('T*"B`@
M("`@("`@<VAI9G0H0$%21U8I.PH*("`@("`@("!I9B`H;7D@)&-M9"`]("1D
M:7-T7V-M9'LD,7TI('L*("`@("`@("`@("`@9&5L971E("1%3E9[)U!!4E]4
M14U0)WT["B`@("`@("`@("`@(&EN:71?:6YC*"D["B`@("`@("`@("`@(')E
M<75I<F4@4$%2.CI$:7-T.PH@("`@("`@("`@("`F>R)005(Z.D1I<W0Z.B1C
M;60B?2@I('5N;&5S<R!`05)'5CL*("`@("`@("`@("`@)GLB4$%2.CI$:7-T
M.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*("`@("`@("`@("`@97AI=#L*("`@
M("`@("!]"B`@("!]"GT*"B,@6%A8("TM(&%D9"`M+7!A<BUD96)U9R!S=7!P
M;W)T(0H*(R!]?7T*"B,@3W5T<'5T(&UO9&4@*"U/*2!H86YD;&EN9R![>WL*
M:68@*"1O=70I('L*("`@('L*("`@("`@("`C;&]C86P@)$E.0WLG0W=D+G!M
M)WT@/2!?7T9)3$5?7R!I9B`D7D\@;F4@)TU35VEN,S(G.PH@("`@("`@(')E
M<75I<F4@24\Z.D9I;&4["B`@("`@("`@<F5Q=6ER92!!<F-H:79E.CI::7`[
M"B`@("!]"@H*("`@(&UY("1P87(@/2!S:&EF="A`05)'5BD["B`@("!M>2`D
M>FEP.PH*"B`@("!I9B`H9&5F:6YE9"`D<&%R*2!["B`@("`@("`@;W!E;B!M
M>2`D9F@L("<\)RP@)'!A<B!O<B!D:64@(D-A;FYO="!F:6YD("<D<&%R)SH@
M)"$B.PH@("`@("`@(&)I;FUO9&4H)&9H*3L*("`@("`@("!B;&5S<R@D9F@L
M("=)3SHZ1FEL92<I.PH*("`@("`@("`D>FEP(#T@07)C:&EV93HZ6FEP+3YN
M97<["B`@("`@("`@*"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D
M<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0H@("`@("`@("`@("!O
M<B!D:64@(E)E860@)R1P87(G(&5R<F]R.B`D(2(["B`@("!]"@H*("`@(&UY
M("5E;G8@/2!D;R!["B`@("`@("`@:68@*"1Z:7`@86YD(&UY("1M971A(#T@
M)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*("`@("`@("`@("`@)&UE
M=&$@/7X@<R\N*EYP87(Z)"\O;7,["B`@("`@("`@("`@("1M971A(#U^(',O
M7EQ3+BHO+VUS.PH@("`@("`@("`@("`D;65T82`]?B`O7B`@*%M>.ETK*3H@
M*"XK*20O;6<["B`@("`@("`@?0H@("`@?3L*"B`@("`C($]P96X@:6YP=70@
M86YD(&]U='!U="!F:6QE<R![>WL*("`@(&QO8V%L("0O(#T@7#0["@H@("`@
M:68@*&1E9FEN960@)'!A<BD@>PH@("`@("`@(&]P96X@4$%2+"`G/"<L("1P
M87(@;W(@9&EE("(D(3H@)'!A<B(["B`@("`@("`@8FEN;6]D92A005(I.PH@
M("`@("`@(&1I92`B)'!A<B!I<R!N;W0@82!005(@9FEL92(@=6YL97-S(#Q0
M05(^(&5Q(")02UPP,#-<,#`T(CL*("`@('T*"B`@("!#<F5A=&50871H*"1O
M=70I(#L*("`@(`H@("`@;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*("`@("`@
M("`D;W5T+`H@("`@("`@($E/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL
M93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"B`@("`@("`@
M,#<W-RP*("`@("D@;W(@9&EE("0A.PH@("`@8FEN;6]D92@D9F@I.PH*("`@
M("0O(#T@*&1E9FEN960@)&1A=&%?<&]S*2`_(%PD9&%T85]P;W,@.B!U;F1E
M9CL*("`@('-E96L@7T9(+"`P+"`P.PH@("`@;7D@)&QO861E<B`]('-C86QA
M<B`\7T9(/CL*("`@(&EF("@A)$5.5GM005)?5D520D%424U](&%N9"`D;&]A
M9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PH@("`@("`@(')E<75I<F4@4$%2
M.CI&:6QT97(Z.E!O9%-T<FEP.PH@("`@("`@(%!!4CHZ1FEL=&5R.CI0;V13
M=')I<"T^;F5W+3YA<'!L>2A<)&QO861E<BP@)#`I"B`@("!]"B`@("!F;W)E
M86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["B`@("`@("`@;7D@)'9A
M;"`]("1E;G9[)&ME>7T@;W(@;F5X=#L*("`@("`@("`D=F%L(#T@979A;"`D
M=F%L(&EF("1V86P@/7X@+UY;)R)=+SL*("`@("`@("!M>2`D;6%G:6,@/2`B
M7U]%3E9?4$%27R(@+B!U8R@D:V5Y*2`N(")?7R(["B`@("`@("`@;7D@)'-E
M="`](")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*("`@("`@("`D;&]A
M9&5R(#U^('-[)&UA9VEC*"`K*7U["B`@("`@("`@("`@("1M86=I8R`N("1S
M970@+B`H)R`G('@@*&QE;F=T:"@D,2D@+2!L96YG=&@H)'-E="DI*0H@("`@
M("`@('UE9SL*("`@('T*("`@("1F:"T^<')I;G0H)&QO861E<BD["B`@("`D
M+R`]('5N9&5F.PH@("`@(R!]?7T*"B`@("`C(%=R:71E(&)U;F1L960@;6]D
M=6QE<R![>WL*("`@(&EF("@D8G5N9&QE*2!["B`@("`@("`@<F5Q=6ER92!0
M05(Z.DAE879Y.PH@("`@("`@(%!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D
M97(H*3L*("`@("`@("!I;FET7VEN8R@I.PH@("`@("`@(')E<75I<F5?;6]D
M=6QE<R@I.PH*("`@("`@("!M>2!`:6YC(#T@<V]R="!["B`@("`@("`@("`@
M(&QE;F=T:"@D8BD@/#T^(&QE;F=T:"@D82D*("`@("`@("!](&=R97`@>PH@
M("`@("`@("`@("`A+T)31%!!3B\*("`@("`@("!](&=R97`@>PH@("`@("`@
M("`@("`H)&)U;F1L92!N92`G<VET92<I(&]R"B`@("`@("`@("`@("@D7R!N
M92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B`@("`@("`@("`@
M("`D7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PH@("`@("`@
M('T@0$E.0SL*"B`@("`@("`@:68@*"1#;VYF:6=[7V1E;&EM?2!E<2`G7%PG
M*2![('-[7%Q]>R]]9R!F;W(@0&EN8R!]"@H@("`@("`@(&UY("5F:6QE<SL*
M("`@("`@("`O7E\\*"XK*20O(&%N9"`D9FEL97-[)#%]*RL@9F]R(&ME>7,@
M)3HZ.PH@("`@("`@("1F:6QE<WLD7WTK*R!F;W(@=F%L=65S("5)3D,["@H@
M("`@("`@(&UY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1]
M.PH@("`@("`@(&UY("5W<FET=&5N.PH*("`@("`@("!F;W)E86-H("AS;W)T
M(&ME>7,@)69I;&5S*2!["B`@("`@("`@("`@(&UY("@D;F%M92P@)&9I;&4I
M.PH*("`@("`@("`@("`@9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["B`@("`@
M("`@("`@("`@("!I9B`H)&YA;64@/2`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE
M>R1??2D@>PH@("`@("`@("`@("`@("`@("`@("1F:6QE(#T@)%\["B`@("`@
M("`@("`@("`@("`@("`@;&%S=#L*("`@("`@("`@("`@("`@('T*("`@("`@
M("`@("`@("`@(&5L<VEF("@O7BA<421D:7)<15PO*"XJ6UY#8UTI*5Q:+VDI
M('L*("`@("`@("`@("`@("`@("`@("`H)&9I;&4L("1N86UE*2`]("@D,2P@
M)#(I.PH@("`@("`@("`@("`@("`@("`@(&QA<W0["B`@("`@("`@("`@("`@
M("!]"B`@("`@("`@("`@("`@("!E;'-I9B`H;2%>+VQO861E<B];7B]=*R\H
M+BI;7D-C72E<6B$I('L*("`@("`@("`@("`@("`@("`@("!I9B`H;7D@)')E
M9B`]("1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD,7TI('L*("`@("`@("`@
M("`@("`@("`@("`@("`@*"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PH@
M("`@("`@("`@("`@("`@("`@("`@("!L87-T.PH@("`@("`@("`@("`@("`@
M("`@('T*("`@("`@("`@("`@("`@("`@("!E;'-I9B`H+68@(B1D:7(O)#$B
M*2!["B`@("`@("`@("`@("`@("`@("`@("`@("@D9FEL92P@)&YA;64I(#T@
M*"(D9&ER+R0Q(BP@)#$I.PH@("`@("`@("`@("`@("`@("`@("`@("!L87-T
M.PH@("`@("`@("`@("`@("`@("`@('T*("`@("`@("`@("`@("`@('T*("`@
M("`@("`@("`@?0H*("`@("`@("`@("`@;F5X="!U;FQE<W,@9&5F:6YE9"`D
M;F%M92!A;F0@;F]T("1W<FET=&5N>R1N86UE?2LK.PH@("`@("`@("`@("!N
M97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT
M7$4D+SL*("`@("`@("`@("`@;W5T<R@@:F]I;B`B(BP*("`@("`@("`@("`@
M("`@('%Q*%!A8VMI;F<@(BDL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@
M.B`D9FEL92P*("`@("`@("`@("`@("`@('%Q*"(N+BXI"B`@("`@("`@("`@
M("D["@H@("`@("`@("`@("!M>2`D8V]N=&5N=#L*("`@("`@("`@("`@:68@
M*')E9B@D9FEL92DI('L*("`@("`@("`@("`@("`@("1C;VYT96YT(#T@)&9I
M;&4M/GMB=69].PH@("`@("`@("`@("!]"B`@("`@("`@("`@(&5L<V4@>PH@
M("`@("`@("`@("`@("`@;W!E;B!&24Q%+"`G/"<L("1F:6QE(&]R(&1I92`B
M0V%N)W0@;W!E;B`D9FEL93H@)"$B.PH@("`@("`@("`@("`@("`@8FEN;6]D
M92A&24Q%*3L*("`@("`@("`@("`@("`@("1C;VYT96YT(#T@/$9)3$4^.PH@
M("`@("`@("`@("`@("`@8VQO<V4@1DE,13L*"B`@("`@("`@("`@("`@("!0
M05(Z.D9I;'1E<CHZ4&]D4W1R:7`M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D
M9FEL92D*("`@("`@("`@("`@("`@("`@("!I9B`A)$5.5GM005)?5D520D%4
M24U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["@H@("`@("`@
M("`@("`@("`@(R!$;R!N;W0@;&5T(%A33&]A9&5R('!I8VL@=7`@875T;R\J
M(&9R;VT@96YV:7)O;FUE;G0*("`@("`@("`@("`@("`@("1C;VYT96YT(#U^
M(',O9V]T;R`K<F5T<GD@*W5N;&5S<R`K+BHO9V]T;R!R971R>3LO"B`@("`@
M("`@("`@("`@("`@("`@:68@;&,H)&YA;64I(&5Q(&QC*")84TQO861E<BYP
M;2(I.PH@("`@("`@("`@("!]"@H@("`@("`@("`@("!O=71S*'%Q*%=R:71T
M96X@87,@(B1N86UE(BDI.PH@("`@("`@("`@("`D9F@M/G!R:6YT*")&24Q%
M(BD["B`@("`@("`@("`@("1F:"T^<')I;G0H<&%C:R@G3B<L(&QE;F=T:"@D
M;F%M92D@*R`Y*2D["B`@("`@("`@("`@("1F:"T^<')I;G0H<W!R:6YT9B@*
M("`@("`@("`@("`@("`@("(E,#AX+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP
M=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M90H@("`@("`@("`@("`I*3L*("`@
M("`@("`@("`@)&9H+3YP<FEN="AP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT
M*2DI.PH@("`@("`@("`@("`D9F@M/G!R:6YT*"1C;VYT96YT*3L*("`@("`@
M("!]"B`@("!]"B`@("`C('U]?0H*("`@(",@3F]W('=R:71E(&]U="!T:&4@
M4$%2(&%N9"!M86=I8R!S=')I;F=S('M[>PH@("`@)'II<"T^=W)I=&54;T9I
M;&5(86YD;&4H)&9H*2!I9B`D>FEP.PH*("`@("1C86-H95]N86UE(#T@<W5B
M<W1R("1C86-H95]N86UE+"`P+"`T,#L*("`@(&EF("@A)&-A8VAE7VYA;64@
M86YD(&UY("1M=&EM92`]("AS=&%T*"1O=70I*5LY72D@>PH@("`@("`@(&UY
M("1C='@@/2!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.E-(03L@1&EG97-T.CI3
M2$$M/FYE=R@Q*2!]"B`@("`@("`@("`@('Q\(&5V86P@>R!R97%U:7)E($1I
M9V5S=#HZ4TA!,3L@1&EG97-T.CI32$$Q+3YN97<@?0H@("`@("`@("`@("!\
M?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.DU$-3L@1&EG97-T.CI-1#4M/FYE
M=R!].PH*("`@("`@("`C(%=O<FMA<F]U;F0@9F]R(&)U9R!I;B!$:6=E<W0Z
M.E-(02`U+C,X(&%N9"`U+C,Y"B`@("`@("`@;7D@)'-H85]V97)S:6]N(#T@
M979A;"![("1$:6=E<W0Z.E-(03HZ5D524TE/3B!]('Q\(#`["B`@("`@("`@
M:68@*"1S:&%?=F5R<VEO;B!E<2`G-2XS."<@;W(@)'-H85]V97)S:6]N(&5Q
M("<U+C,Y)RD@>PH@("`@("`@("`@("`D8W1X+3YA9&1F:6QE*"1O=70L(")B
M(BD@:68@*"1C='@I.PH@("`@("`@('T*("`@("`@("!E;'-E('L*("`@("`@
M("`@("`@:68@*"1C='@@86YD(&]P96XH;7D@)&9H+"`B/"1O=70B*2D@>PH@
M("`@("`@("`@("`@("`@8FEN;6]D92@D9F@I.PH@("`@("`@("`@("`@("`@
M)&-T>"T^861D9FEL92@D9F@I.PH@("`@("`@("`@("`@("`@8VQO<V4H)&9H
M*3L*("`@("`@("`@("`@?0H@("`@("`@('T*"B`@("`@("`@)&-A8VAE7VYA
M;64@/2`D8W1X(#\@)&-T>"T^:&5X9&EG97-T(#H@)&UT:6UE.PH@("`@?0H@
M("`@)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,2`M(&QE;F=T:"`D8V%C:&5?
M;F%M92D["B`@("`D8V%C:&5?;F%M92`N/2`B0T%#2$4B.PH@("`@)&9H+3YP
M<FEN="@D8V%C:&5?;F%M92D["B`@("`D9F@M/G!R:6YT*'!A8VLH)TXG+"`D
M9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BDI*3L*("`@("1F:"T^<')I;G0H
M(EQN4$%2+G!M7&XB*3L*("`@("1F:"T^8VQO<V4["B`@("!C:&UO9"`P-S4U
M+"`D;W5T.PH@("`@(R!]?7T*"B`@("!E>&ET.PI]"B,@?7U]"@HC(%!R97!A
M<F4@)'!R;V=N86UE(&EN=&\@4$%2(&9I;&4@8V%C:&4@>WM["GL*("`@(&QA
M<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*"B`@("!?9FEX7W!R;V=N
M86UE*"D["@H@("`@(R!.;W<@;&]A9"!T:&4@4$%2(&9I;&4@86YD('!U="!I
M="!I;G1O(%!!4CHZ3&EB0V%C:&4@>WM["B`@("!R97%U:7)E(%!!4CL*("`@
M(%!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*"@H@("`@>PH@("`@
M("`@("-L;V-A;"`D24Y#>R=#=V0N<&TG?2`](%]?1DE,15]?(&EF("1>3R!N
M92`G35-7:6XS,B<["B`@("`@("`@<F5Q=6ER92!&:6QE.CI&:6YD.PH@("`@
M("`@(')E<75I<F4@07)C:&EV93HZ6FEP.PH@("`@?0H@("`@;7D@)'II<"`]
M($%R8VAI=F4Z.EII<"T^;F5W.PH@("`@;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=SL*("`@("1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92`B
M)"$Z("1`(CL*("`@("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I(&]R(&1I92`B)"$Z
M("1`(CL*"B`@("!P=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PH@("`@)%!!
M4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["@H@("`@)'%U:65T(#T@
M(21%3E9[4$%27T1%0E5'?3L*("`@(&]U=',H<7$H7"1%3E9[4$%27U1%35!]
M(#T@(B1%3E9[4$%27U1%35!](BDI.PH*("`@(&EF("AD969I;F5D("1%3E9[
M4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0H@
M("`@("`@(&9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2![
M"B`@("`@("`@("`@(&YE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["B`@
M("`@("`@("`@(&UY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE
M.PH@("`@("`@("`@("!N97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*
M("`@("`@("`@("`@("`@(%X*("`@("`@("`@("`@("`@("\_<VAL:6(O"B`@
M("`@("`@("`@("`@("`H/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\*
M("`@("`@("`@("`@("`@("@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M97TO
M*3\*("`@("`@("`@("`@("`@("A;7B]=*RD*("`@("`@("`@("`@("`@("0*
M("`@("`@("`@("`@?7@["B`@("`@("`@("`@(&UY("1E>'1R86-T7VYA;64@
M/2`D,3L*("`@("`@("`@("`@;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PH@("`@
M("`@("`@("`D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M
M92D["B`@("`@("`@("`@(&]U=',H<7$H17AT<F%C=&EN9R`B)&UE;6)E<E]N
M86UE(B!T;R`B)&1E<W1?;F%M92(I*3L*("`@("`@("!]"B`@("!]"B`@("`C
M('U]?0I]"B,@?7U]"@HC($EF('1H97)E)W,@;F\@;6%I;BYP;"!T;R!R=6XL
M('-H;W<@=7-A9V4@>WM["G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N
M86UE?2D@>PH@("`@9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D
M,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N
M<&%R(%T@6R!P<F]G<F%M+G!L(%T*("`@("`@("0P(%L@+4)\+6(@72!;+4]O
M=71F:6QE72!S<F,N<&%R"BX*("`@("1%3E9[4$%27U!23T=.04U%?2`]("1P
M<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"B,@?7U]"@IS=6(@0W)E
M871E4&%T:"!["B`@("!M>2`H)&YA;64I(#T@0%\["B`@("`*("`@(')E<75I
M<F4@1FEL93HZ0F%S96YA;64["B`@("!M>2`H)&)A<V5N86UE+"`D<&%T:"P@
M)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN
M+BHG*2D["B`@("`*("`@(')E<75I<F4@1FEL93HZ4&%T:#L*("`@(`H)1FEL
M93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK
M<&%T:"!D:65S('=I=&@@97)R;W(*?0H*<W5B(')E<75I<F5?;6]D=6QE<R![
M"B`@("`C;&]C86P@)$E.0WLG0W=D+G!M)WT@/2!?7T9)3$5?7R!I9B`D7D\@
M;F4@)TU35VEN,S(G.PH*("`@(')E<75I<F4@;&EB.PH@("`@<F5Q=6ER92!$
M>6YA3&]A9&5R.PH@("`@<F5Q=6ER92!I;G1E9V5R.PH@("`@<F5Q=6ER92!S
M=')I8W0["B`@("!R97%U:7)E('=A<FYI;F=S.PH@("`@<F5Q=6ER92!V87)S
M.PH@("`@<F5Q=6ER92!#87)P.PH@("`@<F5Q=6ER92!#87)P.CI(96%V>3L*
M("`@(')E<75I<F4@17AP;W)T97(Z.DAE879Y.PH@("`@<F5Q=6ER92!%>'!O
M<G1E<CL*("`@(')E<75I<F4@1F-N=&P["B`@("!R97%U:7)E($9I;&4Z.E1E
M;7`["B`@("!R97%U:7)E($9I;&4Z.E-P96,["B`@("!R97%U:7)E(%A33&]A
M9&5R.PH@("`@<F5Q=6ER92!#;VYF:6<["B`@("!R97%U:7)E($E/.CI(86YD
M;&4["B`@("!R97%U:7)E($E/.CI&:6QE.PH@("`@<F5Q=6ER92!#;VUP<F5S
M<SHZ6FQI8CL*("`@(')E<75I<F4@07)C:&EV93HZ6FEP.PH@("`@<F5Q=6ER
M92!005(["B`@("!R97%U:7)E(%!!4CHZ2&5A=GD["B`@("!R97%U:7)E(%!!
M4CHZ1&ES=#L*("`@(')E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PH@
M("`@979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["GT*"B,@5&AE($,@
M=F5R<VEO;B!O9B!T:&ES(&-O9&4@87!P96%R<R!I;B!M>6QD<B]M:W1M<&1I
M<BYC"G-U8B!?<V5T7W!A<E]T96UP('L*("`@(&EF("@D14Y6>U!!4E]414U0
M?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B`@("`@("`@)'!A
M<E]T96UP(#T@)#$["B`@("`@("`@<F5T=7)N.PH@("`@?0H*("`@(&9O<F5A
M8V@@;7D@)'!A=&@@*`H@("`@("`@("AM87`@)$5.5GLD7WTL('%W*"!005)?
M5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`H@("`@("`@('%W
M*"!#.EQ<5$5-4"`O=&UP("X@*0H@("`@*2!["B`@("`@("`@;F5X="!U;FQE
M<W,@)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*("`@("`@("!M
M>2`D=7-E<FYA;64["B`@("`@("`@;7D@)'!W=6ED.PH@("`@("`@(",@9&]E
M<R!N;W0@=V]R:R!E=F5R>7=H97)E.@H@("`@("`@(&5V86P@>R@D<'=U:60I
M(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*"B`@("`@("`@:68@
M*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*("`@("`@("`@("`@
M)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["B`@("`@("`@?0H@("`@
M("`@(&5L<VEF("AD969I;F5D("1P=W5I9"D@>PH@("`@("`@("`@("`D=7-E
M<FYA;64@/2`D<'=U:60["B`@("`@("`@?0H@("`@("`@(&5L<V4@>PH@("`@
M("`@("`@("`D=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U53
M15)]('Q\("=365-414TG.PH@("`@("`@('T*("`@("`@("`D=7-E<FYA;64@
M/7X@<R]<5R]?+V<["@H@("`@("`@(&UY("1S=&UP9&ER(#T@(B1P871H)$-O
M;F9I9WM?9&5L:6U]<&%R+21U<V5R;F%M92(["B`@("`@("`@;6MD:7(@)'-T
M;7!D:7(L(#`W-34["B`@("`@("`@:68@*"$D14Y6>U!!4E]#3$5!3GT@86YD
M(&UY("1M=&EM92`]("AS=&%T*"1P<F]G;F%M92DI6SE=*2!["B`@("`@("`@
M("`@(&]P96X@*&UY("1F:"P@(CPB+B`D<')O9VYA;64I.PH@("`@("`@("`@
M("!S965K("1F:"P@+3$X+"`R.PH@("`@("`@("`@("!S>7-R96%D("1F:"P@
M;7D@)&)U9BP@-CL*("`@("`@("`@("`@:68@*"1B=68@97$@(EPP0T%#2$4B
M*2!["B`@("`@("`@("`@("`@("!S965K("1F:"P@+34X+"`R.PH@("`@("`@
M("`@("`@("`@<WES<F5A9"`D9F@L("1B=68L(#0Q.PH@("`@("`@("`@("`@
M("`@)&)U9B`]?B!S+UPP+R]G.PH@("`@("`@("`@("`@("`@)'-T;7!D:7(@
M+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+2(@+B`D8G5F.PH@("`@("`@("`@
M("!]"B`@("`@("`@("`@(&5L<V4@>PH@("`@("`@("`@("`@("`@;7D@)&-T
M>"`](&5V86P@>R!R97%U:7)E($1I9V5S=#HZ4TA!.R!$:6=E<W0Z.E-(02T^
M;F5W*#$I('T*("`@("`@("`@("`@("`@("`@("!\?"!E=F%L('L@<F5Q=6ER
M92!$:6=E<W0Z.E-(03$[($1I9V5S=#HZ4TA!,2T^;F5W('T*("`@("`@("`@
M("`@("`@("`@("!\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.DU$-3L@1&EG
M97-T.CI-1#4M/FYE=R!].PH*("`@("`@("`@("`@("`@(",@5V]R:V%R;W5N
M9"!F;W(@8G5G(&EN($1I9V5S=#HZ4TA!(#4N,S@@86YD(#4N,SD*("`@("`@
M("`@("`@("`@(&UY("1S:&%?=F5R<VEO;B`](&5V86P@>R`D1&EG97-T.CI3
M2$$Z.E9%4E-)3TX@?2!\?"`P.PH@("`@("`@("`@("`@("`@:68@*"1S:&%?
M=F5R<VEO;B!E<2`G-2XS."<@;W(@)'-H85]V97)S:6]N(&5Q("<U+C,Y)RD@
M>PH@("`@("`@("`@("`@("`@("`@("1C='@M/F%D9&9I;&4H)'!R;V=N86UE
M+"`B8B(I(&EF("@D8W1X*3L*("`@("`@("`@("`@("`@('T*("`@("`@("`@
M("`@("`@(&5L<V4@>PH@("`@("`@("`@("`@("`@("`@(&EF("@D8W1X(&%N
M9"!O<&5N*&UY("1F:"P@(CPD<')O9VYA;64B*2D@>PH@("`@("`@("`@("`@
M("`@("`@("`@("!B:6YM;V1E*"1F:"D["B`@("`@("`@("`@("`@("`@("`@
M("`@("1C='@M/F%D9&9I;&4H)&9H*3L*("`@("`@("`@("`@("`@("`@("`@
M("`@8VQO<V4H)&9H*3L*("`@("`@("`@("`@("`@("`@("!]"B`@("`@("`@
M("`@("`@("!]"@H@("`@("`@("`@("`@("`@)'-T;7!D:7(@+CT@(B1#;VYF
M:6=[7V1E;&EM?6-A8VAE+2(@+B`H("1C='@@/R`D8W1X+3YH97AD:6=E<W0@
M.B`D;71I;64@*3L*("`@("`@("`@("`@?0H@("`@("`@("`@("!C;&]S92@D
M9F@I.PH@("`@("`@('T*("`@("`@("!E;'-E('L*("`@("`@("`@("`@)$5.
M5GM005)?0TQ%04Y](#T@,3L*("`@("`@("`@("`@)'-T;7!D:7(@+CT@(B1#
M;VYF:6=[7V1E;&EM?71E;7`M)"0B.PH@("`@("`@('T*"B`@("`@("`@)$5.
M5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*("`@("`@("!M:V1I<B`D<W1M<&1I
M<BP@,#<U-3L*("`@("`@("!L87-T.PH@("`@?0H*("`@("1P87)?=&5M<"`]
M("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\["GT*"G-U8B!?=&5M<&9I;&4@>PH@("`@;7D@*"1E>'0L("1C<F,I
M(#T@0%\["B`@("!M>2`H)&9H+"`D9FEL96YA;64I.PH*("`@("1F:6QE;F%M
M92`]("(D<&%R7W1E;7`O)&-R8R1E>'0B.PH*("`@(&EF("@D14Y6>U!!4E]#
M3$5!3GTI('L*("`@("`@("!U;FQI;FL@)&9I;&5N86UE(&EF("UE("1F:6QE
M;F%M93L*("`@("`@("!P=7-H($!T;7!F:6QE+"`D9FEL96YA;64["B`@("!]
M"B`@("!E;'-E('L*("`@("`@("!R971U<FX@*'5N9&5F+"`D9FEL96YA;64I
M(&EF("@M<B`D9FEL96YA;64I.PH@("`@?0H*("`@(&]P96X@)&9H+"`G/B<L
M("1F:6QE;F%M92!O<B!D:64@)"$["B`@("!B:6YM;V1E*"1F:"D["B`@("!R
M971U<FXH)&9H+"`D9FEL96YA;64I.PI]"@IS=6(@7W-E=%]P<F]G;F%M92![
M"B`@("!I9B`H)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'
M3D%-17T@/7X@+R@N*RDO*2!["B`@("`@("`@)'!R;V=N86UE(#T@)#$["B`@
M("!]"@H@("`@)'!R;V=N86UE('Q\/2`D,#L*"B`@("!I9B`H)$5.5GM005)?
M5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X]
M(#`I('L*("`@("`@("`D<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R
M:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI("L@,2D["B`@("!]
M"@H@("`@:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R;V=N
M86UE+"`D0V]N9FEG>U]D96QI;7TI(#X](#`I('L*("`@("`@("!I9B`H;W!E
M;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["B`@("`@("`@("`@(')E='5R
M;B!I9B`M<R`D9F@["B`@("`@("`@?0H@("`@("`@(&EF("@M<R`B)'!R;V=N
M86UE)$-O;F9I9WM?97AE?2(I('L*("`@("`@("`@("`@)'!R;V=N86UE("X]
M("1#;VYF:6=[7V5X97T["B`@("`@("`@("`@(')E='5R;CL*("`@("`@("!]
M"B`@("!]"@H@("`@9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D0V]N9FEG
M>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PH@("`@("`@(&YE>'0@:68@
M97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%
M35!].PH@("`@("`@("1D:7(@/7X@<R]<421#;VYF:6=[7V1E;&EM?5Q%)"\O
M.PH@("`@("`@("@H)'!R;V=N86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD
M<')O9VYA;64D0V]N9FEG>U]E>&5](BDL(&QA<W0I"B`@("`@("`@("`@(&EF
M("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE)$-O;F9I9WM?97AE
M?2(["B`@("`@("`@*"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM
M?21P<F]G;F%M92(I+"!L87-T*0H@("`@("`@("`@("!I9B`M<R`B)&1I<B1#
M;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["B`@("!]"GT*"G-U8B!?9FEX7W!R
M;V=N86UE('L*("`@("0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'
M3D%-17T["B`@("!I9B`H:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI
M;7TI(#P@,"D@>PH@("`@("`@("1P<F]G;F%M92`]("(N)$-O;F9I9WM?9&5L
M:6U])'!R;V=N86UE(CL*("`@('T*"B`@("`C(%A86"`M(&AA8VL@=&\@;6%K
M92!05T0@=V]R:PH@("`@;7D@)'!W9"`]("AD969I;F5D("97:6XS,CHZ1V5T
M0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z("1%3E9[4%=$?3L*("`@("1P=V0@
M/2!@<'=D8"!I9B`A9&5F:6YE9"`D<'=D.PH@("`@8VAO;7`H)'!W9"D["B`@
M("`D<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421#;VYF:6=[7V1E;&EM?5Q%
M*2\D<'=D)$-O;F9I9WM?9&5L:6U]+SL*"B`@("`D14Y6>U!!4E]04D]'3D%-
M17T@/2`D<')O9VYA;64["GT*"G-U8B!?<&%R7VEN:71?96YV('L*("`@(&EF
M("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*("`@("`@("!R
M971U<FX["B`@("!](&5L<V4@>PH@("`@("`@("1%3E9[4$%27TE.251)04Q)
M6D5$?2`](#(["B`@("!]"@H@("`@9F]R("AQ=R@@4U!!5TY%1"!414U0($-,
M14%.($1%0E5'($-!0TA%(%!23T=.04U%($%21T,@05)'5E\P("D@*2!["B`@
M("`@("`@9&5L971E("1%3E9[)U!!4E\G+B1??3L*("`@('T*("`@(&9O<B`H
M<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*("`@("`@("`D14Y6
M>R=005)?)RXD7WT@/2`D14Y6>R=005)?1TQ/0D%,7R<N)%]](&EF(&5X:7-T
M<R`D14Y6>R=005)?1TQ/0D%,7R<N)%]].PH@("`@?0H*("`@(&UY("1P87)?
M8VQE86X@/2`B7U]%3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@("`B.PH*
M("`@(&EF("@D14Y6>U!!4E]414U0?2D@>PH@("`@("`@(&1E;&5T92`D14Y6
M>U!!4E]#3$5!3GT["B`@("!]"B`@("!E;'-I9B`H(65X:7-T<R`D14Y6>U!!
M4E]'3$]"04Q?0TQ%04Y]*2!["B`@("`@("`@;7D@)'9A;'5E(#T@<W5B<W1R
M*"1P87)?8VQE86XL(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*("`@("`@("`D
M14Y6>U!!4E]#3$5!3GT@/2`D,2!I9B`D=F%L=64@/7X@+UY005)?0TQ%04X]
M*%Q3*RDO.PH@("`@?0I]"@IS=6(@;W5T<R!["B`@("!R971U<FX@:68@)'%U
M:65T.PH@("`@:68@*"1L;V=F:"D@>PH@("`@("`@('!R:6YT("1L;V=F:"`B
M0%]<;B(["B`@("!]"B`@("!E;'-E('L*("`@("`@("!P<FEN="`B0%]<;B([
M"B`@("!]"GT*"G-U8B!I;FET7VEN8R!["B`@("!R97%U:7)E($-O;F9I9SL*
M("`@('!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O;F9I9SHZ0V]N
M9FEG>R1??2P@<7<H"B`@("`@("`@87)C:&QI8F5X<"!P<FEV;&EB97AP('-I
M=&5A<F-H97AP('-I=&5L:6)E>'`*("`@("`@("!V96YD;W)A<F-H97AP('9E
M;F1O<FQI8F5X<`H@("`@*3L*?0H*(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C
M(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C(R,C
M(R,C"B,@5&AE(&UA:6X@<&%C:V%G92!F;W(@<V-R:7!T(&5X96-U=&EO;@H*
M<&%C:V%G92!M86EN.PH*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"90
M05(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PH*9&EE('%Q
M*'!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@
M3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN*0H@("`@=6YL97-S("UE("1P
M<F]G;F%M93L*"F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PH*?3L*
M"B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0H*0T]213HZ97AI="@D,2D@:68@
M*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ
M7U]%4E)/4B!I9B`D.CI?7T524D]2.PH*,3L*"B-L:6YE(#DT,@H*7U]%3D1?
M7PH``'!A<BYP;```_____]PZ0`#P.D``@)4`````````````!)@``*A```#@
ME`````````````!(F```"$```-B4`````````````&:8````0```\)0`````
M````````:ID``!A``````````````````````````````%:8````````-)@`
M`":8```0F````````/29``#DF0``U)D``,"9``"TF0``I)D``)29``",F0``
M?ID``':9``!>F0``5)D``$:9```ZF0``,)D``"*9```"F@``#ID```29``#T
MF```Z)@``-Z8``#4F```RI@``,"8``"VF```II@``)R8``"2F```B)@``'Z8
M``!TF```%)H``"B:```8F0```````/*7``#<EP``SI<``+R7``"<EP``A)<`
M`&Z7``!@EP``5)<``$27```XEP``)I<``!*7````EP``\)8``-R6``#,E@``
MO)8``*Z6``":E@``BI8``'*6``!@E@``4)8``$"6```LE@``&I8```:6``#X
ME0```````,0#4&5R;%]N97=84P``\0)097)L7V=E=%]C;VYT97AT``#O!&)O
M;W1?1'EN84QO861E<@!<`E!E<FQ?5'-T871B=69?<'1R`+(`4&5R;%])3$E/
M7W!T<@"U`%!E<FQ?24UE;5]P='(`N0!097)L7TE0<F]C7W!T<@``N0)097)L
M7V-R;V%K7VYO8V]N=&5X=```L`!097)L7TE$:7)?<'1R`,\`4&5R;%])8V]M
M<&-V7W!T<@``?@-097)L7VUG7W-E=`"/!%!E<FQ?<W9?<V5T<'8`=@)097)L
M7V%V7W-H:69T`$($4&5R;%]S=E\R<'9?;F]L96X`<`)097)L7V%V7V9E=&-H
M`!`#4&5R;%]G=E]F971C:'!V`&`"4&5R;%]4=&%I;G1E9%]P='(`[01097)L
M7W=I;C,R7W1E<FT`^`1P97)L7V9R964`]P1P97)L7V1E<W1R=6-T`/L$<&5R
M;%]R=6X``/H$<&5R;%]P87)S90``^@-097)L7W-A9F5S>7-M86QL;V,``/``
M4&5R;%])97AI=%]F;&%G<U]P='(``%L!4&5R;%])<&5R;%]D97-T<G5C=%]L
M979E;%]P='(`]@1P97)L7V-O;G-T<G5C=```\`1P97)L7V%L;&]C``"6`%!E
M<FQ?1V1O7W5N9'5M<%]P='(`[`1097)L7W=I;C,R7VEN:70`<&5R;#4X+F1L
M;```)`%'971-;V1U;&5&:6QE3F%M94$``+0`1G)E94QI8G)A<GD`)@%'971-
M;V1U;&5(86YD;&5!``!+15).14PS,BYD;&P``-<`1V5T57-E<DYA;65!``!!
M1%9!4$DS,BYD;&P``+("<W!R:6YT9@"X`G-T<F-M<```QP)S=')T;VL``+X"
M<W1R;&5N``#%`G-T<G-T<@``;0!?7W!?7V5N=FER;VX``)@";65M;6]V90#!
M`G-T<FYC<'D`P`)S=')N8VUP`((!7VUK9&ER``".`5]P8W1Y<&4`%0%?:7-C
M='EP90``80!?7VUB7V-U<E]M87@``)<";65M8W!Y``"9`FUE;7-E=```F0%?
M<FUD:7(``-\`7V9I;F1C;&]S90``W0%?=6YL:6YK`.(`7V9I;F1N97AT`.``
M7V9I;F1F:7)S=```OP%?<W1R9'5P`,4!7W-T<FYI8VUP`$U35D-25"YD;&P`
M`-,`7V5X:70`2`!?6&-P=$9I;'1E<@!)`F5X:70``&0`7U]P7U]?:6YI=&5N
M=@!8`%]?9V5T;6%I;F%R9W,`#P%?:6YI='1E<FT`@P!?7W-E='5S97)M871H
M97)R``"=`%]A9&IU<W1?9F1I=@``:@!?7W!?7V-O;6UO9&4``&\`7U]P7U]F
M;6]D90``@0!?7W-E=%]A<'!?='EP90``R@!?97AC97!T7VAA;F1L97(S``"W
M`%]C;VYT<F]L9G``````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````!`04``/$%``/____]@E$``5E``
M``````````````````$`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````,``P```"@``(`.````2```@!``
M``!@``"````````````````````"``$```!X``"``@```)```(``````````
M``````````$`90```*@``(````````````````````$``0```,```(``````
M``````````````$``````-@```````````````````````$``````.@`````
M``````````````````$``````/@```````````````````````$```````@!
M``"8LP``:`4``````````````+D``*@(`````````````*C!```B````````
M```````@L0``=`(```````````````````````!T`C0```!6`%,`7P!6`$4`
M4@!3`$D`3P!.`%\`20!.`$8`3P``````O03O_@```0``````````````````
M````/P````$````$``0``0```````````````````-0!```!`%,`=`!R`&D`
M;@!G`$8`:0!L`&4`20!N`&8`;P```+`!```!`#``-``P`#``,``T`&(`,```
M`!@````!`$,`;P!M`&T`90!N`'0`<P```"`````!`$,`;P!M`'``80!N`'D`
M3@!A`&T`90``````*`````$`1@!I`&P`90!$`&4`<P!C`'(`:0!P`'0`:0!O
M`&X``````"`````!`$8`:0!L`&4`5@!E`'(`<P!I`&\`;@``````(`````$`
M20!N`'0`90!R`&X`80!L`$X`80!M`&4````D`````0!,`&4`9P!A`&P`0P!O
M`'``>0!R`&D`9P!H`'0````H`````0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`
M80!R`&L`<P``````*`````$`3P!R`&D`9P!I`&X`80!L`$8`:0!L`&4`;@!A
M`&T`90```"`````!`%``<@!I`'8`80!T`&4`0@!U`&D`;`!D````(`````$`
M4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````D`````0!0`'(`;P!D`'4`8P!T
M`%8`90!R`',`:0!O`&X````@`````0!3`'``90!C`&D`80!L`$(`=0!I`&P`
M9````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````5`!R
M`&$`;@!S`&P`80!T`&D`;P!N````````!+`$`````"@````0````(`````$`
M"`````````$```````````````$````!```&!@8`!`0$``("`@``````0D)"
M`)V=G0!N;FX`@8&!`+&QL0#___\`<W-S`)^?GP!86%@`LK*R`/W]_0!K:VL`
M_/S\`-?7UP`6%A8`VMK:`):6E@".CHX`1D9&`,O+RP`+"PL`H*"@`/[^_@#)
MR<D`P<'!``,#`P`1$1$`DY.3`+.SLP#N[NX`]/3T`./CXP#EY>4`HJ*B`+JZ
MN@"OKZ\`!P<'`'%Q<0"UM;4`]?7U`(F)B0"*BHH`-#0T`.CHZ``U-34`6EI:
M`+>WMP#8V-@`T-#0`!T='0`,#`P`CX^/`+Z^O@!M;6T`"0D)`)N;FP`O+R\`
MS<W-`-G9V0`9&1D`34U-`$1$1`!04%``/#P\`&1D9`#"PL(`,C(R``@("``-
M#0T`#P\/````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````4%!04%!04%!04%!04%!`0$!-4`C!4$Q0T%!1$4N
M`0$!`0$!`04!/!<7`0$3`0$!`0$!`0$5`0$F%0$!#P$!`0$!`0$!$P$1`04U
M`28!`0$!`0$!`2(2$#$/!0$@`0$!`0$!`0$0"B`B%14N(2X!`0$!`1\@(1`B
M$",C)28G`0$!`240$!`0$!`0$!`710$!`4$0$!`0$!`0$!`0$!<!`0$/$!`5
M$!`0$!`0$!`!`0$!$A`0$Q\0$!`0$!`'`0$!`2`0$!`!#Q`0$!`1`0$!`0$'
M(!`*`0$%$!`0,0$!`0$!`0$!`0$!`4$%#P$!`0$!`0$!`0$!`0$!`0$!`0$!
M`8`!``#``0``X`,``/`#``#P`P``\`,``.`#``#``0``@`$``(`!``"``P``
M@`,````'````!P``C@\``/\?```H````(````$`````!``@````````$````
M```````````!`````0````````$!`0`"`@(``P,#``0$!``%!04`!@8&``<'
M!P`("`@`"0D)``H*"@`+"PL`#`P,``T-#0`.#@X`#P\/`!`0$``1$1$`$A(2
M`!,3$P`4%!0`%145`!86%@`7%Q<`&!@8`!D9&0`:&AH`&QL;`!P<'``='1T`
M'AX>`!\?'P`@("``(2$A`"(B(@`C(R,`)"0D`"4E)0`F)B8`)R<G`"@H*``I
M*2D`*BHJ`"LK*P`L+"P`+2TM`"XN+@`O+R\`,#`P`#$Q,0`R,C(`,S,S`#0T
M-``U-34`-C8V`#<W-P`X.#@`.3DY`#HZ.@`[.SL`/#P\`#T]/0`^/CX`/S\_
M`$!`0`!!04$`0D)"`$-#0P!$1$0`145%`$9&1@!'1T<`2$A(`$E)20!*2DH`
M2TM+`$Q,3`!-34T`3DY.`$]/3P!04%``45%1`%)24@!34U,`5%14`%5550!6
M5E8`5U=7`%A86`!965D`6EI:`%M;6P!<7%P`75U=`%Y>7@!?7U\`8&!@`&%A
M80!B8F(`8V-C`&1D9`!E964`9F9F`&=G9P!H:&@`:6EI`&IJ:@!K:VL`;&QL
M`&UM;0!N;FX`;V]O`'!P<`!Q<7$`<G)R`'-S<P!T='0`=75U`'9V=@!W=W<`
M>'AX`'EY>0!Z>GH`>WM[`'Q\?`!]?7T`?GY^`']_?P"`@(``@8&!`(*"@@"#
M@X,`A(2$`(6%A0"&AH8`AX>'`(B(B`")B8D`BHJ*`(N+BP",C(P`C8V-`(Z.
MC@"/CX\`D)"0`)&1D0"2DI(`DY.3`)24E`"5E94`EI:6`)>7EP"8F)@`F9F9
M`)J:F@";FYL`G)R<`)V=G0">GIX`GY^?`*"@H`"AH:$`HJ*B`*.CHP"DI*0`
MI:6E`*:FI@"GIZ<`J*BH`*FIJ0"JJJH`JZNK`*RLK`"MK:T`KJZN`*^OKP"P
ML+``L;&Q`+*RL@"SL[,`M+2T`+6UM0"VMK8`M[>W`+BXN`"YN;D`NKJZ`+N[
MNP"\O+P`O;V]`+Z^O@"_O[\`P,#``,'!P0#"PL(`P\/#`,3$Q`#%Q<4`QL;&
M`,?'QP#(R,@`R<G)`,K*R@#+R\L`S,S,`,W-S0#.SLX`S\_/`-#0T`#1T=$`
MTM+2`-/3TP#4U-0`U=75`-;6U@#7U]<`V-C8`-G9V0#:VMH`V]O;`-S<W`#=
MW=T`WM[>`-_?WP#@X.``X>'A`.+BX@#CX^,`Y.3D`.7EY0#FYN8`Y^?G`.CH
MZ`#IZ>D`ZNKJ`.OKZP#L[.P`[>WM`.[N[@#O[^\`\/#P`/'Q\0#R\O(`\_/S
M`/3T]`#U]?4`]O;V`/?W]P#X^/@`^?GY`/KZ^@#[^_L`_/S\`/W]_0#^_OX`
M____`#0T-#P\/#P\/#P\/#P\/#P\/#P\/#P\/"P,!`0$!`0$7+34U-34U-34
MU-34W-34U-34U-34U-34M#11!`0$!`1DS.ST]/3\]/ST]/3\]/ST_/3\_/ST
M].S$-%%1!`0$!&3$Y.ST]/3T]/3T]/3T]/3T]/3T]/3L[+PT45%1!`0$9,3D
M[.SLY.SL[.3DY.3L[.3LY.SL[.SDO#1145%1!`1DO.3L[*R,K-3$E)2<E(R,
ME)24G)R<[-R\-%%145%1!&2\W.3LY,11E*QDY*1,-"Q<Q,Q1M-3DW+PT45%1
M45%19+S<Y.3LY.11A'3DC&RDG&R,O'2LU.3<O#0$!`0$!`1DO-SDY.SDY+Q1
MC-34C*S,=,S<I%'$Y-R\-`0$!`0$!&2\W.3DY.3DW(Q\G+1\A)QLS.2\=,3D
MW+0T!`0$!`0$7+S<W.3DY.34G*1<E(S,/)1DS+R,4=S4M#0$!`0$!`1DM-3<
MW-SDW(R,K"R\M,Q:Q&2<;)2DW-2T-`0$!`0$!%RTU-S<Y-S<=)343*3TC#S4
M=)1TG$S<S+0T!`0$!`0$9+34W-S<Y-QT4?1TA.S,9.1\;*2T;*34M#0$!`0$
M!`1DM,S<W,Q\C(2T_,S$_/2T_+QDS*R$4;RL-`0$!`0$!%RTS-3,C*3<W.S\
M_/S\_/3\[+3LK'R44:PT!`0$!`0$7*S,S&R<]/S\_/S\_/S\_/S\_/S$E(Q1
MK#0$!`0$!`1<K,2\?.3\_/S\_/S\_/S\_/S\_.S,K%&L-`0$!`0$!%RLM$S$
M_/S\[/3\_/S\_/S\_/S\]+Q1Q*PT!`0$!`0$7*RT3-3\_/2TW/S\_/S\_/S\
M_/S,4<S$I#0$!`0$!`1<K,2D4?S\_+2<]/S\_/S\_/S\])11S,2D+`0$!`0$
M!%RDI%&$[/S\S&2T[/S\_/S\_/S4;*S,O*0T!`0$!`0$7*1D=+ST_/S$482<
MY/S\_/S\Y(Q1S,S$I#0$!`0$!`1<I&R,Y/3\_)Q1S%&4[/S\_/R\=,S,Q+R<
M+`0$!`0$!%RDK&1<9*2D4:S,Q'24Y/3\Y%&\S,S$O)PT!`0$!`0$7*2TO+R,
M45'$Q,S,K%%\?)1L?+S,Q,2\G"P$!`0$!`1<G+S$Q,3$Q,3$Q,3$Q%%145&\
MQ,3$Q+2<-`0$!`0$!%2<M,3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,2\M)PL!`0$!`0$
M5)R\Q,2\O+R\O+R\O+R\O+R\O+R\O+RTG"P$!`0$!`1<G+3$O+2TM+2TM+2T
MM+2TM+2TM+2TM*R<+`0$!`0$!%2$E)R<E)24E)24E)24E)24E)24E)24E(0D
M!`0$!`0$1%1<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<5#0$!`0$!`0````_````
M'P````\````'`````P````$`````````/P```#\````_````/P```#\````_
M````/P```#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_````/P```#\````_````/P```#\````_````/P```0`"`!`0```!``@`
M:`4```$`("````$`"`"H"````@``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````$Y",3``````4,S?10$```!#.EQ496UP7%!!4BU086-K
797(M,"XY-S-<;7EL9')<<&%R+G!D8@``
