#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

import re

class MetadataError(Exception):
    pass
    
class NoSeriesName(MetadataError):
    pass    

class NoSeriesIndex(MetadataError):
    pass    

class NoBookTitle(MetadataError):
    pass    
   
class NoValidMajorIndex(MetadataError):
    pass    

class NoValidMinorIndex(MetadataError):
    pass    
      
class metadata:
    
    def __init__(self, bk):
        self.bk = bk
        self.metadata = self.bk.getmetadataxml()
        self.ser_name = None
        self.ser_index = None
        self.booktitle = None
        self.booktitle_original = None;
        self.title_found = False
        
        self.out_trace = False
        self.out_debug = False
        
        self.debug(self.metadata)
        
    def debug(self, p):
        if self.out_debug:
            print(p)   
                 
    def trace(self, ttype, text, tagprefix, tagname, tagtype, tagattr):
        if self.out_trace == True:
            if ttype == 1:
                print("--- text: -{}-\n".format(text))
            else:  
                if tagtype == "begin":
                    print(">>> tagprefix: -{}-".format(tagprefix))
                    print(">>> tagname:   -{}-".format(tagname))
                    print(">>> tagattr:   -{}-\n".format(tagattr))
                if tagtype == "single":
                    print("+++ tagprefix: -{}-".format(tagprefix))
                    print("+++ tagname:   -{}-".format(tagname))
                    print("+++ tagattr:   -{}-\n".format(tagattr))
                if tagtype == "end":
                    print("<<< tagname:   -{}-\n".format(tagname))   
                    
    def scan(self):
        ps = self.bk.qp
        ps.setContent(self.metadata)
        for text, tagprefix, tagname, tagtype, tagattr in ps.parse_iter():
            if text is not None:
                if text != "\n":
                    self.trace(1, text, None, None, None, None)
                    if self.title_found:
                        self.title_found = False
                        self.booktitle = text
            else:
                self.trace(2, None, tagprefix, tagname, tagtype, tagattr)
                
                if tagname == 'meta' and 'name' in tagattr and tagattr['name'] == 'calibre:series' and 'content' in tagattr:
                    self.ser_name = tagattr['content']
                if tagname == 'meta' and 'name' in tagattr and tagattr['name'] == 'calibre:series_index' and 'content' in tagattr:
                    self.ser_index = tagattr['content']
                if tagname == 'meta' and 'name' in tagattr and tagattr['name'] == 'originaltitle' and 'content' in tagattr:
                    self.booktitle_original = tagattr['content']        
                if tagname == 'dc:title' and tagtype == 'begin':
                    self.title_found = True

        if self.ser_name == None:
            raise NoSeriesName
        if self.ser_index == None:
            raise NoSeriesIndex 
        if self.booktitle == None:
            raise NoBookTitle       
              
    def create_new_title(self):
        if self.ser_index.find('.') == -1:
            try:
                self.ser_index = int(self.ser_index)
            except ValueError:
                raise NoValidMajorIndex    
            self.ser_index_str = '({:02d})'.format(self.ser_index)
        else:
            (self.ser_index_major, self.ser_index_minor) = self.ser_index.split(".")
            try:
                self.ser_index_major = int(self.ser_index_major)
            except ValueError:
                raise NoValidMajorIndex
            try:        
                self.ser_index_minor = int(self.ser_index_minor)
            except ValueError:
                raise NoValidMinorIndex    
            if self.ser_index_minor == 0:
                self.ser_index_str = '({:02d})'.format(self.ser_index_major)
            else:    
                self.ser_index_str = '({:02d}.{:02d})'.format(self.ser_index_major, self.ser_index_minor)

        if self.booktitle_original != None:
            self.newtitle = self.booktitle_original
        else:
            self.newtitle = self.booktitle  

        l = len(self.ser_name)
        if ( len(self.newtitle) > l ) and ( self.ser_name == self.newtitle[0:l]):
            self.newtitle  = '{} {} {}'.format(self.ser_name, self.ser_index_str, self.newtitle[l:].strip())
        else:        
            self.newtitle  = '{} {} - {}'.format(self.ser_name, self.ser_index_str, self.newtitle) 
          
        self.debug('>>> Attributes ---')
        self.debug('Booktitle: -{}-'.format(self.booktitle))
        self.debug('Series name: -{}-'.format(self.ser_name))
        self.debug('Series name len: -{}-'.format(l))
        self.debug('Series index: -{}-'.format(self.ser_index))
        self.debug('Series string: -{}-'.format(self.ser_index_str))
        self.debug('Newtitle: -{}-'.format(self.newtitle))
        self.debug('--- Attributes <<<\n')    
            
    def modify(self):
        self.metadata = self.metadata.replace("<dc:title>"+self.booktitle+"</dc:title>", "<dc:title>"+self.newtitle+"</dc:title>")
        if self.booktitle_original == None:
            self.metadata = self.metadata.replace("</metadata>", '<meta content="'+self.booktitle+'" name="originaltitle" />\n</metadata>')
            
        self.debug(self.metadata)
        
    def write(self):
        self.bk.setmetadataxml(self.metadata)        
             
def run(bk):
    try:
        xml = metadata(bk)
        xml.scan()
        xml.create_new_title()
        xml.modify()
        xml.write()
    except NoSeriesName:
        print("No series name information")
        return -1
    except NoSeriesIndex:
        print("No series index information")
        return -1  
    except NoBookTitle:
        print("No booktitle")
        return -1  
    except NoValidMajorIndex:
        print('No valid major index')
        return -1          
    except NoValidMinorIndex:
        print('No valid minor index')
        return -1          
              
    return 0

def main():
    print ('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
