#!/usr/bin/perl -w
use strict;

#####################################################################
# Script to generate a valid atom feed for an online Stanza catlog
# using any epub files in your calibre database.  Resulting atom feed
# XML file is placed at the root of the calibre library.  If you
# mount your calibre library directory in your webserver's directory
# tree you can then access all the epubs in your library
#####################################################################

#####################################################################
# Config Section - Please change to your appropriate settings
#####################################################################
my $libxml = "stanza.xml";
my $libdir = "";
my $webserver_url ="http://www.mywebsite.com/path to/where you/mounted calibre library root directory/";
my $calibre_dir = "C:\\Progra~1\\calibre\\";
my $library_title = "My Library";
my $librarian_name = "Me";
my $librarian_email = "librarian\@mywebsite.com";
#####################################################################

my $separator="XXX";
my $cmdline_tool = "calibredb.exe";
my $option1 ="list";
my $option2="--fields=title,authors,series,series_index,tags,path,size,timestamp";
my $option3 = "--separator=$separator";
my $option4 = "--line-width=2000";

# Strip spaces from URL pieces which would break atom feed
$libxml =~ s/ /%20/;
$webserver_url =~ s/ /%20/;

#Stanza Atom Templates
my $FEED_TEMPLATE = "<feed xmlns=\"http://www.w3.org/2005/Atom\" xmlns:dc=\"http://purl.org/dc/elements/1.1/\">
  <title>$library_title</title>
  <id>_LIBRARY_ID_</id>
  <updated>_LIBRARY_UPDATED_</updated>
  <link rel=\"self\" type=\"application/atom+xml\" href=\"$webserver_url$libxml\"/>
  <author>
    <name>$librarian_name</name>
    <uri>$webserver_url</uri>
    <email>$librarian_email</email>
  </author>
_ENTRIES_
</feed>
";

my $ENTRY_TEMPLATE = "<entry>
    <title>_TITLE_</title>
    <content type=\"xhtml\">
      <div xmlns=\"http://www.w3.org/1999/xhtml\" align=\"center\">_INFO_</div>
    </content>
    <id>_ID_</id>
    <author>
      <name>_AUTHOR_</name>
    </author>
    <updated>_UPDATED_</updated>
    <link type=\"application/epub+zip\" href=\"_BOOK_LINK_\"/>
    <link rel=\"x-stanza-cover-image-thumbnail\" type=\"image/jpg\" href=\"_COVER_LINK_\"/>
    <link rel=\"x-stanza-cover-image\" type=\"image/jpg\" href=\"_COVER_LINK_\"/>
</entry>
";

use constant ID => 0;
use constant TITLE => 1;
use constant AUTHOR => 2;
use constant SERIES => 3;
use constant SERIES_INDEX => 4;
use constant TAGS => 5;
use constant PATH => 6;
use constant SIZE => 7;
use constant TIMESTAMP => 8;

my @book_info = ();
my %library = ();

sub parse_library()
{
    # Dump the interestign info from the Calibre DB and stick it in a hash for later processing
    open (CALIBREDBDUMP, "$calibre_dir$cmdline_tool $option1 $option2 $option3 $option4 |");
    my $epubs = 0;
    my $nonepubs = 0;
    my $i=1;
    while (<CALIBREDBDUMP>)
    {        
        if ( $i > 2 )
        {
            #First couple of lines of caibredb list output are not books 
            my $rawbook = $_;
            $rawbook =~ s/XXX$//;
            $rawbook =~ s/\n//;
            @book_info = split(/XXX/, $rawbook);
            if ( $book_info[PATH] =~ m/epub/)
            {
                # Only put book in hash if we have an epub version
                $book_info[PATH] =~ s/(\[epub,zip\]|\[zip\]|\[epub\])/epub/g;
                $book_info[PATH] =~ s/\s+$//;
                $book_info[PATH] =~ s/ /%20/g;
                $book_info[PATH] =~ s/\\/\//g;
                $book_info[TITLE] =~ s/\s+$//;
                $book_info[AUTHOR] =~ s/\s+$//;
                $book_info[SERIES] =~ s/\s+$//;
                $book_info[TIMESTAMP] =~ s/ /T/;
                $book_info[TIMESTAMP] = $book_info[TIMESTAMP]."+00:00";
                $book_info[TAGS] =~ s/\s+$//;
                print "$book_info[TITLE]\t$book_info[AUTHOR]\t$book_info[SIZE] bytes\n";
                $library{$book_info[ID]} = { Title => $book_info[TITLE],
                                    Author => $book_info[AUTHOR],
                                    Series => $book_info[SERIES],
                                    Series_Index => $book_info[SERIES_INDEX],
                                    Tags => $book_info[TAGS],
                                    Path => $book_info[PATH],
                                    Size => $book_info[SIZE],
                                    Date => $book_info[TIMESTAMP]};
                $epubs++;
            } else
            {
                print "$book_info[TITLE]\t** No Stanza compatible Epub version **";
                $nonepubs++;
            }
        } elsif ( $i == 1 )
        {
            $libdir = $_;
            $libdir =~ s/^Using library at //;
            $libdir =~ s/\s+$//;
            $libdir = "$libdir\\";
        }
        $i++;
    }
    close CALIBREDBDUMP;
    my $totalbooks = $nonepubs + $epubs;
    print "\n\nDetected $epubs Epub books in library of $totalbooks books";
}

sub generate_id
{
    # Generate valid tags using path and date using this guide:
    # http://diveintomark.org/archives/2004/05/28/howto-atom-id
    # Example:tag:mywebserver,2008-10-12:/path%20to/where%20you/mounted%20calibre%20library%20root%20directory/H%20G%20Wells/The%20Time%20Machine%20(8)/The%20Time%20Machine%20-%20H%20G%20Wells.epub
    my($id, $date) = @_;
    my $id_tag = $webserver_url;
    $id_tag =~ s/http:\/\///;
    $id_tag =~ s/\//,$date:\//;
    $id_tag = "tag:$id_tag$id";
    return $id_tag;
}

sub generate_updated
{
    # Generate valid atom datestamp for atom feed
    # Example: 2008-10-12T00:44:20+00:00
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
    if ($isdst == 0)
    {
        $isdst = "+00:00";
    } else
    {
        $isdst = "+01:00";
    }
    my $date = sprintf "%4d-%02d-%02dT%02d:%02d:%02d%s",$year+1900,$mon+1,$mday,$hour,$min,$sec,$isdst;
    return $date;
}

sub generate_xml
{
    my $libfile = $FEED_TEMPLATE;
    my $update_stamp = generate_updated();
    $libfile =~ s/_LIBRARY_UPDATED_/$update_stamp/g;
    my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
    my $date = sprintf"%4d-%02d-%02d",$year+1900,$mon+1,$mday;
    my $library_id = generate_id($libxml,$date);
    $libfile =~ s/_LIBRARY_ID_/$library_id/g;
    
    my $entries = "";

    while ( my ($key) = each(%library) )
    {
        # Add an Entry to the Atom feed for each epub we discovered in our library.
        # Info is in the library hash.
        my $bookentry = $ENTRY_TEMPLATE;
        $bookentry =~ s/_TITLE_/$library{$key}{Title}/;
        
        # Add any available interesting book info to the content section
        my $bookinfo = "";
        if ( $library{$key}{Tags} !~ m/None/ )
        {
            $bookinfo = $bookinfo."Subject: $library{$key}{Tags}<br \/>";
        }
        if ( $library{$key}{Series} !~ m/None/ )
        {
            $bookinfo = $bookinfo."Series: $library{$key}{Series} $library{$key}{Series_Index}<br \/>";
        }
        $bookinfo = $bookinfo."Size: $library{$key}{Size} bytes<br \/>";
        $bookentry =~ s/_INFO_/$bookinfo/;
        
        # Entry dates are based on their calibre timestamps
        my $date = $library{$key}{Date};
        $date =~ s/T.*$//;
        my $id = generate_id($library{$key}{Path},$date);
        $bookentry =~ s/_ID_/$id/;
        $bookentry =~ s/_AUTHOR_/$library{$key}{Author}/;
        $bookentry =~ s/_UPDATED_/$library{$key}{Date}/;
        $bookentry =~ s/_BOOK_LINK_/$webserver_url$library{$key}{Path}/;
        my $coverpath = "${webserver_url}$library{$key}{Path}";
        my $filename = $coverpath;
        $filename =~ s/^.*\///;
        $coverpath =~ s/$filename/cover\.jpg/;
        
        # The thumb and regular image can both use the same path to the cover.jpg
        $bookentry =~ s/_COVER_LINK_/$coverpath/g;
        $entries = "$entries$bookentry";
    }
    $libfile =~ s/_ENTRIES_/$entries/;
    #Write the final XML file to the library location
    my $libfilename = "$libdir$libxml";
    open( LIBFILE, ">$libfilename" ) or die( "Failed to open lib file." );
    printf LIBFILE ("%s", $libfile);
    
}

# Main
print "Books In Library:\n\n";
parse_library;
print "\n\nGenerating Stanza XML file: $libdir$libxml\n";
generate_xml;
print "\nDone!";