#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

import os, os.path, sys, shutil, inspect, re
import options
from PIL import Image
from decimal import *  
import tkinter as tk
import tkinter.messagebox as mbox

try:
    from sigil_bs4 import BeautifulSoup, Comment, Tag
except:
    from bs4 import BeautifulSoup, Comment, Tag
    
    
__all__= ["processAllTasks", "copyTextFiles2Dir", "writeFiles2Epub", "show_msgbox", "cleanExit", "prettifyXHTMLFile", "svgAttributes2CamelCase", "addSelectedHeadings", "removeSelectedHeadings", "checkInputFile", "addSelectedHeading"]

def processAllTasks(bk, wdir, selected_files):
    print('\n -- Processing automatic tasks...')
            
    # copy the epub xhtml files to the working dir
    t_ids, t_fnames = copyTextFiles2Dir(bk, wdir)   
    
    count = 0
    for fname in selected_files:
        print('\n -- Processing next file...' + fname + '\n')
        file = os.path.join(wdir, fname)
        removeSelectedHeadings(wdir, file)
        
        # add a single selected file
        if len(selected_files) == 1:
            addSelectedHeading(wdir, file)  
        elif len(selected_files) > 1:
            # add & reformat a selected group of files    
            count = addSelectedHeadings(wdir, file, count)
            
        # exit on previous error
        if options.SYS_EXIT == True:
            return(0)
        
        # fixes bs4's tendency to convert  
        # all svg attributes to lower case
        svgAttributes2CamelCase(wdir, file)
    
    for filename in t_fnames:
       filename = os.path.join(wdir, filename)
       prettifyXHTMLFile(wdir, filename) 
        
    # write files back to the epub         
    writeFiles2Epub(bk, wdir, t_ids, t_fnames)
    
    return(0)
    
def copyTextFiles2Dir(bk, wdir):

    print('\n -- In copyTextFiles2Dir()...')
    t_ids = list()
    t_hrefs = list()
    t_fnames = list()
    
    for (id, href) in bk.text_iter():
        t_ids.append(id)
        t_hrefs.append(href)
        t_fnames.append(os.path.basename(href))
    
    # copy all xhtml files to the working dir    
    file = str()
    t_fnames_r = list()
    t_ids_r = list()
    i = 0      
    for id in t_ids:
        file = os.path.join(wdir, t_fnames[i])
        print(file)
        with open(file, 'wt', encoding='utf-8') as outfp:
            data = bk.readfile(id)
            t_fnames_r.append(t_fnames[i])
            t_ids_r.append(id)
            outfp.writelines(data)
            i = i + 1
    
    return(t_ids_r, t_fnames_r)     
                
    
def writeFiles2Epub(bk, wdir, ids, fnames):

    print('\n -- In writeFiles2Epub()...' )       
    for file in fnames:    
       print(file)    

    count = len(fnames)
    if count == 0: 
        return(0)
    
    i = 0
    for file in fnames:
        file = os.path.join(wdir, file)
        with open(file, 'rt', encoding='utf-8') as fp:
            data = fp.read()           
            bk.writefile(ids[i], data)
            i = i + 1
    
    return(0)        
            
    
def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
 
def cleanExit(wdir):
    shutil.rmtree(wdir, ignore_errors=True)
    return(0)
    
def prettifyXHTMLFile(wdir, file):
    """ Prettifies the html """
    
    # reformat and prettify the XHTML file
    outfile= os.path.join(wdir, 'final_one.css')
    outfp = open(outfile, 'wt', encoding='utf-8') 
    infp = open(file, 'rt', encoding='utf-8')
    
    for line in infp:
            
        # prettify the html file
        if line.strip().startswith('<?xml') or \
            line.strip().startswith('<!DOCTYPE') or \
            line.strip().startswith('<html') or \
            line.strip().startswith('<head>') or \
            line.strip().startswith('<meta')or \
            line.strip().startswith('<title>') or \
            line.strip().startswith('<link') or \
            line.strip().startswith('</head>') or \
            line.strip().startswith('<body') or \
            line.strip().startswith('<body>'):
            line = line.strip()
            if not line:
                continue
            if line.startswith('<meta') or \
                line.startswith('<title>') or \
                line.startswith('<link'):
                line = '  ' + line      
            if line.startswith('<body'):
                line = '\n' + line
            if line.startswith('</body>'):
                outfp.write('\n' + line.rstrip() + '\n')
            else:
                outfp.write(line.rstrip() + '\n')   
        else:
            line = line.strip() 
            if not line or line == ';':
                continue
            if line.strip().startswith('<img'):
                line = line.strip().replace('/> <img', '/>\n\n<img')   
                line = line.strip().replace('<img', '  <img')
            if '<image' in line:
                line = line.strip().replace('<image', '\n  <image')                
            if line.strip().startswith('<p'):
                line = '  ' + line.strip()           
            outfp.write('\n' + line + '\n')    
            
    infp.close()
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)
    
    # reformat svg tags
    outfile= os.path.join(wdir, 'final.css')
    outfp = open(outfile, 'wt', encoding='utf-8') 
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
            if line.strip().startswith('<svg'):
                line = line.replace('<svg', '  <svg')
            outfp.write(line)
            
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)    
    return(0)                 
                  
def svgAttributes2CamelCase(wdir, file):
    """ I wrote this function because of the problems and frustrations caused by 
        bs4 automatically converting all svg attributes to lowercase, causing 
        various epub svg errors. I also tried using bs4's "xml" parser as a remedy 
        but doing this gave me an even bigger mess. So, in the end, I wrote 
        this simple function which just converts certain essential svg attributes 
        back to proper camel case.
    """       
    
    print(' -- svgAttributes2CamelCase()...')
    #file = os.path.join(wdir, file)
    #file = os.path.join(wdir, os.path.basename(file))       
    output = os.path.join(wdir, 'reformat.html')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()

    soup = BeautifulSoup(html, 'html.parser')
    
    for svg in soup.find_all('svg'):
    
        # reformat the par to camel case
        if svg.has_attr('preserveaspectratio'):
            par = svg['preserveaspectratio']
            del svg['preserveaspectratio']
            svg['preserveAspectRatio'] = par
        
        # reformat the vb to camel case
        if svg.has_attr('viewbox'):
            vb = svg['viewbox']
            del svg['viewbox']
            svg['viewBox'] = vb
            
        if not svg.has_attr('xmlns:xlink'):
            svg['xmlns:xlink'] = "http://www.w3.org/1999/xlink"        
               
    outfp.writelines(str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
    outfp.close()
    os.remove(file)
    os.rename(output, file)        
    return(0)          
    
def addSelectedHeadings(wdir, file, count):
                    
    print('\n>>> In addSelectedHeadings...')                    
                    
    output = os.path.join(wdir, 'out.xhtml')
    outfp = open(output, 'wt', encoding='utf-8')    
    with open(file, 'rt', encoding='utf-8') as infp:    
    
        for line in infp:    
        
            if '<body' in line:
                
                ### create and add the new chapter heading line ###
                
                original_line = line
                new_line = str()
            
                # select the relevant heading style
                if options.HEADING_STYLE_CHOICE == 'Heading 1':
                    heading = 'h1' 
                elif options.HEADING_STYLE_CHOICE == 'Heading 2':
                    heading = 'h2'  
                elif options.HEADING_STYLE_CHOICE == 'Heading 3':
                    heading = 'h3'             
                elif options.HEADING_STYLE_CHOICE == 'Heading 4':
                    heading = 'h4' 
                elif options.HEADING_STYLE_CHOICE == 'Heading 5':
                    heading = 'h5' 
                elif options.HEADING_STYLE_CHOICE == 'Heading 6':
                    heading = 'h6'                              
                
                # Add roman numeral numbering
                if options.NUM_STYLE_CHOICE == 'Roman Numerals':
                    if count > 99:
                        msg = 'You are only allowed to use a maximum of 100 chapters with roman numeral numbering ' + \
                              'using this plugin. The number of your epub chapters has exceeded this limit.'
                        print('\n>>> Overrun Error: ' + msg)
                        show_msgbox('Overrun Error', msg, msgtype='error')
                        options.SYS_EXIT = True
                        return(0)                         
                    else: # add the roman numerals from an ascending list of numerals     
                        new_line = '<' + heading + '>' + options.SUFFIX_CHOICE + ' ' + options.NUMERALS[count] + '</' + heading + '>\n'
                
                # Add chapter number as text
                elif options.NUM_STYLE_CHOICE == 'Number as Text':
                    if count > 99:
                        msg = 'You are only allowed to use a maximum of 100 chapters with number as text headings ' + \
                              'using this plugin. The number of your epub chapters has exceeded this limit.'
                        print('\n>>> Overrun Error: ' + msg)
                        show_msgbox('Overrun Error', msg, msgtype='error')
                        options.SYS_EXIT = True
                        return(0)                     
                    
                    # format chapter numbers as text from a list in titlecase, uppercase or lowercase
                    if options.WORD_CASE_CHOICE == 'titlecase':
                        new_line = '<' + heading + '>' + options.SUFFIX_CHOICE + ' ' + options.WORD_NUMBERS[count].title() + '</' + \
                               heading + '>\n'                    
                    elif options.WORD_CASE_CHOICE == 'uppercase':
                        new_line = '<' + heading + '>' + options.SUFFIX_CHOICE + ' ' + options.WORD_NUMBERS[count].upper() + '</' + \
                               heading + '>\n'
                    elif options.WORD_CASE_CHOICE  == 'lowercase':
                        new_line = '<' + heading + '>' + options.SUFFIX_CHOICE + ' ' + options.WORD_NUMBERS[count].lower() + '</' + \
                               heading + '>\n'                     
                
                else: # Add a simple numeric string chapter heading using a count
                    new_line = '<' + heading + '>' + options.SUFFIX_CHOICE + ' ' + str(count+1) + '</' + heading + '>\n'
                                        
                # hide chapter heading option                        
                if options.HIDE_HEADING_CHOICE == 'Yes':
                    soup = BeautifulSoup(new_line, 'html.parser')  
                    
                    # make h1 invisible handler
                    heading1 = soup.find('h1')
                    if heading1 != None and heading1.name == 'h1':
                        heading1['style'] = "display:none"   # hide the chapter string
                       
                    # make h2 invisible handler    
                    heading2 = soup.find('h2')    
                    if heading2 != None and heading2.name == 'h2':
                        heading2['style'] = "display:none"    # hide the chapter string
                        
                    # make h3 invisible handler    
                    heading3 = soup.find('h3')    
                    if heading3 != None and heading3.name == 'h3':
                        heading3['style'] = "display:none"    # hide the chapter string    
                        
                    # make h3 invisible handler    
                    heading4 = soup.find('h4')    
                    if heading4 != None and heading4.name == 'h4':
                        heading4['style'] = "display:none"    # hide the chapter string        
                        
                    # make h3 invisible handler    
                    heading5 = soup.find('h3')    
                    if heading5 != None and heading5.name == 'h5':
                        heading5['style'] = "display:none"    # hide the chapter string        
                        
                    # make h3 invisible handler    
                    heading6 = soup.find('h6')    
                    if heading6 != None and heading6.name == 'h6':
                        heading6['style'] = "display:none"    # hide the chapter string        
                        
                    new_line = str(soup)    

                # add classes to headings if required
                html = BeautifulSoup(new_line, 'html.parser')
                if len(options.ADD_CLASSES) != 0:
                
                    if heading == 'h1':
                        h1 = html.find('h1')
                        h1['class'] = h1.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)
                    
                    if heading == 'h2':
                        h2 = html.find('h2')
                        h2['class'] = h2.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)
                    
                    if heading == 'h3':
                        h3 = html.find('h3')
                        h3['class'] = h3.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)    
                        
                    if heading == 'h4':
                        h4 = html.find('h4')
                        h4['class'] = h4.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)    

                    if heading == 'h5':
                        h5 = html.find('h5')
                        h5['class'] = h5.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)    

                    if heading == 'h6':
                        h6 = html.find('h6')
                        h6['class'] = h6.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)                            
                        
                # create numbered headings only
                if options.NUM_HEADINGS_CHOICE  == 'Yes':
                    data = BeautifulSoup(new_line, 'html.parser')
                    chap = data.get_text()
                    
                    if 'chapter' in chap.lower():
                        if heading == 'h1':
                            data.h1.string = chap.replace('Chapter', '').replace('CHAPTER', '').replace('chapter', '').upper().strip()
                        elif heading == 'h2':
                            data.h2.string = chap.replace('Chapter', '').replace('CHAPTER', '').replace('chapter', '').upper().strip()
                        elif heading == 'h3':
                            data.h3.string = chap.replace('Chapter', '').replace('CHAPTER', '').replace('chapter', '').upper().strip()
                        elif heading == 'h4':
                            data.h4.string = chap.replace('Chapter', '').replace('CHAPTER', '').replace('chapter', '').upper().strip()
                        elif heading == 'h5':
                            data.h5.string = chap.replace('Chapter', '').replace('CHAPTER', '').replace('chapter', '').upper().strip()
                        elif heading == 'h6':
                            data.h6.string = chap.replace('Chapter', '').replace('CHAPTER', '').replace('chapter', '').upper().strip()                         
                        new_line = str(data)
                                 
                # add the new chapter heading line after the body
                # tag line at the top of the selected xhtml file
                outfp.write(original_line + new_line)   
                count += 1
            
            else:              
                outfp.write(line)
                        
    outfp.close()
    os.remove(file)
    os.rename(output, file)    
    return(count)  
    
def removeSelectedHeadings(wdir, file):
     
    if options.REMOVE_HEADING_CHOICE == '':
        return(0)
    
    print('\n>>> In removeSelectedHeadings...' + options.REMOVE_HEADING_CHOICE)    
     
    output = os.path.join(wdir, 'remove_headings.xhtml')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()
    
    soup = BeautifulSoup(html, 'html.parser')
    
    if options.REMOVE_HEADING_CHOICE == 'Heading 1':
        for h1 in soup.find_all('h1'):
            h1.attrs = {}
            h1.decompose()
            
    if options.REMOVE_HEADING_CHOICE == 'Heading 2':
        for h2 in soup.find_all('h2'):
            h2.attrs = {}
            h2.decompose()    
            
    if options.REMOVE_HEADING_CHOICE == 'Heading 3':
        for h3 in soup.find_all('h3'):
            h3.attrs = {}
            h3.decompose()        
            
    if options.REMOVE_HEADING_CHOICE == 'Heading 4':
        for h4 in soup.find_all('h4'):
            h4.attrs = {}
            h4.decompose()
            
    if options.REMOVE_HEADING_CHOICE == 'Heading 5':
        for h5 in soup.find_all('h5'):
            h5.attrs = {}
            h5.decompose()    
            
    if options.REMOVE_HEADING_CHOICE == 'Heading 6':
        for h6 in soup.find_all('h6'):
            h6.attrs = {}
            h6.decompose()                
            
    outfp.writelines(str(soup))
    outfp.close()
    os.remove(file)
    os.rename(output, file)
    return(0)
    
def checkInputFile(bk):
    
    # get all xhtml filenames    
    text_files = list()    
    for href, id in bk.text_iter():
        ref = bk.id_to_href(id)
        basename = bk.href_to_basename(ref)
        text_files.append(basename)    
        
    # get all css filenames    
    css_files = list()    
    for _, cid in bk.css_iter():
        ref = bk.id_to_href(cid)
        basename = bk.href_to_basename(ref)
        css_files.append(basename)      
        
    # check that the file is an epub
    if len(text_files) == 1 and len(css_files) == 0:  
        msg = 'The epub is either invalid or unusable. Please try again.'
        print("\n>>> Error: " + msg)
        show_msgbox('Invalid Epub', msg, msgtype='error')
        options.SYS_EXIT = True
        return(0)    

    return(0)        
    
def addSelectedHeading(wdir, file):
                    
    print('\n>>> In addSelectedFileHeading...')                    
                    
    output = os.path.join(wdir, 'out.xhtml')
    outfp = open(output, 'wt', encoding='utf-8')    
    with open(file, 'rt', encoding='utf-8') as infp:    
    
        for line in infp:    
        
            if '<body' in line:
                
                ### create and add the new chapter heading line ###
                
                original_line = line
                new_line = str()
            
                # select the relevant heading style
                if options.HEADING_STYLE_CHOICE == 'Heading 1':
                    heading = 'h1'                   
                elif options.HEADING_STYLE_CHOICE == 'Heading 2':
                    heading = 'h2'                 
                elif options.HEADING_STYLE_CHOICE == 'Heading 3':
                    heading = 'h3'             
                elif options.HEADING_STYLE_CHOICE == 'Heading 4':
                    heading = 'h4' 
                elif options.HEADING_STYLE_CHOICE == 'Heading 5':
                    heading = 'h5' 
                elif options.HEADING_STYLE_CHOICE == 'Heading 6':
                    heading = 'h6'    

                #add the heading text name to the heading line
                new_line = '<' + heading + '>' + options.SUFFIX_CHOICE + '</' + heading + '>\n'                         
                
                # hide chapter heading option                        
                if options.HIDE_HEADING_CHOICE == 'Yes':
                    soup = BeautifulSoup(new_line, 'html.parser')  
                    
                    # make h1 invisible handler
                    if heading == 'h1':
                        heading1 = soup.find('h1')    
                        heading1['style'] = "display:none"   # hide the chapter string
                       
                    # make h2 invisible handler    
                    if heading == 'h2':
                        heading2 = soup.find('h2')    
                        heading2['style'] = "display:none"   # hide the chapter string
                        
                    # make h3 invisible handler    
                    if heading == 'h3':
                        heading3 = soup.find('h3')    
                        heading3['style'] = "display:none"   # hide the chapter string
                        
                    # make h3 invisible handler    
                    if heading == 'h4':
                        heading4 = soup.find('h4')    
                        heading4['style'] = "display:none"   # hide the chapter string
                        
                    # make h3 invisible handler    
                    if heading == 'h5':
                        heading5 = soup.find('h5')    
                        heading5['style'] = "display:none"   # hide the chapter string
                        
                    # make h3 invisible handler    
                    if heading == 'h6':
                        heading6 = soup.find('h6')    
                        heading6['style'] = "display:none"   # hide the chapter string
                        
                    new_line = str(soup)  
                    
                # add classes to headings if required
                html = BeautifulSoup(new_line, 'html.parser')
                if len(options.ADD_CLASSES) != 0:
                
                    if heading == 'h1':
                        h1 = html.find('h1')
                        h1['class'] = h1.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)
                    
                    if heading == 'h2':
                        h2 = html.find('h2')
                        h2['class'] = h2.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)
                    
                    if heading == 'h3':
                        h3 = html.find('h3')
                        h3['class'] = h3.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)    
                        
                    if heading == 'h4':
                        h4 = html.find('h4')
                        h4['class'] = h4.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)    

                    if heading == 'h5':
                        h5 = html.find('h5')
                        h5['class'] = h5.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)    

                    if heading == 'h6':
                        h6 = html.find('h6')
                        h6['class'] = h6.get('class', []) + options.ADD_CLASSES
                        new_line = str(html)                            
                                 
                # add the new chapter heading line after the body
                # tag line at the top of the selected xhtml file
                outfp.write(original_line + new_line)   
            
            else:              
                outfp.write(line)
                        
    outfp.close()
    os.remove(file)
    os.rename(output, file)    
    return(0)    
    
