(* $Id: thoPDF.ml 40 2008-10-19 22:29:30Z ohl $ *)
(* Copyright (C) 2008 by Thorsten Ohl <ohl@physik.uni-wuerzburg.de>

   This is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  *)  

type op = Pdfpages.operator

let rgb r g b =
  Pdfpages.Op_RG (r, g, b)

let push_graphics_state =
  Pdfpages.Op_q

let pop_graphics_state =
  Pdfpages.Op_Q

let red = rgb 1.0 0.0 0.0

let line_width width =
  Pdfpages.Op_w width

type line_cap_style =
  | Butt_Cap
  | Round_Cap
  | Projecting_Cap

let line_cap_style style =
  Pdfpages.Op_J
    (match style with
    | Butt_Cap -> 0
    | Round_Cap -> 1
    | Projecting_Cap -> 2)

type line_join_style =
  | Miter_Join
  | Round_Join
  | Bevel_Join

let line_join_style style =
  Pdfpages.Op_j 
    (match style with
    | Miter_Join -> 0
    | Round_Join -> 1
    | Bevel_Join -> 2)

let stroke_path = function
  | [] -> []
  | (x, y) :: ps ->
      Pdfpages.Op_m (x, y) :: List.fold_right
			       (fun (x, y) ops -> Pdfpages.Op_l (x, y) :: ops) ps [Pdfpages.Op_S]
  
let doc_of_pages pages =
  let pagetree, pageroot = Pdfdoc.add_pagetree pages Pdf.empty in
  let doc = Pdfdoc.add_root pageroot [] pagetree in
  doc

let enclose_page_with_streams pre page post =
  let streams =
    [Pdfpages.stream_of_ops pre] @ page.Pdfdoc.content @ [Pdfpages.stream_of_ops post] in
  { page with Pdfdoc.content = streams }

let enclose_page_with_ops pre page post =
  let safe_pre =
    match pre with 
    | [] -> [push_graphics_state]
    | ops -> [push_graphics_state] @ ops @ [pop_graphics_state; push_graphics_state]
  and safe_post =
    match post with
    | [] -> [pop_graphics_state]
    | ops -> [pop_graphics_state; push_graphics_state] @ ops @ [pop_graphics_state] in
  enclose_page_with_streams safe_pre page safe_post

let float_of_pdfobject = function
  | Pdf.Real x -> x
  | Pdf.Integer i -> float i
  | _ -> invalid_arg "float_of_pdfobject"

type rectangle =
    { ll_x : float;
      ll_y : float;
      ur_x : float;
      ur_y : float }

let rectangle = function
  | Pdf.Array [ll_x; ll_y; ur_x; ur_y] ->
      { ll_x = float_of_pdfobject ll_x;
	ll_y = float_of_pdfobject ll_y;
	ur_x = float_of_pdfobject ur_x;
	ur_y = float_of_pdfobject ur_y }
  | _ -> invalid_arg "rectangle"

let mediabox_of_page page =
  rectangle page.Pdfdoc.mediabox

(* The following assumes that the /CropBox will allways be found in a
   [Pdf.Dictionary] stored at the [rest] structure element.  This should
   be a safe assumption, but fails if [Pdfdoc.pages_of_pagetree] doesn't
   take inheritance into account! *)
let cropbox_of_page page =
  match page.Pdfdoc.rest with
  | Pdf.Dictionary dict -> rectangle (List.assoc "/CropBox" dict)
  | _ -> raise Not_found 

type viewport =
  { mediabox : rectangle;
    cropbox : rectangle;
    rotation : Pdfdoc.rotation }

let viewport_of_page page =
  let mediabox = mediabox_of_page page in
  { mediabox = mediabox;
    cropbox = begin try cropbox_of_page page with Not_found -> mediabox end;
    rotation = page.Pdfdoc.rotate }

let viewports pdf =
  let pages = Pdfdoc.pages_of_pagetree pdf in
  List.map viewport_of_page pages

(* *********************************************************************

(* The following works at least for one (landscape) document.  I don't
   understand why, because 1024 is the number of pixels in the x direction... *)
let scale viewport =
  let cb = viewport.cropbox in
  (cb.ur_y -. cb.ll_y) /. 1024. 

let xy_of_point viewport p =
  let cb = viewport.cropbox in
  (cb.ll_x +. float p.Scribble.x *. scale viewport,
   cb.ll_y +. float p.Scribble.y *. scale viewport)

(* The following works at least for one (portrait) document.  I don't
   understand why, because 1024 is the number of pixels in the x direction... *)
let scale viewport =
  let cb = viewport.cropbox in
  (cb.ur_y -. cb.ll_y) /. 1220. 

let xy_of_point viewport p =
  let cb = viewport.cropbox in
  (cb.ll_x +. float p.Scribble.y *. scale viewport,
   cb.ur_y -. float p.Scribble.x *. scale viewport)

********************************************************************* *)

(* The following is a hack, based on two examples only!!! *)

let scale viewport =
  let cb = viewport.cropbox in
  match viewport.rotation with
  | Pdfdoc.Rotate0
  | Pdfdoc.Rotate180 ->
      (cb.ur_y -. cb.ll_y) /. 1220. 
  | Pdfdoc.Rotate90
  | Pdfdoc.Rotate270 ->
      (cb.ur_y -. cb.ll_y) /. 1024. 

let xy_of_point viewport p =
  let cb = viewport.cropbox in
  match viewport.rotation with
  | Pdfdoc.Rotate0 ->
      (cb.ll_x +. float p.Scribble.y *. scale viewport,
       cb.ur_y -. float p.Scribble.x *. scale viewport)
  | Pdfdoc.Rotate90 ->
      (cb.ll_x +. float p.Scribble.x *. scale viewport,
       cb.ll_y +. float p.Scribble.y *. scale viewport)
  | Pdfdoc.Rotate180 ->
      invalid_arg "xy_of_point: Rotate180"
  | Pdfdoc.Rotate270 ->
      invalid_arg "xy_of_point: Rotate270"

let ops_of_stroke viewport s =
  List.concat 
    [ [push_graphics_state;
       line_width
	 (float s.Scribble.stroke_header.Scribble.width *. scale viewport);
       rgb 0.6 0.2 0.2;
       line_cap_style Round_Cap;
       line_join_style Round_Join];
      stroke_path (List.map (xy_of_point viewport) s.Scribble.points);
      [pop_graphics_state] ]

let ops_of_scribble viewport s =
  List.concat (List.map (ops_of_stroke viewport) s.Scribble.strokes)

