(* $Id: scribble.ml 35 2008-10-19 14:34:00Z ohl $ *)
(* Copyright (C) 2008 by Thorsten Ohl <ohl@physik.uni-wuerzburg.de>

   This is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  *)  

let short s i =
  256 * Char.code s.[succ i] + Char.code s.[i]

let write_short s i n =
  s.[i] <- Char.chr (n mod 256);
  s.[succ i] <- Char.chr (n / 256)

type point =
    { p_b01    : string;  (* Always 0x00? *)
      y        : int;
      p_b04_05 : string;  (* Always 0x00? *)
      x        : int;
      p_b08_09 : string;  (* Always 0x00? *)
      p_b10_12 : string   (* Always 0x000000 for the first point,
                             0x803A1E, 0x88D32F or 0xE8021B for the middle points and
                             0xB80E16, 0xF0B115 or 0xB8A215 for the final point? *) }
          
let point_size = 12

let parse_point s offset =
  { p_b01    = String.sub s (offset + 0) 1;
    y        = short      s (offset + 1);
    p_b04_05 = String.sub s (offset + 3) 2;
    x        = short      s (offset + 5);
    p_b08_09 = String.sub s (offset + 7) 2;
    p_b10_12 = String.sub s (offset + 9) 3}

let unparse_point p =
  let s = String.create point_size in
  String.blit p.p_b01    0 s 0 1;
  write_short s 1 p.y;
  String.blit p.p_b04_05 0 s 3 2;
  write_short s 5 p.x;
  String.blit p.p_b08_09 0 s 7 2;
  String.blit p.p_b10_12 0 s 9 3

type stroke_header =
    { s_b01      : string;  (* Always 0x00? *)
      s_b02_03   : int;     (* Always 3?  16bit short or byte?*)
      s_b04_05   : string;  (* Always 0x0000? *)
      s_b06_09   : string;  (* Always 0x996CC142, 0xB4276842 or 0x5A5A3143? *)
      s_b10_17   : string;  (* Always 0x0000000000000000? *)
      width      : int;     (* 16bit short or byte? (the latter would suffice) *)
      s_b20_21   : string;  (* Always 0x0000? *)
      num_points : int;     (* more likely 16bit short than byte *)
      s_b24      : string   (* Always 0x00? *) }

let stroke_header_size = 24

let parse_stroke_header s offset =
  { s_b01      = String.sub s (offset +  0) 1;
    s_b02_03   = short      s (offset +  1);
    s_b04_05   = String.sub s (offset +  3) 2;
    s_b06_09   = String.sub s (offset +  5) 4;
    s_b10_17   = String.sub s (offset +  9) 8;
    width      = short      s (offset + 17);
    s_b20_21   = String.sub s (offset + 19) 2;
    num_points = short      s (offset + 21);
    s_b24      = String.sub s (offset + 23) 1 }

type stroke =
    { stroke_header : stroke_header;
      points : point list }

type header =
    { h_b01_36    : string;
      (* 0x010000000000000000000000000000000000000000000000000000000004000000050000 *)
      num_strokes : int;   (* more likely 16bit short than byte *)
      h_b39       : string (* Always 0x00? *) }

let header_size = 39

let parse_header s =
  { h_b01_36    = String.sub s  0 36;
    num_strokes = short      s 36;
    h_b39       = String.sub s 38  1 }

type scribble =
    { header : header;
      strokes : stroke list }

let rec parse_points s n offset =
  if n <= 0 then
    []
  else
    let point = parse_point s offset in
    point :: parse_points s (pred n) (offset + point_size)
              
let parse_stroke s offset =
  let header = parse_stroke_header s offset in
  { stroke_header = header;
    points = parse_points s header.num_points (offset + stroke_header_size) }
    
let rec parse_strokes s n offset =
  if n <= 0 then
    []
  else
    let stroke = parse_stroke s offset in
    let length = stroke_header_size + point_size * stroke.stroke_header.num_points in
    stroke :: parse_strokes s (pred n) (offset + length)
      
let parse s =
  let header = parse_header s in
  { header = header;
    strokes = parse_strokes s header.num_strokes header_size }

let print_stroke_header i h =
  Printf.printf "  Stroke #%3d:\n" i;
  Printf.printf "    byte  01    = 0x%s\n" (Hex.of_bytes h.s_b01);
  Printf.printf "    bytes 02-03 = %d\n" h.s_b02_03;
  Printf.printf "    bytes 04-05 = 0x%s\n" (Hex.of_bytes h.s_b04_05);
  Printf.printf "    bytes 06-09 = 0x%s\n" (Hex.of_bytes h.s_b06_09);
  Printf.printf "    bytes 10-17 = 0x%s\n" (Hex.of_bytes h.s_b10_17);
  Printf.printf "    width       = %d\n" h.width;
  Printf.printf "    bytes 20-21 = 0x%s\n" (Hex.of_bytes h.s_b20_21);
  Printf.printf "    num_points  = %d\n" h.num_points;
  Printf.printf "    byte 24     = 0x%s\n" (Hex.of_bytes h.s_b24)

let print_point j p = 
  Printf.printf "    %3d: 0x%s, %4d, 0x%s, %4d, 0x%s, 0x%s\n" j
    (Hex.of_bytes p.p_b01) p.y (Hex.of_bytes p.p_b04_05) p.x
    (Hex.of_bytes p.p_b08_09) (Hex.of_bytes p.p_b10_12)

let print_stroke i s =
  print_stroke_header i s.stroke_header;
  let j = ref 1 in
  List.iter (fun p -> print_point !j p; incr j) s.points

let print_header h = 
  Printf.printf "Header:\n";
  Printf.printf "    bytes 01-36 = %s\n" (Hex.of_bytes h.h_b01_36);
  Printf.printf "    num_strokes = %d\n" h.num_strokes;
  Printf.printf "    byte  39    = %s\n" (Hex.of_bytes h.h_b39)

let print scr =
  print_header scr.header;
  let i = ref 1 in
  List.iter (fun r -> print_stroke !i r; incr i) scr.strokes

let strokes scr =
  List.iter
    (fun r ->
      Printf.printf "\n";
      List.iter (fun p -> Printf.printf " %4d %4d\n" p.x p.y) r.points)
    scr.strokes
